/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) cpu.c: version 25.1 created on 11/27/91 at 15:38:57	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)cpu.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
/* cpu.c */

#include "sys/types.h"
#include "sys/sbus.h"
#include "spm_debug.h"
#include "cpu_method.h"
#include "spm.h"
#include "slotdefs.h"
#include "misc.h"
#include "dev.h"

#define	MAX_CPUS	128

typedef cpu_data_t	*element;

/*
 * cpu_list contains an entry for every cpu in the system.
 * active_cpus contains an entry for every cpu which is not deconfigured.
 * The entries in active_cpus are pointers to elements of cpu_list.
 */
static	element		cpu_list[MAX_CPUS + 1];
static	element 	*end_of_list = cpu_list;
static	element		*cur_element;

static	element		active_cpus[MAX_CPUS + 1];
static	element 	*end_of_active = active_cpus;

void
cpu_enroll(new_element)
element	new_element;
{
#if ASDEBUG==YES
	method_t	*m = new_element->cd_methods;

	ASSERT(m->check_printf);
	ASSERT(m->send_clock);
	ASSERT(m->load);
	ASSERT(m->set_pm_id);
	ASSERT(m->prep_for_syms);
	ASSERT(m->load_syms);
	ASSERT(m->prep_boot);
	ASSERT(m->start);
	ASSERT(m->stop);
	ASSERT(m->check_tdb);
	ASSERT(m->brkpt_check);
	ASSERT(m->show_trap);
	ASSERT(m->prepare_to_go);
	ASSERT(m->go);
	ASSERT(m->step);
	ASSERT(m->jump);
	ASSERT(m->get_byte);
	ASSERT(m->put_byte);
	ASSERT(m->get_short);
	ASSERT(m->put_short);
	ASSERT(m->get_long);
	ASSERT(m->put_long);
	ASSERT(m->get_named_reg);
	ASSERT(m->put_named_reg);
	ASSERT(m->display_own);
	ASSERT(m->display_slot);
	ASSERT(m->display_dis);
	ASSERT(m->symbolic_version);
	ASSERT(m->find_bp);
	ASSERT(m->backtrace);
	ASSERT(m->valid_fp);
	ASSERT(m->set_brkpt);
	ASSERT(m->clear_one_brkpt);
	ASSERT(m->clear_brkpts);
	ASSERT(m->display_brkpts);
	ASSERT(m->check_tags);
	ASSERT(m->solo);
	ASSERT(m->deconfigure);
	ASSERT(m->configure);
	ASSERT(m->cpu_copyout);
	ASSERT(m->cpu_copyin);
	ASSERT(m->cpu_malloc);
	ASSERT(m->cpu_zero);
	ASSERT(m->check_coffdata);
	ASSERT(m->check_magic);
	ASSERT(m->pm_first);
	ASSERT(m->print_vtop);
	ASSERT(m->mem_stripe_init);
	ASSERT(m->build_page_table);
	ASSERT(m->get_u_addr);

	ASSERT(end_of_list != cpu_list + MAX_CPUS);
#endif

	*end_of_list = new_element;
	cur_element = end_of_list;
	end_of_list++;

	*end_of_active++ = new_element;
}

method_t *
cpu_cur_method()
{
	ASSERT(cur_element);		/* there is a current element */
	ASSERT(*cur_element);		/* which points to cpu information */
	ASSERT((*cur_element)->cd_methods); /* which points to methods */
	return((*cur_element)->cd_methods);
}

element
cpu_get_cur()
{
	ASSERT(cur_element);
	ASSERT(*cur_element);
	return(*cur_element);
}

cpu_set_cur(desired)
element	desired;
{
	for (cur_element = cpu_list; *cur_element; cur_element++)
		if (*cur_element == desired)
			return;
	ASSERT(0);
}

uint
cpu_cur_sns()
{
	ASSERT(cur_element);
	ASSERT(*cur_element);
	return((*cur_element)->cd_unit.w.sns);
}

/* 
 * hanna FIX: put a nice comment here, as well as throughout code
 * describing how and why this works.
 * how syntax is slot/subslot.
 * how this could later be turned in to some generic parsing routine.
 * hanna FIX: also, talk to james about getting the slot/subslot size
 * settled. is it a char or nibble!
 */
uint
cpu_set_slot(str, sns)
char		*str;
register uint	sns;
{
	register element	*ep;
	iunit_t			unit;

	if (str) {
		if (get_device_info(str, &unit, NULL) != DV_SLOT) {
			printf("Expected <Slot> or <Slot/Subslot>. Got '%s'\n",
			  str);
			return(0);
		}
		sns = unit.w.sns;
	}
	/* else, str was NULL.  sns was supplied */

	for (ep = cpu_list; *ep; ep++)
		if ((*ep)->cd_unit.w.sns == sns) {
			cur_element = ep;
			return(1);
		}

	printf("No enrolled cpu for slot %u", SLOT_FM_SNS(sns));
	if (SUB_SLOT_FM_SNS(sns) != NO_SUB_SLOT)
		printf(", subslot %u", SUB_SLOT_FM_SNS(sns));
	put_char('\n');
	return(0);
}

/*
 * remove current element from active_cpus
 */
cpu_deactivate()
{
	register element *cpu = active_cpus;

	/* find it */
	for ( ; *cpu && *cpu != *cur_element; cpu++);

	if (!*cpu) /* not found */
		return(0);

	/* shift remainder left */
	do
		*cpu = *(cpu + 1);
	while (*cpu++);
	return(1);
}

/*
 * add current element to active_cpus
 * this can be simply added to the end of the active list.
 */
cpu_activate()
{
	register element	*cpu = active_cpus;

	ASSERT(cur_element);
	ASSERT(*cpu); /* must be at least an SPM active */

	/* find end of active_cpus */
	while (*(++cpu));
	*cpu = *cur_element;
}

/************** CPU ROUTINES THAT SCAN ALL ACTIVE CPUS *******/

/*
 * cpu_all_check_printf() - Check all cpu's that might need printf service
 */
void
cpu_all_check_printf()
{
	element	*saved_element = cur_element;

	for (cur_element = active_cpus; *cur_element; cur_element++)
		CPU_check_printf();
	cur_element = saved_element;
}

uint
cpu_all_send_clock(upkern_id)
uint	upkern_id;
{
	element	*saved_element = cur_element;
	uint	upkern_ok = 0;

	for (cur_element = active_cpus; *cur_element; cur_element++)
		if (CPU_send_clock(upkern_id))
			upkern_ok = 1;
	cur_element = saved_element;
	return(upkern_ok);
}

/*
 * cpu_all_prep_boot -- prepare each CPU
 *	function should return ~0 on error, 0 for no-op, or entry point
 */
uint
cpu_all_prep_boot(source_dev, umod_flag)
char	*source_dev;
uint	umod_flag;
{
	register element	*saved_element = cur_element;
	register uint		entry = 0;
	uint			ret;

	for (cur_element = active_cpus; *cur_element; cur_element++) {
		ret = CPU_prep_boot(source_dev, umod_flag);
		if (entry == 0)
			entry = ret;
		if (ret == ~0)
			break;
	}
	cur_element = saved_element;
	return (entry);
}

void
cpu_all_start()
{
	element	*saved_element = cur_element;

	for (cur_element = active_cpus; *cur_element; cur_element++)
		CPU_start();
	cur_element = saved_element;
}

/*
 * cpu_all_stop()
 */
void
cpu_all_stop()
{
	register uint	wrcntl0, wrcntl1;
	register element *saved_element;
	uint		x;

	x = splhi();
	wrcntl0 = *WRCNTL0;
	wrcntl1 = *WRCNTL1;
	*WRCNTL1 = wrcntl1 | WR1_RRDY;
	*WRCNTL0 = wrcntl0 | WR0_BUS_FREEZE;	/* stop all intelligent boards*/
	*WRCNTL0 = wrcntl0;
	*WRCNTL1 = wrcntl1;
	splx(x);

	saved_element = cur_element;
	for (cur_element = active_cpus; *cur_element; cur_element++)
		CPU_stop();
	cur_element = saved_element;
}

uint
cpu_all_check_tdb()
{
	element	*saved_element = cur_element;

	for (cur_element = active_cpus; *cur_element; cur_element++)
		if (CPU_check_tdb())
			return(1);
	cur_element = saved_element;
	return(0);
}

/* build kernel and IOPM page tables */
void
cpu_all_build_page_tables()
{
	element	*saved_element = cur_element;
	static	int	did_build_page_tables = 0;
	
	ASSERT(!did_build_page_tables);

	for (cur_element = active_cpus; *cur_element; cur_element++)
		CPU_build_page_table();

	cur_element = saved_element;
	did_build_page_tables = 1;
}

uint
cpu_all_prepare_to_go()
{
	element	*saved_element = cur_element;

	for (cur_element = active_cpus; *cur_element; cur_element++)
		if (!CPU_prepare_to_go())
			return(0); /* something has gone wrong */
	cur_element = saved_element;
	return(1);
}

void
cpu_all_go()
{
	element	*saved_element = cur_element;

	for (cur_element = active_cpus; *cur_element; cur_element++)
		CPU_go();
	cur_element = saved_element;
}

void
cpu_all_clear_brkpts()
{
	element	*saved_element = cur_element;

	for (cur_element = active_cpus; *cur_element; cur_element++)
		CPU_clear_brkpts();
	cur_element = saved_element;
}

void
cpu_all_display_brkpts()
{
	element	*saved_element = cur_element;

	for (cur_element = active_cpus; *cur_element; cur_element++)
		CPU_display_brkpts();
	cur_element = saved_element;
}

/*
 * cpu_all_set_pm_id - initialize the pm_id for each pm which is to be loaded
 */
cpu_all_set_pm_id()
{
	element	*saved_element = cur_element;

	for (cur_element = active_cpus; *cur_element; cur_element++)
		CPU_set_pm_id();
	cur_element = saved_element;
}

/* hanna FIX: these routines, ask joe if they can be methodized */

static uint	num_pm_in_sys;
static ushort	sns_array[SBUS_MAX_NUM_PM];
static uchar	pm_id_base[SBUS_NUM_SLOT];

/* 
 * called -- in order of PMs in the machine, at start of load
 * this fills in an array to translate pm_id's to slot id's and viceeversee.
 * the number of PMs in the system is also tallied.
 */
uint
set_pm_id() 
{
	uint	sns = cpu_cur_sns();

	ASSERT(num_pm_in_sys < SBUS_MAX_NUM_PM);
	/* initialize pm_id to sns translation arrays */
	if (num_pm_in_sys == 0) {
		register int	num_pm;

		for (num_pm = NEL(sns_array); --num_pm >= 0; )
			sns_array[num_pm] = SNS_NO_PM;
		for (num_pm = NEL(pm_id_base); --num_pm >= 0; )
			pm_id_base[num_pm] = SNS_NO_PM;
	}
	sns_array[num_pm_in_sys] = sns;
	/* declan how does this work for slot A of a dpm ? */
	if (SUB_SLOT_FM_SNS(sns) == NO_SUB_SLOT)
		pm_id_base[SLOT_FM_SNS(sns)] = num_pm_in_sys;
	return(num_pm_in_sys++);
}

/*
 * The sns_array is indexed by pm_id and contains an sns, a combination
 * of Slot aNd Subslot.  The caller, can then extract whatever it
 * wants from that information.
 */
uint
sns_fm_pm_id(pm_id)
uint	pm_id;
{
	return (pm_id < num_pm_in_sys ? sns_array[pm_id] : SNS_NO_PM);
}

/*
 * The pm_id_base array is indexed by slot and holds the base
 * pm id for that slot.  To compute the pm_id, add the subslot
 * information to that base and return.
 * returns ~0 for invalid sns
 */
uint
pm_id_fm_sns(sns)
uint	sns;
{
	uint	slot = SLOT_FM_SNS(sns);

	return (
	  (slot >= SBUS_NUM_SLOT || pm_id_base[slot] >= num_pm_in_sys) ? ~0 :
	   (pm_id_base[slot] + (SUB_SLOT_FM_SNS(sns) == NO_SUB_SLOT)));
}
