/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) rtc.c: version 25.1 created on 11/27/91 at 15:37:23	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)rtc.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/

/*
 *	rtc.c -- real time clock interface
 *
 *
 *	rtc_stime(date)
 *	time_t date;
 *		Takes the input in seconds from the epoch, tranlates
 *		it into year, hour, minute, etc. and sets the
 *		real time clock.  On successful completion, a 0 is
 *		returned, -1 on error.
 *
 *	time_t rtc_time()
 *		Reads the real time clock, translates it into seconds
 *		from the epoch and returns that data.
 *
 *	Error condidions:
 *		If the RTC is not installed, the SPM will panic.
 *
 *	Note:  The epoch is 1/1/1970.
 *
 *	The SPM will keep a count of seconds elapsed since 1/1/1970 GMT
 *	in spm_mem.h. The kernel will maintain a seconds count which, on
 *	command to the SPM, will be set as the new time and date.
 *
 */

#include "misc.h"
#include "sys/types.h"
#include "spm_debug.h"
#include "spm.h"
#include "rtc.h"

#define	leapyear(Y)	((Y) % 4 == 0)			/* works until 2100 */
#define	dysize(Y)	(leapyear(Y) ? 366 : 365)

extern uint	n_seconds, time_stamp_window;

/* Days of the month for regular and leap years */
static int	dmsize[] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };
static int	lmsize[] = { 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

static char	*days[] = {
	"???",
	"Sun",
	"Mon",
	"Tue",
	"Wed",
	"Thu",
	"Fri",
	"Sat"
};

static char	*months[] = {
	"???",
	"Jan",
	"Feb",
	"Mar",
	"Apr",
	"May",
	"Jun",
	"Jul",
	"Aug",
	"Sep",
	"Oct",
	"Nov",
	"Dec"
};

/* Calculate seconds since the epoch from clock chip */

time_t
rtc_time()
{
	register time_t	timebuf;
	register int	d0, month, year;
	register int	*msize;
	struct rtc	rtc;

	from_rtc(&rtc);

	month = rtc.month - 1;
	year = rtc.year;
	if (year < 70)		/* wraparound: only works up to Dec 31, 2069 */
		year += 100;
	year += 1900;

	timebuf = rtc.date - 1;			/* Calculate days so far */
	msize = leapyear(year) ? lmsize : dmsize;
	for (d0 = 0; d0 < month; d0++)
		timebuf += msize[d0];

	for (d0 = 1970; d0 < year; d0++)	/* add up days since epoc */
		timebuf += dysize(d0);

	timebuf *= 24;				/* convert from days to hours */
	timebuf += rtc.hour;

	timebuf *= 60;				/* hours to minutes */
	timebuf += rtc.min;

	timebuf *= 60;				/* minutes to seconds */
	timebuf += rtc.sec;

	return (timebuf);
}

rtc_stime(date)
register unsigned long	date;
{
    	register uint	year, val, month;
	register int	*msize;
	struct rtc	rtc;

	/*
	 * generate hours:minutes:seconds
	 */
	rtc.sec  = date % 60;		/* secs	*/
	date /= 60;
	rtc.min  = date % 60;		/* minutes */
	date /= 60;
	rtc.hour = date % 24;		/* hours */
	date /= 24;

	/*
	 * date is the day number.
	 * generate day of the week.
	 * The addend is 4 mod 7 (1/1/1970 was Thursday)
	 */
	rtc.day = (date + 4) % 7 + 1;

	/*
	 * year number
	 */
	for (year = 1970; date >= (val = dysize(year)); year++)
		date -= val;

	rtc.year = year % 100;

	/*
	 * generate month
	 */
	msize = leapyear(year) ? lmsize : dmsize;
	for (month = 0; date >= msize[month]; month++)
		date -= msize[month];

	rtc.date = date + 1;
	rtc.month = month + 1;
	to_rtc(&rtc);
}

/*
 * print_time_stamp
 */

print_time_stamp(always)
uint	always;
{
	static uint last_time_stamp;

	if (!always && (n_seconds - last_time_stamp <= time_stamp_window))
		return;

	last_time_stamp = n_seconds;
	prnt_time_stamp();
}

prnt_time_stamp()
{
	struct rtc	rtc;
	uint		year;

	from_rtc(&rtc);
	year = rtc.year + 1900;
	if (year < 1970)
		year += 100;
	if (rtc.day > NEL(days))
		rtc.day = 0;
	if (rtc.month > NEL(months))
		rtc.month = 0;

	printf("%s %s %u %02u:%02u:%02u GMT %u\n", days[rtc.day],
	  months[rtc.month], rtc.date, rtc.hour, rtc.min, rtc.sec, year);
}

#if 0
/*
 * get_range_str -- return a number between high and low from the given string
 */

get_range_str(str, low, high, name)
char	*str, *name;
int	low, high;
{
	int	val;

	if ((val = atoi(str)) < low || val > high)
		err("%s value out of range! (%d to %d)", name, low, high);

	return (val);
}
#endif


/*
 * from_rtc -- loads a rtc structure with the contents of the clock, converted
 *		to binary from BCD, but skipping the control field
 */

from_rtc(rtcp)
struct rtc	*rtcp;
{
	register uchar	*srcp, *destp;
	register int	n;
	register uint	v;

	srcp = &RTC->sec;
	destp = &rtcp->sec;
	n = sizeof(*RTC) - 1;
	v = splhi();
	RTC->control |= RTC_READ_BIT;			/* halt updating */

	while (--n >= 0)
		*destp++ = *srcp++;

	RTC->control &= ~(RTC_WRITE_BIT | RTC_READ_BIT);	/* done */
	splx(v);

	srcp = &rtcp->sec;				/* convert to binary */
	n = sizeof(*RTC) - 1;
	while (--n >= 0) {
		v = *srcp;
		*srcp++ = (v >> 4) * 10 + (v & 0xf);
	}
}

/*
 * to_rtc -- loads a rtc structure into the the clock, converting
 *		from binary to BCD, but skipping the control field
 */

to_rtc(rtcp)
struct rtc	*rtcp;
{
	register uchar	*srcp, *destp;
	register int	n;
	register uint	v;

	srcp = &rtcp->sec;				/* convert to BCD */
	n = sizeof(*RTC) - 1;
	while (--n >= 0) {
		v = *srcp;
		*srcp++ = ((v / 10) << 4) | (v % 10);
	}

	srcp = &rtcp->sec;
	destp = &RTC->sec;
	n = sizeof(*RTC) - 1;
	v = splhi();
	RTC->control |= RTC_WRITE_BIT;		/* set time (no updates) */

	while (--n >= 0)
		*destp++ = *srcp++;

	RTC->control &= ~(RTC_WRITE_BIT | RTC_READ_BIT);	/* done */
	splx(v);
}

/*
 * get_rtc_cal -- return the scaled clock chip calibration value
 */

get_rtc_cal()
{
	int	ctrl, val;

	ctrl = RTC->control;
	val = (ctrl & RTC_CAL_BITS) * RTC_SCALE;
	if (ctrl & RTC_SIGN_BIT)
		val = -val;

	return (val);
}

/*
 * set_rtc_cal -- set and return the scaled clock chip calibration value
 */

set_rtc_cal(val)
register int	val;
{
	int	sign;
	uint	spl;

	if (val < 0) {
		val = -val;
		sign = RTC_SIGN_BIT;
	}
	else
		sign = 0;

	val = (val + RTC_SCALE / 2) / RTC_SCALE;	/* round it */

	if (val > RTC_CAL_BITS)
		val = RTC_CAL_BITS;			/* clip it */
	val |= sign;
	
	spl = splhi();
	RTC->control = val | RTC_WRITE_BIT;		/* set it */
	RTC->control = val;
	splx(spl);

	return (get_rtc_cal());
}
