/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) symbol.c: version 25.1 created on 11/27/91 at 15:30:33	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)symbol.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
/*
 * symbol.c
 */

#include	"sys/types.h"
#include	"sys/spm_mem.h"
#include	"sys/sysarix.h"
#include	"ctype.h"
#include	"symbol.h"
#include	"misc.h"
#include	"coff_hdr.h"
#include	"polyio.h"
#include	"spm_debug.h"

/* #define TIME_LOAD_SYMS		/* print symbol load time */

#define SYM_BIND_LIMIT	0x7ffff		/* symbol binding range limit	*/
#define SYM_LOW_LOCKOUT	0x30000		/* don't expand addrs less than this */
#define SYM_ARRAY_NUM	512
#define SYM_ARRAY_BYTES	(SYM_ARRAY_NUM * sizeof(kern_sym_t))
#define SYM_CACHE_NUM	8		/* size of symbol cache		*/
#define MERGE_CHUNKS	16		/* number of merge sort chunks	*/

typedef struct _sym_cache {
	struct _sym_cache	*next;
	struct _sym_cache	*prev;
	uint			range;
	ksym_t			sym;
} sym_cache_t;

typedef struct _merge_chunk {
	kern_sym_t	*tptr;		/* SPM table pointer		*/
	kern_sym_t	*tstart;	/* SPM table start		*/
	kern_sym_t	*tend;		/* SPM table end		*/
	uchar		*kvaddr;	/* kernel virtual temp address	*/
	int		bytes;		/* number of bytes in kv	*/
} merge_chunk_t;

static kern_sym_t	*ksyms;
static int		nksyms;
static sym_cache_t	ksymcache[SYM_CACHE_NUM];
static sym_cache_t	*ksymhead = ksymcache;

char	*symstr();

extern char	*strcpy(), *strncpy();
extern uint	km_allocate();
extern void	qsort();
extern uint	Atox();
extern uchar	*km_map();
extern char	*comm_args[];


/*
 * getsym -- returns a pointer to a ksym_t, given a kernel virtual address
 *		in the kernel symbol table
 */

ksym_t *
getsym(vaddr)
kern_sym_t	*vaddr;
{
	kern_sym_t	*kp;
	uint		saved_map = iomap_save();
	static ksym_t	sym;

	/* can use kp instead of copy_from_km 'cuz we've aligned the ksyms */
	kp = (kern_sym_t *)km_map((uchar *)vaddr);
	sym.value = kp->value;
	copy_from_km((uchar *)kp->name, (uchar *)sym.name, sizeof(sym.name)-1);
	sym.name[sizeof(sym.name) - 1] = '\0';
	iomap_restore(saved_map);

	return (&sym);
}

/*	Symbol display corresponds to the 'symbol' command.	*/

symdsp(argc)
int	argc;
{
	register kern_sym_t	*kp;
	register ksym_t		*sym;
	register int		n, len;
	uint			sym_val;

	if (argc >= 1) {
		/*
		 * Print the symbol(s) that match the name string
		 */

		len = strlen(comm_args[1]);

		if (comm_args[1][0] == '$') {		/* remove a $ prefix */
			strcpy(comm_args[1], comm_args[1] + 1);
			--len;
		}

		if (comm_args[1][len - 1] != '*') {
			sym_val = Atox(comm_args[1]);	/* do exact match */
			printf("%s = 0x%08x\n", symstr(sym_val), sym_val);
			return;
		}
							/* do partial match */
		comm_args[1][--len] = '\0';
		for (kp = ksyms, n = nksyms; --n >= 0; kp++) {
			if (get_possible_char() >= 0)
				break;
			sym = getsym(kp);
			if (strncmp(sym->name, comm_args[1], len))
				continue;

			printf("%-31.31s 0x%08x\n", sym->name, sym->value);
		}
		return;
	}

	/*
	 * print all symbols
	 */

	for (kp = ksyms, n = 0; n < nksyms; n++, kp++) {
		sym = getsym(kp);
		printf("%-31.31s%08x%c", sym->name, sym->value,
		  ((n & 1) ? '\n' : ' '));
		if (get_possible_char() >= 0)
			break;
	}
	printf("\n");
}


/*
 * used_cache -- just used this cache entry, update its LRU status
 */

static void
used_cache(cp)
register sym_cache_t	*cp;
{
	if (cp != ksymhead) {
		/* unlink from old position */
		cp->next->prev = cp->prev;
		cp->prev->next = cp->next;
		/* relink to head of list */
		cp->next = ksymhead;
		cp->prev = ksymhead->prev;
		cp->prev->next = cp;
		cp->next->prev = cp;
		ksymhead = cp;
	}
}


/*
 * cache_add -- add the given ksym, found at kvaddr, to the cache
 */

static sym_cache_t *
cache_add(ksym, kvaddr)
ksym_t		*ksym;
kern_sym_t	*kvaddr;
{
	register sym_cache_t	*cp;

	cp = ksymhead->prev;	/* reuse the least recently used cache entry */
	ksymhead = cp;
	cp->sym = *ksym;	/* save the symbol */

	/* we can always get lowp+1 because nksyms doesn't include the ~0 sym */
	cp->range = getsym(kvaddr + 1)->value - cp->sym.value;

	return (cp);
}


/*
 * atov -- ascii to value function
 *	If an exact match to string s is found, put the symbol's value in xp,
 *	and return non-zero, else return 0.
 */

atov(xp, name)
uint		*xp;
register char	*name;
{
	register kern_sym_t	*kp;
	register ksym_t		*sym;
	register sym_cache_t	*cp;
	register int		n = nksyms;

	if (n <= 0)
		return (0);

	/* search cache for name */
	cp = ksymhead;
	do {
		if (strcmp(cp->sym.name, name) == 0) {
			used_cache(cp);
			*xp = cp->sym.value;
			return (1);
		}
	} while ((cp = cp->next) != ksymhead);

	/* linear search of symbol table */
	for (kp = ksyms; --n >= 0; kp++) {
		sym = getsym(kp);
		if (strcmp(sym->name, name) == 0) {
			cp = cache_add(sym, kp);
			*xp = cp->sym.value;
			return (1);
		}
	}
	return (0);
}

/*
 * findsym -- returns a pointer to the symbol with the given value, or the
 *	one with the next lower value (needs end symbols to always work),
 *	or NULL.
 *
 *	Requires that value be the first field in kern_sym_t.
 */

ksym_t *
findsym(value)
uint	value;
{
	register kern_sym_t	*lowp, *highp, *midp;
	register sym_cache_t	*cp;
	register int		diff;
	register uint		val, midv, saved_map;

	ASSERT((uint)&ksymcache->sym == (uint)&ksymcache->sym.value);

	if (nksyms <= 0)
		return (NULL);

	/*
	 * check cached symbols for a match
	 */
	cp = ksymhead;
	do {
		if (cp->sym.value <= value && cp->sym.value+cp->range > value) {
			used_cache(cp);
			return (&cp->sym);		/* was cached */
		}
	} while ((cp = cp->next) != ksymhead);

	saved_map = iomap_save();

	lowp = ksyms;
	highp = &ksyms[nksyms - 1];
	val = value >> 1;

	do {
		midp = lowp + (((highp - lowp) + 1) >> 1);
		midv = *(uint *) km_map((uchar *)midp);
							/* avoid wraparound */
		if ((diff = (midv >> 1) - val) == 0)
			diff = midv - value;		/* same to LSB */

		if (diff > 0)
			highp = midp - 1;
		else if (diff < 0)
			lowp = midp;
		else {
			lowp = midp;			/* exact match */
			break;
		}
	} while (lowp < highp);

	if (highp <= ksyms)
	{
		iomap_restore(saved_map);
		return (NULL);				/* too small */
	}

	cp = cache_add(getsym(lowp), lowp);

	iomap_restore(saved_map);

	return (&cp->sym);
}


/*
 * symstr -- converts value to string (symbol name)
 *	returns a string with the best symbolic representation of the address
 *
 *	The symbol can have three representations:
 *		exact match:	symbol_name
 *		close match:	symbol_name+offset
 *		no/bad match:	0xValue
 *
 *	A bad match occurs when the offset is greater than SYM_BIND_LIMIT,
 *	or when not an exact match and the address is less than SYM_LOW_LOCKOUT.
 */

char *
symstr(x)
register uint	x;
{
	register ksym_t	*sp;
	register uint	delta;
	static char	str[KSYM_NAME_LEN + 12];

	sp = findsym(x);
	if (sp == NULL || (delta = x - sp->value) > SYM_BIND_LIMIT ||
	    (x < SYM_LOW_LOCKOUT && delta) || sp->name[0] == '\0')
		sprintf(str, "0x%04x", x);		/* no/bad match */
	else if (delta == 0) {
		strncpy(str, sp->name, KSYM_NAME_LEN - 1); /* exact match */
		str[KSYM_NAME_LEN - 1] = '\0';
	}
	else
		sprintf(str, "%s+0x%x", sp->name, delta);

	return (str);
}

/*
 * sym_compare -- comparison function for qsort that avoids wraparound
 */

sym_compare(p1, p2)
kern_sym_t	*p1, *p2;
{
	int	value;

	if (value = (int)(p1->value >> 1) - (int)(p2->value >> 1))
		return (value);
	if (value = p1->value - p2->value)	/* identical to LSB */
		return (value);
	return (strcmp(p1->value, p2->value));	/* identical values */
}

/*
 * spm_mem_syms -- put the symbol into Spm_Mem
 */

spm_mem_syms()
{
	Spm_Mem->symbols = ksyms;
	Spm_Mem->num_syms = nksyms;
}


/*
 * init_cache -- initialize the kernel symbol cache
 */

static void
init_cache()
{
	register sym_cache_t	*cp = ksymcache - 1;
	register int		n = SYM_CACHE_NUM;

	do {
		++cp;
		(cp->next = cp + 1)->prev = cp;
	} while (--n > 0);
	ksymcache->prev = cp;
	cp->next = ksymcache;
}

/*
 * fill_chunk -- fill the SPM buffer of a merge sort from kernel virtual
 *		returns non-zero if nothing left, else zero
 */

static uint
fill_chunk(mp)
register merge_chunk_t	*mp;
{
	register int	nbytes;

	ASSERT(mp->tend > mp->tstart);
	ASSERT(mp->bytes % sizeof(kern_sym_t) == 0);

	if (mp->bytes <= 0)
		return (1);

	nbytes = (mp->tend - mp->tstart) * sizeof(kern_sym_t);
	nbytes = MIN(nbytes, mp->bytes);
	mp->tptr = mp->tend - nbytes / sizeof(kern_sym_t);
	copy_from_km(mp->kvaddr, (uchar *)mp->tptr, nbytes);
	mp->kvaddr += nbytes;
	mp->bytes -= nbytes;

	return (0);
}


/*
 * load_syms -- load the kernel symbols into the SPM's memory, the strings
 *		into kernel memory, and reserve space for the symbol table
 *		(which will be written into kernel memory after sorting)
 *
 * Arguments:
 *	ap	-- polyopen'ed file pointer
 *	hd	-- contains COFF header info
 *
 * Returns 0 on error, else non-zero.
 */

#define xby	xfer_bytes	/* for convenience */
#define xbuf(ap, buf, size)	xfer_bytes(TO_BUF, ap, buf, size)
#define xcpu(ap, buf, size)	xfer_bytes(TO_CPU, ap, buf, size)

uint
load_kern_syms(ap, hd)
struct args_buf		*ap;
struct coff_hdr_data	*hd;
{
	register kern_sym_t	*kp, *kend;	/* SPM table ptr and its end */
	register uchar		*vaddr;
	register merge_chunk_t	*mp;
	register int		i, n, v;
	register uint		val;
	register int		sym_cnt;	/* total symbol count	*/
	int			sym_bytes;	/* total symbol bytes	*/
	register int		merge_cnt;	/* merge chunk count	*/
	kern_sym_t		tbl[SYM_ARRAY_NUM];	/* SPM table	*/
	int			str_tbl_bytes;	/* string table size	*/
	uint			km_bytes;	/* total km allocated	*/
	uchar			*kv_addr;
	uchar			*str_table;
	uchar			*str_start;
	kern_sym_t		*mout_start;	/* merge output start	*/
	struct syment		lsym, ltmp;
	merge_chunk_t		msort[MERGE_CHUNKS];
	static uchar		last_str[] = "~0";
	extern int		end;
#ifdef TIME_LOAD_SYMS
	extern uint		int_counter;
	uint			start_ticks = int_counter;
#endif

	ASSERT((uint)&msort[MERGE_CHUNKS] > (uint)&end + 0x1000);
	init_cache();

	if ((n = hd->nsyms) == 0)
		return (1);

	/*
	 * allocate enough kernel memory for the worst case
	 *
	 * Usage:
	 *	1) The strings that fit in a syment structure will be stored at
	 *	   the beginning.
	 *	2) Any merge chunks will be temporarily stored in decending
	 *	   order, starting at the high end.
	 *
	 * low addr						   high addr
	 * +---------------------------------------------------------------+
	 * |syment strings.....                |chunk N|...|chunk 1|chunk 0|
	 * +---------------------------------------------------------------+
	 */
	km_bytes = val = n * (KSYM_NAME_LEN + sizeof(kern_sym_t));
	str_start = vaddr = (uchar *)km_allocate(val, sizeof(long));
	kv_addr = vaddr + val - SYM_ARRAY_BYTES;	/* chunk 0 kv addr */

	mp = msort;
	bzero((caddr_t)mp, sizeof(msort));
	merge_cnt = sym_cnt = 0;
	kp = tbl;
	kend = &kp[SYM_ARRAY_NUM];

	for (i = hd->nsyms; --i >= 0; ) {
		if (!xbuf(ap, (uchar *)&lsym, sizeof(lsym))) {
			printf("EOF on lsym\n");
			break;
		}

		/* eat up auxillary entries */
		for (n = lsym.n_numaux; --n >= 0 && --i >= 0; )
			if (!xbuf(ap, (uchar *)&ltmp, sizeof(ltmp))) {
				printf("EOF on ltmp\n");
				goto err;		/* can't break */
			}

		if (*lsym.n_name == '.' || lsym.n_scnum > 3 ||
		    (lsym.n_scnum <= 0 && lsym.n_scnum != -1))
			continue;	/* reject file and section info */

		/*
		 * else, good symbol
		 */
		kp->value = lsym.n_value;
		if (lsym.n_zeroes == 0)
			kp->name = (char *)lsym.n_offset; /* in string table */
		else {
			/* can overflow lsym.n_name because already got value */
			lsym.n_name[sizeof(lsym.n_name)] = '\0';
			n = strlen(lsym.n_name) + 1;	  /* in syment */
if (n <= 1) printf("load_syms: 0x%08x has null name!\n", kp->value);
			kp->name  = (char *)vaddr;
			copy_to_km((uchar *)lsym.n_name, vaddr, (uint)n);
			vaddr += n;
		}
		++sym_cnt;

		if (++kp < kend)
			continue;
		/*
		 * table overflow; sort it and dump it into kmem
		 */
		kp = tbl;
		qsort(kp, SYM_ARRAY_NUM, sizeof(*kp), sym_compare);
		copy_to_km((uchar *)kp, kv_addr, SYM_ARRAY_BYTES);
		/*
		 * store start and size of this chunk in msort
		 */
		mp->kvaddr = kv_addr;
		mp->bytes = SYM_ARRAY_BYTES;
		kv_addr -= SYM_ARRAY_BYTES;
		++merge_cnt;
		if (++mp <= &msort[MERGE_CHUNKS - 1])
			continue;
		/*
		 * too many symbols to merge!
		 */
		printf("Partial symbol table! (only %d syms)\n",
		  MERGE_CHUNKS * SYM_ARRAY_NUM);
		/*
		 * eat the other symbols
		 */
		for ( ; i > 0; i -= n) {
			n = MIN(i, sizeof(tbl) / sizeof(ltmp));
			if (!xbuf(ap, (uchar *)kp, n * sizeof(ltmp))) {
				printf("EOF in eat loop\n");
				break;
			}
		}
		++i;				/* indicate failure */
		break;
	}
	if (i >= 0) {
err:		km_freen((uint)str_start, km_bytes);
		return (0);				/* failed */
	}

	/* alloc a symbol with maximum uint value */
	kp->name  = (char *)vaddr;
	kp->value = ~0;
	copy_to_km(last_str, vaddr, sizeof(last_str));
	vaddr += sizeof(last_str);
	++kp;
	++sym_cnt;

	sym_bytes = sym_cnt * sizeof(kern_sym_t);
				/* sizeof(kern_sym_t) must be a power of 2 */
	vaddr = (uchar *)ALIGN(vaddr, sizeof(kern_sym_t));
	ksyms = (kern_sym_t *)vaddr;

	/*
	 * The chunks (if any) will be sorted and stored:
	 *
	 * low addr						   high addr
	 * +---------------------------------------------------------------+
	 * |syment strings|sorted kern_sym table|string table|freed...	   |
	 * +---------------------------------------------------------------+
	 */
	if (merge_cnt == 0) {
		/*
		 * everything fit in the table -- sort and store it
		 */
		kp = tbl;
		qsort(kp, sym_cnt, sizeof(*kp), sym_compare);
		copy_to_km((uchar *)kp, vaddr, sym_bytes);
		vaddr += sym_bytes;
	}
	else {
		/*
		 * must merge -- sort and save the table's contents into kmem
		 */
		n = kp - tbl;
		kp = tbl;
		qsort(kp, n, sizeof(*kp), sym_compare);
		n *= sizeof(kern_sym_t);
		copy_to_km((uchar *)kp, kv_addr, n);
		/*
		 * store start and size of this chunk in msort
		 */
		mp->kvaddr = kv_addr;
		mp->bytes = n;
		++merge_cnt;
		/*
		 * alloc table space for merge buffers: the output buffer gets
		 *  half, the chunks divide up the rest (in ascending order so
		 *  buffers can be combined when a chunk is finished)
		 */
		i = SYM_ARRAY_NUM / merge_cnt / 2;
		for (n = merge_cnt, mp = msort; --n >= 0; mp++) {
			mp->tstart = kp;
			mp->tend  = (kp += i);
			(void) fill_chunk(mp);
		}
		mout_start = kp;
		kv_addr = (uchar *)ksyms;
		v = (kend - kp) * sizeof(kern_sym_t); /* output bytes */
		/*
		 * merge it
		 */
		for (;;) {
			/* find smallest symbol in the chunks */
			n = merge_cnt;
			mp = msort;
			val = mp->tptr->value;
			i = val >> 1;
			vaddr = (uchar *)mp;		/* ptr to min msort */
			for (mp++; --n > 0; mp++) {
				if ((mp->tptr->value >> 1) > (uint)i)
					continue;
				if (mp->tptr->value < val) {
					val = mp->tptr->value;
					i = val >> 1;
					vaddr = (uchar *)mp;
				}
			}
			/* and put it in the output */
			mp = (merge_chunk_t *)vaddr;
			kp->value = val;
			kp->name = mp->tptr->name;
			if (++kp >= kend) {
				/* flush full output buffer */
				kp = mout_start;
				copy_to_km((uchar *)kp, kv_addr, v);
				kv_addr += v;
			}
			if (++mp->tptr >= mp->tend && fill_chunk(mp)) {
				/* finished this chunk -- remove from array */
				n = mp - msort;
				++mp;
				mp->tstart = msort[n].tstart;	/* combine */
				for ( ; n < merge_cnt; n++)
					msort[n] = *mp++;	/* move rest */
				if (--merge_cnt <= 1)
					break;			/* down to 1 */
			}
		}
		/*
		 * down to one chunk -- flush output, input, then rest of input
		 */
		vaddr = kv_addr;
		if ((n = kp - mout_start) > 0) {
			n *= sizeof(kern_sym_t);
			copy_to_km((uchar *)mout_start, vaddr, n);
			vaddr += n;
		}
		if ((n = msort->tend - msort->tptr) > 0) {
			n *= sizeof(kern_sym_t);
			copy_to_km((uchar *)msort->tptr, vaddr, n);
			vaddr += n;
		}
		if ((n = msort->bytes) > 0) {
			kp = tbl;
			copy_from_km(msort->kvaddr, (uchar *)kp, n);
			copy_to_km((uchar *)kp, vaddr, n);
			vaddr += n;
		}
	}

	/*
	 * load string table after kern_sym table
	 */
	if (!xbuf(ap, (uchar *)&str_tbl_bytes, sizeof(long))) {
		printf("Missing string table size\n");
		goto err;
	}
	if (str_tbl_bytes < 0 || str_tbl_bytes > km_bytes) {
		printf("Invalid string table size %d\n", str_tbl_bytes);
		goto err;
	}

	str_table = vaddr;
	str_tbl_bytes -= sizeof(long);
	if (!xcpu(ap, vaddr, str_tbl_bytes)) {
		printf("String table EOF!\n");
		goto err;
	}

	/* clear and free any left over memory */
	vaddr = (uchar *)ALIGN(vaddr + str_tbl_bytes, sizeof(long));
	i = km_bytes - (vaddr - str_start);
	ASSERT(i >= 0);
	km_clear(vaddr, i);
	km_freen((uint)vaddr, i);

	/*
	 * update symbols with strings in the string table
	 */
	kv_addr = (uchar *)ksyms;
	vaddr = str_start;
	str_table -= sizeof(long);		/* allow for str_tbl_bytes */
	for (n = sym_cnt; n > 0; n -= v, kv_addr += val) {
		v = MIN(n, SYM_ARRAY_NUM);
		val = v * sizeof(kern_sym_t);
		kp = tbl;
		copy_from_km(kv_addr, (uchar *)kp, val);
		for (i = val / sizeof(kern_sym_t); --i >= 0; kp++) {
			if (kp->name < (char *)vaddr)
				kp->name += (uint)str_table;
		}
		copy_to_km((uchar *)tbl, kv_addr, val);
	}

	nksyms = sym_cnt - 1;				/* ready to use */
#ifdef TIME_LOAD_SYMS
	printf("Symbol processing took %u ticks\n", int_counter - start_ticks);
#endif
	return (1);
}
