/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) pm20_tdb.c: version 23.5 created on 4/2/91 at 20:52:25	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)pm20_tdb.c	23.5	4/2/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/

/*
 * pm20_tdb.c -- pm20 driver
 */

#include "sys/types.h"
#include "sys/own.h"
#include "sys/sbus.h"
#include "sys/sbus_pm.h"
#include "sys/spm_mem.h"
#include "sys/trap.h"
#include "sys/kmem.h"
#include "sys/ints.h"
#include "sys/sysmacros.h"
#include "sys/vmem.h"

#include "symbol.h"
#include "spm_debug.h"
#include "cpu_method.h"
#include "sbus_conf.h"
#include "slotdefs.h"
#include "filehdr.h"
/*	#include "global.h"	*/

#define	MAX_PM20	12

/* externals */
extern uint	spm_slot;
extern uint	pm_booting_done;
extern uint	kernel_etext;
extern uint	km_used;
extern uint	enable_clock_on_start;
extern uint	kern_clock_enable;

extern unchar	*iomap();
extern unchar	*kmmap();
extern uint	m68k_display_dis();
extern int	copy_from_km();
extern int	copy_to_km();
extern uint	km_allocate();
extern uint	km_clear();
extern uint	load_kern_syms();
extern char	*symstr();
extern uint	iomap_save();

extern uint	pm20_rde_to_km();
extern uint	pm20_sde_to_km();
extern uint	pm20_mpfn_to_spfn();
extern void	pm20_print_vtop();
extern uint	pm20_mem_stripe_init();
extern void	pm20_build_page_table();
extern rde_t	pm20_km_to_rde();

typedef struct _pm20_info {
	/* methods must be first */
	cpu_data_t	cpu_data;
	uint		pm_id;		/* CPU ID */
	uint		state;		/* state of this PM */
	uint		own;		/* own structure in kmem */
	uint		own_map;	/* saved own struct map info */
	own_t		*own_ptr;
} pm20_info_t;

static pm20_info_t	pm20_infos[MAX_PM20];
static pm20_info_t	*next_pm_info = pm20_infos;
extern uint		pm_entry;

#define	STATE_INIT	0	/* Proms are in control */
#define	STATE_DECONFIG	1	/* Board has been disabled */
#define	STATE_LOADED	2	/* Program loaded in memory */
#define	STATE_START	3	/* PM started, but hasn't responded */
#define	STATE_RUNNING	4	/* Running kernel */
#define	STATE_STOPPED	5	/* Stopped or stopping in debugger */
#define	STATE_STOP_BRK	6	/* Stopped in debugger at breakpint */

/* flags for break points */
#define	BRK_INMEM	1	/* break point is active */
#define	BRK_TEMP	2	/* break point is temporary */

static uint
pm20_pm_first()
{
	register pm20_info_t	*pm_desc;

	for (pm_desc = pm20_infos; pm_desc != next_pm_info; pm_desc++)
		if (pm_desc->state != STATE_DECONFIG)
			return(MAKE_SNS(pm_desc->cu_unit.s.slot, SBUS_NO_BOARD));
	return(SNS_NO_PM);
}

static pm20_info_t *
pm20_cur()
{
	pm20_info_t	*pm_desc;

	pm_desc = (pm20_info_t *) cpu_get_cur();
	ASSERT(pm_desc >= pm20_infos);
	ASSERT(pm_desc < next_pm_info);
	return(pm_desc);
}

static uint
pm20_tdb_done()
{
	uint			saved_map = iomap_save();
	register pm20_info_t	*pm_desc = pm20_cur();
	uint			retval;

	iomap_restore(pm_desc->own_map);	/* now own_ptr is valid */
	retval = pm_desc->own_ptr->o_tdb.done;

	iomap_restore(saved_map);
	return(retval);
}

static uint
pm20_tdb_cmd(cmd, arg, value)
uint	cmd;
uint	arg;		/* optional */
uint	value;		/* optional */
{
	uint			saved_map = iomap_save();
	register pm20_info_t	*pm_desc = pm20_cur();
	register own_t		*o;
	uint			retval;
	void			pm20_check_printf();

	if (pm_desc->state != STATE_STOPPED &&
	  pm_desc->state != STATE_STOP_BRK) {
		printf("PM has not been stopped\n");
		rtn_to_monitor();
	}
		
	iomap_restore(pm_desc->own_map);	/* now own_ptr is valid */
	o = pm_desc->own_ptr;

	if (!o->o_tdb.done) {
		printf("PM is not responding\n");
		rtn_to_monitor();
	}

	o->o_tdb.value = value;
	o->o_tdb.arg = arg;
	o->o_tdb.cmd = cmd;
	o->o_tdb.done = 0;

	if (cmd == TDB_CMD_GO || cmd == TDB_CMD_STEP) {
		iomap_restore(saved_map);
		return(0);
	}

	delay_and_poll_until(HZ * 5 + 1, pm20_tdb_done, NULL);
	pm20_check_printf();

	if (!o->o_tdb.done) {
		printf("PM %u not responding\n", pm_desc->cu_unit.s.slot);
		rtn_to_monitor();
	}
	if (o->o_tdb.error) {
		printf("tdb error %#x\n", o->o_tdb.error);
		rtn_to_monitor();
	}
	retval = o->o_tdb.value;
	iomap_restore(saved_map);
	return(retval);
}

/******** PM20 METHODS ********/

static void
pm20_display_slot()
{
	register pm20_info_t	*pm_desc = pm20_cur();
	register char		*state;

	switch (pm_desc->state) {
	case STATE_INIT:
		state = "under PROM control";
		break;
	case STATE_DECONFIG:
		state = "deconfigured";
		break;
	case STATE_LOADED:
		state = "loaded, ready to run";
		break;
	case STATE_START:
		state = "started and has yet to respond";
		break;
	case STATE_RUNNING:
		state = "running";
		break;
	case STATE_STOPPED:
		state = "stopped in debugger";
		break;
	case STATE_STOP_BRK:
		state = "stopped at breakpoint";
		break;
	default:
		ASSERT(0);
	}

	printf("PM in slot %u is %s.\n", pm_desc->cu_unit.s.slot, state);
}

/*
 * kern_printf_check -- check for kernel printfs -- works for all PMs
 *	Call after checking ok_to_put_char().  Note:  does a spl1().
 */
#define MAX_KERN_READ	64 /* max chars to read from kernel in one poll */

void
kern_check_printf(o, unit)
register own_t	*o;
iunit_t		unit;
{
	register uint	i, count;
	register int	c;
	uint		spl_level;
	extern char	*pm_name();

	spl_level = spl1();
	i = o->o_spm_printf_count;
	count = 0;
	while (i != o->o_printf_count && count != MAX_KERN_READ) {
		c = o->o_char_buff[i];
		if (last_put_char() == '\n' && c != '\n') {
			print_time_stamp(0);
			printf("PM%s: ", pm_name(unit));
		}
		put_char(c);
		o->o_spm_printf_count = (i = (i + 1) & CharBuffMask);
		count++;
	}
	splx(spl_level);
}

static void
pm20_check_printf()
{
	register pm20_info_t	*pm_desc = pm20_cur();
	uint			savemap;
	register own_t	 	*o;

	switch (pm_desc->state) {
	case STATE_INIT:
	case STATE_DECONFIG:
	case STATE_LOADED:
		return;
	case STATE_START:
	case STATE_RUNNING:
	case STATE_STOPPED:
	case STATE_STOP_BRK:
		break;
	default:
		ASSERT(0);
	}
	savemap = iomap_save();
	iomap_restore(pm_desc->own_map);	/* now own_ptr is valid */
	o = pm_desc->own_ptr;

	if (o->o_spm_printf_count != o->o_printf_count &&
	    ok_to_put_char((void *) pm_desc)) {
		kern_check_printf(o, pm_desc->cu_unit);
		sticky_put_char((void *) pm_desc);
	}
	iomap_restore(savemap);
}

static uint
pm20_send_clock(upkern_id)
uint	upkern_id;
{
	register own_t	 	*o;
	register pm20_info_t	*pm_desc = pm20_cur();
	uint			saved_map = iomap_save();

	if (pm_desc->state != STATE_RUNNING)
		return(0);

	iomap_restore(pm_desc->own_map);	/* now own_ptr is valid */
	o = pm_desc->own_ptr;

	switch (o->o_clock) {
	case CLOCK:
	case CLOCK_LEADER:			/* not yet consumed */
		o->o_clocks_dropped++;
		/* fall through */
	case	0: 				/* not yet enabled */
		iomap_restore(saved_map);
		return(0);
	case CLOCK_ENABLE:			/* ready for another */
		break;
	default:
		ASSERT(0);
	}

	if (upkern_id == pm_desc->pm_id)
		o->o_clock = CLOCK_LEADER;
	else
		o->o_clock = CLOCK;

	*(uint *)iomap(pm_desc->cu_unit.s.slot, PM_INT_REQ_REG) =
	  PM_LEVEL_SIX_INT_REQ;
	iomap_restore(saved_map);
	return(upkern_id == pm_desc->pm_id);
}

/**************** LOW LEVEL DEBUGGER OPERATIONS *****************/

static uint
pm20_get_byte(addr)
uint	addr;
{
	return(pm20_tdb_cmd(TDB_CMD_GET_CHAR, addr));
}

static void
pm20_put_byte(addr, value)
uint	addr;
uint	value;
{
	pm20_tdb_cmd(TDB_CMD_PUT_CHAR, addr, value);
}

static uint
pm20_get_short(addr)
uint	addr;
{
	return(pm20_tdb_cmd(TDB_CMD_GET_SHORT, addr));
}

static void
pm20_put_short(addr, value)
uint	addr;
uint	value;
{
	pm20_tdb_cmd(TDB_CMD_PUT_SHORT, addr, value);
}

static uint
pm20_get_long(addr)
uint	addr;
{
	return(pm20_tdb_cmd(TDB_CMD_GET_LONG, addr));
}

static void
pm20_put_long(addr, value)
uint	addr;
uint	value;
{
	pm20_tdb_cmd(TDB_CMD_PUT_LONG, addr, value);
}

static uint
pm20_get_named_reg(name)
unchar	*name;
{
	return(pm20_tdb_cmd(TDB_CMD_GET_REG, m68k_valid_reg_name(name)));
}

static void
pm20_put_named_reg(name, value)
unchar	*name;
uint	value;
{
	pm20_tdb_cmd(TDB_CMD_PUT_REG, m68k_valid_reg_name(name), value);
}


/*****************************************/

static unchar *
pm20_symbolic_version(addr)
uint	addr;
{
	return ((unchar *)symstr(addr));
}

static BRK_TAB *
pm20_find_bp(addr)
uint	addr;
{
	return(bp_search((void *) pm20_infos, addr));
}

static void
pm20_start()
{
	register pm20_info_t	*pm_desc = pm20_cur();
	uint			saved_map = iomap_save();
	register own_t		*o;

	if (pm_desc->state == STATE_INIT) {
		printf("PM must be loaded first\n");
		rtn_to_monitor();
	}
	if (pm_desc->state != STATE_LOADED)
		return;

	pm_desc->own = sbus_config.pm_own[pm_desc->pm_id];
	/* pre-map and save own structure map */
	pm_desc->own_ptr = (own_t *)km_map(pm_desc->own);
	pm_desc->own_map = iomap_save();

	/* send kernel root pointer */

	*(rde_t *) iomap(pm_desc->cu_unit.s.slot, (spm_slot << 28) | 0x8180) =
	  pm20_km_to_rde(sbus_config.pm_stbl[pm_desc->pm_id]);

	delay_check_iomap(5);

	iomap_restore(pm_desc->own_map);	/* now own_ptr is valid */
	o = pm_desc->own_ptr;
	o->o_tdb.cmd = TDB_CMD_GO;

	*(uint *) iomap(pm_desc->cu_unit.s.slot, (spm_slot << 28) | 0x2580) =
	  pm_entry;

	pm_desc->state = STATE_START;

	/* wait for PM boot response flag */

	delay_check_iomap(5);

	iomap_restore(pm_desc->own_map);	/* now own_ptr is valid */
	o = pm_desc->own_ptr;
	if (o->o_boot_response) {
		printf("PM in slot %u booted.\n", pm_desc->cu_unit.s.slot);
		pm_desc->state = STATE_RUNNING;
	} else if ( Spm_Mem->pm_version != SPM_MEM_VERSION ) {
		printf("Mismatched versions: spm: %#x, kernel:%#x\n",
		  SPM_MEM_VERSION, Spm_Mem->pm_version);

		/* treat this condition as a booted kernel */
		pm_booting_done = 1;
		rtn_to_monitor();
	} else {
		printf("slot %u: PM to SPM boot response timeout\n",
		  pm_desc->cu_unit.s.slot);
		rtn_to_monitor();
	}
	iomap_restore(saved_map);
}

static void
pm20_stop()
{
	register pm20_info_t	*pm_desc = pm20_cur();
	uint		saved_map = iomap_save();
	own_t		*o;

	if (pm_desc->state != STATE_RUNNING)
		return;

	pm_desc->state = STATE_STOPPED;

	iomap_restore(pm_desc->own_map);	/* now own_ptr is valid */
	o = pm_desc->own_ptr;

	if (!o->o_tdb.done ) {
		*(uint *)iomap(pm_desc->cu_unit.s.slot, PM_INT_REQ_REG) = 
		  PM_LEVEL_SEVEN_INT_REQ;

		delay_no_poll(2);

		iomap_restore(pm_desc->own_map);
		o = pm_desc->own_ptr;
		if ( !o->o_tdb.done )
			printf("PM %u didn\'t stop\n", pm_desc->cu_unit.s.slot);
		else
			o->o_tdb.cmd = TDB_CMD_NONE;
	}
	iomap_restore(saved_map);
}

static uint
pm20_check_tdb()
{
	uint			saved_map = iomap_save();
	register pm20_info_t	*pm_desc = pm20_cur();
	register own_t	*o;
	uint	retval = 0;

	if (pm_desc->state != STATE_RUNNING)
		return(0);

	iomap_restore(pm_desc->own_map);	/* now own_ptr is valid */
	o = pm_desc->own_ptr;

	if (o->o_tdb.done && o->o_tdb.cmd == TDB_CMD_GO ) {
		retval = 1;
		pm_desc->state = STATE_STOPPED;
	}
	iomap_restore(saved_map);
	return(retval);
}

static void
pm20_brkpt_check()
{
	uint			saved_map = iomap_save();
	register pm20_info_t	*pm_desc = pm20_cur();
	uint			kernel_pc;

	kernel_pc = pm20_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_PC);

	if (pm20_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_VECTOR) == INSTERR &&
	  (bp_search((void *) pm20_infos, kernel_pc))) {

		pm_desc->state = STATE_STOP_BRK;
		printf("PM %u hit breakpoint ", pm_desc->cu_unit.s.slot);
		printf("(%s)\n", pm20_symbolic_version(kernel_pc));

	}
	iomap_restore(saved_map);
}

static	void
pm20_show_trap()
{
	pm20_tdb_cmd(TDB_CMD_SHOW_TRAP);
	pm20_check_printf();
	printf("\n");

	m68k_display_dis(pm20_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_PC));
	printf("\n");
}

static void
pm20_step()
{
	uint			saved_map = iomap_save();
	register pm20_info_t	*pm_desc = pm20_cur();
	register BRK_TAB	*brkp = NULL;
	uint			kernel_pc;
	
	kernel_pc = pm20_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_PC);
	if (pm_desc->state == STATE_STOP_BRK) {
		brkp = bp_search((void *) pm20_infos, kernel_pc);
		if (brkp)
			*(ushort *)km_map(kernel_pc) = brkp->text;
		pm_desc->state = STATE_STOPPED;
	}

	pm20_tdb_cmd(TDB_CMD_STEP);

	delay_and_poll_until(HZ * 5, pm20_tdb_done, NULL);
	pm20_check_printf();

	if (brkp)
		*(ushort *)km_map(kernel_pc) = ILLEGAL_INSTR;

	if (!pm20_tdb_done()) {
		printf("PM did not return from single step\n");
		pm_desc->state = STATE_RUNNING;
		rtn_to_monitor();
	}
	iomap_restore(saved_map);
}

static void
pm20_jump()
{
	register pm20_info_t	*pm_desc = pm20_cur();
	uint			saved_map = iomap_save();
	uint			kernel_pc;
	register ushort		*mapped_pc;
	register int		ilen;
	register ushort		text;

	if (!m68k_at_jsr()) {
		pm20_step();
		return;
	}
	kernel_pc = pm20_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_PC);
	if (pm_desc->state == STATE_STOP_BRK)
		pm20_step();

	ilen = m68k_dis(kernel_pc) * sizeof(ushort);
	mapped_pc = (ushort *)km_map(kernel_pc + ilen);
	text = *mapped_pc;
	*mapped_pc = ILLEGAL_INSTR;

	pm20_tdb_cmd(TDB_CMD_GO);
	delay_and_poll_until(HZ * 5, pm20_tdb_done, NULL);
	pm20_check_printf();
	*mapped_pc = text;

	if (!pm20_tdb_done()) {
		printf("PM did not hit temporary breakpoint\n");
		pm_desc->state = STATE_RUNNING;
		rtn_to_monitor();
	}
	iomap_restore(saved_map);
}

static uint
pm20_prepare_to_go()
{
	register pm20_info_t	*pm_desc = pm20_cur();

	if (pm_desc->state == STATE_STOP_BRK) {
		pm20_step();
		pm20_brkpt_check();
	}
	return(pm_desc->state != STATE_STOP_BRK);
}

static void
pm20_go()
{
	register pm20_info_t	*pm_desc = pm20_cur();

	if (pm_desc->state != STATE_STOPPED)
		return;

	pm20_tdb_cmd(TDB_CMD_GO);
	pm_desc->state = STATE_RUNNING;
}

static void
pm20_display_own()
{
	register pm20_info_t	*pm_desc = pm20_cur();
	uint			saved_map = iomap_save();
	register struct own	*o;

	switch (pm_desc->state) {
	case STATE_INIT:
	case STATE_DECONFIG:
	case STATE_LOADED:
		printf("Own structure not yet available\n");
		return;
	case STATE_START:
	case STATE_RUNNING:
	case STATE_STOPPED:
	case STATE_STOP_BRK:
		break;
	default:
		ASSERT(0);
	}
	iomap_restore(pm_desc->own_map);	/* now own_ptr is valid */
	o = pm_desc->own_ptr;

	printf("slot   %8x  lock_id  %8x  fpu_type   %8x  cpu_type     %8x\n",
	  o->o_slot, o->o_lock_id, o->o_fpu_type, o->o_cpu_type);

	printf("curpri %8x  runrun   %8x  curproc    %8x  switching    %8x\n",
	  o->o_curpri, o->o_runrun, o->o_curproc, o->o_switching);

	printf("lticks %8x  trap_hap %8x  fpu_loaded %8x  is_idle      %8x\n",
	  o->o_lticks, o->o_trap_hap, o->o_fpu_loaded, o->o_is_idle);

	printf(
	  "spin_lock_cnt             %8x  flush_tlb  %8x  stbl         %8x\n",
	  o->o_spin_lock_cnt, o->o_flush_tlb, o->o_stbl);

	printf(
	  "in_int_service            %8x  upkern_proc%8x\n",
	  o->o_in_int_service, o->o_upkern_proc);

	printf(
	  "clocks_dropped            %8x  clock      %8x\n\n",
	o->o_clocks_dropped, o->o_clock);
	iomap_restore(saved_map);
}


static void
pm20_backtrace(fp)
uint	*fp;
{
	if (fp == NULL)
		fp = (uint *) pm20_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_A6);
	m68k_backtrace(fp);
}

static uint
pm20_valid_fp(fp)
uint	fp;
{
	return (fp >= ADDR_EXTRA_STK && fp < ADDR_U_STRUCT ||
		fp >= ADDR_OWN_EXTRA && fp < ADDR_OWN_U_STR);
}

static void
pm20_set_brkpt(bp_addr)
uint	bp_addr;
{
	uint			saved_map = iomap_save();
	register BRK_TAB	*brkp;

	if (bp_addr < ADDR_KERNEL || bp_addr > kernel_etext) {
		printf("Bad kernel breakpoint address:%#x\n", bp_addr);
		return;
	}
	if (bp_search((void *) pm20_infos, bp_addr)) {
		printf("Breakpoint already set\n");
		return;
	}
	brkp = bp_search(NULL, 0);
	if (!brkp) {
		printf("No more breakpoints available\n");
		return;
	}
	cpu_all_stop();

	brkp->flags = 0;
	brkp->pc = bp_addr;
	brkp->text = *(ushort *)km_map(bp_addr);
	brkp->unique_id = (void *) pm20_infos;
	*(ushort *)km_map(bp_addr) = ILLEGAL_INSTR;

	iomap_restore(saved_map);
}

static void
pm20_clear_one_brkpt(bp_addr)
uint	bp_addr;
{
	register BRK_TAB	*brkp;
	uint			saved_map = iomap_save();

	brkp = bp_search((void *) pm20_infos, bp_addr);
	if (!brkp) {
		printf("Breakpoint not found\n");
		return;
	}
	cpu_all_stop();
	*(ushort *)km_map(brkp->pc) = brkp->text;
	brkp->pc = 0;
	brkp->flags = 0;
	brkp->unique_id = NULL;
	printf("Kernel breakpoint (%s) cleared\n",
	  pm20_symbolic_version(bp_addr));

	iomap_restore(saved_map);
}

static void
pm20_clear_brkpts()
{
	register BRK_TAB	*brkp;
	uint	saved_map = iomap_save();

	for (brkp = brk_tab; brkp != brk_tab + NUM_BREAKPOINTS; brkp++) {
		if (brkp->unique_id != (void *) pm20_infos)
			continue;
		cpu_all_stop();
		*(ushort *)km_map(brkp->pc) = brkp->text;
		brkp->pc = 0;
		brkp->flags = 0;
		brkp->unique_id = NULL;
	}
	iomap_restore(saved_map);
}

static void
pm20_display_brkpts()
{
	register pm20_info_t	*pm_desc = pm20_cur();
	register BRK_TAB	*brkp;

	if (pm_desc != pm20_infos)
		return;		/* only do for first pm */

	for (brkp = brk_tab; brkp != brk_tab + NUM_BREAKPOINTS; brkp++) {
		if (brkp->unique_id != (void *) pm_desc)
			continue;
		printf("        kernel (%#x)    %s\n", brkp->pc,
		  pm20_symbolic_version(brkp->pc));
	}
}

static void
pm20_check_tags()
{
	register uint	saved_kern_clock_enable = kern_clock_enable;

	kern_clock_enable = 0;

	delay_no_poll(2);

	pm20_tdb_cmd(TDB_CMD_CHECK_TAGS);

	kern_clock_enable = saved_kern_clock_enable;
}

static void
pm20_deconfig_any(pm_desc)
register pm20_info_t	*pm_desc;
{
	if (pm_desc->state != STATE_INIT) 
		return;

	pm_desc->state = STATE_DECONFIG;
	printf("deconfiguring PM in slot %u\n", pm_desc->cu_unit.s.slot);
	cpu_set_slot(NULL, pm_desc->cu_unit.w.sns);
	cpu_deactivate();
	BOARD_DECONFIG(pm_desc->cu_unit.s.slot);
}

static void
pm20_solo()
{
	register pm20_info_t	*pm_desc = pm20_cur();
	register pm20_info_t	*desc;

	if (pm_desc->state != STATE_INIT) {
		printf("PM is not at initial state\n");
		rtn_to_monitor();
	}
	for (desc = pm20_infos; desc != next_pm_info; desc++) {
		if (desc == pm_desc)
			continue;
		if (desc->state != STATE_INIT)
			continue;
		pm20_deconfig_any(desc);
	}
}

static void
pm20_deconfigure()
{
	register pm20_info_t	*pm_desc = pm20_cur();

	if (pm_desc->state != STATE_INIT) {
		printf("PM is not at initial state\n");
		rtn_to_monitor();
	}
	pm20_deconfig_any(pm_desc);
}

static void
pm20_configure()
{
	register pm20_info_t	*pm_desc = pm20_cur();

	if (pm_desc->state != STATE_DECONFIG) {
		printf("PM is not deconfigured\n");
		rtn_to_monitor();
	}

	pm_desc->state = STATE_INIT;
	printf("configuring PM in slot %u\n", pm_desc->cu_unit.s.slot);
	BOARD_CONFIG(pm_desc->cu_unit.s.slot);
	cpu_activate();
	check_pm_config();
}

static uint
pm20_prep_for_syms(bss_end, text_end)
unchar	*bss_end;
unchar	*text_end;
{
	km_used = (uint)bss_end;	/* End of kernel bss. */
	kernel_etext = (uint)text_end;	/* Used by PM_build() */
	return(1);
}


/*
 * pm20_load -- return 0 for no-op, ~0 for error, or kernel entry point
 *		Sets maincpu_mpfn_to_spfn
 */
uint
pm20_load(source_dev, umod_flag)
char	*source_dev;
uint	umod_flag;
{
	register pm20_info_t	*desc = pm20_cur();
	uint			entry;
	extern uint		load_kernel();

	if (desc->state == STATE_RUNNING) {
		printf("The kernel has already been booted.\n");
		return (~0);
	} else if (desc->state == STATE_LOADED) {
		printf("The kernel has already been loaded.\n");
		return (~0);
	} else if (!(desc->state == STATE_INIT)) {
		printf("PM %u in wrong state for downloading Kernel (%#x).\n",
		  desc->cu_unit.s.slot, desc->state);
		return (~0);
	}

	maincpu_mpfn_to_spfn = pm20_mpfn_to_spfn;
	sbus_config.cpu_type = 68020;
	if ((entry = load_kernel(source_dev, umod_flag)) == 0 || entry == ~0)
		printf("Download failed!\n");
	else {
		desc->state = STATE_LOADED;
		pm20_all_set_loaded();
		printf("\rdownload completed.\n");
	}
	return (entry);
}

uint
pm20_check_coffdata(addr)
uint	addr;
{
	if (addr < MAINSTORE || addr >= KIO_START) {
		printf("0x%08x bad program address for the kernel!\n", addr);
		printf("Indicates wrong slot or wrong kernel code!\n");
		return(0);
	}
	return(1);
}

static uint
pm20_check_magic(number)
short	number;
{
	if (number == A68020MAGIC)
		return(1);
	printf("file is not a PM20 binary\n");
	return(0);
}

static uint
pm20_prep_boot(source_dev, umod_flag)
char	*source_dev;
uint	umod_flag;
{
	register pm20_info_t	*desc = pm20_cur();

	if (desc->state == STATE_LOADED)
		return (0);

	return (pm20_load(source_dev, umod_flag));
}

/*
 * pm20_print_vtop
 *
 *	break down virtual address into translation components
 *	print root pointer, page table pointer, page descriptor.
 */

void
pm20_print_vtop(addr, proc_addr)
uint	addr, proc_addr;
{
	uint	kv, seg_offset;
	uint	page_table_offset;
	rde_t	rde;
	sde_t	sde;
	pde_t	pde;

	/* 
	 * get root pointer 
	 */

	if (is_kern_addr(addr)) {
		rde.rde_all = CPU_get_long(KROOT);
		/* 
		 * calulate offset within segment table 
		 */
		seg_offset = (addr - KMEM_START) / (NPGPT * NBPP);
		seg_offset *= 4;
		page_table_offset = ((addr - KMEM_START) & 0x1ff000) / NBPP;
		page_table_offset *= 4;
	}
	else if (proc_addr == 0) {
		printf("Illegal address (0x%08x) for kernel!\n", addr);
		return;
	}
	else {
		copy_from_km(proc_addr + ((uint)&((proc_t *)0)->p_urde),
		  &rde.rde_all, sizeof(rde_t));
	/****	rde.rde_all = CPU_get_long(UROOT);	****/
		/* calulate offset within segment table */
		seg_offset = addr / (NPGPT * NBPP);
		seg_offset *= 4;
		page_table_offset = (addr & 0x1ff000) / NBPP;
		page_table_offset *= 4;
	}

	/* kernel virtual to base of segment table */
	kv = pm20_rde_to_km(rde);
	kv += seg_offset;

	/* get sde */
	copy_from_km(kv, &sde.sg_sge, sizeof(sde.sg_sge));

	/* get page table pointer */
	kv = pm20_sde_to_km(sde);

	/* get page table entry */
	kv += page_table_offset;
	copy_from_km(kv, &pde.pgi.pg_pde, sizeof(pde.pgi.pg_pde));

	printf("Address: %8x  rde:  %8x  sde: %8x pde: %8x, pde at %8x\n",
		addr, rde.rde_all, sde.sg_sge, pde.pgi.pg_pde, kv);
}

static uint
pm20_get_u_addr()
{
	return (ADDR_U_STRUCT);
}

static void
pm20_set_pm_id()
{
	register pm20_info_t	*desc = pm20_cur();

	ASSERT(desc->state == STATE_INIT);
	desc->pm_id = set_pm_id();
}

/************** END OF METHODS ***************/

static void
pm20_setup_methods(mp)
register method_t	*mp;
{
	mp->check_printf = pm20_check_printf;
	mp->send_clock = pm20_send_clock;
	mp->stop = pm20_stop;

	mp->display_slot = pm20_display_slot;

	mp->load = pm20_load;
	mp->check_coffdata = pm20_check_coffdata;
	mp->check_magic = pm20_check_magic;
	mp->prep_for_syms = pm20_prep_for_syms;
	mp->load_syms = load_kern_syms;
	mp->set_pm_id = pm20_set_pm_id;
	mp->prep_boot = pm20_prep_boot;
	mp->start = pm20_start;
	mp->check_tdb = pm20_check_tdb;
	mp->prepare_to_go = pm20_prepare_to_go;
	mp->go = pm20_go;
	mp->show_trap = pm20_show_trap;

	mp->pm_first = pm20_pm_first;
	mp->print_vtop = pm20_print_vtop;
	mp->mem_stripe_init = pm20_mem_stripe_init;
	mp->build_page_table = pm20_build_page_table;
	mp->get_u_addr = pm20_get_u_addr;

	mp->get_byte = pm20_get_byte;
	mp->put_byte = pm20_put_byte;
	mp->get_short = pm20_get_short;
	mp->put_short = pm20_put_short;
	mp->get_long = pm20_get_long;
	mp->put_long = pm20_put_long;
	mp->get_named_reg = pm20_get_named_reg;
	mp->put_named_reg = pm20_put_named_reg;

	mp->step = pm20_step;
	mp->jump = pm20_jump;
	mp->brkpt_check = pm20_brkpt_check;
	mp->display_own = pm20_display_own;
	mp->display_dis = m68k_display_dis;
	mp->symbolic_version = pm20_symbolic_version;
	mp->find_bp = pm20_find_bp;
	mp->backtrace = pm20_backtrace;
	mp->valid_fp = pm20_valid_fp;
	mp->set_brkpt = pm20_set_brkpt;
	mp->clear_one_brkpt = pm20_clear_one_brkpt;
	mp->clear_brkpts = pm20_clear_brkpts;
	mp->display_brkpts = pm20_display_brkpts;

	mp->check_tags = pm20_check_tags;
	mp->solo = pm20_solo;
	mp->deconfigure = pm20_deconfigure;
	mp->configure = pm20_configure;
	mp->cpu_copyin = (uint (*)())copy_from_km;
	mp->cpu_copyout = (uint (*)())copy_to_km;
	mp->cpu_malloc = (uint (*)())km_allocate;
	mp->cpu_zero = km_clear;
}

pm20_setup(slot)
uint	slot;
{
	register pm20_info_t	*pm_desc = next_pm_info;
	register uint		n, id;
	static method_t		pm20_methods;

	ASSERT(pm_desc >= pm20_infos);
	ASSERT(pm_desc < pm20_infos + MAX_PM20);

	if (pm_desc == pm20_infos) {
		pm20_setup_methods(&pm20_methods);

	}

	pm_desc->cu_methods = &pm20_methods;
	pm_desc->cu_unit.i = IUNIT_NO_DEV;
	pm_desc->cu_unit.s.slot = slot;
	pm_desc->state = STATE_INIT;
	next_pm_info++;

	cpu_enroll(&pm_desc->cpu_data);
	return (0);
}

pm20_all_set_loaded()
{
	register pm20_info_t	*pm_desc;

	for (pm_desc = pm20_infos; pm_desc != next_pm_info; pm_desc++)
		if (pm_desc->state == STATE_INIT)
			pm_desc->state = STATE_LOADED;
}
