/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) iopm_tdb.c: version 25.1 created on 11/27/91 at 15:29:59	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)iopm_tdb.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
/* cpu/iopm_tdb.c */

#include "global.h"
#include "sys/types.h"
#include "spm_debug.h"
#include "sys/spm_mem.h"
#include "spm_tdb.h"

#include "sys/iopmcomm.h"
#include "dev.h"
#include "cpu_method.h"
#include "iopm_data.h"
#include "iomap.h"
#include "novram.h"
#include "iom_config.h"
#include "filehdr.h"

#include "sys/trap.h"


/* externals */
extern uint	iom_count;

extern uint	iomap_save(), alt_iomap_save();
extern void	iopm_start();
extern char	*extract_iopm_dev();
extern char	*strcpy();
extern void	iopm_build_page_table(), iopm_start_hardware(), iopm_init();
extern uchar 	*iomap();

extern iom_config_t	*find_iom();

/* Forwards */
char		*gs_str(), *s_str();
void 		iopm_check_printf();
static void	iopm_put_short();
static uint	iopm_get_short();

#define MAX_IOPMS	16	/* FIX IOSBA */
#define	FIRST_IOPM	0

static	iopm_data_t	iopm_data[MAX_IOPMS];
static  iopm_data_t	*next_iopm_data = iopm_data;

/* Used when downloading to keep track of the last IOPM downloaded */
static	iopm_data_t	*last_iopm_loaded;

static	uint	iopm_already_offline;

/* FIX THIS joe, make many of these routines static */
/* FIX THIS, JPC:  this should go into an include file somewhere */
	/* Description of individual IOSB */
#define IOSB_OFFSET_BITS	28	/* Bits behind each slot. */


uchar *
sio_map(unit, address)
iunit_t	unit;
uint	address;
{
	ASSERT(unit.s.slot != SBUS_NO_BOARD);

	/* if the board is in a subslot, map it through the IOSBA */
	if (unit.s.subslot != SBUS_NO_BOARD)
		address = (unit.s.subslot << IOSB_OFFSET_BITS) +
			    LowBits(IOSB_OFFSET_BITS, address);

	return(iomap(unit.s.slot, address));
}


check_ios_bd(unit, header)
iunit_t	unit;
char	*header;
{
	uint		bd_id = sbus_config.slot_id[unit.s.slot];
	iom_config_t	*iomp;
	css_bd_dat_t	*csdp;

	if (unit.s.slot == SBUS_NO_BOARD && bd_id != SBUS_IOPM) {
		printf("%s: board in slot %u is not an IOPM!\n",
		  header, unit.s.slot);
		return(0);
	}
	if (unit.s.subslot != SBUS_NO_BOARD) {
		if (bd_id != SBUS_NIOM || !(iomp = find_iom(unit.s.slot))) {
			printf("%s: board in slot %u is not a new IOM!\n",
			  header, unit.s.slot);
			return(0);
		}
		csdp = &iomp->ex_card_cage.css_bd_dat[unit.s.subslot];
		if (csdp->css_slot_id != SBUS_IOPM ||
		    csdp->dev_board_id != IOPM_DB_ID_DSDB) {
			printf("%s: board in %s is not an IOPM/DSDB!\n",
			  header, s_str(unit));
			return(0);
		}
	}
	return(1);
}

uchar *
iopm_map(desc, addr)
iopm_data_t	*desc;
uint		addr;
{
	uint	iosb_slot; 
	uint	bd_id;

/* FIX joe, use above routine after it is tested. */
	iosb_slot = desc->cu_unit.s.subslot;
	bd_id = sbus_config.slot_id[desc->cu_unit.s.slot];

	if (iosb_slot == SBUS_NO_BOARD && bd_id != SBUS_IOPM ||
	    iosb_slot != SBUS_NO_BOARD && bd_id != SBUS_NIOM) {
		printf("iopm_map: desc=%#x, iunit=0x%08x, slot_id=0x%02x\n",
		  desc, desc->cu_unit.i, bd_id);
		ASSERT(0);
	}

	return(sio_map(desc->cu_unit, addr));
}

/*
 * gs_str -- return a pointer to a string containing the slot or slot/subslot
 */

char *
gs_str(iunit)
iunit_t	iunit;
{
	static char	slot_str[8];

	sprintf(slot_str,
	  ((iunit.s.subslot != SBUS_NO_BOARD) ? "%u/%u" : "%u"),
	  iunit.s.slot, iunit.s.subslot);

	return(slot_str);
}

/*
 * s_str -- call gs_str for an iopm_data_t
 */

char *
s_str(i_desc)
iopm_data_t	*i_desc;
{
	return(gs_str(i_desc->cu_unit));
}

/*
 * scan_for_iopms -- returns non-zero if any IOPMs are enrolled
 */

uint
scan_for_iopms()
{
	return((uint)iopm_data[0].cu_methods);
}

/* FIX joe, this routine is used by cmd reset and cmd_unreset.
 * I have to talk to cmw. */
iopm_data_t *
iopm_slot_to_desc(iunit)
iunit_t	iunit;
{
	register iopm_data_t	*i_desc = iopm_data;
	register uint		sns = iunit.w.sns;

	/* FIX joe, get rid of slot and access to sbus_config */
	for (; i_desc != next_iopm_data; i_desc++) {
		if (i_desc->cu_unit.w.sns == sns)
			return(i_desc);
	}

	printf("Unable to locate IOPM in slot %s.\n", gs_str(iunit));
	return ((iopm_data_t *)NULL);
}

static iopm_data_t *
iopm_cur()
{
	iopm_data_t	*iopm_desc;

	iopm_desc = (iopm_data_t *) cpu_get_cur();

	ASSERT(iopm_desc <= next_iopm_data);
	ASSERT(iopm_desc >= iopm_data);
	return(iopm_desc);
}

/* Due to the iopm hardware going out to lunch (OTL) for a few seconds every
 * once in awhile, iopm_offline_handler() give the board a chance to recover
 * before permenently deconfiguring it.
 */

static void
iopm_offline_handler()
{
	register iopm_data_t	*i_desc = iopm_cur();

	if (iopm_already_offline && (i_desc->offline_delay == 0))
		i_desc->state = STATE_DECONFIG;

	i_desc->offline_delay = HZ * 10;
	iopm_already_offline = 1;
	printf("Off line delay invoked.\n");
}

/*
 * NOTE: both copy_from_iopm and copy_to_iopm assume that the caller has
 *	 ensured that addr + size will not fall outside of the SPM (and
 *	 maybe IOSBA) map windows.
 */

uint
copy_from_iopm(from, to, size)
uchar		*from;		/* SPM virtual address */
uchar		*to;		/* IOPM physical offset */
uint		size;		/* number bytes to transfer */
{
	uint	saved_map = iomap_save();

	bcopy(iopm_map(iopm_cur(), from), to, size);

	iomap_restore(saved_map);
}

uint
copy_to_iopm(from, to, size)
uchar		*from;		/* SPM virtual address */
uchar		*to;		/* IOPM physical offset */
uint		size;		/* number bytes to transfer */
{
	uint	saved_map = iomap_save();

	bcopy(from, iopm_map(iopm_cur(), to), size);

	iomap_restore(saved_map);
}


static uint
iopm_zero(addr, size)
uchar	*addr;		/* IOPM virtual address */
uint	size;		/* number bytes to clear */
{
	uint	saved_map = iomap_save();

	bzero(iopm_map(iopm_cur(), addr), size);

	iomap_restore(saved_map);
	return (0);
}


static uint
iopm_tdb_done()
{
	uint		saved_map = iomap_save();
	uint		retval;
	iopm_data_t	*i_desc = iopm_cur();

	iomap_restore(i_desc->iopm_comm_map);	/* iopm_comm_ptr now valid */
	retval = i_desc->iopm_comm_ptr->o_tdb.done;

	iomap_restore(saved_map);
	return(retval);
}

static uint
iopm_malloc(count, boundry)
uint	count;
uint	boundry;
{
	return(0);
}

/*
 * iopm_check_tdb() - While running the TDB_CMD_GO command is in effect.
 *	When the IOPM is stopped in tdb, the go command is 'done'.
 */
static uint
iopm_check_tdb()
{
	uint			saved_map = iomap_save();
	register iopm_data_t	*i_desc = iopm_cur();
	register iopm_comm_t	*o;
	uint			retval;
	uint			tdb_cmd_done;

	if (i_desc->state != STATE_RUNNING)
		return(0);

	if (i_desc->offline_delay)
		return(0); /* Delay tdb check for now. */

	iomap_restore(i_desc->iopm_comm_map);	/* now iopm_comm_ptr is valid */
	o = i_desc->iopm_comm_ptr;

	if (!probe_rd_long(&o->o_tdb.done, &tdb_cmd_done)) {
		printf("Bus error reading tdb data, IOPM %s.\n", s_str(i_desc));
		iopm_offline_handler();	/* Board may be temporarily offline. */
		iomap_restore(saved_map);
		return(0);
	}
	if (tdb_cmd_done && o->o_tdb.cmd == TDB_CMD_GO) {
		retval = 1;
		i_desc->state = STATE_STOPPED;
	}
	else
		retval = 0;
	iopm_already_offline = 0;
	iomap_restore(saved_map);
	return(retval);
}

static void
iopm_display_slot()
{
	iopm_data_t	*i_desc = iopm_cur();
	register char	*state;

	switch (i_desc->state) {
	case STATE_INIT:
		state = "holding in reset state";
		break;
	case STATE_ENABLED:
		state = "able to run";
		break;
	case STATE_LOADED:
		state = "loaded and ready to run";
		break;
	case STATE_RUNNING:
		state = "running";
		break;
	case STATE_STOPPED:
		state = "stopped in debugger";
		break;
	case STATE_DECONFIG:
		state = "deconfigured";
		break;
	default:
		ASSERT(0);
	}

	printf("IOPM in slot %s is %s.\n", s_str(i_desc), (state));
}

static unchar *
iopm_symbolic_version(addr)
uint	addr;
{
	static char	str[12];

	sprintf(str, "0x%08x", addr);
	return((uchar *)str);		/* NO IOPM SYMBOLS FOR NOW */
}

static BRK_TAB *
iopm_find_bp(addr)
uint	addr;
{
	return(bp_search((void *) iopm_cur(), addr));	
}

/*VARARGS1*/
static uint
iopm_tdb_cmd(cmd, arg, value)
uint	cmd;
uint	arg;	/* optional */
uint	value;	/* optional */
{
	register iopm_data_t	*i_desc = iopm_cur();
	register iopm_comm_t	*o;
	uint			retval;
	uint			saved_map = iomap_save();

	if (i_desc->state != STATE_STOPPED && !i_desc->at_brkpt) {
		printf("IOPM has not been stopped\n");
		rtn_to_monitor();
	}

	iomap_restore(i_desc->iopm_comm_map);	/* now iopm_comm_ptr is valid */
	o = i_desc->iopm_comm_ptr;

/* FIX NOW joe, possibly set the state variable. */
	if (!o->o_tdb.done) {
		printf("IOPM is not responding\n");
		rtn_to_monitor();
	}

	o->o_tdb.value = value;
	o->o_tdb.arg = arg;
	o->o_tdb.cmd = cmd;
	o->o_tdb.done = 0;

	if (cmd == TDB_CMD_GO || cmd == TDB_CMD_STEP) {
		iomap_restore(saved_map);
		return(0);
	}

	delay_and_poll_until(HZ * 5 + 1, iopm_tdb_done, NULL);
	iopm_check_printf();

/* FIX NOW joe, possibly set the state variable. */
	if (!o->o_tdb.done) {
		printf("IOPM %s not responding\n", s_str(i_desc));
		rtn_to_monitor();
	}

	if (o->o_tdb.error) {
		printf("tdb error %#x\n", o->o_tdb.error);
		rtn_to_monitor();
	}

	retval = o->o_tdb.value;
	iomap_restore(saved_map);
	return(retval);
}

uint
iopm_valid_fp(fp)
uint	*fp;
{
	return(fp != NULL);
}


static void
iopm_backtrace(fp)
uint	*fp;
{
	if (fp == NULL)
		fp = (uint *) iopm_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_A6);
	m68k_backtrace(fp);
}

static void
iopm_set_brkpt(bp_addr)
uint	bp_addr;
{
	register iopm_data_t		*i_desc = iopm_cur();
	uint				saved_map = iomap_save();
	register BRK_TAB		*brkp;

	if (bp_addr >= MAINSTORE && bp_addr <= IOPM_RAM_START ||
	    bp_addr >= IOPM_RAM_START + i_desc->memsize ||
	    (bp_addr & (sizeof(ushort) - 1))) {
		printf("0x%08x is NOT a valid bp address for an IOPM!\n",
		  bp_addr);
		iomap_restore(saved_map);
		return;
	}
	if (bp_search((void *) i_desc, bp_addr)) {
		printf("Breakpoint already set\n");
		iomap_restore(saved_map);
		return;
	}
	brkp = bp_search(NULL, 0);
	if (!brkp) {
		printf("No more breakpoints available\n");
		iomap_restore(saved_map);
		return;
	}
	cpu_all_stop();

	brkp->flags = 0;
	brkp->pc = bp_addr;
	brkp->text = iopm_get_short(bp_addr);

	brkp->unique_id = (void *) i_desc;
	iopm_put_short(bp_addr, ILLEGAL_INSTR);

	iomap_restore(saved_map);
}

static void
iopm_clear_one_brkpt(bp_addr)
uint	bp_addr;
{
	register iopm_data_t	*i_desc = iopm_cur();
	register BRK_TAB	*brkp;
	uint			saved_map = iomap_save();

	brkp = bp_search((void *) i_desc, bp_addr);
	if (!brkp) {
		printf("Breakpoint not found\n");
		return;
	}
	iopm_put_short(brkp->pc, brkp->text);

	brkp->pc = 0;
	brkp->flags = 0;
	brkp->unique_id = NULL;
	printf("IOPM %s breakpoint (%#x) cleared\n",
		s_str(i_desc), bp_addr);

	iomap_restore(saved_map);
}

static void
iopm_clear_brkpts()
{
	register iopm_data_t	*i_desc = iopm_cur();
	register BRK_TAB	*brkp;
	uint	saved_map = iomap_save();

	for (brkp = brk_tab; brkp != brk_tab + NUM_BREAKPOINTS; brkp++) {

		if (brkp->unique_id != (void *) i_desc)
			continue;

		iopm_put_short(brkp->pc, brkp->text);

		brkp->pc = 0;
		brkp->flags = 0;
		brkp->unique_id = NULL;
	}
	iomap_restore(saved_map);
}

static void
iopm_display_brkpts()
{
	register iopm_data_t	*i_desc = iopm_cur();
	register BRK_TAB	*brkp;

	for (brkp = brk_tab; brkp != brk_tab + NUM_BREAKPOINTS; brkp++) {
		if (brkp->unique_id != (void *) i_desc)
			continue;
		printf("        IOPM %s (%#x)\n", s_str(i_desc), brkp->pc);
	}
}


static void
iopm_brkpt_check()
{
	uint			saved_map = iomap_save();
	register iopm_data_t	*i_desc = iopm_cur();
	uint			iopm_pc;
	uchar			*name;

	iopm_pc = iopm_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_PC);

	if (iopm_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_VECTOR) == INSTERR &&
	  (bp_search((void *) i_desc, iopm_pc))) {

		printf("IOPM %s hit breakpoint ", s_str(i_desc));
		i_desc->at_brkpt = 1;
		name = iopm_symbolic_version(iopm_pc);
		if (!name)
			printf("(0x%08x)\n", iopm_pc);
		else
			printf("(%s)\n", name);

	}
	iomap_restore(saved_map);
}

static uint
iopm_dis(addr)
uint	addr;
{
	register uint		i;
	register ushort		*mp;
	uint			nwords;

	char		acode[80];
	extern char	atext[];
	extern ushort	mem[];

	printf("%08lx:  ", addr);
	nwords = m68k_dis(addr);
	acode[0] = '\0';
	for (i = 0, mp = mem; i < nwords; i++) {
		sprintf(acode, "%s%04x ", acode, *mp);
		mp++;
	} 
	printf("%-32s%s\n", acode, atext);
	return(nwords);
}

static void
iopm_step()
{
	uint			saved_map = iomap_save();
	register iopm_data_t	*i_desc = iopm_cur();
	register BRK_TAB	*brkp = NULL;
	uint			iopm_pc;

	iopm_pc = iopm_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_PC);

	if (i_desc->at_brkpt) {
		brkp = bp_search((void *) i_desc, iopm_pc);
		if (brkp) {
			iopm_put_short(iopm_pc, brkp->text);
		} else
			i_desc->at_brkpt = 0;
		i_desc->state = STATE_STOPPED;
	}

	(void)iopm_tdb_cmd(TDB_CMD_STEP);
	delay_and_poll_until(HZ * 5, iopm_tdb_done, NULL);
	iopm_check_printf();

	if (i_desc->at_brkpt) {
		i_desc->at_brkpt = 0;
		iopm_put_short(iopm_pc, ILLEGAL_INSTR);

	}
	if (!iopm_tdb_done()) {
		printf("IOPM did not return from single step\n");
		i_desc->state = STATE_DECONFIG;
		rtn_to_monitor();
	}
	iomap_restore(saved_map);
}

static void
iopm_jump()
{
	register iopm_data_t	*i_desc = iopm_cur();
	uint			iopm_pc;
	register int		ilen;
	register ushort		text;

	if (!m68k_at_jsr()) {
		iopm_step();
		return;
	}

	iopm_pc = iopm_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_PC);

	if (i_desc->at_brkpt) 
		iopm_step();

	ilen = m68k_dis(iopm_pc) * sizeof(ushort);
	text = iopm_get_short(iopm_pc + ilen);

	iopm_put_short(iopm_pc + ilen, ILLEGAL_INSTR);

	(void)iopm_tdb_cmd(TDB_CMD_GO);
	delay_and_poll_until(HZ * 5, iopm_tdb_done, NULL);
	iopm_check_printf();
	iopm_put_short(iopm_pc + ilen, text);

	i_desc->at_brkpt = 0;

	if (!iopm_tdb_done()) {
		printf("IOPM did not hit temporary breakpoint\n");
		i_desc->state = STATE_RUNNING;
		rtn_to_monitor();
	}
}

static uint
iopm_prepare_to_go()
{
	register iopm_data_t	*i_desc = iopm_cur();

	if (i_desc->at_brkpt)
		iopm_step();
	return(!i_desc->at_brkpt);
}

static void
iopm_go()
{
	register iopm_data_t	*i_desc = iopm_cur();
	
	ASSERT(!i_desc->at_brkpt);
	if (i_desc->state != STATE_STOPPED)
		return;

	(void)iopm_tdb_cmd(TDB_CMD_GO);

	i_desc->state = STATE_RUNNING;
}

static void
iopm_stop()
{
	uint			saved_map = iomap_save();
	register iopm_data_t	*i_desc = iopm_cur();
	register iopm_comm_t	*iopcommp;

	if (i_desc->state != STATE_RUNNING)
		return;
	i_desc->state = STATE_STOPPED;

	iomap_restore(i_desc->iopm_comm_map);	/* now iopm_comm_ptr is valid */
	iopcommp = i_desc->iopm_comm_ptr;

	if (!iopcommp->o_tdb.done ) {
		*(uchar *)iopm_map(i_desc, INT_CTRL_REG) = INTR7 | INTR_ON;

		delay_no_poll(5);

		iomap_restore(i_desc->iopm_comm_map);

		if (!iopcommp->o_tdb.done) {
			printf("IOPM %s didn\'t stop\n", s_str(i_desc));
			i_desc->state = STATE_RUNNING;
		} else
			iopcommp->o_tdb.cmd = TDB_CMD_NONE;
	}
	iomap_restore(saved_map);
}

static	void
iopm_show_trap()
{
	(void)iopm_tdb_cmd(TDB_CMD_SHOW_TRAP);
	printf("\n");
	iopm_dis(iopm_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_PC));
	printf("\n");
}

/*
 * iopm_clock() - must return zero because the upkern pm returns 1.
 */
uint
iopm_clock(cant_own_upkern)
uint cant_own_upkern;
{
	uint  			savmap = iomap_save();
	register iopm_data_t	*i_desc = iopm_cur();
	register iopm_comm_t 	*iopcommp;
	uint 			clock_ok;

	if (i_desc->state != STATE_RUNNING)
		return(0);

	if (i_desc->offline_delay) {		/* If glitch causes iopm to */
		i_desc->offline_delay--;	/* go offline, delay clock  */
		return(0);			/* for about 10 seconds.    */
	}

	iomap_restore(i_desc->iopm_comm_map);	/* now iopm_comm_ptr is valid */
	iopcommp = i_desc->iopm_comm_ptr;

	if (!probe_rd_long(&iopcommp->o_reprime_clock, &clock_ok)) {
		printf("bus error reading iopm clock flag\n");
		iopm_offline_handler();	/* Board may be temporarily offline. */
		iomap_restore(savmap);
		return(0);		/* Because upkern returns 1. */
	} else if ( !clock_ok )
		iopcommp->o_clocks_dropped++;
	else if (!probe_wr_long(&iopcommp->o_reprime_clock, 0))
		printf("bus error clearing iopm clock flag\n");
	else if (!probe_wr_byte(iopm_map(i_desc, INT_CTRL_REG), 
						INTR6 | INTR_ON))
		printf("bus error setting iopm clock interrupt \n");

	iopm_already_offline = 0;
	iomap_restore(savmap);
	return(0);		/* Because upkern returns 1. */
}

void
iopm_check_printf()
{
	register uint		i, spl_level;
	register iopm_data_t	*i_desc = iopm_cur();
	uint			savemap = iomap_save();
	register iopm_comm_t 	*iopm_o;
	uint			cnt;

	if (i_desc->offline_delay)
		return;	/* delay checking for printf. */

	switch (i_desc->state) {
	case STATE_INIT:
	case STATE_DECONFIG:
	case STATE_LOADED:
		return;
	case STATE_ENABLED:
	case STATE_RUNNING:
	case STATE_STOPPED:
		break;
	default:
		ASSERT(0);
	}

	iomap_restore(i_desc->iopm_comm_map);	/* now iopm_comm_ptr is valid */
	iopm_o = i_desc->iopm_comm_ptr;

	if (!probe_rd_long(&iopm_o->o_spm_printf_count, &cnt)) {
		printf("Error reading local memory of IOPM %s\n",
			s_str(i_desc));
		iopm_offline_handler();	/* Board may be temporarily offline. */
		iomap_restore(savemap);
		return;
	} else if (cnt && ok_to_put_char((void *) i_desc)) {
		K_ASSERT(cnt <= sizeof(iopm_o->o_char_buff));
		spl_level = spl1();
		for (i=0; i < cnt; i++) {
			if (last_put_char() == '\n' &&
				iopm_o->o_char_buff[i] != '\n') {

				print_time_stamp(0);
				printf("IOPM%s: ", s_str(i_desc));
			}
			put_char(iopm_o->o_char_buff[i]);
		}
		splx(spl_level);

		iopm_o->o_spm_printf_count = 0;  /* signal we are done */
		sticky_put_char((void *) i_desc);
	}
	iopm_already_offline = 0;
	iomap_restore(savemap);
}

static uint
iopm_prep_for_syms(bss_end, text_end)
unchar	*bss_end;
unchar	*text_end;
{
	return(0);
}

static uint
iopm_load_syms(ap, hd)
uchar	*ap, *hd;	/* use real declarations when really implementing */
{
	return (1);	/* succeed by doing nothing (Politics in action!) */
}

/*
 * iopm_load -- Returns 0 for no-op, ~0 on error, or entry point.
 */

uint
iopm_load(source_dev, umod_flag)
char	*source_dev;
uint	umod_flag;
{
	register iopm_data_t	*desc = iopm_cur();

	printf("Loading IOPM in slot %s\n", s_str(desc));

	if (!(desc->state == STATE_INIT) && !(desc->state == STATE_LOADED)) {
		printf("IOPM %s: wrong state for downloading.\n", s_str(desc));
		return (~0);
	}

	if (desc->state == STATE_LOADED)
		printf("IOPM %s: overwriting previously downloaded code.\n",
			s_str(desc));

	if (!umod_flag)
		desc->entry = load_m68k(source_dev);
	else
		desc->entry = load_m68k("umod");

	if (desc->entry == ~0)
		printf("Download failed!\n");
	else {
		desc->state = STATE_LOADED;
		last_iopm_loaded = desc;
		printf("\rdownload completed.\n");
	}
	return (desc->entry);
}

uint
iopm_check_coffdata(addr)
uint	addr;
{
	uint				saved_map = iomap_save();
	register iopm_data_t		*i_desc = iopm_cur();

	if (addr < IOPM_RAM_START ||
	    addr >= (IOPM_RAM_START + i_desc->memsize)) {
		printf("0x%08x is a bad program address for an IOPM.\n", addr);
		printf("Indicates download code is not an IOPM O.S.!\n");
		iomap_restore(saved_map);
		return(0);
	}
	iomap_restore(saved_map);
	return(1);
}

uint
iopm_check_magic(number)
short	number;
{
	uint	sns = find_first_pm();
	unchar	slot_id;

	if (sns == SNS_NO_PM) {
		printf("no processor modules configured\n");
		return(0);
	}
	slot_id = sbus_config.slot_id[SLOT_FM_SNS(sns)];
	switch (number) {
		case A68020MAGIC :
			if (slot_id == SBUS_PM20)
				return(1);
		ASSERT(slot_id == SBUS_DPM40_DBL || slot_id == SBUS_DPM40_SGL);
			printf("68020 iopm code cannot run on 68040 system\n");
			return(0);
		case M68040MAGIC_OS :
			if (slot_id == SBUS_DPM40_DBL ||
				slot_id == SBUS_DPM40_SGL)
				return(1);
			ASSERT(slot_id == SBUS_PM20);
			printf("68040 iopm code cannot run on 68020 system\n");
			return(0);
		default :
			printf("unrecognized magic number for iopm code\n");
			return(0);
	}
}

/* iopm_prep_boot() - load the iopm OS from the kernel boot device to the
 *		      current instance of an IOPM.  This routine checks 
 * 		      that the device board is a dsdb.
 * Returns 0 for no-op, ~0 on error, or entry point.
 */

uint
iopm_prep_boot(k_root_dev, umod_flag)
char	*k_root_dev;
uint	umod_flag;
{
	register iopm_data_t	*desc = iopm_cur();
	char			*iopm_boot_dev;

	if (desc->state == STATE_LOADED)
		return (0);

	if (desc->db_id != IOPM_DB_ID_DSDB)
		return (0);

	iopm_boot_dev = extract_iopm_dev(k_root_dev);
	return (iopm_load(iopm_boot_dev, umod_flag /*UMOD_OFF*/));
}

void
iopm_monitor_start()
{
	register iopm_data_t	*desc = iopm_cur();
	register uint           saved_map = iomap_save();
	register iopm_comm_t 	*iopcommp;
	uint			b_response;
	register uint		loopcnt;

	/* Spm_Mem must be previously set up by spm_mem_copy in PM_start */
	ASSERT(spm_mem_copy_called);

	if (!(desc->state == STATE_LOADED) && !(desc->state == STATE_RUNNING)) {
		printf("IOPM %s present but not loaded.\n", s_str(desc));
		return;
	}

	iomap_restore(desc->iopm_comm_map);	/* now iopm_comm_ptr is valid */
	iopcommp = desc->iopm_comm_ptr;
	iopcommp->o_tdb.cmd = TDB_CMD_GO;
	iopcommp->o_boot_response = 0;

	iopm_start_hardware(desc);

	/* wait for IOPM boot response flag */
	loopcnt = HZ * 2;
/* FIX NOW joe, possibly set the state variable. */
	do {
		delay_check_iomap(1);
		if (--loopcnt == 0) {
			printf("slot %s: IOPM to SPM boot response timeout\n",
				s_str(desc));
			rtn_to_monitor();
		}
/* FIX NOW joe, possibly set the state variable. */
		if (!probe_rd_long(&iopcommp->o_boot_response, &b_response)) {
			printf("slot %s: Bad read of IOPM boot response\n",
				s_str(desc));
			rtn_to_monitor();
		}
	} while (!b_response);

/* FIX NOW joe, possibly set the state variable. */
	if ( iopcommp->iopm_version != IOPM_VERSION )
		printf("Mismatched iopm_os version numbers: iopm:%x, spm:%x\n",
		       iopcommp->iopm_version, IOPM_VERSION );
	else {
		printf("IOPM in slot %s booted.\n", s_str(desc));
		desc->state = STATE_RUNNING;
	}
	iomap_restore(saved_map);
}

/* FIX THIS joe, needs more thought. */
void
iopm_start()
{
	iopm_monitor_start();
}

void
iopm_display_own()
{
	register iopm_data_t	*desc = iopm_cur();
	register iopm_comm_t 	*o;
	uint          		 saved_map = iomap_save();

	switch (desc->state) {
	case STATE_INIT:
	case STATE_DECONFIG:
	case STATE_LOADED:
		printf("Iopm common (own) structure not available yet!\n");
		return;
	case STATE_ENABLED:
	case STATE_RUNNING:
	case STATE_STOPPED:
		break;
	default:
		ASSERT(0);
	}

	iomap_restore(desc->iopm_comm_map);	/* now iopm_comm_ptr is valid */
	o = desc->iopm_comm_ptr;

	printf("\nIOPM COMMON (OWN) STRUCTURE:\n\n");

	printf("slot   %8s   dev brd id    %2x   ",
		s_str(desc), o->dbtype);
	printf("mem size  0x%x   mem free  0x%x\n",
		o->memsize, o->freemem);

	printf("magic  %8x   stbl    %8x   dropped clocks %3d\n",
		o->iopm_version, o->stblp, o->o_clocks_dropped);

	printf("ktoib  %8x   itokbp  %8x   ktois     %8x   itoksp   %8x\n",
		o->ktoib, o->itokbp, o->ktois, o->itoksp);

	iomap_restore(saved_map);
}

static void
iopm_deconfig_any(i_desc)
register iopm_data_t	*i_desc;
{
	if (i_desc->state != STATE_INIT)
		return;

	i_desc->state = STATE_DECONFIG;
	/* FIX joe, get rid of slot stuff. */
	/* declan : yes , but how ? */
	printf("deconfiguring IOPM in slot %s\n", s_str(i_desc));
	cpu_set_slot(NULL, i_desc->cu_unit.w.sns);
	cpu_deactivate();
	BOARD_DECONFIG(i_desc->cu_unit.s.slot);
}

static void
iopm_deconfigure()
{
	register iopm_data_t	*i_desc = iopm_cur();

	if (i_desc->state != STATE_INIT) {
		printf("IOPM is not at initial state\n");
		rtn_to_monitor();
	}

	iopm_deconfig_any(i_desc);
	cpu_set_cur(&i_desc->cpu_data); /* declan question : why this */
}

static void
iopm_solo()
{
	register iopm_data_t	*i_desc = iopm_cur();
	register iopm_data_t	*desc = iopm_data;

	if (i_desc->state != STATE_INIT) {
		printf("IOPM is not at initial state\n");
		rtn_to_monitor();
	}
	for ( ; desc != next_iopm_data; desc++) {
		if (desc == i_desc)
			continue;
		if (desc->state != STATE_INIT)
			continue;
		iopm_deconfig_any(desc);
	}
}

static void
iopm_configure()
{
	register iopm_data_t	*i_desc = iopm_cur();

	if (i_desc->state != STATE_DECONFIG) {
		printf("IOPM is not deconfigured\n");
		rtn_to_monitor();
	}

	i_desc->state = STATE_INIT;
	printf("configuring IOPM in slot %s\n", s_str(i_desc));
	BOARD_CONFIG(i_desc->cu_unit.s.slot);
	cpu_activate();
}

/**************** LOW LEVEL DEBUGGER OPERATIONS *****************/

static uint
iopm_get_byte(addr)
uint	addr;
{
	return(iopm_tdb_cmd(TDB_CMD_GET_CHAR, addr));
}

static void
iopm_put_byte(addr, value)
uint	addr;
uint	value;
{
	iopm_tdb_cmd(TDB_CMD_PUT_CHAR, addr, value);
}

static uint
iopm_get_short(addr)
uint	addr;
{
	return(iopm_tdb_cmd(TDB_CMD_GET_SHORT, addr));
}

static void
iopm_put_short(addr, value)
uint	addr;
uint	value;
{
	(void)iopm_tdb_cmd(TDB_CMD_PUT_SHORT, addr, value);
}

static uint
iopm_get_long(addr)
uint	addr;
{
	return(iopm_tdb_cmd(TDB_CMD_GET_LONG, addr));
}

static void
iopm_put_long(addr, value)
uint	addr;
uint	value;
{
	(void)iopm_tdb_cmd(TDB_CMD_PUT_LONG, addr, value);
}

static uint
iopm_get_named_reg(name)
unchar	*name;
{
	return(iopm_tdb_cmd(TDB_CMD_GET_REG, (uint)m68k_valid_reg_name(name)));
}

static void
iopm_put_named_reg(name, value)
unchar	*name;
uint	value;
{
	(void)iopm_tdb_cmd(TDB_CMD_GET_REG, m68k_valid_reg_name(name), value);
}


/******************************************************/
/*** ==>>> called when method not implemented <<<== ***/
void
not_implemented()
{
	printf("Not implemented yet!\n");
	rtn_to_monitor();
}
uint
u_not_implemented()
{
	not_implemented();
	return(0);
}

static void
iopm_return_void()
{
	return;
}
/*** ============================================== ***/

static void
iopm_setup_methods(mp)
register method_t	*mp;
{
	/* SPM provided services. */
	mp->check_printf = iopm_check_printf;
	mp->send_clock = iopm_clock;

	/* Other routines called anytime. */
	mp->stop = iopm_stop;
	mp->check_tdb = iopm_check_tdb;

	/* Basic control functions. */
	mp->display_slot = iopm_display_slot;
	mp->load = iopm_load;
	mp->check_coffdata = iopm_check_coffdata;
	mp->check_magic = iopm_check_magic;
	mp->prep_for_syms = iopm_prep_for_syms;
	mp->load_syms = iopm_load_syms;
	mp->set_pm_id = iopm_return_void;
	mp->prep_boot = iopm_prep_boot;
	mp->start = iopm_start;
	mp->prepare_to_go = iopm_prepare_to_go;
	mp->go = iopm_go;

	mp->pm_first = u_not_implemented;
	mp->print_vtop = not_implemented /* iopm_print_vtop?? */;
	mp->mem_stripe_init = u_not_implemented;
	mp->build_page_table = iopm_build_page_table;
	mp->get_u_addr = u_not_implemented;

	/* Debugger primitives */
	mp->get_byte = iopm_get_byte;
	mp->put_byte = iopm_put_byte;
	mp->get_short = iopm_get_short;
	mp->put_short = iopm_put_short;
	mp->get_long = iopm_get_long;
	mp->put_long = iopm_put_long;
	mp->get_named_reg = iopm_get_named_reg;
	mp->put_named_reg = iopm_put_named_reg;

	/* Debugger menu operations. */
	mp->show_trap = iopm_show_trap;
	mp->step = iopm_step;
	mp->jump = iopm_jump;
	mp->display_dis = iopm_dis;
	mp->backtrace = iopm_backtrace;
	mp->valid_fp = iopm_valid_fp;

	/* Breakpoint operations. */
	mp->find_bp = iopm_find_bp;
	mp->brkpt_check = iopm_brkpt_check;
	mp->set_brkpt = iopm_set_brkpt;
	mp->clear_one_brkpt = iopm_clear_one_brkpt;
	mp->clear_brkpts = iopm_clear_brkpts;
	mp->display_brkpts = iopm_display_brkpts;

	/* Misc unrelated operations. */
	mp->display_own = iopm_display_own;
	mp->symbolic_version = iopm_symbolic_version;
	mp->check_tags = not_implemented;
	mp->solo = iopm_solo;
	mp->deconfigure = iopm_deconfigure;
	mp->configure = iopm_configure;
	mp->cpu_copyout = copy_to_iopm;
	mp->cpu_copyin = copy_from_iopm;
	mp->cpu_malloc = iopm_malloc;
	mp->cpu_zero = iopm_zero;
}

/* iopm_comm_init(desc) - Spm will initialize parts of the iopm_comm
 * structure on 'reset'.
 */
void
iopm_comm_init(desc)
iopm_data_t	*desc;
{
	uint			saved_map = iomap_save();
	register iopm_comm_t	*iopm_o;

	iomap_restore(desc->iopm_comm_map);	/* now iopm_comm_ptr is valid */
	iopm_o = desc->iopm_comm_ptr;
	iopm_o->memsize = desc->memsize;

	iopm_o->slot = (desc->cu_unit.s.slot << 4) |
	  (desc->cu_unit.s.subslot == IUNIT_NONE ?
	   0xf : desc->cu_unit.s.subslot);
	iopm_o->dbtype = desc->db_id;

	iomap_restore(saved_map);
}

uint
iopm_any_setup(slot, sub_slot)
uint		slot;
uint		sub_slot;
{
	register iopm_data_t	*i_desc = next_iopm_data;
	static method_t		iopm_methods;
	iunit_t			iunit;
	uint			save;

	ASSERT(i_desc >= iopm_data);
	ASSERT(i_desc < iopm_data + MAX_IOPMS);

	/* FIX joe, get rid of macros and slot stuff. */
	iunit.i = IUNIT_NO_DEV;
	iunit.s.slot = slot;
	iunit.s.subslot = sub_slot;

	if (i_desc == iopm_data)
		iopm_setup_methods(&iopm_methods);
	i_desc->cu_methods = &iopm_methods;
	cpu_enroll(&i_desc->cpu_data);

	i_desc->cu_unit = iunit;
	i_desc->state = STATE_INIT;
	i_desc->offline_delay = 0;
	i_desc->db_id = IOPM_NO_DB;

	/* save map info for iopm_comm */
	save = iomap_save();
	i_desc->iopm_comm_ptr = (iopm_comm_t *)iopm_map(i_desc, IOPM_COMM);
	i_desc->iopm_comm_map = iomap_save();
	iomap_restore(save);

	iopm_init(i_desc);
	iopm_comm_init(i_desc);

	next_iopm_data++;
	return (i_desc->db_id);
}

uint
iopm_setup(slot)
uint		slot;
/* FIX joe, get rid of slot stuff. */
{
	return (iopm_any_setup(slot, SBUS_NO_BOARD));
}

/*
 * iopm_download_dsdb -- pre-load any DSDBs, returns non-zero on error
 */

uint
iopm_download_dsdb(src_dev)
char	*src_dev;
{
	register int		slot, iom;
	register iom_config_t	*iomp;
	register css_bd_dat_t	*csbp;
	char			path[DEV_NAMLEN];

	strcpy(path, src_dev);
	if (!extract_iopm_dev(path))
		return (1);

	for (slot = 0; slot < SBUS_NUM_SLOT; slot++) {
		if (sbus_config.slot_id[slot] != SBUS_IOPM ||
		    sbus_config.dev_board_id[slot] != IOPM_DB_ID_DSDB ||
		    cpu_set_slot(NULL, MAKE_SNS(slot, NO_SUB_SLOT)) == 0)
			continue;
		if (iopm_cur()->state == STATE_LOADED)
			continue;
		if (iopm_load(path, UMOD_OFF) == ~0)
			return (1);
	}

	for (iomp = iom_config, iom = 0; iom < iom_count; iom++, iomp++) {
		csbp = iomp->ex_card_cage.css_bd_dat;
		for (slot = 0; slot < SBUS_NUM_SLOT; slot++, csbp++) {
			if (csbp->css_slot_id != SBUS_IOPM ||
			    csbp->dev_board_id != IOPM_DB_ID_DSDB ||
			    !cpu_set_slot(NULL, MAKE_SNS(iomp->iom_slot, slot)))
				continue;
			if (iopm_cur()->state == STATE_LOADED)
				continue;
			if (iopm_load(path, UMOD_OFF) == ~0)
				return (1);
		}
	}

	return (0);
}
