/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) dpm40_tdb.c: version 25.1 created on 11/27/91 at 15:29:50	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)dpm40_tdb.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
/*  add SCCS header here */

/*
 * dpm40_tdb.c -- dpm40 tdb driver
 *
 * This driver is one of many drivers through which the SPM common code will
 * access some boards on a system 90.  These drivers were created to keep 
 * the SPM common code de-coupled from cpu specific handling routines.
 */

#ifdef M68020
#undef M68020
#define M68040
#endif

#include "sys/types.h"
#include "sys/own.h"
#include "sys/sbus.h"
#include "sys/sbus_pm.h"
#include "sys/spm_mem.h"
#include "sys/trap.h"
#include "sys/kmem.h"
#include "sys/sysmacros.h"
#include "sys/ipcc.h"
#include "sys/ints.h"
#include "sys/lio.h"

#include "symbol.h"
#include "spm_debug.h"
#include "cpu_method.h"
#include "sbus_conf.h"
#include "global.h"
#include "dpm40_tdb.h"
#include "filehdr.h"



/* externals */
extern unchar	*iomap();
extern unchar	*kmmap();
extern uint	m68k_display_dis();
extern int	copy_from_km();
extern int	copy_to_km();
extern uint	km_allocate();
extern uint	km_clear();
extern uint	load_kern_syms();
extern char	*symstr();

extern uint	dpm40_mem_stripe_init();
extern uint	dpm40_mpfn_to_spfn();
extern void	dpm40_print_vtop();
extern void	dpm40_build_page_table();

static void	dpm40_ipcc_send();

static pm40_info_t	dpm40_data[SBUS_MAX_NUM_PM];
static pm40_info_t	*next_dpm40_data = dpm40_data;
extern uint		pm_entry;


static uint
dpm40_pm_first()
{
	register pm40_info_t *pm_desc;

	for (pm_desc = dpm40_data; pm_desc != next_dpm40_data; pm_desc++)
		if (pm_desc->state != STATE_DECONFIG)
			return(pm_desc->cu_unit.w.sns);
	return(SNS_NO_PM);
}

/* If the current instance is a cpu on a dpm40, pm40_cur will return a
 * descriptor containing information about that cpu.
 */
static pm40_info_t *
dpm40_cur()
{
	pm40_info_t	*pm40_desc;

	pm40_desc = (pm40_info_t *) cpu_get_cur();
	ASSERT(pm40_desc >= dpm40_data);
	ASSERT(pm40_desc < next_dpm40_data);
	return(pm40_desc);
}

static uint
dpm40_tdb_done()
{
	uint			saved_map = iomap_save();
	register pm40_info_t	*pm40_desc = dpm40_cur();
	uint			retval;

	iomap_restore(pm40_desc->own_map);	/* now own_ptr is valid */
	retval = pm40_desc->own_ptr->o_tdb.done;

	iomap_restore(saved_map);
	return(retval);
}

char *
pm_name(unit)
iunit_t	unit;
{
	static char	str[8];

	switch (sbus_config.slot_id[unit.s.slot]) {
	case SBUS_DPM40_DBL:
	case SBUS_DPM40_SGL:
		sprintf(str, "%u/%c", unit.s.slot,
		  (unit.s.subslot == PMA_SUB_SLOT) ? 'A' : 'B');
		break;
	default:
		sprintf(str, "%u", unit.s.slot);
	}
	return (str);
}

char *
dpm40_name(pm40_desc)
pm40_info_t	*pm40_desc;
{
	return (pm_name(pm40_desc->cu_unit));
}

static uint
dpm40_tdb_cmd(cmd, arg, value)
uint	cmd;
uint	arg;		/* optional */
uint	value;		/* optional */
{
	uint			saved_map = iomap_save();
	register pm40_info_t	*pm40_desc = dpm40_cur();
	register own_t		*o;
	uint			retval;
	void			dpm40_check_printf();

	if (pm40_desc->state != STATE_STOPPED &&
	  pm40_desc->state != STATE_STOP_BRK) {
		printf("PM:%s has not been stopped\n", dpm40_name(pm40_desc));
		rtn_to_monitor();
	}
		
	iomap_restore(pm40_desc->own_map);	/* now own_ptr is valid */
	o = pm40_desc->own_ptr;

	if (!o->o_tdb.done) {
		printf("PM:%s is not responding\n", dpm40_name(pm40_desc));
		rtn_to_monitor();
	}

	o->o_tdb.value = value;
	o->o_tdb.arg = arg;
	o->o_tdb.cmd = cmd;
	o->o_tdb.done = 0;

	if (cmd == TDB_CMD_GO || cmd == TDB_CMD_STEP) {
		iomap_restore(saved_map);
		return(0);
	}

	delay_and_poll_until(HZ * 5 + 1, dpm40_tdb_done, NULL);
	dpm40_check_printf();

	if (!o->o_tdb.done) {
		printf("PM:%s is not responding\n", dpm40_name(pm40_desc));
		rtn_to_monitor();
	}
	if (o->o_tdb.error) {
		printf("tdb error %#x\n", o->o_tdb.error);
		rtn_to_monitor();
	}
	retval = o->o_tdb.value;
	iomap_restore(saved_map);
	return(retval);
} /* dpm40_tdb_cmd */

/******** PM40 METHODS ********/


static void
dpm40_check_printf()
{
	register pm40_info_t	*pm40_desc = dpm40_cur();
	uint			savemap;
	register own_t	 	*o;
	extern void		kern_check_printf();

	switch (pm40_desc->state) {
	case STATE_INIT:
	case STATE_DECONFIG:
	case STATE_LOADED:
		return;
	case STATE_START:
	case STATE_RUNNING:
	case STATE_STOPPED:
	case STATE_STOP_BRK:
		break;
	default:
		ASSERT(0);
	}
	savemap = iomap_save();
	iomap_restore(pm40_desc->own_map);	/* now own_ptr is valid */
	o = pm40_desc->own_ptr;

	if (o->o_spm_printf_count != o->o_printf_count &&
	    ok_to_put_char((void *) pm40_desc)) {
		kern_check_printf(o, pm40_desc->cu_unit);
		sticky_put_char((void *) pm40_desc);
	}
	iomap_restore(savemap);
} /* dpm40_check_printf */

static uint
dpm40_send_clock(upkern_id)
uint	upkern_id;
{
	register own_t	 	*o;
	register pm40_info_t	*pm40_desc = dpm40_cur();
	uint			saved_map = iomap_save();

	if (pm40_desc->state != STATE_RUNNING)
		return(0);

	iomap_restore(pm40_desc->own_map);	/* now own_ptr is valid */
	o = pm40_desc->own_ptr;

	switch (o->o_clock) {
	case CLOCK:
	case CLOCK_LEADER:			/* not yet consumed */
		o->o_clocks_dropped++;
		/* fall through */
	case	0: 				/* not yet enabled */
		iomap_restore(saved_map);
		return(0);
	case CLOCK_ENABLE:			/* ready for another */
		break;
	default:
		ASSERT(0);
	}

	if (upkern_id == pm40_desc->pm_id)
		o->o_clock = CLOCK_LEADER;
	else
		o->o_clock = CLOCK;

/* hanna FIX: this shift by 29 bits */
/* hanna FIX: actually, make a macro to do this, like sbus_pm.h's send_pm_int */
	*(uint *)iomap(pm40_desc->cu_unit.s.slot, PM_INT_REQ_REG) =
	  PM_LEVEL_SIX_INT_REQ |
	  ((pm40_desc->cu_unit.s.subslot == PMA_SUB_SLOT) << 29);
	iomap_restore(saved_map);
	return(upkern_id == pm40_desc->pm_id);
}

/**************** LOW LEVEL DEBUGGER OPERATIONS *****************/


static uint
dpm40_get_byte(addr)
uint	addr;
{
	return(dpm40_tdb_cmd(TDB_CMD_GET_CHAR, addr));
}

static void
dpm40_put_byte(addr, value)
uint	addr;
uint	value;
{
	dpm40_tdb_cmd(TDB_CMD_PUT_CHAR, addr, value);
}

static uint
dpm40_get_short(addr)
uint	addr;
{
	return(dpm40_tdb_cmd(TDB_CMD_GET_SHORT, addr));
}

static void
dpm40_put_short(addr, value)
uint	addr;
uint	value;
{
	dpm40_tdb_cmd(TDB_CMD_PUT_SHORT, addr, value);
}

static uint
dpm40_get_long(addr)
uint	addr;
{
	return(dpm40_tdb_cmd(TDB_CMD_GET_LONG, addr));
}

static void
dpm40_put_long(addr, value)
uint	addr;
uint	value;
{
	dpm40_tdb_cmd(TDB_CMD_PUT_LONG, addr, value);
}

static uint
dpm40_get_named_reg(name)
unchar	*name;
{
	return(dpm40_tdb_cmd(TDB_CMD_GET_REG, m68k_valid_reg_name(name)));
}

static void
dpm40_put_named_reg(name, value)
unchar	*name;
uint	value;
{
	dpm40_tdb_cmd(TDB_CMD_PUT_REG, m68k_valid_reg_name(name), value);
}


/*****************************************/

static unchar *
dpm40_symbolic_version(addr)
uint	addr;
{
	return ((unchar *)symstr(addr));
}

static BRK_TAB *
dpm40_find_bp(addr)
uint	addr;
{
	return(bp_search((void *) dpm40_data, addr));
}

/* MAJOR FIX joe, do we want to check for ack of ipcc command. */
static void
dpm40_start()
{
	register pm40_info_t	*pm40_desc = dpm40_cur();
	uint			saved_map = iomap_save();
	register own_t		*o;


	if (pm40_desc->state == STATE_INIT) {
		printf("PM %s must be loaded first\n", dpm40_name(pm40_desc));
		rtn_to_monitor();
	}
	if (pm40_desc->state != STATE_LOADED)
		return;

	pm40_desc->own = sbus_config.pm_own[pm40_desc->pm_id];
	pm40_desc->own_ptr = (own_t *)km_map(pm40_desc->own);
	pm40_desc->own_map = iomap_save();	/* save own mapping */

	/* send kernel root pointer */
	dpm40_ipcc_send(pm40_desc->cu_unit.s.slot, IPCC_PM_NKRP, 
		dpm40_km_to_rde(sbus_config.pm_qtbl[pm40_desc->pm_id]),
		  (pm40_desc->cu_unit.s.subslot == PMA_SUB_SLOT) ?
		   IPCC_ADDR_SLOT_A : IPCC_ADDR_SLOT_B);

	delay_and_poll(5);

	iomap_restore(pm40_desc->own_map);	/* now own_ptr is valid */
	o = pm40_desc->own_ptr;
	o->o_tdb.cmd = TDB_CMD_GO;

	dpm40_ipcc_send(pm40_desc->cu_unit.s.slot, IPCC_EXECUTE, pm_entry,
	  (pm40_desc->cu_unit.s.subslot == PMA_SUB_SLOT) ?
	   IPCC_ADDR_SLOT_A : IPCC_ADDR_SLOT_B);
	/* any check to see that it started ok? */
	delay_and_poll(5);

	pm40_desc->state = STATE_START;

	/* wait for PM boot response flag */

	delay_check_iomap(5);
	delay_and_poll(HZ / 2);		/* needs a little time to start... */

	if (o->o_boot_response) {
		printf("PM %s booted.\n", dpm40_name(pm40_desc));
		pm40_desc->state = STATE_RUNNING;
	} else if ( Spm_Mem->pm_version != SPM_MEM_VERSION ) {
		printf("Mismatched versions: spm: %08x, kernel: %08x\n",
		  SPM_MEM_VERSION, Spm_Mem->pm_version);

		/* treat this condition as a booted kernel */
		pm_booting_done = 1;
		rtn_to_monitor();
	} else {
		printf("PM %s : SPM boot response timeout.\n",
		  dpm40_name(pm40_desc));
		rtn_to_monitor();
	}
	iomap_restore(saved_map);
} /* dpm40_start */

static void
dpm40_stop()
{
	register pm40_info_t	*pm40_desc = dpm40_cur();
	register own_t		*o;
	uint			saved_map = iomap_save();

	if (pm40_desc->state != STATE_RUNNING)
		return;

	pm40_desc->state = STATE_STOPPED;

	iomap_restore(pm40_desc->own_map);	/* now own_ptr is valid */
	o = pm40_desc->own_ptr;

	if (!o->o_tdb.done ) {

/* hanna FIX: this shift by 29 bits */
/* hanna FIX: actually, make a macro to do this, like sbus_pm.h's send_pm_int */
		*(uint *)iomap(pm40_desc->cu_unit.s.slot, PM_INT_REQ_REG) = 
		  PM_LEVEL_SEVEN_INT_REQ |
		  ((pm40_desc->cu_unit.s.subslot == PMA_SUB_SLOT) << 29);

		delay_no_poll(2);

		iomap_restore(pm40_desc->own_map);	/* back to own_ptr */
		if ( ! o->o_tdb.done )
		    printf("PM %s didn\'t stop.\n", dpm40_name(pm40_desc));
		else
			o->o_tdb.cmd = TDB_CMD_NONE;
	}
	iomap_restore(saved_map);
}

static uint
dpm40_check_tdb()
{
	uint			saved_map = iomap_save();
	register pm40_info_t	*pm40_desc = dpm40_cur();
	register own_t	*o;
	uint	retval = 0;

	if (pm40_desc->state != STATE_RUNNING)
		return(0);

	iomap_restore(pm40_desc->own_map);	/* now own_ptr is valid */
	o = pm40_desc->own_ptr;

	if (o->o_tdb.done && o->o_tdb.cmd == TDB_CMD_GO ) {
		retval = 1;
		pm40_desc->state = STATE_STOPPED;
	}
	iomap_restore(saved_map);
	return(retval);
}

static void
dpm40_brkpt_check()
{
	uint			saved_map = iomap_save();
	register pm40_info_t	*pm40_desc = dpm40_cur();
	uint			kernel_pc;

	kernel_pc = dpm40_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_PC);

	if (dpm40_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_VECTOR) == INSTERR &&
	  (bp_search((void *) dpm40_data, kernel_pc))) {

		pm40_desc->state = STATE_STOP_BRK;
		printf("PM %s hit breakpoint (%s)\n",
		  dpm40_name(pm40_desc), dpm40_symbolic_version(kernel_pc));
	}
	iomap_restore(saved_map);
}

static	void
dpm40_show_trap()
{
	dpm40_tdb_cmd(TDB_CMD_SHOW_TRAP);
	dpm40_check_printf();
	printf("\n");

	m68k_display_dis(dpm40_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_PC));
	printf("\n");
}

static void
dpm40_step()
{
	uint			saved_map = iomap_save();
	register pm40_info_t	*pm40_desc = dpm40_cur();
	register BRK_TAB	*brkp = NULL;
	uint			kernel_pc;
	
	kernel_pc = dpm40_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_PC);
	if (pm40_desc->state == STATE_STOP_BRK) {
		brkp = bp_search((void *) dpm40_data, kernel_pc);
		if (brkp)
			*(ushort *)km_map(kernel_pc) = brkp->text;
		pm40_desc->state = STATE_STOPPED;
	}

	dpm40_tdb_cmd(TDB_CMD_STEP);

	delay_and_poll_until(HZ * 5, dpm40_tdb_done, NULL);
	dpm40_check_printf();

	if (brkp)
		*(ushort *)km_map(kernel_pc) = ILLEGAL_INSTR;

	if (!dpm40_tdb_done()) {
		printf("PM %s did not return from single step.\n",
		  dpm40_name(pm40_desc));
		pm40_desc->state = STATE_RUNNING;
		rtn_to_monitor();
	}
	iomap_restore(saved_map);
} /* dpm40_step */

static void
dpm40_jump()
{
	register pm40_info_t	*pm40_desc = dpm40_cur();
	uint			saved_map = iomap_save();
	uint			kernel_pc;
	register ushort		*mapped_pc;
	register int		ilen;
	register ushort		text;

	if (!m68k_at_jsr()) {
		dpm40_step();
		return;
	}
	kernel_pc = dpm40_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_PC);
	if (pm40_desc->state == STATE_STOP_BRK)
		dpm40_step();

	ilen = m68k_dis(kernel_pc) * sizeof(ushort);
	mapped_pc = (ushort *)km_map(kernel_pc + ilen);
	text = *mapped_pc;
	*mapped_pc = ILLEGAL_INSTR;

	dpm40_tdb_cmd(TDB_CMD_GO);
	delay_and_poll_until(HZ * 5, dpm40_tdb_done, NULL);
	dpm40_check_printf();
	*mapped_pc = text;

	if (!dpm40_tdb_done()) {
		printf("PM %s did not hit temporary breakpoint.\n",
		  dpm40_name(pm40_desc));
		pm40_desc->state = STATE_RUNNING;
		rtn_to_monitor();
	}
	iomap_restore(saved_map);
} /* dpm40_jump */

static uint
dpm40_prepare_to_go()
{
	register pm40_info_t	*pm40_desc = dpm40_cur();

	if (pm40_desc->state == STATE_STOP_BRK) {
		dpm40_step();
		dpm40_brkpt_check();
	}
	return(pm40_desc->state != STATE_STOP_BRK);
}

static void
dpm40_go()
{
	register pm40_info_t	*pm40_desc = dpm40_cur();

	if (pm40_desc->state != STATE_STOPPED)
		return;

	dpm40_tdb_cmd(TDB_CMD_GO);
	pm40_desc->state = STATE_RUNNING;
}

static void
dpm40_display_own()
{
	register pm40_info_t	*pm40_desc = dpm40_cur();
	uint			saved_map = iomap_save();
	register struct own	*o;

	switch (pm40_desc->state) {
	case STATE_INIT:
	case STATE_DECONFIG:
	case STATE_LOADED:
		printf("Own structure not yet available\n");
		return;
	case STATE_START:
	case STATE_RUNNING:
	case STATE_STOPPED:
	case STATE_STOP_BRK:
		break;
	default:
		ASSERT(0);
	}
	iomap_restore(pm40_desc->own_map);	/* now own_ptr is valid */
	o = pm40_desc->own_ptr;

	printf("pm_id: %d slot %u/%c\n", o->o_pm_id, o->o_slot,
	  (o->o_pm_id == o->o_base_pm_id) ? 'A' : 'B');
	printf("lock_id  %8x  fpu_type   %8x  cpu_type     %8x\n",
	  o->o_lock_id, o->o_fpu_type, o->o_cpu_type);

	printf("curpri %8x  runrun   %8x  curproc    %8x  switching    %8x\n",
	  o->o_curpri, o->o_runrun, o->o_curproc, o->o_switching);

	printf("lticks %8x  trap_hap %8x  fpu_loaded %8x  is_idle      %8x\n",
	  o->o_lticks, o->o_trap_hap, o->o_fpu_loaded, o->o_is_idle);

	printf(
	  "spin_lock_cnt             %8x  flush_tlb  %8x  stbl         %8x\n",
	  o->o_spin_lock_cnt, o->o_flush_tlb, o->o_stbl);

	printf(
	  "in_int_service            %8x  upkern_proc%8x\n",
	  o->o_in_int_service, o->o_upkern_proc);

	printf(
	  "clocks_dropped            %8x  clock      %8x\n\n",
	o->o_clocks_dropped, o->o_clock);
	iomap_restore(saved_map);
}


static void
dpm40_backtrace(fp)
uint	*fp;
{
	if (fp == NULL)
		fp = (uint *) dpm40_tdb_cmd(TDB_CMD_GET_REG, TDB_REG_A6);
	m68k_backtrace(fp);
}

static uint
dpm40_valid_fp(fp)
uint	fp;
{
	return (fp >= ADDR_EXTRA_STK && fp < ADDR_U_STRUCT ||
		fp >= ADDR_OWN_EXTRA && fp < ADDR_OWN_U_STR);
}

static void
dpm40_set_brkpt(bp_addr)
uint	bp_addr;
{
	uint			saved_map = iomap_save();
	register BRK_TAB	*brkp;

	if (bp_addr < ADDR_KERNEL || bp_addr > kernel_etext) {
		printf("Bad kernel breakpoint address:%#x\n", bp_addr);
		return;
	}
	if (bp_search((void *) dpm40_data, bp_addr)) {
		printf("Breakpoint already set\n");
		return;
	}
	brkp = bp_search(NULL, 0);
	if (!brkp) {
		printf("No more breakpoints available\n");
		return;
	}
	cpu_all_stop();

	brkp->flags = 0;
	brkp->pc = bp_addr;
	brkp->text = *(ushort *)km_map(bp_addr);
	brkp->unique_id = (void *) dpm40_data;
	*(ushort *)km_map(bp_addr) = ILLEGAL_INSTR;

	iomap_restore(saved_map);
}

static void
dpm40_clear_one_brkpt(bp_addr)
uint	bp_addr;
{
	register BRK_TAB	*brkp;
	uint			saved_map = iomap_save();

	brkp = bp_search((void *) dpm40_data, bp_addr);
	if (!brkp) {
		printf("Breakpoint not found\n");
		return;
	}
	cpu_all_stop();
	*(ushort *)km_map(brkp->pc) = brkp->text;
	brkp->pc = 0;
	brkp->flags = 0;
	brkp->unique_id = NULL;
	printf("Kernel breakpoint (%s) cleared\n",
	  dpm40_symbolic_version(bp_addr));

	iomap_restore(saved_map);
}

static void
dpm40_clear_brkpts()
{
	register BRK_TAB	*brkp;
	uint	saved_map = iomap_save();

	for (brkp = brk_tab; brkp != brk_tab + NUM_BREAKPOINTS; brkp++) {
		if (brkp->unique_id != (void *) dpm40_data)
			continue;
		cpu_all_stop();
		*(ushort *)km_map(brkp->pc) = brkp->text;
		brkp->pc = 0;
		brkp->flags = 0;
		brkp->unique_id = NULL;
	}
	iomap_restore(saved_map);
}

static void
dpm40_display_brkpts()
{
	register pm40_info_t	*pm40_desc = dpm40_cur();
	register BRK_TAB	*brkp;

	if (pm40_desc != dpm40_data)
		return;		/* only do for first pm */

	for (brkp = brk_tab; brkp != brk_tab + NUM_BREAKPOINTS; brkp++) {
		if (brkp->unique_id != (void *) pm40_desc)
			continue;
		printf("        kernel (%#x)    %s\n", brkp->pc,
		  dpm40_symbolic_version(brkp->pc));
	}
}

static void
dpm40_check_tags()
{
	register uint	saved_kern_clock_enable = kern_clock_enable;
/* hanna FIX: 40 doesn't currently support this, but the kernel will tell you */

	kern_clock_enable = 0;

	delay_no_poll(2);

	dpm40_tdb_cmd(TDB_CMD_CHECK_TAGS);

	kern_clock_enable = saved_kern_clock_enable;
}

static void
dpm40_deconfig_any(pm40_desc)
register pm40_info_t	*pm40_desc;
{
	uchar	cur_slot = pm40_desc->cu_unit.s.slot;

	if ((pm40_desc->cu_unit.s.subslot == PMA_SUB_SLOT) &&
	    (sbus_config.slot_id[cur_slot] == SBUS_DPM40_DBL)) {
		printf("All PMs in slot %u will be deconfigured.\n",
		  pm40_desc->cu_unit.s.slot);
		dpm40_deconfig_any(pm40_desc + 1); /* next descriptor is B */
	}
	
	pm40_desc->state = STATE_DECONFIG;
	printf("deconfiguring PM %s\n", dpm40_name(pm40_desc));

	cpu_set_slot(NULL, pm40_desc->cu_unit.w.sns);
	cpu_deactivate();
	if (pm40_desc->cu_unit.s.subslot == PMA_SUB_SLOT) {
		BOARD_DECONFIG(cur_slot);
	}
	else
		sbus_config.slot_id[cur_slot] = SBUS_DPM40_SGL;
}

static void
dpm40_solo()
{
	register pm40_info_t	*pm40_desc = dpm40_cur();
	register pm40_info_t	*desc;

	if (pm40_desc->state != STATE_INIT) {
		printf("PM %s is not at initial state.\n",
		  dpm40_name(pm40_desc));
		rtn_to_monitor();
	}
	if (pm40_desc->cu_unit.s.subslot != PMA_SUB_SLOT) {
		printf("PM %s cannot run solo.\n", dpm40_name(pm40_desc));
		rtn_to_monitor();
	}

	for (desc = dpm40_data; desc != next_dpm40_data; desc++)
		if (desc != pm40_desc && desc->state == STATE_INIT)
			dpm40_deconfig_any(desc);
}

static void
dpm40_deconfigure()
{
	register pm40_info_t	*pm40_desc = dpm40_cur();

	if (pm40_desc->state != STATE_INIT) {
		printf("PM %s is not at initial state.\n",
		  dpm40_name(pm40_desc));
		rtn_to_monitor();
	}
	dpm40_deconfig_any(pm40_desc);
} /* dpm40_deconfigure */

static void
dpm40_configure()
{
	register pm40_info_t	*pm40_desc = dpm40_cur();
	uchar	cur_slot = pm40_desc->cu_unit.s.slot;

	if (pm40_desc->state != STATE_DECONFIG) {
		printf("PM is not deconfigured\n");
		return;
	}

	/* check for subslot B being configured when A isn't */
	if ((pm40_desc->cu_unit.s.subslot == PMB_SUB_SLOT) &&
		(sbus_config.slot_id[cur_slot] != SBUS_DPM40_SGL)) {
		printf("Can't configure subslot B if A is not configured\n");
		printf("slot id = %x \n", sbus_config.slot_id[cur_slot]);
		return;
	}

	pm40_desc->state = STATE_INIT;
	printf("configuring PM %s\n", dpm40_name(pm40_desc));
	sbus_config.slot_id[cur_slot] =
		(pm40_desc->cu_unit.s.subslot == PMB_SUB_SLOT) ?
		SBUS_DPM40_DBL : SBUS_DPM40_SGL;
	cpu_activate();
	check_pm_config();
}

static uint
dpm40_prep_for_syms(bss_end, text_end)
unchar	*bss_end;
unchar	*text_end;
{
	km_used = (uint)bss_end;	/* End of kernel bss. */
	kernel_etext = (uint)text_end;	/* Used by PM_build() */
	return(1);
}


/*
 * dpm40_load -- return 0 for no-op, ~0 for error, or kernel entry point
 *		Sets maincpu_mpfn_to_spfn
 */
uint
dpm40_load(source_dev, umod_flag)
char	*source_dev;
uint	umod_flag;
{
	register pm40_info_t	*desc = dpm40_cur();
	uint			entry;
	extern uint		load_kernel();

	if (desc->state == STATE_RUNNING) {
		printf("PM %s: The kernel has already been booted.\n",
		  dpm40_name(desc));
		return (~0);
	} else if (desc->state == STATE_LOADED) {
		printf("PM %s: The kernel has already been loaded.\n",
		  dpm40_name(desc));
		return (~0);
	} else if (desc->state == STATE_DECONFIG) {
		/* "This is not the PM you're looking for...  You don't need
		    to check its ID...  Move along." */
		return (0);
	} else if (!(desc->state == STATE_INIT)) {
		printf("PM %s: Wrong state for downloading Kernel.\n",
		  dpm40_name(desc));
		return (~0);
	}

	maincpu_mpfn_to_spfn = dpm40_mpfn_to_spfn;
	sbus_config.cpu_type = 68040;
	if ((entry = load_kernel(source_dev, umod_flag)) == 0 || entry == ~0) {
		printf("PM %s: Download failed!\n", dpm40_name(desc));
		return (~0);
	}
	else {
		desc->state = STATE_LOADED;
		dpm40_all_set_loaded();
		printf("\rPM %s: download completed.\n", dpm40_name(desc));
	}
	return (entry);
}

uint
dpm40_check_coffdata(addr)
uint	addr;
{
	if (addr < MAINSTORE || addr >= KIO_START) {
		printf("0x%08x is bad program address for the kernel!\n", addr);
		printf("Indicates wrong slot or wrong kernel code!\n");
		return(0);
	}
	return(1);
}

uint
dpm40_check_magic(number)
short	number;
{
	if (number == M68040MAGIC_OS)
		return(1);
	printf("file is not a DPM40 binary\n");
	return(0);
}

uint
dpm40_prep_boot(source_dev, umod_flag)
char	*source_dev;
uint	umod_flag;
{
	register pm40_info_t	*desc = dpm40_cur();

	if (desc->state == STATE_LOADED)
		return(0);

	return (dpm40_load(source_dev, umod_flag));
}

static void
dpm40_display_slot()
{
	pm40_info_t	*pm40_desc = dpm40_cur();
	char		*state;

	switch (pm40_desc->state) {
	case STATE_INIT:
		state = "under PROM control";
		break;
	case STATE_DECONFIG:
		state = "deconfigured";
		break;
	case STATE_LOADED:
		state = "loaded, ready to run";
		break;
	case STATE_START:
		state = "started and has yet to respond";
		break;
	case STATE_RUNNING:
		state = "running";
		break;
	case STATE_STOPPED:
		state = "stopped in debugger";
		break;
	case STATE_STOP_BRK:
		state = "stopped at breakpoint";
		break;
	default:
		ASSERT(0);
	}

	printf("PM %s is %s.\n", dpm40_name(pm40_desc), state);
}


/*
 * dpm40_map_own -- return the own pointer for the current 68040
 */

own_t *
dpm40_map_own()
{
	pm40_info_t	*pm40_desc = dpm40_cur();

	iomap_restore(pm40_desc->own_map);	/* now own_ptr is valid */
	return (pm40_desc->own_ptr);
}


/*
 * dpm40_get_pm_id -- return the current PM ID
 */

uint
dpm40_get_pm_id()
{
	return (dpm40_cur()->pm_id);
}

static uint
dpm40_get_u_addr()
{
	return (ADDR_U_STRUCT);
}

static void
dpm40_set_pm_id()
{
	register pm40_info_t	*pm40_desc = dpm40_cur();

	ASSERT(pm40_desc->state == STATE_INIT);
	pm40_desc->pm_id = set_pm_id();
}

/************** END OF METHODS ***************/

static void
dpm40_setup_methods(mp)
register method_t	*mp;
{
	mp->check_printf = dpm40_check_printf;
	mp->send_clock = dpm40_send_clock;
	mp->stop = dpm40_stop;

	mp->display_slot = dpm40_display_slot;

	mp->load = dpm40_load;
	mp->set_pm_id = dpm40_set_pm_id;
	mp->check_coffdata = dpm40_check_coffdata;
	mp->check_magic = dpm40_check_magic;
	mp->prep_for_syms = dpm40_prep_for_syms;
	mp->load_syms = load_kern_syms;
	mp->prep_boot = dpm40_prep_boot;
	mp->start = dpm40_start;
	mp->check_tdb = dpm40_check_tdb;
	mp->prepare_to_go = dpm40_prepare_to_go;
	mp->go = dpm40_go;
	mp->show_trap = dpm40_show_trap;

	mp->pm_first = dpm40_pm_first;
	mp->print_vtop = dpm40_print_vtop;
	mp->mem_stripe_init = dpm40_mem_stripe_init;
	mp->build_page_table = dpm40_build_page_table;
	mp->get_u_addr = dpm40_get_u_addr;

	mp->get_byte = dpm40_get_byte;
	mp->put_byte = dpm40_put_byte;
	mp->get_short = dpm40_get_short;
	mp->put_short = dpm40_put_short;
	mp->get_long = dpm40_get_long;
	mp->put_long = dpm40_put_long;
	mp->get_named_reg = dpm40_get_named_reg;
	mp->put_named_reg = dpm40_put_named_reg;

	mp->step = dpm40_step;
	mp->jump = dpm40_jump;
	mp->brkpt_check = dpm40_brkpt_check;
	mp->display_own = dpm40_display_own;
	mp->display_dis = m68k_display_dis;
	mp->symbolic_version = dpm40_symbolic_version;
	mp->find_bp = dpm40_find_bp;
	mp->backtrace = dpm40_backtrace;
	mp->valid_fp = dpm40_valid_fp;
	mp->set_brkpt = dpm40_set_brkpt;
	mp->clear_one_brkpt = dpm40_clear_one_brkpt;
	mp->clear_brkpts = dpm40_clear_brkpts;
	mp->display_brkpts = dpm40_display_brkpts;

	mp->check_tags = dpm40_check_tags;
	mp->solo = dpm40_solo;
	mp->deconfigure = dpm40_deconfigure;
	mp->configure = dpm40_configure;
	mp->cpu_copyin = (uint (*)())copy_from_km;
	mp->cpu_copyout = (uint (*)())copy_to_km;
	mp->cpu_malloc = (uint (*)())km_allocate;
	mp->cpu_zero = km_clear;
}


/* Called when polling the system bus.  'dpm40_setup(slot)' sets methods
 * associated with a pm40's and a descriptor for each cpu on the board.
 */
void
dpm40_setup(slot)
uint	slot;
{
	register pm40_info_t	*pm40_desc = next_dpm40_data;
	static method_t		pm40_methods;
	uint			num_pm, sub_slot;

	ASSERT(pm40_desc >= dpm40_data);
	ASSERT(pm40_desc < dpm40_data + SBUS_MAX_NUM_PM);
	if (sbus_config.slot_id[pm40_desc->cu_unit.s.slot] == SBUS_DPM40_SGL)
	    num_pm = 1;
	else
	    num_pm = 2;
	

	/* Setup methods once */
	if (pm40_desc == dpm40_data)
		dpm40_setup_methods(&pm40_methods);
	for (sub_slot=0; sub_slot < num_pm; sub_slot++) {

		pm40_desc->cu_methods = &pm40_methods;
		pm40_desc->cu_unit.s.slot = slot;
		pm40_desc->cu_unit.s.subslot =
		  (sub_slot ? PMB_SUB_SLOT: PMA_SUB_SLOT);
		/* inverted B=1; A=0; hanna FIX: get symbolic reference here */
		pm40_desc->interrupt_addr = sub_slot ? 0 : 1;
		pm40_desc->state = STATE_INIT;
		cpu_enroll(&pm40_desc->cpu_data);
		pm40_desc = ++next_dpm40_data;
	} /* for */

} /* dpm40_setup */


dpm40_all_set_loaded()
{
	register pm40_info_t	*pm40_desc;

	for (pm40_desc = dpm40_data; pm40_desc != next_dpm40_data; pm40_desc++)
		if (pm40_desc->state == STATE_INIT)
			pm40_desc->state = STATE_LOADED;
}



/*************************DPM40 specific routines *****************/



/* dpm40_send_map_data() - used to send data to PROMS */
/* send once per slot					*/
dpm40_send_map_data(pm_slot, mm_slot, map_address, map_size)
uint	pm_slot;	/* pm's map to be filled */
uint	mm_slot;	/* memory module slot number. */
uint	map_address;	/* offset of memory map in memory module. */
uint	map_size;	/* size of map data (bytes) */
{
	/* 1. Tell PROM which mm-slot the mapping register data resides... */
	dpm40_ipcc_send(pm_slot, IPCC_MM_SLOT, mm_slot, IPCC_ADDR_SLOT_A);

	/* 2. ...give PROM offset into that mm-slot... */
	dpm40_ipcc_send(pm_slot, IPCC_ADDR, map_address, IPCC_ADDR_SLOT_A);

	/* 3. ...give PROM size of mapping register data. */
	dpm40_ipcc_send(pm_slot, IPCC_MAP, map_size, IPCC_ADDR_SLOT_A);
}



static void
dpm40_ipcc_send(pm_slot, cmd, data, cpu)
uint	pm_slot;
uint	cmd;
uint	data;
uint	cpu;
{
	uint old_map = iomap_save();
/* hanna FIX: put in macros for the 28 and 8  (see ipcc.h) */

	*(uint *)iomap(pm_slot, (spm_slot << 28) | (cmd << 8) | cpu) = data;
	delay_check_iomap(5);
	iomap_restore(old_map);
}
