/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) mkfs.c: version 25.1 created on 11/27/91 at 15:28:25	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)mkfs.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/

/*
 * mkfs.c -- file system stuff
 */

#include "mkfs.h"
#include "sys/param.h"
#include "sys/sysmacros.h"
#include "sys/immu.h"
#include "sys/fs/s5filsys.h"
#include "sys/fs/s5fblk.h"
#include "sys/ioctl.h"

/*** int inodesize;	***/		/* if we ever support multiple inode */
#define inodesize	SEC_INODESIZE	/* sizes, turn this back into an int */

#define	NIPB		(BSIZE/inodesize) /* number of inodes per fs block */
#define	MAXFN		2000		/* max blocks per cyl (for free list) */

#define LADDR		10		/* number of direct blocks in inode */

#define BLK_OFFSET(off)	((uint)(off) & (NBPSCTR - 1))

#define	UTIME		0x999999	/* phony last mount time */

typedef union _filsys {
	struct filsys	fs;
	char		c[BLKSIZ];	/* filsys goes in 512 byte block */
} FILSYS;

typedef union _fbuf {
	struct fblk	fb;
	char		c[BSIZE];	/* free blocks are in 1K blocks */
} FBUF;

static FILSYS	filsys;

static long	ino_num;		/* inode number counter */

static MINO	*rootino;

int		error;			/* multiple error eliminator flag */

extern int	fsd;			/* file system file descriptor */

extern off_t	lseek();
extern void	ltol3();

/*
 * lseekck -- error trapped lseek
 */

static void
lseekck(fd, p)
int	fd;
off_t	p;
{
	off_t	rc;

	DBUGl2("lseekck(%d, %d) ", fd, p);
	if ((rc = lseek(fd, p, 0)) < 0) {
		err("lseek(%d, 0x%x, 0) error=%d", fd, p, rc);
		die();
	}
}

/*
 * rdfs -- read a MBLK from the file system
 */

static void
rdfs(bp)
MBLK	*bp;
{ 
	DBUGm1("rdfs(%d) ", bp->b.number);
	lseekck(fsd, bp->b.number * BSIZE);
	if (read(fsd, bp->b_baddr, BSIZE) != BSIZE) {
		err("read error at block %ld", bp->b.number);
		die();
	}
}

/*
 * wrfs -- write a MBLK to the file system
 */

void
wrfs(bp)
MBLK	*bp;
{
	DBUGm1("wrfs(%d) ", bp->b.number);
	lseekck(fsd, bp->b.number * BSIZE);
	if (write(fsd, bp->b_baddr, BSIZE) != BSIZE) {
		err("write error at block %ld", bp->b.number);
		die();
	}
}

/*
 * getblk -- return a block # bnum, filled from the file system
 */

MBLK *
getblk(bnum)
daddr_t	bnum;
{
	MBLK	*bp;

	DBUGm1("getblk(%d) ", bnum);
	if (bp = find_mblk(bnum))
		return (bp);

	bp = get_mblk(bnum);
	rdfs(bp);
	return (bp);
}

static void
bfree(bnum)
daddr_t	bnum;
{
	FBUF	*fbufp;
	MBLK	*bp;

	DBUGm1("bfree(%d) ", bnum);
	filsys.fs.s_tfree++;
	if (filsys.fs.s_nfree >= NICFREE) {
		bp = new_mblk(bnum);
		fbufp = (FBUF *)bp->b_baddr;
		fbufp->fb.df_nfree = filsys.fs.s_nfree;
		bcopy((caddr_t)filsys.fs.s_free, (caddr_t)fbufp->fb.df_free,
		  sizeof(fbufp->fb.df_free));
		wrfs(bp);
		filsys.fs.s_nfree = 0;
		free_mblk(bp);
	}
	filsys.fs.s_free[filsys.fs.s_nfree++] = bnum;
}

/*
 * dump_indir -- dump an indirect block's blocks then it,
 */

static void
dump_indir(ibp)
register MBLK	*ibp;
{
	register int	i;
	register daddr_t blknum;
	register MBLK	*bp;

	DBUGl1("dump_indir(%d) ", ibp->b.number);
	for (i = 0; i < NINDIR; i++)
		if ((blknum = ibp->b_daddr[i]) && (bp = find_mblk(blknum))) {
			wrfs(bp);
			free_mblk(bp);
		}

	wrfs(ibp);
	free_mblk(ibp);
}

/*
 * iput -- write the inode to disk and free it
 */

void
iput(ip)
register MINO	*ip;
{
	register MBLK	*bp, *ibp;
	struct dinode	*dp;
	daddr_t		bno;
	register uint	off;
	register int	i;

	DBUGh1("\niput(%d) ", ip->i.number);
	bno = sec_itod(ip->i.number, inodesize);
	if (bno >= filsys.fs.s_isize) {
		if (!error)
			err("iput: ilist too small for inode %d", ip->i.number);
		error = 1;
		free_mino(ip);
		return;
	}
	off = inodesize * sec_itoo(ip->i.number, inodesize);
	bp = getblk(bno);
	dp = (struct dinode *)&bp->b_baddr[off];

	bzero((caddr_t)dp, sizeof(*dp)); /* zero unused portion for security */

	dp->di_mode = ip->i_mode;
	dp->di_nlink = ip->i_nlink;
	dp->di_uid = ip->i_uid;
	dp->di_gid = ip->i_gid;
	dp->di_size = ip->i_size;
	dp->di_atime = dp->di_mtime = dp->di_ctime = ip->i_mtime;
	ltol3(dp->di_addr, ip->i_baddr, NADDR);

	/* i_baddr[0] of character and block special files contains major
	 * and minor numbers. Free the inode here to avoid this being 
	 * interpreted as a block number */

	if ((ip->i_mode & IFMT) == IFBLK || (ip->i_mode & IFMT) == IFCHR) {
		free_mino(ip);
		return;
	}

	DBUGl("{direct blks} ");
	for (i = 0; i < LADDR; i++)
		if ((bno = ip->i_baddr[i]) && (bp = find_mblk(bno))) {
			wrfs(bp);
			free_mblk(bp);
		}

	/* write out a single indirect block */
	DBUGl("{single indir} ");
	if (bno = ip->i_baddr[LADDR]) {
		if ((ibp = getblk(bno)) == NULL) {
			err("iput: can't find single indirect %d!", bno);
			die();
		}
		dump_indir(ibp);
	}

	/* write out any double indirect blocks */
	DBUGl("{double indir} ");
	if (bno = ip->i_baddr[LADDR + 1]) {
		if ((ibp = getblk(bno)) == NULL) {
			err("iput: can't find double indirect %d!", bno);
			die();
		}
		for (i = 0; i < NINDIR; i++) {
			if (bno = ibp->b_daddr[i]) {
				if ((bp = getblk(bno)) == NULL) {
					err("iput: can't find indirect %d", bno);
					die();
				}
				dump_indir(bp);
			}
		}
		wrfs(ibp);
		free_mblk(ibp);
	}

	free_mino(ip);
}

/*
 * ialloc -- allocate an inode from the file system
 */

static MINO *
ialloc()
{
	if (filsys.fs.s_tinode-- == 0) {
		err("out of inodes!");
		die();
	}
	return (get_mino(++ino_num));
}

/*
 * balloc -- allocate a block from the file system
 */

static MBLK *
balloc()
{
	MBLK	*bp;
	FBUF	*fbufp;
	daddr_t	bno;

	filsys.fs.s_tfree--;
	if ((bno = filsys.fs.s_free[--filsys.fs.s_nfree]) == 0) {
		err("out of free space");
		die();
	}
	bp = get_mblk(bno);
	if (filsys.fs.s_nfree <= 0) {
		rdfs(bp);
		fbufp = (FBUF *)bp->b_baddr;
		filsys.fs.s_nfree = fbufp->fb.df_nfree;
		bcopy((caddr_t)fbufp->fb.df_free, (caddr_t)filsys.fs.s_free,
		  sizeof(filsys.fs.s_free));
	}
	DBUGm1("balloc=%d\n", bp->b.number);
	bzero(bp->b_baddr, BSIZE);
	return (bp);
}

/*
 * bmap -- return a pointer to the daddr_t for a given offset
 *		will alloc single and double indirect blocks
 */

static daddr_t *
bmap(ip, offset)
MINO	*ip;
off_t	offset;
{
	register uint	blknum = btodt(offset);
	MBLK		*bp, *pb;
	daddr_t		indir;

	DBUGl2("bmap(%d, %d)=", ip->i.number, offset);
	if (blknum < LADDR) {
		DBUGl1("dir[%d] ", blknum);
		return (&ip->i_baddr[blknum]);		/* direct block */
	}

	if ((blknum -= LADDR) < NINDIR) {
		DBUGl("single ");
		if (indir = ip->i_baddr[LADDR])		/* single indirect */
			bp = getblk(indir);
		else {
			bp = balloc();
			ip->i_baddr[LADDR] = bp->b.number;
			bzero(bp->b_baddr, BSIZE);
		}
	}
	else if ((blknum -= NINDIR) < NINDIR * NINDIR) {
		DBUGl("double ");
		if (indir = ip->i_baddr[LADDR + 1])	/* double indirect */
			bp = getblk(indir);
		else {
			bp = balloc();
			ip->i_baddr[LADDR + 1] = bp->b.number;
			bzero(bp->b_baddr, BSIZE);
		}
		pb = bp;
		DBUGl1("[%d] ", blknum / NINDIR);
		if (indir = pb->b_daddr[blknum / NINDIR])
			bp = getblk(indir);
		else {
			bp = balloc();
			pb->b_daddr[blknum / NINDIR] = bp->b.number;
			bzero(bp->b_baddr, BSIZE);
		}
		blknum %= NINDIR;
	}
	else {						/* no triple indirect */
		err("bmap: inode %d too large (off=%d)", ip->i.number, offset);
		die();
	}

	DBUGl1("[%d] ", blknum);
	return (&bp->b_daddr[blknum]);
}

/*
 * seeki -- return the MBLK which contains offset -- alloc as needed
 *	Caller is responsible for finding the offset within the MBLK.
 *	Note: non-sequential writes will leave holes.
 */

MBLK *
seeki(ip, offset)
MINO	*ip;
off_t	offset;
{
	register MBLK	*bp;
	register daddr_t *blkp;
	daddr_t		bnum;

	DBUGm2("seeki(%d, %d) ", ip->i.number, offset);
	blkp = bmap(ip, offset);
	if ((bnum = *blkp) == 0) {
		bp = balloc();
		*blkp = bp->b.number;
	}
	else if ((bp = find_mblk(bnum)) == NULL) {
		err("seeki: cached block # %d for inode %d not found!",
		  bnum, ip->i.number);
		die();
	}

	return (bp);
}


/*
 * mkentry -- make a directory entry in par of ent with name
 */

void
mkentry(par, ent, name)
register MINO	*par, *ent;
char		*name;
{
	struct direct	*dp;
	MBLK		*bp;

	DBUGh3("mkentry(%d, %d, '%s') ",
	  par->i.number, ent->i.number, name);
	bp = seeki(par, par->i_size);
	dp = (struct direct *)&bp->b_baddr[BLK_OFFSET(par->i_size)];
	dp->d_ino = ent->i.number;
	strncpy(dp->d_name, name, sizeof(dp->d_name));
	par->i_size += sizeof(*dp);
	ent->i_nlink++;
	DBUGm1("nlink=%d ", ent->i_nlink);
}

/*
 * set_fs_size -- get the file system size
 */
static
get_fs_size()
{
	int	slice_size;
	int	n, fs_size = 0;

	if ((slice_size = ioctl(fsd, GET_DISK_SIZE, 0)) < 0) {
		err("ioctl GET_DISK_SIZE failed!");
		die();
	}

	do {
		printf("\nfile system size [default = %d 1K blocks]: ", slice_size);
		gets(string);
		n = atoi(string);

		/* 
		 * compute inode from file system size 
		 */
		if (n == 0 || n == slice_size)
			fs_size = slice_size;
		else if (n > slice_size)
			err("size (%d) is larger than slice (%d)", n, slice_size);
		else if (n < FS_MIN_BLKS)
			err("size (%d) is smaller than minimum (%d)",
			  n, FS_MIN_BLKS);
		else if (n < slice_size) {
			err("Note: size (%d) smaller than slice (%d)", n, slice_size);
			fs_size = n;
		}
	} while (fs_size == 0);
	return(fs_size);
}

static void
get_name(name, prompt, max_len)
char	*name, *prompt;
uint	max_len;
{
	printf("%s", prompt);
	gets(string);
	strncpy(name, string, max_len);
}

/*
 * mkfs -- initialize stuff and make a file system
 */

void
mkfs()
{
	register MINO		*ip;
	register MBLK		*bp;
	register long		n;
	register int		i, j;
	register daddr_t	f, d;
	int			i_lace, sec_per_cyl;
	char			flg[MAXFN];
	daddr_t			adr[MAXFN];
	char	bars[] = "|/-\\";
	char	*c = bars;
	int	whirl = 1;

#ifdef	DEBUG
	if	(debug > 0)
		whirl = 0;
#endif	/* DEBUG */

	i_lace = 1;
	if ((sec_per_cyl = ioctl(fsd, GET_CYLINDER_SIZE, 0)) < 0) {
		err("ioctl GET_CYLINDER_SIZE failed!");
		die();
	}

	if (sec_per_cyl > MAXFN) {
		err("can't handle %d sectors per cylinder (max=%d)",
		  sec_per_cyl, MAXFN);
		die();
	}

	filsys.fs.s_fsize = get_fs_size();

	filsys.fs.s_time = UTIME;
	filsys.fs.s_state = FsOKAY - filsys.fs.s_time;

	get_name(filsys.fs.s_fname, "\nfile system name: ",
		sizeof(filsys.fs.s_fname));
	get_name(filsys.fs.s_fpack, "\nvolume name: ", sizeof(filsys.fs.s_fpack));

	filsys.fs.s_magic = FsMAGIC;
/***	inodesize = SEC_INODESIZE; (changed to define above)	***/
	filsys.fs.s_type  = (inodesize == FsINODESIZE) ? Fs2b : FsSEC1;

	printf("\nfsize = %d\n", filsys.fs.s_fsize);
	n = filsys.fs.s_fsize / (NIPB * 4);
	if (n <= 0)
		n = 1;
	else if (n > 65530 / NIPB)
		n = 65530 / NIPB;
	filsys.fs.s_isize = n + 2;	/* boot, super, and inode blocks */
	filsys.fs.s_tinode = n * NIPB;	/* total free inodes */
	printf("isize = %d\n", filsys.fs.s_tinode);

	filsys.fs.s_dinfo[0] = i_lace;
	filsys.fs.s_dinfo[1] = sec_per_cyl;

	printf("interlace %d, sectors/cylinder %d\n", i_lace, sec_per_cyl);

	/* inode number can not be greater than file system size */

	if (filsys.fs.s_isize >= filsys.fs.s_fsize) {
		err("bad ratio of isize to fsize:  %ld/%ld",
		  filsys.fs.s_fsize, filsys.fs.s_isize - 2);
		die();
	}

	filsys.fs.s_tfree = 0;		/* total free block */

	DBUG("\nclear inode blocks on disk\n");
	bp = new_mblk(-1L);
	bzero(bp->b_baddr, BSIZE);
	for (n = 0; n < filsys.fs.s_isize; n++) {
		if (whirl)
			if (!(n % 8)) {
				printf("%c\b", *c);
				if (!*(++c))
					c = bars;
			}
		bp->b.number = n;
		wrfs(bp);
	}
	free_mblk(bp);

	DBUG("\nbuild free list\n");

	/* fill adr[i] with interlace gap: 
	 * adr[0]=1 adr[1]=4 adr[2]=7
	 */

	bzero(flg, sec_per_cyl);
	i = 0;
	for (j = 0; j < sec_per_cyl; j++) {
		while (flg[i])
			i = (i + 1) % sec_per_cyl;
		adr[j] = i + 1;
		flg[i]++;
		i = (i + i_lace) % sec_per_cyl;
	}

	/* free block 0 */
	bfree((daddr_t)0);

	/* total free block decrement 1 */
	filsys.fs.s_tfree--;	

	d = filsys.fs.s_fsize - 1;	/* total file system */
	while (d % sec_per_cyl)  /* round up to cylinder boundary */
		d++;                  /* will only work if slice is aligned */
	if (whirl)
		c = bars + strlen(bars) - 1;
	for ( ; d > 0; d -= sec_per_cyl) {
		if (whirl) {
			if (c-- == bars)
				c = bars + strlen(bars) - 1;
			printf("%c\b", *c);
		}

		for (i = 0; i < sec_per_cyl; i++) {
			f = d - adr[i];
			if (f < filsys.fs.s_fsize && f >= filsys.fs.s_isize)
				bfree(f);
		}
	}

	/* set up inode 1 (bad block holder inode) */

	ip = ialloc();
	ip->i_mode = IFREG;		/* regular file */
	iput(ip);

	/* set up inode 2 (root inode) */

	rootino = ip = ialloc();
	ip->i_mode = IFDIR | 0755;	/* directory: drwxr-xr-x */
	mkentry(ip, ip, ".");
	mkentry(ip, ip, "..");
}

/*
 * scan_dir -- scan a directory for a given entry, return its inumber or 0
 */

static uint
scan_dir(ip, str)
register MINO	*ip;
register char	*str;
{
	register MBLK	*bp;
	struct direct	*dp;
	register off_t	offset = 0;
	register int	bcnt = 0;

	DBUGm2("scan_dir(%d, '%s') ", ip->i.number, str);
	while (offset < ip->i_size) {
		if (bcnt <= 0) {
			if ((bp = seeki(ip, offset)) == NULL) {
				err("scan_dir: ino %d offset %d not found",
				  ip->i.number, offset);
				return (0);
			}
			dp = (struct direct *)bp->b_baddr;
			bcnt = BSIZE;
		}

		if (strncmp(str, dp->d_name, sizeof(dp->d_name)) == 0)
			return (dp->d_ino);

		offset += sizeof(*dp);
		bcnt -= sizeof(*dp);
		dp++;
	}

	return (0);
}

/*
 * namei -- turn a pathname into a MINO pointer, or NULL
 *		returns parent ptr if path was ok up to the last part, or NULL.
 *		part must be at least DIRSIZE long
 *
 *		Note:  assumes '.' is root!  Parent ptr can't cope with some
 *			possible uses of '..'!
 */

MINO *
namei(path, parpp, part)
char	*path;
MINO	**parpp;
char	part[DIRSIZ];
{
	register MINO	*dp, *ip;
	register char	*src = path, *dest;
	register long	i;
	register int	c;

	DBUG1("namei('%s') ", path);
	dp = ip = rootino;
	part[0] = '\0';

	if (src == NULL || *src == '\0') {
		*parpp = NULL;		/* no null pathnames allowed */
		return (NULL);
	}

	while ((c = *src) == '/')	/* skip leading slashes */
		++src;

	while (c) {
		dest = part;			/* get path component */
		i = DIRSIZ;
		while (c && c != '/' && --i >= 0) {
			*dest++ = c;
			c = *++src;
		}
		if (i > 0)
			*dest = '\0';		/* null terminate if < DIRSIZ */
		else
			while (c && c != '/')
				c = *++src;	/* eat excess path component */

		while ((c = *src) == '/')	/* skip trailing slashes */
			++src;

		dest = part;			/* skip '.' */
		if (dest[0] == '.' && dest[1] == '\0')
			continue;

		dp = ip;			/* traverse directory */
		if ((dp->i_mode & IFMT) != IFDIR) {
			err("namei(%s): inode %d '%.*s' not a directory!",
			  path, dp->i.number, DIRSIZ, dest);
			*parpp = NULL;
			return (NULL);
		}

		if (i = scan_dir(dp, dest)) {	/* search directory */
			if ((ip = find_mino(i)) == NULL) {
				err("namei(%s): inode %d '%.*s' not cached",
				  path, i, DIRSIZ, dest);
				*parpp = NULL;
				return (NULL);
			}
		}
		else if (c) {
			err("namei(%s): directory '%.*s' not found!",
			  path, i, DIRSIZ, dest);
			*parpp = NULL;
			return (NULL);
		}
		else {
			*parpp = dp;		/* not found */
			return (NULL);
		}
	}

	*parpp = dp;
	return (ip);
}

/*
 * openi -- open / create an inode
 */

MINO *
openi(path, mode)
char	*path;
ushort	mode;
{
	register uint	type;
	register MINO	*ip;
	MINO		*par;
	char		part[DIRSIZ];

	DBUG2("openi('%s', %o) ", path, mode);
	if ((ip = namei(path, &par, part)) == NULL) {
		if (par == NULL)
			return (NULL);
		ip = ialloc();
		mkentry(par, ip, part);
		if ((mode & IFMT) == IFDIR) {
			mkentry(ip, ip, ".");
			mkentry(ip, par, "..");
		}
	}
	else if ((type = ip->i_mode & IFMT) != (mode & IFMT)) {
		err("openi: existing inode type 0%05o != new type 0%05o",
		  ip->i_mode, mode);
		die();
	}
	else if (type == IFREG) {
		err("openi('%s', %05o): reopened a regular file!", path, mode);
		die();
	}
	ip->i_mode = mode;

	return (ip);
}

/*
 * flush_mino -- find and update non-directory inodes, then free them
 */

void
flush_mino(flg)
uint	flg;
{
	register MINO	*ip;
	register int	i;
	register uint	type;

	DBUG1("flush_mino(%d) ", flg);

	type = (uint)(flg == FLUSH_SOME ? IFDIR : -1);

	for (i = 1; i <= ino_num; i++) {
		if ((ip = find_mino(i)) == NULL)
			continue;
		if (((uint)ip->i_mode & IFMT) == type)
			continue;
		iput(ip);
	}
}

/*
 * flush_mblk -- find, write out, and free, either inode blocks, or all blocks
 */

void
flush_mblk(flg)
int	flg;
{
	register MBLK	*bp;
	register long	i, limit;

	DBUG1("flush_mblk(%d) ", flg);

	limit = (flg == FLUSH_SOME ? filsys.fs.s_isize : filsys.fs.s_fsize);

	for (i = 0; i < limit; i++)
		if (bp = find_mblk(i)) {
			wrfs(bp);
			free_mblk(bp);
		}
}

/*
 * finish -- flush caches and finish up the file system
 */

void
finish()
{
	MBLK	*bp;

	DBUG("\nfinish() ");
	flush_mino(FLUSH_ALL);
	flush_mblk(FLUSH_ALL);

	if ((bp = get_mblk(0L)) == NULL) {
		err("finish: can't alloc block for superblock!");
		die();
	}

	/* write superblock to disk at second half of block 0 */

	bzero(bp->b_baddr, BLKSIZ);
	bcopy(filsys.c, bp->b_baddr + BLKSIZ, BLKSIZ);
	wrfs(bp);
	free_mblk(bp);
}
