/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) auth.c: version 25.1 created on 11/27/91 at 15:13:24	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)auth.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
/*
 * 	auth.c  authentication routines for the kernel
 *	(C)Copyright 1990 by ARIX Corp.
 *
 *	Note: this module contains the PRIV, MAC and DAC security
 *	policy implementation for the ARIX default, C2 and B1 kernels.
 *	It is essentially the major single authorization point in 
 *	the entire kernel. 
 *
 *	Compile with SECON undef'd for default Unix authorizations
 *	Compile with SECON and B1 undefs for C2 Unix operation
 *	Compile with SECON and B1 defined for B1 Unix authorizations
 *	Compile with SECDEBUG for debug information 
 *
 *	Version 2: This module has been updated to match the new privilege
 * 		   implementation - JTOF
 */

#include "sys/types.h"
#include "sys/sysmacros.h"
#include "sys/systm.h"
#include "sys/tuneable.h"
#include "sys/immu.h"
#include "sys/user.h"
#include "sys/errno.h"
#include "sys/file.h"
#include "sys/nami.h"
#include "sys/inode.h"
#include "sys/ino.h"
#include "sys/fstyp.h"
#include "sys/region.h"
#include "sys/proc.h"
#include "sys/var.h"
#include "sys/clock.h"
#include "sys/conf.h"
#include "sys/spl.h"
#include "sys/mount.h"
#include "sys/debug.h"
#include "sys/spm_mem.h"
#include "sys/kmem.h"
#include "sys/param.h"
#include "sys/fs/s5inode.h"
#include "sys/cmn_err.h" 
#include "sys/acct.h"
#include "sys/acl.h"
#include "sys/priv.h"
#include "sys/mls.h"
#include "sys/security.h"

/* Security Debug flag - allows certian security debug messages to 
 * appear. Should normally be turned off for distribution
 */
#ifdef SECDEBUG
uchar	secdebug = 1;
#else
uchar	secdebug = 0;
#endif

/*
 *	auth_conf   return the state of the security options. This call
 *	            is placed in the sysconf call, returning the proper
 *		    state of the security options        
 */

auth_conf( val )
int val;
{
#ifdef SECON
	if ( val == _SC_SECURITY ) {	
		u.u_rval1 = sec_mode;
		return 1;
	} 
	else
		return 0;
#else
	return 0;
#endif
}
/*
 *	auth_p0init  assign the intial privileges and labels to the 
 *		   first process, proc[0];
 * 
 */

auth_p0init( pp )
struct proc *pp;
{
#ifdef SECON
	
	/* don't let the idiots user macs without privs 
         * we can't start su or init in this case so die 
	 * real gracefully so he knows!!!!
 	 */
	if ( (! sec_mode & PRIVON) && (sec_mode & MACSON)) {
		panic( "MLS Labels enabled but not privileges" );
	}

	/* JTOF - we don't need to set initial labels for proc 0
	 * because will have P_ALL (MAC/DAC overrides)
 	 */
	if ( sec_mode & PRIVON ) {
		pp->p_auth->a_priv.real = P_ALL;
		pp->p_auth->a_priv.mask = P_ALL;
		pp->p_auth->a_priv.eff = P_ALL;
	}

	/* default is to always have all audit events so do the
	 * overkill kinda thing and make entire mask set to on 
	 */
	if ( sec_mode & SATON ) {
		pp->p_auth->a_amask[0] = ~0;
		pp->p_auth->a_amask[1] = ~0;
	}
#endif
}
	
/*
 *	auth_acct  auth user request to enable and disable system 
 *		   accounting. This should be handled by the SYSADM
 *		   only.    
 */
auth_acct()
{
	return auth_adm();
}

/*
 *	auth_pacct  copy the p_used field from the uauth structure to
 *		    the process accounting strucuture
 */
auth_pacct( p )
struct acct *p;
{
#ifdef SECON
	p->ac_pused = uauth->a_pused;
#endif
} 

/*
 *	auth_audcntl auth user request to make audit control calls via
 *		   the audit system call. This is restricted to the 
 *		   SECADM only.
 */
auth_audcntl()
{
	return auth_sec();
}

/*
 *	auth_audit  auth user to download a audit record throught the
 *		    audit system call 
 */
auth_audit()
{ 
#ifdef SECON
	/* user must be security administrator. Effective not checked */
	if ( p_chkall( P_AUDIT ) ) {
		u.u_acflag |= ASU;	/* make as having priv */
		uauth->a_pused |= P_AUDIT;
		return 1;
	} else
		return 0;
#else
	/* if not security mode then never let the user in */
	return 0;      
#endif
}

/*
 *	auth_kill   auth a users ability to override the checks made to kill
 *		    another process. This call isn't allowed to override
 *		    the labels pn the other process unless the MAC override
 *		    privileges exist 
 */
auth_kill(q,p)
struct proc *q,*p;
{
#ifdef SECON

	/* if we have have security privileges then kill the guy
 	 * unconditionally !!!!
 	 */	
	if ( p_chkreal( P_SECURITY  )) {
		u.u_acflag |= ASU;
		return 1;
	} 
	/* if we have P_ADM and our labels match allow the kill
 	 * if labels don't match don't allow, and if we are ADM
         * don't allow the kill
 	 */
	return p_chkall( P_ADM ) && 
	       mls_equ(&q->p_auth->a_minlabel, &p->p_auth->a_minlabel);
#else
	return u.u_uid == 0;
#endif
}

/*
 *	auth_nice  auth user to alter his priority downward via the
 *		   nice system call.
 */
auth_nice()
{
	return auth_adm();
} 

/*
 *	auth_dac_chk    check to see if user has a DAC exemptions 
 */
auth_dac_chk( mode )
register uint mode;
{
#ifdef SECON
 	if ( mode & IWRITE )  { 
		if (p_chkall(P_DAC_WRITE)) {
			u.u_acflag |= ASU;	/* mark as policy override */
			uauth->a_pused |= P_DAC_WRITE;
			return 1;
		} 
		else 
			return 0;
	}
	else {
		if ( p_chkall(P_DAC_READ)) {
			u.u_acflag |= ASU;	/* mark as policy override */
			uauth->a_pused |= P_DAC_READ;
			return 1;
		}	
		else 
			return 0;
	}
#else
	return (u.u_uid == 0);
#endif
}

/*
 *	auth_fsnami  do special label checks for operations like 
 *		     rm, rename etc. These operation typically need
 *		     access to the directory that contains a file,
 *		     not to the file themselves! 
 *
 */
auth_fsnami(ip)
struct inode *ip;
{
#ifdef SECON
register struct s5inode *s5ip;

	/* get the s5ip */
	s5ip = (struct s5inode *) ip->i_fsptr;
	return auth_label_chk( s5ip, IWRITE );
#else
	return 1;
#endif
}

/*
 *	auth_s5iread  copy the priv and label from the disk inode to the
 *		      incore inode
 *
 */
auth_s5iread( ip, dp )
register struct inode *ip;
register struct dinode *dp;
{
#ifdef SECON
register struct s5inode *s5ip;

	/* if the filesystem is secure one copy the label and priv */
	if (ip->i_mntdev->m_isize == SECURE_INODE) {
		if ( sec_mode & MACSON ) {
			s5ip = (struct s5inode *) ip->i_fsptr;
			bcopy( &dp->di_label, &s5ip->s5i_label, 
				sizeof( slabel_t));
		}
		ip->i_priv = dp->di_priv;
	}
#endif
}

/*
 *	auth_s5ialloc   copy the process label to the newly created
 *			file. This should be controlled by the option
 *		        settings in the sec_mode variable.
 */	
auth_s5ialloc(s5ip)
register struct s5inode *s5ip;
{
#ifdef SECON 
	/* if macs enabled and inheritance enabled */
	if ((sec_mode & (MACSON|MACSINH)) == (MACSON|MACSINH)) {
		bcopy((caddr_t) &uauth->a_minlabel, (caddr_t)&s5ip->s5i_label, 
			sizeof( slabel_t ));
	} 
	else  
		/* not options so clear the label */
		bzero( &s5ip->s5i_label, sizeof( slabel_t) );
#endif
}

/*
 *	auth_s5iupdate  set the files label and inherited privilege
 *		        from the incore copy
 *
 */

auth_s5iupdate( ip, dp )
register struct inode *ip;
register struct dinode *dp;
{
#ifdef SECON
register struct s5inode *s5ip;

	/* if the filesystem is secure one copy the label and priv */
	if (ip->i_mntdev->m_isize == SECURE_INODE ) {
		if ( sec_mode & MACSON ) {
			s5ip = (struct s5inode *) ip->i_fsptr;
			bcopy( &s5ip->s5i_label, &dp->di_label, 
				sizeof( slabel_t ));
		}
		dp->di_priv = ip->i_priv;
	}
#else
	/* see sys/ino.h for exact size in case it is modified */
	bzero( &dp->di_lpad[0], 18);
#endif
}

/*
 *	auth_label_chk  check the file label against the process
 *			label to determine if access to the file. 
 *
 */
auth_label_chk( s5ip, mode )
register struct s5inode *s5ip;
register int mode;
{
#ifdef SECON

	/* is macs disabled return true for label checks */
	if ( ! (sec_mode & MACSON) ) 
		return 1;
	
	if ( mode & IWRITE || mode & ILABELW ) {
		if (p_chkall( P_MAC_WRITE )) {
			u.u_acflag |= ASU;
			uauth->a_pused |= P_MAC_WRITE;
			return 1;
		}
		return mls_equ( &uauth->a_minlabel, &s5ip->s5i_label );
	} 
	
	if ( p_chkall( P_MAC_READ )) {
		u.u_acflag |= ASU;
		uauth->a_pused |= P_MAC_READ;
		return 1;
	}
	return mls_dom( &uauth->a_minlabel, &s5ip->s5i_label );
#else
	return 1;	/* return okay if security implemented */
#endif

}

/*
 *	auth_label_obj auth user requrest to create label on 
 *	                a file or ipc
 */
auth_label_obj()
{
	return auth_sec();
}

/*
 *	auth_label_subj  auth user requrest to create label on 
 *	                the current process 
 */
auth_label_subj()
{
	return auth_sec();
}

/*
 * 	auth_priv_subj  auth user request to change privileges on
 *	                a process 
 */
auth_priv_subj()
{
	return auth_sec();
}

/*
 * 	auth_priv_objj  auth user request to change privileges on
 *	                a file         
 */
auth_priv_obj()
{
	return auth_sec();
}

/*
 *	auth_suid  auth user capability to execute suid files.
 *	Note: remember that the P_ID_EXEC is a reverse privilege.
 */
auth_suid( s5ip, mode )
register struct s5inode *s5ip;
register int mode;
{
#ifdef SECON
	/* if P_ID_EXEC restriction on then return no canna do!! */
	if ( p_chkall( P_ID_EXECR ))
		return 0;
#endif
	return s5ip->s5i_mode & mode;
}

/*
 *	auth_sgid   auth user capability to execute sgid files.
 *	Note: remember that the P_ID_EXEC is a reverse privilege.
 */
auth_sgid( s5ip, mode )
register struct s5inode *s5ip;
register int mode;
{

#ifdef SECON
	/* if P_ID_EXEC restriction on then return no canna do!! */
	if ( p_chkall( P_ID_EXECR ))
		return 0;
#endif
	return s5ip->s5i_mode & mode;	          
}

/* 
 *	auth_sysmask auth user request to change system wide privilege
 *	             mask. This is a highly security conscious operation
 */
auth_sysmask()
{
	return auth_sec();
}

/*
 *	auth_fork   copy the proper privileges to the child process.
 *		
 */
auth_fork( cp, pp )
struct proc *cp, *pp;
{
#ifdef SECON
	
	if ( sec_mode & PRIVON || sec_mode & MACSON ) {
		cp->p_auth->a_role = pp->p_auth->a_role;
	}
	if ( sec_mode & PRIVON ) {
		/* copy the parent real privileges ANDed with the system mask */
		cp->p_auth->a_priv.real = pp->p_auth->a_priv.real & 
					  sys_priv_mask;	

		/* copy the parent mask to the child */
		cp->p_auth->a_priv.mask = pp->p_auth->a_priv.mask;

		/* now do the effective privileges. The rules are that we only
		 * copy the effective privielges if the parent priv is P_INHERIT
		 * and then, we mask it against the system mask before we actual
		 * copy it. If the P_INHERIT isn't set, then we zero out the 
		 * effective privilege
		 */ 
		if (( pp->p_auth->a_priv.real & P_INHERIT ) || 
		    ( pp->p_auth->a_priv.eff & P_INHERIT )) {
			cp->p_auth->a_priv.eff = pp->p_auth->a_priv.eff & 
						sys_priv_mask;
			}
		else
			cp->p_auth->a_priv.eff = P_NULL;
	}
	if ( sec_mode & MACSON ) {
		bcopy((caddr_t)&pp->p_auth->a_minlabel, 
		(caddr_t)&cp->p_auth->a_minlabel, sizeof(slabel_t)*2);
	}
	if ( sec_mode & SATON ) {
		/* make sure we copy the audit id */
		cp->p_auth->a_auid = pp->p_auth->a_auid;
		cp->p_auth->a_amask[0] = pp->p_auth->a_amask[0];
		cp->p_auth->a_amask[1] = pp->p_auth->a_amask[1];
	}
#endif
}
	
	
/*
 *	auth_execok  auth user to execute files that aren't part of
 *	             the TCB.  
 */
auth_execok(ip)
register struct inode *ip;
{
#ifdef SECON
	if ( p_chkall( P_EXECR ))
		return mls_exec(ip);
#endif
	return 1;	/* if no security always return true */
}

/*
 *	auth_exec  perform the privileged actions necessary to 
 *	           handle the suid/sgid events. This involves
 *		   first checking the ISUID and ISGID bits to 
 *		   see if were allowd to do the suid/sgid. Once
 *		   we are cleared then in the case of suid, we
 *		   must find the accompaning role and set the
 *		   current process's effective privilege to the
 *		   sum of the role read and effective.
 */
auth_exec(ip)
register struct inode *ip;
{
#ifdef SECON
register uint priv;
kern_role_t *rp, *getuidinfo();
#endif

	/* make sure we have the abilitiy to do suid 
	 * if we are cleared, then set up the effective privs
         */
	if (!FS_ACCESS(ip, ISUID)) {
#ifdef SECON 		
		/* make sure we know the user and the role. If we don't
		 * or if the user account is locked, we will skip the 
		 * suid-on-exec thing to do
		 */
		if ( setrole( ip->i_uid, 1 )) {
			u.u_uid = ip->i_uid;
			if ( sec_mode & PRIVON ) {
				uauth->a_priv.eff &= (uauth->a_priv.mask |
						      sys_priv_mask );
			}
		}
#else
		u.u_uid = ip->i_uid;
#endif
	}
	if (!FS_ACCESS(ip, ISGID)) {
#ifdef SECON
		/* Check to see if we know about the group. If
		 * the group is locked or unknown, we won't do the
 		 * sgid-on-exec kinda thing
		 */
		if ( getgidinfo( ip->i_gid ))
			u.u_gid = ip->i_gid;
#else
		u.u_gid = ip->i_gid;
#endif
	}
#ifdef SECON
	uauth->a_priv.eff |= (ip->i_priv & sys_priv_mask);
#endif
	u.u_procp->p_saved_set_uid = u.u_uid;
	u.u_procp->p_saved_set_gid = u.u_gid;
	u.u_procp->p_suid = u.u_uid;
	u.u_procp->p_sgid = u.u_gid;
}
/*
 *	auth_plock  auth user request to lock a process in memory 
 */
auth_plock()
{
	return auth_adm();
}

/*
 *	auth_link  auth a user request to link a directory 
 */
auth_link()
{
#ifdef SECON
	if ( p_chkall( P_LINK ) ) {
		u.u_acflag |= ASU;
		uauth->a_pused |= P_LINK;
		return 1;
	} 
	else {
		u.u_error = EPERM;
		return 0;
	}
#else
	return suser();
#endif
}

/*
 *	auth_ulink  auth a user request to ulink a directory 
 */
auth_unlink()
{
#ifdef SECON
	if ( p_chkall( P_LINK ) ) {
		u.u_acflag |= ASU;
		uauth->a_pused |= P_LINK;
		return 1;
	} 
	else {
		u.u_error = EPERM;
		return 0;
	}
#else
	return suser();
#endif
}

/*
 *	auth_mknod  auth a user request to create nodes 
 */
auth_mknod()
{
#ifdef SECON
	if ( p_chkall( P_MKNOD ) ) {
		u.u_acflag |= ASU;
		uauth->a_pused |= P_MKNOD;
		return 1;
	} 
	else {
		u.u_error = EPERM;
		return 0;
	}
#else
	return suser();
#endif
}

/*
 *	auth_chroot  auth a user requrest to change process idea
 *		     of the root directory 
 */
auth_chroot()
{
#ifdef SECON
	if ( p_chkall( P_CHROOT ) ) {
		u.u_acflag |= ASU;
		uauth->a_pused |= P_CHROOT;
		return 1;
	} 
	else {
		u.u_error = EPERM;
		return 0;
	}
#else
	return suser();
#endif
}

/*
 *	auth_utime   auth a user to change the update times on a file
 */
auth_utime()
{
#ifdef SECON
	if ( p_chkall( P_UTIME )) {
		u.u_acflag |= ASU;
		uauth->a_pused |= P_UTIME;
		return 1;
	} else
		return 0;
#else
	return suser();
#endif
}

/*
 *	auth_msg  privilege to override msg DAC controls 
 */
auth_msg()
{
	return auth_ipc();
}

/*
 *	auth_sem  privilege to override sem DAC controls
 */
auth_sem()
{
	return auth_ipc();
}

/*
 *	auth_shm  privilege to override shm DAC controls
 */
auth_shm()
{
	return auth_ipc();
}

/*
 *	auth_sat   auth user request to access the audit node for read
 *		   access. This is a security critical probe.
 */
auth_sat()
{
	return auth_sec();
}


/*
 *	auth_stime auth a user to change system time 
 */
auth_stime()
{
	return auth_adm();
}


/* 
 *	auth_utsname  auth a user to change the sytem node name
 *		      This is considered a security sensitive operation
 *		      because we could fool users and the net about
 *		      who we are!
 */
auth_utsname()
{
	return auth_sec();
}

/*
 *	auth_power  auth a user to adjust the power margining.  
 */
auth_power()
{
	return auth_adm();
}

/*
 *	auth_kstr  auth to read a kernel string 
 */
auth_kstr()
{	
	return auth_adm();
}

/* 
 * 	auth_kdump  auth a user to dump kernel memory to a floppy. This
 *	            is considered a security relevant event because data
 *		    could be transfered that gives away passwds etc.
 */
auth_kdump()
{
	return auth_sec();
}

/*
 *	auth_phmat  auth a user to attach physical memory
 */
auth_phmat()
{
	return	auth_adm();
}

/*
 *	auth_phmdt  auth a user to detach physical memory
 */
auth_phmdt()
{
	return	auth_adm();
}

/*
 *	auth_clkcal  auth a user requrest to change clock calibrations
 */
auth_clkcal()
{
	return auth_adm();
}

/* 
 *	auth_cdebug  auth a user to enter debug mode automatically
 *		     This is considered a security relevant event 
 */
auth_cdebug()
{
	return auth_sec();
}

/*
 *	auth_posix   auth a user requrest to change the state of the
 *		     process SVID and NOJOBS flags 
 */
auth_posix()
{
	return auth_adm();
}

/*
 *	auth_chquantum
 *	Authorize a user request to change file's quantum time slice.
 *	This is not a security relevant event.
 */
auth_chquantum()
{
#ifdef	PERF_NOSUCHK
	return 1;
#else	/* PERF_NOSUCHK */
	return auth_adm();
#endif	/* PERF_NOSUCHK */
}

/*
 *	auth_chslice
 *	Authorize a user request to change file's time slice clamp.
 *	This is not a security relevant event.
 */
auth_chslice()
{
#ifdef	PERF_NOSUCHK
	return 1;
#else	/* PERF_NOSUCHK */
	return auth_adm();
#endif	/* PERF_NOSUCHK */
}

/*
 *	auth_chpri
 *	Authorize a user request to change file's pri clamp.
 *	This is not a security relevant event.
 */
auth_chpri()
{
#ifdef	PERF_NOSUCHK
	return 1;
#else	/* PERF_NOSUCHK */
	return auth_adm();
#endif	/* PERF_NOSUCHK */
}

/*
 *	auth_chusrpri
 *	Authorize a user request to change file's usrpri clamp.
 *	This is not a security relevant event.
 */
auth_chusrpri()
{
#ifdef	PERF_NOSUCHK
	return 1;
#else	/* PERF_NOSUCHK */
	return auth_adm();
#endif	/* PERF_NOSUCHK */
}

/*
 *	auth_chcpu
 *	Authorize a user request to change file's cpu clamp.
 *	This is not a security relevant event.
 */
auth_chcpu()
{
#ifdef	PERF_NOSUCHK
	return 1;
#else	/* PERF_NOSUCHK */
	return auth_adm();
#endif	/* PERF_NOSUCHK */
}

/*
 *	auth_chpriadj
 *	Authorize a user request to change file's pri adj value.
 *	This is not a security relevant event.
 */
auth_chpriadj()
{
#ifdef	PERF_NOSUCHK
	return 1;
#else	/* PERF_NOSUCHK */
	return auth_adm();
#endif	/* PERF_NOSUCHK */
}

/*
 *	auth_chflags
 *	Authorize a user request to change file's performance flags.
 *	This is not a security relevant event.
 */
auth_chflags()
{
#ifdef	PERF_NOSUCHK
	return 1;
#else	/* PERF_NOSUCHK */
	return auth_adm();
#endif	/* PERF_NOSUCHK */
}

/*
 *	auth_swap   auth a user request to add or remove swap
 *		    devices to the kernel. This is a security
 *		    relevant event beacuse the swap device chosen
 *		    could be a unsecure device...furhter swapping
 *		    could catch utilities that do passwd stuff on
 *		    swap!!
 */
auth_swap()
{	
	return auth_sec();
}

/*
 * 	auth_errpt  auth a user requrest to toggle the error reporting
 *		    on and off. This isn't a security relevant event
 *		    but maybe it should be!!!
 */
auth_errpt()
{
	return auth_adm();
}

/*
 *	auth_chown  auth a user requrest to chown a file he doesn't
 *		    own. This is not a security relevant event cause
 *		    this priv doesn't allow user to bypass MACs.     
 */ 
auth_chown()
{
	return auth_owner();
}

/*
 *	auth_chmode auth a user to change the modes on a files he 
 *		    doesn't own. Not a security relevant event cause
 *		    it doesn't allow user to bypass MACs.
 */
auth_chmod()
{
	/* JTOF - remember to change this to P_DAC_READ and P_DAC_WRITE
	 */
	return auth_owner();
}



/*
 *	auth_ulimit  auth a user request to change the file limit 
 *		     for his process
 */
auth_ulimit()
{
#ifdef SECON
	if ( p_chkall( P_ULIMIT ) ) {
		u.u_acflag |= ASU;
		uauth->a_pused |= P_ULIMIT;
		return 1;
	} 
	else {
		u.u_error = EPERM;
		return 0;
	}
#else
	return suser();
#endif
}

/*
 *	auth_acl   auth a user requrest to create or update an ACL
 */
auth_acl()
{
#ifdef SECON 
	if ( p_chkall( P_DAC_ACL )) {
		u.u_acflag |= ASU;
		uauth->a_pused |= P_DAC_ACL;
		return 1;
	} else
		return 0;
#else
	return suser();
#endif
}

/*
 *	auth_advfs  uth a user request to advertise rfs filesystem. This
 *		    This isn't a security relevant event because it's 
 *		    assumed that the security stuff done already and were
 *		    only letting others see the filesys.
 */
auth_advfs() 
{
	return auth_adm();
}

/*
 *	auth_unadvfs auth a user requset to unadvertise an rfs filesystem.
 *		     This request isn't security relevant 
 */
auth_unadvfs()
{
	return auth_adm();
}

/*
 *	auth_idmap  auth ability to download rfs uid/gid maps to the 
 *		    kernel. This is a highly security relevent event 
 */
auth_idmap()
{
	return auth_sec();
}

/*
 *	auth_rfstart   auth users request to start rfs. This is 
 *		        considered security relevant cause RFS could
 *		        provide excellent break-ins with privs and 
 *			labels setup on nonsecure remote filesystems. 
 */
auth_rfstart()
{
	return auth_sec();
}

/*
 *	auth_rfstop   auth user to stop rfs. This isn't security important
 *		       cause we are closing a hole
 */
auth_rfstop()
{
	return auth_adm();
}


/* 
 * 	auth_rmount   auth a user to mount a remote filesystem 
 */
auth_rmount()
{
	return auth_sec();
}


/* 
 * 	auth_rumount   auth a user to unmount a remote filesystem 
 */
auth_rumount()
{
	return auth_adm();
}

/* 
 * 	auth_mount   auth a user to do mount a filesystem 
 */
auth_mount()
{
#ifdef SECON
	if (p_chkall( P_FSYS )) {
		u.u_acflag |= ASU;
		uauth->a_pused |= P_FSYS;
		return 1;
	} 
	else {
		u.u_error = EPERM;
		return 0;
	}
#else
	return suser();
#endif
}

/* 
 * 	auth_umount   auth a user to do umount a filesystem 
 */
auth_umount()
{
	return auth_mount();
}

/*
 *	auth_vd    auth a user request to access stripe driver
 */
auth_vd()
{
	return auth_mount();
}

/*
 *	auth_md    auth a user request to access mirror driver
 */
auth_md()
{
	return auth_mount();
}

/*
 *	auth_rw   provide any r/w authorization functionality necessary
 */
auth_rw(ip,mode)
struct inode *ip;
int mode;
{
	/* DO NOTHING TILL B1 DAYS */
}

/*
 *	auth_copen   provide additional open authorization
 */
auth_copen(nmarg)
struct argnamei *nmarg;
{
	/* DO NOTHING TILL B1 DAYS */
}

/*
 *	auth_stat1  
 */
auth_stat1(ip, ub)
struct inode *ip;
struct stat *ub;
{
	/* DO NOTHING TILL B1 DAYS */
}

/*
 *	auth_nami   allow the insertion of multilevel directory rerouting
 */
auth_nami(dp,cp,np,limit)
struct inode *dp;
caddr_t cp, np, limit;
{
	/* DO NOTHING TILL B1 DAYS */
}
/*
 *	auth_sec  auth a user based on his ownership of P_SECURITY
 *		  in the real priv only. This is intend to be called
 *		  by other auth calls.
 */
auth_sec()
{
#ifdef SECON 
	if ( p_chkreal( P_SECURITY)) {
		u.u_acflag |= ASU;
		uauth->a_pused |= P_SECURITY;
		return 1;
	} 
	else {
		u.u_error = EPERM; 
		return 0;
	}
#else
	return suser();
#endif
}

/*
 *	auth_adm_ne   do the same thing as auth_adm but don't set the
 *		      errno's 
 */
auth_adm_ne()
{
#ifdef SECON
	return p_chkall( P_ADM );
#else
	return (u.u_uid == 0);
#endif
}

/*
 *	auth_chown_ne   do the same thing as auth_chown, but don't bother
 *		        setting errno's 
 */
auth_chown_ne()
{
#ifdef SECON
	return p_chkall( P_OWNER );
#else
	return (u.u_uid == 0);
#endif
}

/*
 *	auth_chmod_ne  do the same thing as auth_chmod, but don't bother
 *		       setting errno's 
 */
auth_chmod_ne()
{
#ifdef SECON
	return p_chkall( P_OWNER );
#else
	return (u.u_uid == 0);
#endif
}

 

/*
 *	auth_adm   auth a user based on his possession of the P_ADM
 *		   privilege. This call will be used by several other
 *		   auth calls and is intended to reduce code size only!
 */
auth_adm()
{
#ifdef SECON 
	if ( p_chkall( P_ADM )) {
		u.u_acflag |= ASU;
		uauth->a_pused |= P_ADM;
		return 1;
	} 
	else {
		u.u_error = EPERM;
		return 0;
	}
#else
	return suser();
#endif
}

/*
 *	auth_owner auth a user to perform operations on files he doesn't
 *		   own. This call is intended to be used by other auth_calls
 */
auth_owner()
{
#ifdef SECON 
	if ( p_chkall( P_OWNER )) {
		u.u_acflag |= ASU;
		uauth->a_pused |= P_OWNER;
		return 1;
	} 
	else {
		u.u_error = EPERM;
		return 0;
	}
#else
	return suser();
#endif
}

/*
 *	auth_setgrps  auth user request to set supplementarty groups 
 */
auth_setgrps()
{
#ifdef SECON 
	if ( p_chkall( P_ADDGRPS )) {
		u.u_acflag |= ASU;
		uauth->a_pused |= P_ADDGRPS;
		return 1;
	} 
	else {
		u.u_error = EPERM;
		return 0;
	}
#else
	return suser();
#endif
}


/*
 *	auth_ipc  auth request to override IPC DAC mechanism to 
 *	  	  set, remove or modify and IPC. This is intended
 *		  to be used by other auth calls 
 */
auth_ipc()
{
#ifdef SECON
	if ( p_chkall( P_DAC_EXEMPT )) {
		u.u_acflag |= ASU;
		uauth->a_pused |= P_DAC_EXEMPT;
		return 1;
	} 
	else {
		u.u_error = EPERM;
		return 0;
	}
#else
	return suser();
#endif
}

/*
 *	auth_suser   return the security version of the suser 
 *		     call. 
 */

auth_suser()
{
	/* currently this guy does same thing as original suser
 	 * call. It is intended for compatability with any drivers
 	 * that have not been properly converted to a security
         * scheme. It will complain about such drivers if the 
         * proper flags are on. This should display a partial 
 	 * bactrace so the driver can be easily identified!! 
         */
	if ( secdebug ) {
		printf( "suser() called by driver\n" );
	}

	if (u.u_uid == 0) {
		u.u_acflag |= ASU;
		return 1;
	}
	u.u_error = EPERM;
	return 0;
}
		

/*
 *	auth_setuid  perform the real setuid function for a process. In
 *	             a privilege kernel, this involves looking up the 
 *		     role definition for a uid and setting the rolenun,
 *		     privileges and macs 
 */
#ifdef SECON 
auth_setuid( uid )
uint uid;
{
	kern_role_t *rp, *getuidinfo();
	int ret = 0;	/* presume innocent */

	/* if we aren't privileges then if we're setting uid to our uid
	 * or toggling back check, set up the role stuff
 	 */ 
	if ( !p_chkall(P_SETUID) ) {
		if (uid == u.u_ruid || uid == u.u_procp->p_saved_set_uid) {
			/* try to setup the users environment from the
			 * internal role database. If we can't either 
			 * the user or role isn't known, so don't honor
			 * the requests 
			 */
			if (setrole( uid, 0 ))
				u.u_uid = uid;
			else 
				ret = u.u_error = EPERM;
		}
		else
			ret = u.u_error = EPERM;
	}
	else {
		u.u_acflag |= ASU;
		uauth->a_pused |= P_SETUID;
		/* try to setup the users environment from the
 		 * internal role database. If we can't either 
		 * the user or role isn't known, so don't honor
		 * the requests, and set the user id to some 
 		 * very funky number (check this out..-1 may be a booboo) 
		 */
		if (!setrole( uid, 0 )) {
			uid = -1;
			ret = u.u_error = EPERM;
		}
		u.u_uid = uid;
		u.u_ruid = uid;
		u.u_procp->p_uid = uid;
		u.u_procp->p_suid = uid;
		u.u_procp->p_saved_set_uid = uid;
		if ( uauth->a_auid == 0 )
			uauth->a_auid = uid;
	}

	return ( ret == 0 );
}
#else
auth_setuid( uid )
uint uid;
{
	if (u.u_uid && (uid == u.u_ruid || uid == u.u_procp->p_saved_set_uid)) 
		u.u_uid = uid;
	else if ( suser()) {
		u.u_uid = uid;
		u.u_procp->p_uid = uid;
		u.u_procp->p_suid = uid;
		u.u_procp->p_saved_set_uid = uid;
		u.u_ruid = uid;
	}
	return 1;
}
#endif

/*
 *	auth_setgid  perform the setgid fucntion. The whole body of the
 *		     system call is here so we can do whatever to it in 
 *		     the future 
 */
#ifdef SECON
auth_setgid(gid)
uint gid;
{
	kern_gidmap_t *gp;
	int ret = 0;	/* presume innocent */

	if ( !p_chkall(P_SETGID) ) {
		if (gid == u.u_rgid || gid == u.u_procp->p_saved_set_gid) {
			/* see if the gid database knows about this
		         * group or if the group has been locked. If
			 * so, then disallow the setgid
			 */
			if ( getgidinfo( gid ) ) 
				u.u_gid = gid;
			else
				ret = u.u_error = EPERM;
		}
		else
			ret = u.u_error = EPERM;
	}
	else {
		u.u_acflag |= ASU;
		uauth->a_pused |= P_SETGID;
		/* see if the gid database knows about this
		 * group or if the group has been locked. If
		 * so, then disallow the setgid and set the gid
		 * to some very funcky number (-1 may be bad choice)
		 */
		if ( !getgidinfo( gid ) ) {
			ret = u.u_error = EPERM;
			gid = -1;
		}
		u.u_gid = gid;
		u.u_rgid = gid;
		u.u_procp->p_sgid = gid;
		u.u_procp->p_saved_set_gid = gid;
	}

	return ( ret == 0 );
}
#else
auth_setgid(gid)
uint gid;
{
	if (u.u_uid && (gid == u.u_rgid || gid == u.u_procp->p_saved_set_gid)) 
		u.u_gid = gid;
	else if ( suser() ) {
		u.u_gid = gid;
		u.u_procp->p_sgid = gid;
		u.u_procp->p_saved_set_gid = gid;
		u.u_rgid = gid;
	}
	return 1;
}
#endif

/*
 *	fileref   return the number of references for a file
 */

fileref()
{
#ifdef SECON
struct file *fp, *getf();
int fd;
register struct a {
	uint	fd;	
	} *uap = (struct a *) u.u_ap;

	u.u_error = 0;
	if ((fp=getf(uap->fd)) == NULL)
		return;
	u.u_rval1 = fp->f_inode->i_count;
	return;
#else
	u.u_rval1 = 1;
	return; 
#endif
}

#ifdef SECON
/* 
 *	p_chkall check to see if the privilege (p) is owned by the 
 *	         process in any way, shape or form. 
 *
 */
p_chkall( priv )
register uint priv;
{
	if ( sec_mode & PRIVON )
		return (((uauth->a_priv.real | 
			uauth->a_priv.eff ) & priv ) == priv );
	else
		return suser();
}

/*
 *	p_chkreal  check to see if the privilege (p) is owned by the
 *		   process as a real privilege. 
 */
p_chkreal( priv )
register uint priv;
{
	if ( sec_mode & PRIVON )
		return ( (uauth->a_priv.real & priv ) == priv );
	else
		return suser();
}

/*
 *	p_chkeff  check to see if the privilege (p) is owned by the
 *	          process as an effective privilege 
 */
p_chkeff( priv )
register uint priv;
{
	if ( sec_mode & PRIVON )
		return ((uauth->a_priv.eff & priv ) == priv );
	else
		return suser();
}
#endif
