/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) synch.c: version 25.1 created on 11/27/91 at 15:12:30	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)synch.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/

/*
 * synch.c
 *
 *	multiprocessor synchronization support routines
 */


#include "sys/param.h"
#include "sys/types.h"
#include "sys/synch.h"
#include "sys/mfs.h"
#include "sys/debug.h"
#include "sys/sbus.h"
#include "sys/sbus_pm.h"
#include "sys/spm_mem.h"
#include "sys/pm_iomap.h"
#include "sys/lio.h"
#include "sys/own.h"
#include "sys/cmn_err.h"
#include "sys/systm.h"
#include "sys/buf.h"
#include "sys/ints.h"


/*
 * Most spin_locks are initialized here so that they will be placed
 * together.  When they were in bss they ended up linked at addresses that
 * were far apart.
 */

spin_lock_t
	runqs_lock,
	proc_lock,
	flock_sleeplist_sem,
	freelocklist_sem,
	inode_sem,
	file_locking_sem,
	region_lock_sem,
	bio_lock,
	s54kbio_lock,
	super_block_flock,
	super_block_ilock,
	callout_lock,
	swap_lock,
	swapbuf_lock,
	delay_lock,
	file_freelist_lock,
	errlog_sem,
	sat_sem;


#define	BUSY	0x80
#define	WANT	0x40

/*
 * suspend_lock -- get a high level lock, otherwise suspend
 */

suspend_lock(lp)
register suspend_lock_t	*lp;
{
	register int	waited = 0;

	while (atom_or_byte(&lp->s_lock, BUSY) & BUSY) {
		spin_lock(&lp->s_spin_lock);
		if (lp->s_proc == (uint)u.u_procp) {
#ifdef SUSPEND_DEBUG
			suspend_lock_t	lt;

			suscheckpoint(&lt);
			susbadmsg("suspend_lock: already has lock", lp, &lt);
#else
			cmn_err(CE_PANIC,
			  "%s: proc=%x already has lock=%x (lcks=%d, col=%d)\n",
			  "suspend_lock",
			  u.u_procp, lp, lp->s_numlocks, lp->s_collisions);
#endif /* SUSPEND_DEBUG */
		}
		if (atom_or_byte(&lp->s_lock, WANT) & BUSY) {
			waited = 1;
			mfs_sleep(lp, lp->s_priority, &lp->s_spin_lock);
		}
		else
			(void) atom_and_byte(&lp->s_lock, ~WANT);
		spin_unlock(&lp->s_spin_lock);
	}
	lp->s_proc = (uint)u.u_procp;
	lp->s_numlocks++;
	lp->s_collisions += waited;
#ifdef SUSPEND_DEBUG
	suscheckpoint(lp);		/* remember locker */
#endif
}

#ifdef SUSPEND_DEBUG

#include "sys/kmem.h"

/*
 * suscheckpoint -- record short stack trace of the locker/unlocker
 */
suscheckpoint(lp)
suspend_lock_t	*lp;
{
	register suspend_lock_t	*lockp;
	register uint		*fp;
	register uint		nfp;
	register int		n;
	extern uint		sys_stk_end[];

	nfp = (uint)(&lp - 2);	/* stack frame is fp, pc, arg, .... */
	n = 0;
	for ( ; n < SUSPEND_DEBUG && nfp >= ADDR_EXTRA_STK &&
	  nfp <= (uint)sys_stk_end - 4; n++) {
		fp = (uint *)nfp;
		nfp = fp[0];
		lockp->s_func[n] = fp[1];	/* store return pc */
	}

	for ( ; n < SUSPEND_DEBUG; n++)
		lockp->s_func[n] = ~0;	/* invalid pc */
}

/*
 * susbadmsg -- print out a failure message and report the procs and traces
 *		lp0 is the suspend_lock, lp1 is a temp checkpoint suspend_lock
 */
susbadmsg(msg, lp0, lp1)
char		*msg;
suspend_lock_t	*lp0, *lp1;
{
	register int		i, n;
	register suspend_lock_t	*lockp;
	static char		*rpass[] = { "was", "now" };

	printf("%s\n", msg);
	for (lockp = lp0, i = 0; i <= 1; lockp = lp1, i++) {
		printf("%s: proc=%x, locks=%d, coll=%d, trace: ", rpass[i],
		  lockp->s_proc, lockp->s_numlocks, lockp->s_collisions);
		for (n = 0; n < SUSPEND_DEBUG; n++)
			printf("%s%x", (n ? ", " : ""), lockp->s_func[n]);
		printf("\n");
	}
	cmn_err(CE_PANIC, "suspend lock=%x", lp0);
}
#endif /* SUSPEND_DEBUG */

/*
 * try_suspend_lock -- try to get a high level lock, return result
 */

uint
try_suspend_lock(lp)
register suspend_lock_t	*lp;
{
	if (lp->s_proc == (uint)u.u_procp) {
#ifdef SUSPEND_DEBUG
		suspend_lock_t	lt;

		suscheckpoint(&lt);
		susbadmsg("try_suspend_lock: already have lock", lp, &lt);
#else
		cmn_err(CE_PANIC,
		  "%s: proc=%x already has lock=%x (locks=%d, coll=%d)\n",
		  "try_suspend_lock",
		  u.u_procp, lp, lp->s_numlocks, lp->s_collisions);
#endif /* SUSPEND_DEBUG */
	}

	if (atom_or_byte(&lp->s_lock, BUSY) & BUSY)
		return(0);	/* already in use */

	lp->s_proc = (uint)u.u_procp;
	++lp->s_numlocks;
#ifdef SUSPEND_DEBUG
	suscheckpoint(lp);		/* remember locker */
#endif

	return(1);	/* we got it */
}


/*
 * suspend_unlock -- release a high level lock, waking up any others
 *	that might want it.
 */

suspend_unlock(lp)
register suspend_lock_t	*lp;
{
	if (lp->s_proc != (uint)u.u_procp) {
#ifdef SUSPEND_DEBUG
		suspend_lock_t	lt;

		suscheckpoint(&lt);
		susbadmsg("suspend_unlock: don't own lock", lp, &lt);
#else
		cmn_err(CE_PANIC,
		  "%s: proc=%x didn't own lock=%x, %x did (locks=%d, col=%d)\n",
		  "suspend_unlock",
		  u.u_procp, lp, lp->s_proc, lp->s_numlocks, lp->s_collisions);
#endif /* SUSPEND_DEBUG */
	}

	lp->s_proc = 0;				/* lock released */

	if (atom_and_byte(&lp->s_lock, ~(WANT | BUSY)) & WANT) {
		spin_lock(&lp->s_spin_lock);
		mfs_wakeup_all(lp);
		spin_unlock(&lp->s_spin_lock);
	}
}

/*
 * suspend_islocked -- return 1 if the lock is owned by the current process
 */

uint
suspend_islocked(lp)
register suspend_lock_t	*lp;
{
	if ((lp->s_lock & BUSY) && lp->s_proc == (uint)u.u_procp)
		return (1);			/* we already own the lock */
	return(0);
}

/*
** dizzy_lock is motivated by the desire to have lockers w/o
** spying caches (IOPM) spin in their local memory, not on the CSS. Even
** among lockers with spying caches (PM) this new lock forces FIFO behaviour
** on requestors instead of the free-for-all behaviour with sem_lock.
**
** The dizzy_lock, spin and lock_req_array structures are manipulated
** by dizzy_lock and dizzy_unlock routines.
** There is one dizzy_lock struct per lockable resource.
** There is one lock req per locker (eg PM20, DPM40, or IOPM).
** For PM's the lock req is the lock_req_array element indexed by own_lock_id.
** For IOPM's part of the lock req is the lock_req_array element indexed
** by own_lock_id and part is "spin" in local memory for the IOPM.
**
** The lock is free if "head_lock_req" is zero.
** To aquire the lock, the requester adds its lock req to the head of the
** list of requesters. If the list transistioned from empty, the requester
** is the new owner, otherwise the requester waits.
** When releasing the lock, the owner passes the lock to the next requester.
** The next_lock_req entry gives the address of the array portion of the lock
** req as a link to the previous requestor.
** Requestors add themselves
** to the head of the list (near dizzy_lock). The owner of the lock removes
** themself from the tail of the list (far from dizzy_lock) and
** passes the lock to the next requestor (new tail of the list).
** Requesters stay on the list until they give up the lock.
** Requesters can be on more than one list. They may own one lock and be waiting
** for one other lock.
** This gives rise to lock lists that logically look like this:
**
**	LOCK A -> req 1 -> req 2 -> req 3
**	LOCK B -> req 3 -> req 4
**
** Note that req 3 owns LOCK A and is waiting for LOCK B. Note also that
** even though req 3 points to req 4, req 4 is not on the LOCK A list and
** is not the owner of LOCK A. This is not apparent to an outside observer
** traveling down the list for LOCK A without knowlegde of the LOCK B list.
** But req 3 knows it owns LOCK A and can give it to req 2 at will.
*/

struct lock_req_array  lock_req_array[LLOCK_MAX];

valid_lock_req_array(lrap)
register struct lock_req_array  *lrap;
{
	if ( lrap >= lock_req_array &&
	  lrap < &lock_req_array[LLOCK_MAX] &&
	  ((uint)lrap - (uint)lock_req_array) % sizeof(struct lock_req_array) )
		return 0;

	return 1;
}

/*****************************************************************************/
/* If dizzy_unlock calls dizzy_unlock_c we know there is at least one requestor
/* on the list. Requestors never come off the list except by being taken off by
/* the owner of the lock (us).
/* Walk down the list to the last requestor and give the lock to requestor.
*/

dizzy_unlock_c(lockp)
register struct dizzy_lock  *lockp;
{
	register struct lock_req_array  *lrap;
	register uint                   chain_len = 1;
	uint                            savmap;
	extern struct lock_req_array    *own_lock_req_ptr;
	extern uchar                    own_lock_id;
	extern struct iopm_comm         *iopm_base[];

	ASSERT(lockp->lock_owner == own_lock_id);	/* belongs to us */

	lrap = lockp->head_lock_req;	/* must be atomic, hence aligned */
	ASSERT(lrap != own_lock_req_ptr);	/* someone waiting*/
	ASSERT(valid_lock_req_array(lrap));

	/* Travel down the list of dizzy requestors and remove the last req */
	while ( lrap->next_lock_req != own_lock_req_ptr )
	{
		chain_len++;
		lrap = lrap->next_lock_req;
		ASSERT(valid_lock_req_array(lrap));
	}

	/* lrap is the lock req pointing to us. It will be the new lock owner*/
	/* lrap->next_lock_req is us. We are the end of the request chain. */
	ASSERT(own_lock_req_ptr->next_lock_req == 0);
	lockp->max_chain = max(lockp->max_chain, chain_len);

	/* remove our req from list */
	lrap->next_lock_req = 0;

	lockp->lock_owner = lrap - lock_req_array;

	ASSERT(lrap->lra.addr);		/* either spin is set or addr is set */
	/* alert next lock owner */
	if ( lrap->remote_spin )
	{
		ASSERT(lrap->lra.addr != (ulong *)1);	/* valid remote addr */
		savmap = iomap_save();
		/* non IOPM specific alternative */
		*(ulong *)iopm_map(lrap->slot, lrap->lra.addr) = 0; 
		iomap_restore(savmap);
	}
	else
		lrap->lra.spin = 0;
}

stop_all_processors()
{
#if 0
	register uint 	pm_id;

	for ( pm_id = 0; pm_id < spm_mem.num_pm; pm_id++ ) {
		if ( pm_id != own.o_pm_id )
			send_pm_interrupt(pm_id, PM_LEVEL_SEVEN_INT_REQ);
	}

	while (1)
		send_pm_interrupt(own.o_pm_id, PM_LEVEL_SEVEN_INT_REQ);
#endif /* 0 */
	for (;;)
		do_bus_freeze();
}


/* UPKERN support */

uint
is_upkern_lock()
{
	upkern_t	upkern;

	upkern = spm_mem.upkern;

	return (upkern.up_pm_id == own.o_pm_id && upkern.up_cnt);
}

/* Higher level upkern routines */

extern RUNQ	upkern_runqs;


upkern_lock()
{
	ASSERT(!own.o_in_int_service);
	ASSERT(u.u_procp->p_upkern_cnt <= UPKERN_MAX);
	if (u.u_procp->p_upkern_cnt == UPKERN_MAX)
		return;					/* max-ed out	*/
	u.u_procp->p_upkern_cnt++;

	if (!own.o_upkern_proc) {
		if (upkern_try_inc()) {
			own.o_upkern_proc = 1;
			spm_mem.upkern.up_lock_cnt++;
		}
		else {
			atom_inc(&spm_mem.upkern.up_qswtch_cnt);
			qswtch();
		}
	}
	ASSERT(own.o_upkern_proc);
}

upkern_unlock()
{
	extern uint	upkern_runq_cnt;

	ASSERT(!own.o_in_int_service);
	ASSERT(u.u_procp->p_upkern_cnt);
	ASSERT(own.o_upkern_proc);
	ASSERT(u.u_procp->p_upkern_cnt <= UPKERN_MAX);

	if (u.u_procp->p_upkern_cnt == UPKERN_MAX)
		return;					/* max-ed out	*/
	if (--u.u_procp->p_upkern_cnt == 0) {
		ASSERT(is_upkern_lock());
		own.o_upkern_proc = 0;
		upkern_dec();
		ASSERT(!own.o_upkern_proc);
	}
}

upkern_unlock_all()
{
	extern uint	upkern_runq_cnt;

	ASSERT(!own.o_in_int_service);
	ASSERT(own.o_upkern_proc);
	ASSERT(u.u_procp->p_upkern_cnt);
	ASSERT(is_upkern_lock());
	u.u_procp->p_upkern_cnt = 0;
	own.o_upkern_proc = 0;
	upkern_dec();
	if ( upkern_runqs.rq_cnt )
		own.o_runrun = 1;
}

/*
 * upkern_save -- return the current upkern level (much like splN)
 */

uint
upkern_save()
{
	ASSERT(!own.o_in_int_service);
	return (u.u_procp->p_upkern_cnt);
}

/*
 * upkern_restore -- restore a previously saved upkern level (like splx)
 */

upkern_restore(cnt)
uint	cnt;
{
	ASSERT(!own.o_in_int_service);
	ASSERT(u.u_procp->p_upkern_cnt >= cnt);	/* never increase level */
	ASSERT(cnt <= UPKERN_MAX);

	if (u.u_procp->p_upkern_cnt == 0)
		return;

	ASSERT(own.o_upkern_proc);
	ASSERT(is_upkern_lock());
	if (cnt == 0)
		upkern_unlock_all();
	else
		u.u_procp->p_upkern_cnt = cnt;
}

sl_fquit_slave()
{
	sl_quit_slave();
}

sl_quit_slave()
{
	ASSERT(!own.o_in_int_service);
	ASSERT(u.u_procp->p_upkern_cnt <= UPKERN_MAX);

	/*
	 * set p_upkern_cnt to a never unlockable level
	 */
	u.u_procp->p_upkern_cnt = UPKERN_MAX;
	if (!own.o_upkern_proc) {
		if (upkern_try_inc()) {
			own.o_upkern_proc = 1;
		}
		else
			qswtch();
	}
	ASSERT(own.o_upkern_proc);
}

spin_enter_upkern()
{
	while (!upkern_try_inc())
		;
}

queue_level_one(o, int_data)
register own_t	*o;
disp_int_t	int_data;
{
	register disp_int_t	*old_prdcr, *new_prdcr;

	do {
		old_prdcr = o->o_queue_1.producer;

		new_prdcr = old_prdcr + 1;
		/* handle wrap around */
		if (new_prdcr == o->o_queue_1.end)
			new_prdcr = o->o_queue_1.data;

		/* handle queue full */
		while (*(uint *)new_prdcr)
			;

	} while (! cas_long(&o->o_queue_1.producer, old_prdcr, new_prdcr));

	*old_prdcr = int_data;

	send_pm_interrupt(o->o_pm_id, PM_LEVEL_ONE_INT_REQ);
}

queue_level_four(o, int_data)
register own_t	*o;
disp_int_t	int_data;
{
	register disp_int_t	*old_prdcr, *new_prdcr;

	do {
		old_prdcr = o->o_queue_4.producer;

		new_prdcr = old_prdcr + 1;
		/* handle wrap around */
		if (new_prdcr == o->o_queue_4.end)
			new_prdcr = o->o_queue_4.data;

		/* handle queue full */
		while (*(uint *)new_prdcr)
			;

	} while (! cas_long(&o->o_queue_4.producer, old_prdcr, new_prdcr));

	*old_prdcr = int_data;

	send_pm_interrupt(o->o_pm_id, PM_LEVEL_FOUR_INT_REQ);
}

/************* uniprocessor device driver support *********************/

#define b_strategy	av_back

static buf_t		*upkern_strat_head;
static disp_int_t	upkern_strategy_int_data;

extern buf_t		*revervse_bp_list();

/* 
 * initialize interrupt data to be used
 * for upkern_strategy interrupts.
 */

upkern_strategy_init()
{
	upkern_strategy_int_data.fields.vector = UPKERN_STRATEGY;
	upkern_strategy_int_data.fields.level = MOT_LEVEL_ONE;
	upkern_strategy_int_data.fields.directed = NON_DIRECTED;
}

upkern_strategy(strat, bp)
void	(*strat)();
buf_t	*bp;
{
	bp->b_strategy = (buf_t *)strat;
	if (add_chain(&upkern_strat_head, bp, &bp->av_forw))
		return;

	queue_level_one(spm_mem.pm_own[upkern_inc()], upkern_strategy_int_data);
}

upkern_do_strategy()
{
	register buf_t	*bp, *bp_list;
	extern uint	remove_chain();
	buf_t		*reverse_bp_list();

	bp_list = (buf_t *)remove_chain(&upkern_strat_head);

	if (bp_list && bp_list->av_forw)
		bp_list = reverse_bp_list(bp_list);
	while (bp = bp_list) {
		bp_list = bp->av_forw;
		ASSERT(is_upkern_lock());
		ASSERT(bp->b_strategy);
		(*(void (*)())bp->b_strategy)(bp);
	}
	upkern_dec();
}
