/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) buserr.c: version 25.6 created on 6/17/92 at 10:51:48	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)buserr.c	25.6	6/17/92 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/

/*
 * buserr.c -- deal with different kinds of bus errors
 */


#include "sys/types.h"
#include "sys/lio.h"
#include "sys/psl.h"
#include "sys/sysmacros.h"
#include "sys/immu.h"
#include "sys/systm.h"
#include "sys/user.h"
#include "sys/errno.h"
#include "sys/inode.h"
#include "sys/mount.h"
#include "sys/fstyp.h"
#include "sys/var.h"
#include "sys/buf.h"
#include "sys/utsname.h"
#include "sys/sysinfo.h"
#include "sys/pfdat.h"
#include "sys/region.h"
#include "sys/proc.h"
#include "sys/state.h"
#include "sys/map.h"
#include "sys/swap.h"
#include "sys/getpages.h"
#include "sys/debug.h"
#include "sys/cmn_err.h"
#include "sys/conf.h"
#include "sys/spm_mem.h"
#include "sys/own.h"
#ifdef	PERF
#include "sys/schedcpu.h"
#include "sys/schedext.h"
#endif	/* PERF */

/* hanna FIX: how can some of the functionality of the findpde be replaced
 * hanna FIX: by the ptest?
 * hanna FIX: perhaps make findpde(&ptestval) return the pde address, if ther
 * hanna FIX: is one, but also, do the ptest and return the results in the
 * hanna FIX: ptestval.
 * hanna FIX: The poopy thing is that the ptestval has all the info, but
 * hanna FIX: we want the pde location to make mods to.
 */
extern pde_t	*findpde();
extern ulong	ptest_addr();

/*
 * bus fault handlers
 * the following bus fault cases will be fixed:
 * 
 *	page not present
 * 	write protect    (user violation)
 * 	copy on write	 (data page is shared)
 *	stack growth
 *	written bit simulation ?
 *	access bit simulation ?
 *	Data cache push fault (68040)
 *	Mov16 fault
 *	various hardware & "impossible" faults will be recognized.
 */

/*
 * k_buserr -- Motorola bus error while in kernel mode
 */
/* hanna DEBUG */extern int hanflag;
k_buserr(stk_frame)
struct	state stk_frame;		/* stk_frame is value/result */
{
	register int	signo;
	register uint	statusreg;
	register int	caddrsave;
	uint		saved_cpu_dests = own.o_cpu_same;

	statusreg = *(unsigned *)STATUS_REG;

/* hanna FIX -- put in stuff for stack frame crunching? */

	/*
	 * Try to correct the fault.
	 * usrxmemflt will return:
	 *	SIGBUS if hardware or kernel problem
	 *	SIGSEGV if attempting to access an invalid user page
	 *	0 if a page fault that has been fixed
	 */

	caddrsave = u.u_caddrflt;
	u.u_caddrflt = 0;
	signo = usrxmemflt(&stk_frame, statusreg);

	u.u_caddrflt = caddrsave;

	switch (signo) {
	case SIGSEGV:
		if (caddrsave) {
			stk_frame.s_pc = caddrsave;
			stk_frame.s_crunch = 1;
			break;
		}
		/* no break */

	default:
	case SIGBUS:
		showbuserr(&stk_frame, statusreg);
		panic("Kernel mode bus error");
		/* NOTREACHED */

	case 0:
		/* Force reschedule.  This is needed to deal with the
		 * case where we are called from an interrupt routine
		 * (e.g. addupc from clock interrupt) or some other
		 * kernel routine that doesn't call u_trap_done before
		 * returning to user mode.
		 */
		own.o_runrun = 1;
	}
}

/*
 * u_buserr -- Motorola bus error while in user mode 
 */

u_buserr(stk_frame)
struct	state stk_frame;		/* stk_frame is value/result */
{
	register int	signo;
	register time_t	syst;
	register uint	statusreg;
	uint		saved_cpu_dests = own.o_cpu_same;
	extern uint	user_pm_bits;

	statusreg = *(unsigned *)STATUS_REG;

	atom_and(&user_pm_bits, ~own.o_id_bit);

	syst = u.u_stime;
	u.u_astk = &stk_frame;
	signo = usrxmemflt(&stk_frame, statusreg);

	if (signo) {
		if (!u.u_signal[signo - 1])
			showbuserr(&stk_frame, statusreg);
		fpu_stuck_check();
		psignal(u.u_procp, signo);

		if (ISSIG(u.u_procp, FORREAL)) {
			upkern_lock();
			psig();
		}
	}
	else if (ISSIG(u.u_procp, FORREAL)) {
		/*
		 * If we have a pending signal, we can't process it now or we
		 * will lose information in the extended stack frame, so
		 * arrange to take a trace trap at the end of this instruction.
		 *[This isn't true for the 68040, since we don't stack crunch]
		 */
		stk_frame.s_ps |= PS_T;
		atom_or(&u.u_procp->p_flag, SBUSTRC);
	}

	if (u.u_prof.pr_scale)
		addupc(stk_frame.s_pc, &u.u_prof,
			(int)(u.u_stime-syst));

#ifdef	PERF
 	USERPRI(u.u_procp); /* modifies p_pri & own.o_curpri */
#else	/* PERF */
 	own.o_curpri = u.u_procp->p_pri = calcppri(u.u_procp);
#endif /* PERRF */

	/* check for cpu mask number mismatch */

	if (stk_frame.s_format == 0x7 && !stk_frame.s_crunch) {
		u.u_procp->p_dests &= saved_cpu_dests;
		if (! (own.o_id_bit & u.u_procp->p_dests))
			own.o_runrun++;
	}

	u_trap_done();
}


usrxmemflt(stk_ptr, statusreg)
register state_t	*stk_ptr;
uint			statusreg;
{
	ulong		vaddr;
	register pde_t	*pde;
	register uint	signo = 0;
	register uint	ssw = stk_ptr->s_proc.format7.f7_ssw;
	register uint	tt, tm, user_ref;
	mmu_sr_t	mmu_sr;
	static char	fatal_error[] = "Fatal error!";
/* hanna FIX: later, when you are sure of what bits (like ATC) get set under
 * hanna FIX: what conditions, then you may want to reorganize this code to
 * hanna FIX: try to catch page faults first. -- faster.
 */

	tt = ssw & SSW_TT;	/* break out commonly used fields */
	tm = ssw & SSW_TM;
	user_ref = USERMODE(stk_ptr->s_ps);

	ASSERT(stk_ptr->s_format == 7);	/* if not format 7, we're hosed */

	vaddr = busviol(stk_ptr);
	pde = findpde(vaddr, u.u_procp);

	/*
	 * buserr if there was any hardware "unhandleable" kernel fault
	 */
		    /* The hardware went south */
	if (statusreg & (STS_NORSP | STS_NOGRANT | STS_MAPFAULT)) {
		printf("usrxmemflt: unexpected fault: statusreg=%x, ssw=%x\n",
		  statusreg, ssw);
		break_out_pm_sts(statusreg, 1);
		hard_error(1, stk_ptr);
		/*
		 * The optimizer will fold these four lines together
		 * throughout this function, so I don't have to use an
		 * evil goto.  Yay!
		 */
		showbuserr(stk_ptr, statusreg);
		backtrace();
		panic(fatal_error);
		return (SIGBUS);	/*NOTREACHED*/
	}

		    /* Got a fault type we shouldn't have */
	if (tt == TT_ACK || tt == TT_ALT) {
		printf("usrxmemflt: %s Access fault: ssw=0x%x\n",
		  (tt == TT_ACK) ? "Acknowledge" : "Alt. Function Code", ssw);
		/*
		 * The optimizer will fold these four lines together
		 * throughout this function, so I don't have to use an
		 * evil goto.  Yay!
		 */
		showbuserr(stk_ptr, statusreg);
		backtrace();
		panic(fatal_error);
		return (SIGBUS);	/*NOTREACHED*/
	}

		    /* kernel should always be accessible. */
	else if (tm == TM_SDATA || tm == TM_SCODE) {
		printf("usrxmemflt: Kernel %s Access fault ssw=0x%x\n",
		  (tm == TM_SDATA) ? "Data" : "Code", ssw);
		/*
		 * The optimizer will fold these four lines together
		 * throughout this function, so I don't have to use an
		 * evil goto.  Yay!
		 */
		showbuserr(stk_ptr, statusreg);
		backtrace();
		panic(fatal_error);
		return (SIGBUS);	/*NOTREACHED*/
	}

		    /* kernel should always be accessible.
		     * All user page tables should be ok, except stack.
		     */
	else if ((!user_ref && tm == TM_MMUDATA) || (tm == TM_MMUCODE)) {
		printf("usrxmemflt: unexpected MMU %s Table fault ssw=0x%x\n",
		  (tm == TM_MMUDATA) ? "Data" : "Code", ssw);
		/*
		 * The optimizer will fold these four lines together
		 * throughout this function, so I don't have to use an
		 * evil goto.  Yay!
		 */
		showbuserr(stk_ptr, statusreg);
		backtrace();
		panic(fatal_error);
		return (SIGBUS);	/*NOTREACHED*/
	}

	else if (tm == TM_PUSH) {	/* data cache push error -- only */
		signo = cache_push(stk_ptr);
	}

	/* On most systems, we would have to look at function codes
	 * or use other methods to determine if a reference is user or
	 * kernel.  In our case, however, User Virtual and Kernel
	 * Virtual do not overlap and the kernel is not demand paged.
	 * This means is most cases, we don't care about function
	 * codes.
	 */

	else if (pde == NULL || pg_isuninit(pde)) {
		if ((vaddr > KMEM_START &&
		     !(user_ref || tm == TM_UDATA || tm == TM_UCODE)) ||
		    (u.u_procp == &own.o_proc)) {
			signo = SIGSEGV;	/* kernel problem */
		}
		/*
		 * Check for A1000 stack probing and grow stack
		 */
		if (USERMODE(stk_ptr->s_ps) && syncpc(stk_ptr->s_pc)) {
			if (!grow(vaddr))
				signo = SIGSEGV;
		}
		/* If the user SP is below the stack segment,
		 * grow the stack automatically.
		 */
		else if (!grow(stk_ptr->s_sp)) {
			signo = SIGSEGV;
		}
	}
		/* page fault */
	else if (!pg_isvalid(pde)) {
		signo = vfault(vaddr, pde);
		/*
		 * Take care of any potential write fault now.
		 * In case you need to do a writeback, r/m/w(locked op), mov16
		 */
		if (!signo && pg_iscw(pde) &&
		    ((ssw & SSW_RW) == RW_WRITE || (ssw & SSW_LK)))
			signo = pfault(vaddr, pde);
	}

/* hanna FIX: how do I figure out a read permission error */

	/* page preesent, but no write perm for a write or read/modify/write */
	else if (((ssw & SSW_RW) == RW_WRITE) || (ssw & SSW_LK)) {
		signo = pfault(vaddr, pde);
	}

	/*
	 * We can safely ignore Instruction ATC faults.
	 * (see M68040 User's Man section 9.6, last paragraph)
	 */
	else if ((ssw & SSW_ATC) && tm == TM_UCODE) {
		signo = 0;
	}

	/*
	 * We have been getting spurious data ATC faults during user read
	 * cycles, even though the page really is resident, and none of
	 * the causes of the SSW_ATC bit seem to be present.  If so, then
	 * go back and try again.
	 */
	else if (((mmu_sr.msr_all = ptest_addr(ssw, vaddr)),
		   mmu_sr.msr.msr_resident) &&
		 (ssw & SSW_ATC) && tt == TT_NORM && tm == TM_UDATA &&
		 (ssw & SSW_RW) != RW_WRITE) {
		flush_all_user_tlb();	/* clean the TLB out and try again */
		signo = 0;
	}

	/* The page is present, it's not a write, must be a read error? */
	else {
/* hanna FIX: have I missed any permutations of ATC and other SSW bits? */
/* hanna FIX: what about the triple fault problem, could it be mistaken here */
		printf(
"usrxmemflt: I'm confused! status=%x ssw=%x tt=%x tm=%x vaddr=%x pde=%x *pde=%x dbd=%x MMU=%x\n",
		  statusreg, ssw, tt, tm, vaddr, pde, pde->pde_all,
		  pde[NPGPT].pde_all, mmu_sr.msr_all);
		signo = SIGSEGV;		/* no read permission */
	}

/* clean up code */

/* hanna FIX: what error do you get if the mov16 address is not aligned */
/* hanna FIX: should i scream here, if the mov16 address is not aligned? */
					/* should be a usr's mv16 err */
	if (!signo && tt == TT_MOV16 && (ssw & SSW_RW) == RW_WRITE) {
		if (tm == TM_UMOV16) {	/* user's mov16 error */
			if (copyout(&stk_ptr->s_proc.format7.f7_pd0, vaddr, 16))
				if (!signo)
					signo = SIGSEGV;
		}
		else {
			/* kernel's mov16 error */
			printf("usrxmemflt: System mov16 fault. statusreg=%x, ssw=%x\n",
			  statusreg, ssw);
#if 0 /* hanna FIX: take out when you've confirmed this shouldn't happen... */
			bcopy(vaddr, &(stk_ptr->s_proc.format7.f7_pd0), 16);
#endif /* hanna FIX: take out when you've confirmed this shouldn't happen... */
		}
	} /* if mov16 error */

	    /* check for valid writebacks to perform, regardless */
	if ((stk_ptr->s_proc.format7.f7_wb1s & WBS_VALID) ||
	    (stk_ptr->s_proc.format7.f7_wb2s & WBS_VALID) ||
	    (stk_ptr->s_proc.format7.f7_wb3s & WBS_VALID)) {
		if ((user_ref = write_back(stk_ptr)) && !signo)
			signo = user_ref;
        }

#if 0
/* hanna FIX: does it matter to do excess flush if signo is set? */
	/* flush this entry from tlb */
/* hanna FIX */flush_all_user_tlb(); /* you don't know what writebacks??? */
#else
	flush_user_tlb_entry(vaddr);
#if 0
	if (!signo)
		flush_user_tlb_entry(vaddr);
#endif
#endif /* 1 */

	/*
	 * There is a pending floating point trap on an instruction that
	 * is being traced.  Handle the trace trap now.
	 */
	if ((ssw & (SSW_CU | SSW_CP)) && (stk_ptr->s_ps & PS_T)) {
		stk_ptr->s_ps &= ~PS_T;		/*turn off trace bit */
		if (((u.u_procp->p_flag & SSIGTRC) == 0) && !signo)
			signo = SIGTRAP;
		atom_and(&u.u_procp->p_flag, ~SSIGTRC);
	}

	return(signo);
} /* usrxmemflt */

busviol(stk_ptr)
struct state	*stk_ptr;
{
	/* If access is "misaligned", real fault addr is on next page */

	if (stk_ptr->s_proc.format7.f7_ssw & SSW_MA)  {
		return (
		  (uint)(stk_ptr->s_proc.format7.f7_fa + 0x100) & ~POFFMASK);
	}

	return((uint)stk_ptr->s_proc.format7.f7_fa);
}


showbuserr(stk_ptr, statusreg)
register struct state *stk_ptr;
register statusreg;
{
	ulong	vaddr;

	showtrap(stk_ptr);

	vaddr = busviol(stk_ptr);

	if ((stk_ptr->s_proc.format7.f7_ssw & SSW_TM) == TM_PUSH)
	    printf("PHYSICAL fault address %x\n",vaddr);
	else
	    printf("LOGICAL fault address %x\n",vaddr);

	if ((!USERMODE(stk_ptr->s_ps)) && is_iomap(vaddr))
		printf("Address mapped to system bus address %x:%x\n",
			iomap_to_slot(vaddr), iomap_to_offset(vaddr));

	break_out_pm_sts(statusreg, 1);

	break_out_ssw(stk_ptr->s_proc.format7.f7_ssw, 1);

	if (!USERMODE(stk_ptr->s_ps)) {
		spm_mem.panic_pm_id = own.o_pm_id;
		spm_mem.panic_slot = (uint)-1; /* to signify panic from pm bd */
		spm_mem.panic_flag = 1;
		stop_all_processors();
	}
}

break_out_ssw(ssw, verbose)
register unsigned	ssw;
{
	register uint	tm = ssw & SSW_TM;
	register uint	tt = ssw & SSW_TT;

	printf("SSW:%x: ", ssw);
	if (!verbose) {
		printf("\n");
		return;
	}

	/* hanna FIX: can't really tell user or sys mode without seeing ps */
	if (tm == TM_UDATA || tm == TM_UCODE)
		printf("USER ");
	else
		printf("SYS ");

	switch (ssw & SSW_SZ) {
	case SZ_BYTE:
		printf("(1 byte)"); break;
	case SZ_WORD:
		printf("(2 byte)"); break;
	case SZ_LONG:
		printf("(4 byte)"); break;
	case SZ_LINE:
		printf("(line)"); break;
	}

	if (tt == TT_ACK)
		printf(" ACK,");

	if (ssw & SSW_ATC)
		printf("ATC,");

	switch (tm) {
	case TM_PUSH:
		switch (tt) {
		case TT_MOV16:	break;
		case TT_ALT:	printf(" FC0");break;
		default:	printf(" Push"); break;
		}
		break;
	case TM_UDATA:
		switch (tt) {
		case TT_MOV16:	printf(" Usr Mov16"); break;
		case TT_ALT:	printf(" Bad FC1");break;
		default:	printf(" Usr Data"); break;
		}
		break;
	case TM_UCODE:
		switch (tt) {
		case TT_MOV16:	break;
		case TT_ALT:	printf(" Bad FC2");break;
		default:	printf(" Usr Code"); break;
		}
		break;
	case TM_MMUDATA:
		switch (tt) {
		case TT_MOV16:	break;
		case TT_ALT:	printf(" FC3");break;
		default:	printf(" MMU Data"); break;
		}
		break;
	case TM_MMUCODE:
		switch (tt) {
		case TT_MOV16:	break;
		case TT_ALT:	printf(" FC4");break;
		default:	printf(" Code MMU"); break;
		}
		break;
	case TM_SDATA:
		switch (tt) {
		case TT_MOV16:	printf(" Sys Mov16"); break;
		case TT_ALT:	printf(" Bad FC5");break;
		default:	printf(" Sys Data"); break;
		}
		break;
	case TM_SCODE:
		switch (tt) {
		case TT_MOV16:	break;
		case TT_ALT:	printf(" Bad FC6");break;
		default:	printf(" Sys Code"); break;
		}
		break;
	case TM_INVALID:
		switch (tt) {
		case TT_MOV16:	break;
		case TT_ALT:	printf(" FC7");break;
		default:	printf(" Bad TM"); break;
		}
		break;
	}

	printf(" Access on ");
	if (ssw & SSW_MA)
		printf("misaligned ");

	if ((ssw & SSW_LK))
		printf("Atomic ");

	if ((ssw & SSW_RW) == RW_READ)
		printf("read");
	else
		printf("write");

	if (ssw & SSW_CP)
		printf(" w/Post-FP");
	if (ssw & SSW_CU)
		printf(" w/Unimp-FP");
	if (ssw & SSW_CT)
		printf(" w/Trace");
	if (ssw & SSW_CM)
		printf(" w/Movm");

	printf(".\n");
} /* break_out_ssw */

/*
 * Write Backs.
 *  For certain access errors (ATC faults or bus errors) there may be
 *  pending writebacks, that is, writes that could not be finished due to
 *  the buserr.  The handler, consequently, must complete them.
 *  There may be up to three writebacks, the addresses, data, and status
 *  information concering the writeback are in the error stack frame type 7.
 *  For the first writeback, data is in the "proper memory alignment".
 *  For this, shift the heck out of the data to get it the data in the proper
 *  bytes for a single write, or movs.
 *  For the 2nd and 3rd, data is in "register" type alignment.
 *  Also, for the first writeback, the address is guaranteed to be present,
 *  however, this may not be true of the successive writebacks.
 *  Check that the addresses are within some kind of reasonable user range,
 *  and grow the stack here, if necessary, then let the writes
 *  fault and the recursively called buserr will deal with the pages
 *  (page not present, page copy on write).
 *  User's Manual, page 8-8 states:
 *  "Misaligned operand accesses that miss in the data cache or non-cachable
 *   are converted by the data memory unit in the M68040 to a sequence of
 *   aligned accesses.  These aligned access requests are sent to the bus
 *   controller for completion, resulting in bus transfers which are always
 *   aligned."
 *   So, it should do no harm to try writing misaligned accesses as a single
 *   write and let the processor align them.
 *
 * QUESTIONS:
 * . Will the Transfer Size (wb_status->wbs_size) ever equal 3 (16byte line)?
 * . For Transfer type, what are the legal options in here?
 *   How do I determine whether this is a valid write by a user or sup?
 *   That is, how do I determine whether to do an su* write (in user space) or
 *   a regular assignment (in kernel space)?
 */

/* well, it's one way to deal with it. */
/* this array describes the arrangement of the data for writeback one and
 * gives shift and mask values for turning the data into something that can
 * be writ out "easily" in one swell foop.
 */
typedef struct wb_shift {
	uint	lshift, lmask;
	uint	rshift, rmask;
} wb_shift_t;

wb_shift_t wb_align_tbl[4][4] = { /* [Transfer Size] [Transfer Alignment] */
/*          Lshift, Lmask, Rshift, Rmask     A1 A0 (alignment)  BYTE ORDER   */
    {
/* LONG */ { 0,         0,  0,       ~0}, /*  0  0              31: 0        */
	   { 8,     ~0xff, 24,     0xff}, /*  0  1              23: 0, 31:24 */
           {16,   ~0xffff, 16,   0xffff}, /*  1  0              15: 0, 31:16 */
	   {24, ~0xffffff,  8, 0xffffff}, /*  1  1               7: 0, 31: 8 */
    },

    {
/* BYTE */ { 0,         0, 24,     0xff}, /*  0  0              31:24,       */
	   { 0,         0, 16,     0xff}, /*  0  1              23:16,       */
           { 0,         0,  8,     0xff}, /*  1  0              15: 8,       */
	   { 0,         0,  0,     0xff}, /*  1  1               7: 0        */
    },

    {
/* WORD */ { 0,         0, 16,   0xffff}, /*  0  0              31:16,       */
	   { 0,         0,  8,   0xffff}, /*  0  1              23: 8        */
           { 0,         0,  0,   0xffff}, /*  1  0              15: 0        */
	   { 8,    0xff00, 24,     0xff}, /*  1  1               7: 0, 31:24 */
    },

    {
/* LINE */ { 0,         0,  0,       ~0}, /*  0  0              31: 0        */
	   { 8,     ~0xff, 24,     0xff}, /*  0  1              23: 0, 31:24 */
           {16,   ~0xffff, 16,   0xffff}, /*  1  0              15: 0, 31:16 */
	   {24, ~0xffffff,  8, 0xffffff}, /*  1  1               7: 0, 31: 8 */
   },
};

write_back(stk_ptr)
struct state	*stk_ptr;
{
	typedef struct wb_addr_data {	/* address/data layout in type 7 */
		ulong	wb_addr;	/* stack frame's write back area */
		ulong	wb_data;
	} wb_a_d_t;
	register wb_a_d_t	*wb_a_d;
	register ulong		addr;
	register ulong		data;
	register ushort		*wb_status;
	register uint		status;
	int			wb_cnt;
	wb_shift_t		*sh;
	register pde_t		*pde;
	register uint		user_ref;
	uint			user_mode, kern_addr;
	uint			tm, tt, size;
	register int		signo;
	mmu_sr_t		mmu;

	wb_a_d = (wb_a_d_t *)&stk_ptr->s_proc.format7.f7_wb1a;
	wb_status = &(stk_ptr->s_proc.format7.f7_wb1s);
	user_mode = USERMODE(stk_ptr->s_ps);
	signo = 0;
/* the "ptr -= 1" stuff on the next line is to kludge around a compiler bug */
	for (wb_cnt = 3; --wb_cnt >= 0; wb_status -= 1, wb_a_d -= 1) {
		if (!((status = *wb_status) & WBS_VALID))
			continue;	/* "This is not the writeback you're
					 * looking for....  Move along." */
		size = status & WBS_SZ;
		tt = status & WBS_TT;
		tm = status & WBS_TM;
		user_ref = (user_mode || tm == TM_UDATA) ? 1 : 0;
		addr = wb_a_d->wb_addr;

		/* check for error Transfer Types and Modifiers */
		if (tt != TT_NORM || size == SZ_LINE ||
		    (!user_ref && tm != TM_SDATA)) {
			cmn_err(CE_WARN,
			  "Writeback: Unexpected Mode WB%dS=0x%x, addr=0x%x\n",
			  3 - wb_cnt, status, addr);
			signo = SIGBUS;
			continue;
		}

		/*
		 * This write-back is invalid if we don't have write
		 * permission, am a user writing to kernel space, or in kernel
		 * mode writing to user space
		 */
		kern_addr = is_kern_addr(addr) ? 1 : 0;
		if (user_ref == kern_addr) {
			signo = SIGSEGV;		/* no permission */
			continue;
		}
		mmu.msr_all = ptest_addr(status, addr);
		if (kern_addr) {
			if (!mmu.msr.msr_resident) {
				signo = SIGSEGV;	/* kernel problem */
				continue;
			}
		}
		else {
			pde = findpde(addr, u.u_procp);
			if (pde == NULL || pg_isuninit(pde)) {
				if ((kern_addr && !user_ref) ||
				    (u.u_procp == &own.o_proc)) {
					signo = SIGSEGV;/* kernel problem */
					continue;
				}

				/* If the user SP is below the stack segment,
				 * grow the stack automatically.
				 */
				else if (!grow(stk_ptr->s_sp)) {
					signo = SIGSEGV;/* couldn't grow */
					continue;
				}
/* hanna FIX */flush_all_user_tlb();
				if (fubyte(addr) < 0) {
					signo = SIGSEGV;/* bad address */
					continue;
				}
				if (!pde && !(pde = findpde(addr, u.u_procp))) {
					signo = SIGSEGV;/* very bad address */
					continue;
				}
				mmu.msr_all = ptest_addr(status, addr);
			}
			if ((mmu.msr.msr_wp || pg_iswp(pde)) && !pg_iscw(pde)) {
				signo = SIGSEGV;	/* no permission */
				continue;
			}
			data = addr;
			switch (size) {
			case SZ_BYTE:
				break;
			case SZ_WORD:
				data += sizeof(ushort) - 1;
				break;
			case SZ_LONG:
				data += sizeof(ulong) - 1;
				break;
			}
			if (pnum(addr) != pnum(data) && fubyte(data) < 0) {
				signo = SIGSEGV;	/* misaligned write */
				continue;		/* to bad page */
			}
		}

		data = wb_a_d->wb_data;
		if (wb_cnt == 2) {
			/*
			 * wb_cnt == 2 means WB # 1, the "memory aligned" WB
			 * massage the "memory aligned" data for a single write
			 */
			sh = &wb_align_tbl[size >> 5][addr & 0x3];
			data = ((data << sh->lshift) & sh->lmask) |
			       ((data << sh->rshift) & sh->rmask);
		}

		switch (size) {
		case SZ_BYTE:
			if (user_ref ? subyte(addr, data) : kwrbyte(addr, data))
				signo = SIGSEGV;
			break;
		case SZ_WORD:
			if (user_ref ? sushort(addr,data) : kwrshort(addr,data))
				signo = SIGSEGV;
			break;
		case SZ_LONG:
			if (user_ref ? suword(addr, data) : kwrword(addr, data))
				signo = SIGSEGV;
			break;
		}
	}
	return(signo);
} /* write_back */

/*
 * lpfn_to_uva -- search the current process for the given lpfn and return a
 *		pointer to the cooresponding user virtual address, or NULL
 */

ulong *
lpfn_to_uva(lpfn, pdepp)
register ulong	lpfn;
pde_t		**pdepp;
{
	register int	j, pglim, i, seglim;
	register pde_t	*pt;
	register reg_t	*rp;
	register preg_t	*prp;

	for (prp = u.u_procp->p_region; (rp = prp->p_reg); prp++) {
		seglim = ctos(rp->r_pgsz);

		for (i = 0; i < seglim; i++) {
			/* Look through segment's page table for valid
			 * pages to dump.
			 */
			pt = rp->r_list[i];
			pglim = rp->r_pgsz - stoc(i);
			if (pglim > NPGPT)
				pglim = NPGPT;
			else { /* set up "pt" for stack region */
				if (rp->r_flags & RG_STACK)
					pt += NPGPT - pglim;
			}
			ASSERT(pglim >= 0 && pglim <= NPGPT);

			for (j = 0; j < pglim; j++, pt++) {
				if (pg_valpfn(pt) == lpfn) {
					*pdepp = pt;
					return (ulong *)(prp->p_regva +
						stob(i) + ctob(j));
				}
			}
		}
	}

	return (NULL);
}

/*
 * cache_push:  This routine completes a faulted cache push.
 * The page has not been checked for presence or write permission.
 */

cache_push(stk_ptr)
state_t	*stk_ptr;
{
	register uint	ssw;
	register ulong	*vaddr;
	pde_t		*pde;
	mmu_sr_t	mmu;
/*
 * The manual (section 9.6.4) says:
 * "Note that memory is now incoherent since the push buffer is invalidated
 *  after the fault -- the only valid copy of the cache line now resides on
 *  the stack and cannot be snooped."
 * This is not a problem so long as copyback pages are truly private to one
 * process and/or CPU.
 */

	/* for now, we should never get push bus errors in kernel mode */
	ssw = stk_ptr->s_proc.format7.f7_ssw;
	ASSERT(USERMODE(stk_ptr->s_ps) || (ssw & WBS_TM) == TM_UDATA);

	vaddr = lpfn_to_uva(pnum(stk_ptr->s_proc.format7.f7_fa), &pde);
	ASSERT(vaddr);

	mmu.msr_all = ptest_addr(ssw, vaddr);
	if ((mmu.msr.msr_wp || pg_iswp(pde)) && !pg_iscw(pde))
		return (SIGSEGV);
	if (fubyte(vaddr) < 0)			/* make sure page is present */
		return (SIGSEGV);
	/*
	 * cache pushes are either one or four long words, in the same page
	 */
	if ((ssw & SSW_SZ) == SZ_LONG)
		ssw = 4;
	else {
		if (fubyte(vaddr + 15) < 0)	/* make sure page is present */
			return (SIGSEGV);
		ssw = 16;
	}
	if (copyout(&(stk_ptr->s_proc.format7.f7_pd0), vaddr, ssw))
		return (SIGSEGV);

	return (0);
}

print_stk(stk)
register struct	state	*stk;
{
	printf("D:%x %x %x %x %x %x %x %x\n",
	    stk->s_d0, stk->s_d1, stk->s_d2, stk->s_d3,
	    stk->s_d4, stk->s_d5, stk->s_d6, stk->s_d7);
	printf("A:%x %x %x %x %x %x %x %x\n",
	    stk->s_a0, stk->s_a1, stk->s_a2, stk->s_a3,
	    stk->s_a4, stk->s_a5, stk->s_a6, stk->s_sp);
	printf("PS:%x PC:%x  FMT: %x V: %x\n",
	    stk->s_ps, stk->s_pc, stk->s_format, stk->s_vector);
	switch(stk->s_format) {
	    case 0: break;
	    case 1: break;
	    case 2:
		printf("INST ADDR %x\n", stk->s_proc.format2.f2_instaddr);
		break;
	    case 3:
		printf("EA: %x\n", stk->s_proc.format2.f2_instaddr);
		break;
	    case 7:
		printf("EA:%x FA:%x \n",
		    stk->s_proc.format7.f7_ea, stk->s_proc.format7.f7_fa);
		break_out_ssw(stk->s_proc.format7.f7_ssw, 1);
		printf("WB:S/D/A: 1:%x/%x/%x  2:%x/%x/%x  3:%x/%x/%x\n",
		    stk->s_proc.format7.f7_wb1s,
		    stk->s_proc.format7.f7_wb1d, stk->s_proc.format7.f7_wb1a,
		    stk->s_proc.format7.f7_wb2s,
		    stk->s_proc.format7.f7_wb2d, stk->s_proc.format7.f7_wb2a,
		    stk->s_proc.format7.f7_wb3s,
		    stk->s_proc.format7.f7_wb3d, stk->s_proc.format7.f7_wb3a);
		printf("PD: 0) %x 1) %x 2) %x 3) %x\n",
		    stk->s_proc.format7.f7_pd0, stk->s_proc.format7.f7_pd1,
		    stk->s_proc.format7.f7_pd2, stk->s_proc.format7.f7_pd3);
		break;
	} /* switch */
} /* print_stk */

/*
 * syncpc is taken directly from the A1000 syncpc
 * it returns 1 if the user was executing a tst.b or similar
 * instruction.
 */

#define TST 0x4a2f8000   /* tstw, tstb, or tstl for NCR tower */
#define TSTEXT 0x4a370170   /*  extended offset */

syncpc(pc)
register short *pc;
{
	register i,temp;

	for (i = 0; i < 4; i++)  {
		temp = fuword(pc--);
		if (temp == -1) return(0);
		if (((temp & 0xff3f8000) == TST)  ||
		    ((temp & TSTEXT) == temp))
			 return(1);
	}
	return(0);
}
