/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) vd.c: version 25.1 created on 11/27/91 at 15:00:26	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)vd.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
/*
 * virtual slice driver
 */

#include "sys/types.h"
#include "sys/sysmacros.h"
#include "sys/user.h"
#include "sys/errno.h"
#include "sys/conf.h"
#include "sys/buf.h"
#include "sys/vd.h"
#include "sys/iobuf.h"
#include "sys/ioctl.h"
#include "sys/debug.h"
#include "sys/synch.h"
#include "sys/open.h"
#include "sys/file.h"
#include "sys/cmn_err.h"

#ifdef SECON
#define VD_PERM_CK	auth_vd
#else
#define VD_PERM_CK	suser
#endif

#define b_kern_bp	b_back		/* this field should only be accessed
					 * in the vdbuf pool, where there
					 * is no other use for b_back
					 */
#define b_pending	av_back

#define io_waitbuf	io_s1		/* # of buffer requests that waited */
#define io_total	io_s2		/* total # of buffer requests */

#define BUF_FLAG_MASK	(B_READ | B_BUSY | B_PHYS | B_ASYNC | B_READAHEAD)

static struct iobuf	vdtab;		 /* vd buf freelist and request queue */
static spin_lock_t	vdmulti_lock;
static spin_lock_t	vdfreelist_lock;
void			vdiodone();
void			vdsend();
buf_t			*vdgetbuf();

extern vd_t		virt_slices[];
extern uint		max_virt_slices;
extern buf_t		vdbufhdrs[];
extern uint		num_vd_bufs;


/*
 * initialize the virtual buf headers:
 *
 *	link as a singly linked free list on vdtab.b_forw.
 */

vdinit()
{
	register buf_t	*vbp;
	register vd_t	*vd;
	register int	n;
	void		vdiodone();

	for (vbp = vdbufhdrs, n = num_vd_bufs; --n >= 0; vbp++) {
		vbp->b_dev = NODEV;
		vbp->b_iodone = vdiodone;
		vbp->b_forw = vbp + 1;
	}
	(vbp - 1)->b_forw = NULL;
	vdtab.b_forw = vdbufhdrs;

	/* initialize suspend locks for open */
	for (vd = virt_slices, n = max_virt_slices; --n >= 0; vd++)
		vd->v_lock.s_priority = PZERO;
}

/*
 * vdopen()
 *
 *	open a virtual slice (raw and blocked)
 *
 * Parameters:
 *
 *	minor device number
 *
 *	Flag (unused)
 *
 * Algorithm:
 *
 *	Check to see if this is a legimate device.
 *
 *	If they haven't already been called, called the real open
 *	routines for each component of the virtual slice.
 *
 * No Return Values
 *
 * Notes:
 *
 *	Errors are marked in u.u_error
 */

vdopen(minordev, flag, otype)
dev_t	minordev;
int	flag, otype;
{
	register vd_t		*vd;

	if (minordev >= max_virt_slices) {
		u.u_error = ENXIO;
		return;
	}

	vd = &virt_slices[minordev];
	suspend_lock(&vd->v_lock);


	if (ot_openchk(&vd->v_opentyp, flag, otype)) {
		suspend_unlock(&vd->v_lock);
		return;
	}

	suspend_unlock(&vd->v_lock);
}

/*
 * vdclose()
 *
 *	close a virtual slice (raw and blocked)
 *
 * Parameters:
 *
 *	device number
 *
 *	Flag (unused)
 *
 * Algorithm:
 *
 *	Call the driver close routines for each component of the virtual slice.
 *
 * No Return Values
 *
 * Notes:
 *
 *	Errors are marked in u.u_error
 */

vdclose(minordev, flag, otype)
dev_t	minordev;
int	flag, otype;
{
	register vd_t	*vd;
	register int	i;
	register dev_t	dev;

	vd = &virt_slices[minordev];
	suspend_lock(&vd->v_lock);

	ot_closechk(&vd->v_opentyp, otype);

	/* only call close routines for last close */
	if (vd->v_opentyp.ot_all_opens == 0 && vd->v_self_destruct) {
		for (i = 0; i < vd->v_num_stripe_components; i++) {
			dev = vd->v_sd[i].s_cdev;
			/*
			 * OTYP_MIR will fail if opened for any other use,
			 * and only allows read opens hereafter
			 */
			(*cdevsw[major(dev)].d_close)(minor(dev), FREAD|FWRITE,
			  OTYP_MIR);
		}
		bzero((caddr_t)vd, (uint)&vd->v_sd[MAX_STRIPE_COMPONENTS] -
		  (uint)vd);	/* clear up to the stripe components */
	}

	suspend_unlock(&vd->v_lock);
}

vdprint(dev, str)
dev_t dev;
char *str;
{
	printf("%s on STRIPE dev %d\n", str, minor(dev));
}


/*
 * vdstrategy
 *
 * Parameters:
 *
 *	kernel buffer header pointer.
 *
 * Algorithm:
 *
 *	attempt to allocate a driver buffer header pointer.
 *	failing to do so, queue the request for vdiodone to send out later.
 *
 *	call vdsend.
 */

vdstrategy(kern_bp)
register buf_t	*kern_bp;
{
	register vd_t	*vd;
	buf_t		*driver_bp;

	ASSERT(kern_bp->b_flags & B_BUSY);

	vd = &virt_slices[minor(kern_bp->b_dev)];

	if (! vd->v_configured) {
		kern_bp->b_driver_flags |= B_ERROR;
		kern_bp->b_error = ENXIO;
		kern_bp->b_resid = kern_bp->b_bcount;
		iodone(kern_bp);
		return;
	}

	kern_bp->b_resid = kern_bp->b_bcount;
	kern_bp->b_pending = 0;

	/* EOF check for block device (vdread/vdwrite called physck) */

	if (kern_bp->b_blkno + btod(kern_bp->b_bcount) > vd->v_size) {
		if (! (kern_bp->b_flags & B_READ) ||
		  (kern_bp->b_blkno != vd->v_size)) {
			kern_bp->b_error = ENXIO;
			kern_bp->b_driver_flags |= B_ERROR;
		}
		iodone(kern_bp);
		return;
	}
			
	if ((driver_bp = vdgetbuf(kern_bp)) == NULL)
		return;

	vdsend(driver_bp, vd);
}

/*
 * vdsend
 *
 *	break request up into multiple requests if required.
 *	calculate and copy next level driver info into driver bp(s)
 *	and send them on their way.
 *
 *	Return so that caller does an iowait() on the kernel bp.
 */

static void
vdsend(driver_bp, vd)
register buf_t	*driver_bp;
register vd_t	*vd;
{
	register buf_t		*kern_bp;
	register stripe_t	*sd;
	register uint		vcyl_offset, vtrack_size;
	register uint		vblock, count;
	uint			vcyl, sblock;

	ASSERT(driver_bp->b_kern_bp);
	ASSERT(vd);

	kern_bp = driver_bp->b_kern_bp;

	do {
		ASSERT(kern_bp->b_resid);

		/* find stripe component */
		vblock = kern_bp->b_blkno +
		  btod(kern_bp->b_bcount - kern_bp->b_resid);
		vcyl = vblock / vd->v_cyl_size;
		vcyl_offset = vblock % vd->v_cyl_size;
		sd = vd->v_sd;
		while (vcyl_offset >= (vtrack_size = sd->s_track_size)) {
			vcyl_offset -= vtrack_size;
			sd++;
			vtrack_size = sd->s_track_size;
		}

		sblock = (vcyl * vtrack_size) + vcyl_offset;

		/* count is max xfer from this stripe component */
		count = dtob(sd->s_track_size - (sblock % sd->s_track_size));
		/* b_bcount is now max for remainder of request */
		driver_bp->b_bcount = kern_bp->b_resid;

		if (count > driver_bp->b_bcount)
			count = driver_bp->b_bcount;
		else
			driver_bp->b_bcount = count;

		driver_bp->b_un.b_addr = kern_bp->b_un.b_addr +
		  (kern_bp->b_bcount - kern_bp->b_resid);
		driver_bp->b_dev = sd->s_bdev;
		driver_bp->b_blkno = sblock;
		driver_bp->b_flags = kern_bp->b_flags & BUF_FLAG_MASK;
		driver_bp->b_proc = kern_bp->b_proc;

		spin_lock(&vdmulti_lock);
		(*(uint *)(&(kern_bp->b_pending)))++; 
		kern_bp->b_resid -= count;
		spin_unlock(&vdmulti_lock);

		(*bdevsw[major(driver_bp->b_dev)].d_strategy)(driver_bp);

	} while (kern_bp->b_resid && (driver_bp = vdgetbuf(kern_bp)));
}

vdread(minordev)
dev_t	minordev;
{
	vd_t	*vd;

	vd = &virt_slices[minordev];

	if (! vd->v_configured) {
		u.u_error = ENXIO;
		return;
	}

	if (physck(vd->v_size, B_READ))
		physio(vdstrategy, 0, minordev, B_READ);
}

vdwrite(minordev)
dev_t	minordev;
{
	vd_t	*vd;

	vd = &virt_slices[minordev];

	if (! vd->v_configured) {
		u.u_error = ENXIO;
		return;
	}

	if (physck(vd->v_size, B_WRITE))
		physio(vdstrategy, 0, minordev, B_WRITE);
}

/*
 * vdgetbuf
 *
 *	Allocate a virtual slice buffer header for "kern_bp"
 *	return a pointer to the buffer header
 *	If none are avail queue kern_bp and return NULL
 *	
 */

static buf_t *
vdgetbuf(kern_bp)
register buf_t	*kern_bp;
{
	register buf_t	*vbp;

	spin_lock(&vdfreelist_lock);
	vdtab.io_total++;
	if (vbp = vdtab.b_forw) {
		vdtab.b_forw = vbp->b_forw;
		spin_unlock(&vdfreelist_lock);

		ASSERT(!vbp->b_kern_bp);
		vbp->b_kern_bp = kern_bp;
		return(vbp);
	}

	vdtab.io_waitbuf++;
	kern_bp->av_forw = NULL;

	if (vdtab.b_actl) {
		vdtab.b_actl->av_forw = kern_bp;
		vdtab.b_actl = kern_bp;
	}
	else
		vdtab.b_actf = vdtab.b_actl = kern_bp;

	spin_unlock(&vdfreelist_lock);
	return (NULL);
}

/*
 * vdrelease
 *
 *	put a vd buffer header back onto the freelist.
 */

static
vdrelease(vbp)
register buf_t	*vbp;
{
	register buf_t	*kern_bp;

	vbp->b_error = vbp->b_flags = vbp->b_driver_flags = 0;

	spin_lock(&vdfreelist_lock);

	/* if there are any requests queued up, send one out with this vbp */
	if (kern_bp = vdtab.b_actf) {

		if (kern_bp == vdtab.b_actl)
			vdtab.b_actf = vdtab.b_actl = 0;
		else
			vdtab.b_actf = kern_bp->av_forw;

		spin_unlock(&vdfreelist_lock);

		vbp->b_kern_bp = kern_bp;
		vdsend(vbp, &virt_slices[minor(kern_bp->b_dev)]);
	}
	else {
		vbp->b_kern_bp = 0;
		vbp->b_forw = vdtab.b_forw;
		vdtab.b_forw = vbp;
		spin_unlock(&vdfreelist_lock);
	}
}

/*
 * vdioctl:
 *
 *	cmd:
 *		GET_DISK_SIZE:		Returns the number of 1k sectors
 *					in this device.  arg ignored.
 *
 *		GET_CYLINDER_SIZE:	returns the number of 1k sectors
 *				  	per virtual cylinder.  arg ignored.
 *
 *		GET_TRACK_SIZE:		returns the average track size.
 *
 *		GET_NUM_TRACKS:		returns the total number of average 
 *					tracks used by this striped slice.
 *
 *		GET_INTERLACE:		returns a recommended filesystem
 *					interlace in 1k sectors.  arg ignored.
 *
 *		GET_OPEN_TYPES:		returns the uint version of the
 *					v_opentyp struct, which keeps track of
 *					how the device has been opened.
 *
 *		SET_VIRT_DISK:		copy in user configuration.  arg is a
 *					pointer to a vd_t structure.
 *
 *		CLEAR_VIRT_DISK:	unset a virtual slice.
 *
 *		GET_VIRT_DISK:		copy out virtual slice config info.
 */

vdioctl(minordev, cmd, arg, mode)
register dev_t	minordev;
{
	register vd_t	*vd;
	register int	i;
	register uint	tmp1, tmp2;
	register dev_t	dev;
	vd_t		vd_set;

	vd = &virt_slices[minordev];
	suspend_lock(&vd->v_lock);

	if ((vd->v_configured == 0) && 
	  ((cmd != SET_VIRT_DISK) && (cmd != GET_VIRT_DISK))) {
		u.u_error = ENXIO;
		suspend_unlock(&vd->v_lock);
		return;
	}

	tmp1 = tmp2 = 0;

	switch ( cmd ) {

	case  GET_DEV_TYPE:
		u.u_rval1 = STRIPE_TYPE;
		break;

	case  GET_LOG_TYPE:
		u.u_rval1 = LDT_UNIX;	/* unix file system type	*/
		break;

	case  GET_DISK_SIZE:
		u.u_rval1 = vd->v_size;
		break;

	case  GET_CYLINDER_SIZE:
	case  GET_TRACK_SIZE:		/* striped track == striped cylinder */
		u.u_rval1 = vd->v_cyl_size;
		break;

	case  GET_NUM_TRACKS:
		u.u_rval1 = vd->v_size / vd->v_cyl_size;
		break;

	case  GET_INTERLACE:
		for (i = 0; i < vd->v_num_stripe_components; i++) {
			dev = vd->v_sd[i].s_cdev;
			(*cdevsw[major(dev)].d_ioctl)(minor(dev),
			  GET_INTERLACE, arg, mode);
			if (u.u_error)
				break;
			tmp1 = u.u_rval1;

			if (tmp1 > tmp2)
				tmp2 = tmp1;
		}
		u.u_rval1 = tmp2;
		break;

	case  GET_OPEN_TYPES:
		u.u_rval1 = vd->v_opentyp.ot_all_opens;
		break;

	case  SET_VIRT_DISK:
		if (! VD_PERM_CK())
			break;
		if (vd->v_configured) {
			u.u_error = EBUSY;
			break;
		}

		if (copyin((caddr_t)arg, (caddr_t)&vd_set, sizeof(vd_set))) {
			u.u_error = EFAULT;
			break;
		}

		if (vd_set.v_num_stripe_components == 0 ||
		    vd_set.v_num_stripe_components > MAX_STRIPE_COMPONENTS) {
			u.u_error = EINVAL;
			break;
		}

		if (vd_verify(&vd_set))
			break;

		vd->v_configured = VD_CONFIGURED | VD_OPENED;
		vd->v_num_stripe_components = vd_set.v_num_stripe_components;
		vd->v_cyl_size = vd_set.v_cyl_size;
		vd->v_size = vd_set.v_size;
		/*
		 * now use the v_num_stripe_components to determine how many
		 * bytes to copy for the rest of the structure.
		 */
		tmp1 = vd->v_num_stripe_components * sizeof(*vd->v_sd);
		bcopy((caddr_t)vd_set.v_sd, (caddr_t)vd->v_sd, tmp1);
		break;

	case  CLEAR_VIRT_DISK:
		if (! VD_PERM_CK())
			break;
		if (vd->v_opentyp.ot_lyrcnt ||
		    (vd->v_opentyp.ot_flag & ~OFLG_CHR)) {
			u.u_error = EBUSY;
			break;
		}
		vd->v_self_destruct = 1;	/* clear on last close */
		break;

	case  GET_VIRT_DISK:
		vd_set = *vd;
		for (i = vd->v_num_stripe_components; --i >= 0; ) {
			vd_set.v_sd[i].s_bdev &= UNMINOR_MASK;
			vd_set.v_sd[i].s_cdev &= UNMINOR_MASK;
		}
		if (copyout((caddr_t)&vd_set, (caddr_t)arg, sizeof(vd_set)))
			u.u_error = EFAULT;
		break;

	default:
		u.u_error = EINVAL;
	}
	suspend_unlock(&vd->v_lock);
}

/*	utility only passes in sd_cdev, we must copy to bdev	*/

static
vd_verify(vd)
register vd_t	*vd;
{
	register stripe_t	*sd;
	register struct cdevsw	*cdp;
	register int		i;
	register uint		min;

	for (i = 0, sd = vd->v_sd; i < vd->v_num_stripe_components; i++, sd++) {

		sd->s_cdev = notminored(sd->s_cdev);	/* passed full dev_t */
		sd->s_bdev = sd->s_cdev;

		if ((min = major(sd->s_cdev)) > cdevcnt) {
			u.u_error = EINVAL;
			break;
		}
		cdp = &cdevsw[min];
		min = minor(sd->s_cdev);
		/*
		 * OTYP_MIR will fail if opened for any other use, and only
		 * allows read opens hereafter
		 */
		(*cdp->d_open)(min, FREAD | FWRITE, OTYP_MIR);
		if (u.u_error)
			break;

		(*cdp->d_ioctl)(min, GET_DISK_SIZE, 0, 0);
		if (u.u_rval1 <= 0 && u.u_error == 0)
			u.u_error = EINVAL;
		if (u.u_error) {
			(*cdp->d_close)(min, FREAD | FWRITE, OTYP_MIR);
			break;
		}
	}

	if (u.u_error) {
		/*
		 * close previous opens
		 */
		while (--i >= 0) {
			--sd;
			cdp = &cdevsw[major(sd->s_cdev)];
			min = minor(sd->s_cdev);
			(*cdp->d_close)(min, FREAD | FWRITE, OTYP_MIR);
		}
		return(1);
	}

	return(0);
}

static void
vdiodone(driver_bp)
register buf_t	*driver_bp;
{
	register buf_t	*kern_bp;
	register uint	do_iodone;

	do_iodone = 0;
	kern_bp = driver_bp->b_kern_bp;
	ASSERT(kern_bp);

	spin_lock(&vdmulti_lock);
	kern_bp->b_driver_flags |= (driver_bp->b_driver_flags & B_ERROR);
	if (! kern_bp->b_error)
		kern_bp->b_error = driver_bp->b_error;
	ASSERT(driver_bp->b_bcount);
	ASSERT(kern_bp->b_pending);

	if (--(*(uint *)(&(kern_bp->b_pending))) == 0 && kern_bp->b_resid == 0)
		do_iodone = 1;
	spin_unlock(&vdmulti_lock);

	vdrelease(driver_bp);

	/*
	 * b_resid doesn't need to be reset for errors
	 * since u.u_error clobbers u.u_rval1
	 */
	if (do_iodone)
		iodone(kern_bp);
}
