/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) spmtydvr.c: version 25.2.1.2 created on 3/23/92 at 21:16:25	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)spmtydvr.c	25.2.1.2	3/23/92 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
/*------------------------------------------------------------------*
 *	spmtydvr.c - streams tty driver for the ACDB & SPM async interface board
 *
 *	this is a streams tty driver used to drive the spm and the ACDB
 *
 *	- on the upstream size it interfaces with modules that handle most
 *	  of the character processing
 *	- on the downstream side it interfaces with the microsequencer (uSeq)
 *	  through the SPM or directly on the ACDB.
 *	  The uSeq then takes care of the actual
 *	  transmitting and receiving of characters,
 *	  as well as the uSeq doing some limited character processing
 *
 *	the primary functions of this driver are:
 *	- to handle the setting of the uart registers (e.g., baud rate)
 *	- move output data from streams messages to uSeq ram buffers
 *	  for output to the device by the uSeq
 *	- move data received from the device from uSeq ram buffers
 *	  to streams messages and send them upstream
 *	- process all eia changes (e.g., HUP signal sent by when carrier drops)
 *	- maintain timers for the ARIX add-on dcd timeout and activity timeout
 *
 *------------------------------------------------------------------*/
#include "sys/types.h"
#include "sys/user.h"
#include "sys/proc.h"
#include "sys/param.h"
#include "sys/errno.h"
#include "sys/sysmacros.h"
#include "sys/systm.h"
#include "sys/stream.h"
#include "sys/strstat.h"
#include "sys/stropts.h"
#include "sys/signal.h"
#include "sys/file.h"
#include "sys/gctio.h"
#include "sys/termio.h"
#include "sys/stermio.h"
#include "sys/strlog.h"
#include "sys/ttold.h"
#include "sys/debug.h"
#include "sys/cmn_err.h"
#include "sys/tyd_clog.h"
#include "sys/cir_log.h"
#include "sys/tyd_sram.h"
#ifdef SAK
#include "sys/sakioctl.h"
#endif /* SAK */
#include "sys/tyd_tty.h"
#include "sys/sysinfo.h"
/*----------------------------------------------------------------------*/
int	TTYDVR5;	/* version of streams tty driver		*/
/*----------------------------------------------------------------------*/
extern	long	time;
int	tyd_cir_log_debug = 1;	/* set non-zero to enable tyd_cir_log */
int	tyd_cir_log_init_flag = 0; /* non-zero if cur_log initialized */
int	tyd_cir_log_only_one_port = 0; /* set non-zero to log only one port */
int	tyd_cir_port_to_log = 0; /* single port no. to log */

int	tyd_cir_index;		/* number of next tyd_cir_log_buf entry */

struct  cir_log	*tyd_cir_ptr;  /* ptr to next tyd_cir_log_buf entry */
struct	cir_log	tyd_cir_log_buf[TYD_CIR_LOG_SIZE];
/*------------------------------------------------------------------*/
int	tyd_debug = 0;		/* set non-zero to enable tyd_log */
/*------------------------------------------------------------------*/
#define LOG_DEBUG	tyd_debug
#define	LOG_RTN		tyd_log

#define CIRLOG_DEBUG	tyd_cir_log_debug
#define	CIRLOG_RTN	tyd_cir_logit

#include "sys/debuglog.h"
/*------------------------------------------------------------------*/
#ifdef IOPM
#define TYD_MOD		123
#else
#define TYD_MOD		120
#endif
#define TYD_MSIZE	1024		/* max data msg size we will send up */

int tyd_open(), tyd_close(), tyd_rput(), tyd_rsrv(), tyd_wput(), tyd_wsrv();

struct module_info	tyd_minfo = {
/*			 packet size
   mod id   module name  min    max	hi-water  lo_water
   ------   -----------  -------------  --------  --------*/
#ifdef IOPM
   TYD_MOD, "ttydvr",    0,     INFPSZ, 2048,     1024 };
#else
   TYD_MOD, "spmtydvr",  0,     INFPSZ, 2048,     1024 };
#endif

struct qinit tyd_ri = {
/* put       service   open      close      unused  mod_info   stats
   ---       -------   ----      -----      ------  --------   ----- */
   tyd_rput, tyd_rsrv, tyd_open, tyd_close, NULL,  &tyd_minfo, NULL };
struct qinit tyd_wi = {
/* put       service   open      close      unused  mod_info   stats
   ---       -------   ----      -----      ------  --------   ----- */
   tyd_wput, tyd_wsrv, NULL,     NULL,      NULL,  &tyd_minfo, NULL };

#ifdef IOPM
struct streamtab ttydvrinfo = { &tyd_ri, &tyd_wi, NULL, NULL };
#else
struct streamtab spmtydvrinfo = { &tyd_ri, &tyd_wi, NULL, NULL };
#endif
/*------------------------------------------------------------------*/
typedef struct	termio  termio_t;

mblk_t *tyd_alloc();
mblk_t *tyd_wsrv_ioc();
mblk_t *tyd_send_gctio_to_user();
mblk_t *tyd_process_msg_in_steps();
#ifdef SAK
mblk_t *tyd_send_sak_to_user();
#endif /* SAK */
extern	mblk_t *tyd_wsrv_data();

int	tyd_out_pending_expired();
int	tyd_process_msg_delay_elapsed();
int	tyd_send_tact_sig();
int	tyd_dcd_timeout();
int	tyd_wait_for_dsr();
int	tyd_timer();
int	tyd_cancel_check();

#ifdef TTYDEBUG
int	tyd_timer_time = 0;
#endif /* TTYDEBUG */
int	tyd_timer_elapsed_time = 0;
int	tyd_timer_count = 0;
int	tyd_poll_timer_id = 0;				/* id of tyd_timer */
int	tyd_poll_timer_init = 0;
uint	tyd_input_count = 0;		/* no. characters received */
uint	tyd_output_count = 0;		/* no. characters written */
/*------------------------------------------------------------------*/
extern	tyd_tty_t tyd[];		/* one struct per port */
extern	tyd_tty_t *tyd_tp_tab[];	/* check for activity in poll*/
extern	struct module_stat r_stat[];	/* one read module stat per port */
extern	struct module_stat w_stat[];	/* one write module stat per port */
extern	int	tyd_num_ports;

/*------------------------------------------------------------------*/
/*
 *		OPEN
 *
 *------------------------------------------------------------------*
 *
 * tyd_open -- open a stream tty driver on the SPM Async interface board
 *
 * Arguments:
 *	rq	if stream open, read queue pointer
 *	dev	major/minor device number
 *		module open		- 0
 *		regular driver open	- major/minor
 *		clone driver open	- major only
 *	flag	flags device was opened with
 *	sflag	open type flag
 *		0		- driver open
 *		MODOPEN		- normal module open
 *		CLONEOPEN	- clone driver open
 *
 * Returns minor device number or OPENFAIL
 */
tyd_open(rq, dev, flag, sflag)
queue_t		*rq;
dev_t		dev;
int	flag, sflag;
{
	register tyd_tty_t	*tp;
	queue_t		*wq;
	struct proc	*pp;
	int	cflag;
	dev_t	min_dev;
	int	first_open = 1;
#ifndef IOPM
	int	s;
#endif
	int	sleep_return;		/* return code from sleep
					   0 = requested item occurred
					   1 = unexpected signal occurred */
	tyd_init();
	if ((min_dev = minor(dev)) >= tyd_num_ports)
	{
		u.u_error = ENXIO;
		return (OPENFAIL);
	}
	BUGCIR_LOG(min_dev, flag, sflag, TYD_CIR_OPEN, (int)rq);
	if (tyd_poll_timer_init == 0)
	{
#ifdef TTYDEBUG
#ifdef IOPM
		tyd_timer_time = get_time_stamp();
#else
		tyd_timer_time = lbolt;
#endif /* IOPM */
#endif /* TTYDEBUG */
		tyd_timer_elapsed_time = 0;
		tyd_poll_timer_init++;
		tyd_poll_timer_id = timeout(tyd_timer, (tyd_tty_t *)NULL,1);
	}
#ifndef NOT_REAL_HARDWARE	
	if (sflag != 0)
	{
		BUGLOG1(NULL,BUG_LERR,"tyd_open bad sflag=%x", sflag);
		u.u_error = EINVAL;
		return (OPENFAIL);
	}
#endif /* NOT_REAL_HARDWARE */	
	if ((tp = (tyd_tty_t *)rq->q_ptr) != NULL)
	{			/* already attached */
		tp->rq_stat->ms_ocnt++;		/* bump count of open calls */
		if ((tp == (tyd_tty_t *)OTHERQ(rq)->q_ptr)
		 && (tp->t_device == dev)	/* major/minor device no. */
		 && (tp->state & TTUSE))	/* double check */
		{
			first_open = 0;
			goto open_exit;
		}
		else
		{
			/* CMW: this should be an ASSERT(0), not OPENFAIL */
			cmn_err(CE_WARN,"tyd_open: multiple open\n");
			rq->q_ptr = OTHERQ(rq)->q_ptr = NULL;
			u.u_error = EIO;
			return(OPENFAIL);
		}			
	}
	if (tyd_check_if_port_available(min_dev))
	{
		u.u_error = ENXIO;
		return(OPENFAIL);	/* port not available */
	}
	tp = &tyd[min_dev];
	if (tp->state & TTUSE)
	{
		/* CMW: this should be an ASSERT(0), not OPENFAIL */
		u.u_error = EIO;
		return(OPENFAIL);	/* port not available */
	}
#ifdef SAK
	tp->sak.flags = 0;
#endif /* SAK */
	BUGLOG2(tp, BUG_LOPEN, "tyd_open rq=%x, flag=%x", rq, flag);
	bzero(tp, sizeof(tyd_tty_t));
	tp->myname[0] = 'T';
	tp->myname[1] = 'Y';
	tp->myname[2] = 'D';
	tp->myname[3] = min_dev;	/* use lower byte of minor device */

	wq = OTHERQ(rq);
	rq->q_ptr = (char *)tp;
	wq->q_ptr = (char *)tp;
	tp->rq = rq;				/* queue for upstream */
	tp->wq = wq;				/* queue for downstream */
	tp->state = TTUSE;			/* double check */
	tp->t_openf = flag;			/* type of open - info only */
	tp->t_device = dev;			/* major/minor device no. */
	tp->min_dev = min_dev;			/* minor device no. */
	tp->xonc  = CSTART;			/* default X-ON character */
	tp->xofc  = CSTOP;			/* default X-OFF character */
	tp->break_interval = 250;		/* break interval of .25 sec */

	tyd_init_stats(tp);			/* init statistics buffers */
	tp->rq_stat->ms_ocnt++;			/* bump count of open calls */

	tyd_init_pointers(tp);			/* init ptrs/tbls this port*/
	cflag = B9600|CS8|CREAD|HUPCL;		/* SANE settings */

#ifdef IOPM
	tyd_cflag_program(tp, cflag);		/* program hardware */
	tyd_iflag_program(tp);			/* set uSeq iflag=0*/
	tyd_tp_tab[min_dev] = tp;		/* check in tyd_timer */
#else
	s = splstr();
	tyd_cflag_program(tp, cflag);		/* program hardware */
	tyd_iflag_program(tp);			/* set uSeq iflag=0*/
	tyd_set_flag_cmd(tp);
	tyd_tp_tab[min_dev] = tp;		/* check in tyd_timer */
	splx(s);

	tp->state |= WAIT_FOR_FLAG_ON_OPEN;
	while (tp->state & WAIT_FOR_FLAG_ON_OPEN)
	{
		sleep_return = sleep(&tp->state, STIPRI|PCATCH);
		BUGLOG2(tp, BUG_LOPEN, "tyd_open sleep_return=%x rq=%x",
						  sleep_return,   rq);
		if (sleep_return != 0)
		{
			if ( first_open )
				tyd_shut_down_port(tp);
			return (OPENFAIL);
		}
	}
#endif /* IOPM */

open_exit:
	tp->state &= ~(CARR_ON | T_DCD | T_DSR);
	tp->state |= tyd_get_eia_status(tp);
	if ( tp->state & T_DCD && tp->state & T_DSR ||
	     tp->termio.c_cflag & CLOCAL )
		tp->state |= CARR_ON;

	if ( !(tp->state & CARR_ON) && !(flag & FNDELAY) )
	{
		/* if not "open with no delay", then wait for carrier */
		if (tp->t_gctio.gt_tdcd != 0 && !(tp->state & T_DSR))
		/* DSR is hi but DCD is low, so start connect timer */
			tyd_start_dcd_timer(tp);
		while (!(tp->state & CARR_ON))
		{
			sleep_return = sleep(&tp->state, STIPRI|PCATCH);
			BUGLOG2(tp, BUG_LOPEN, "tyd_open sleep_return=%x rq=%x",
							  sleep_return,   rq);
			if (sleep_return != 0)
			{
				if ( first_open )
					tyd_shut_down_port(tp);
				return (OPENFAIL);
			}
		}
	}
	tp->state |= ISOPEN;

	pp = u.u_procp;
	if (pp->p_pid == pp->p_pgrp && u.u_ttyp == NULL && tp->t_pgrp == 0)
	{
                u.u_ttyp  = (short *)&tp->t_pgrp;
                tp->t_pgrp = pp->p_pgrp;
	}
#ifndef IOPM
	tyd_log_status(tp, 0);
#endif
	return(min_dev);		/* flag successful open */
}/*------------------------------------------------------------------*/
/*
 *		CLOSE
 *
 *------------------------------------------------------------------*
 *
 * tyd_close -- close a stream tty driver
 *
 * Arguments:
 *	rq	read queue pointer
 *	flag	flags device opened with, 0 for modules
 */
tyd_close(rq, flag)
register queue_t	*rq;
{
	register tyd_tty_t	*tp;
	int	delay;

	tp = (tyd_tty_t *)rq->q_ptr;
	ASSERT(tp != NULL);
	tp->rq_stat->ms_ccnt++;		/* bump count of close calls */
	BUGCIR_LOG(tp->min_dev, flag, 0, TYD_CIR_CLOSE, (int)rq);
	BUGLOG2(tp, BUG_LCLOSE, "tyd_close rq=%x, flag=%x", rq, flag);

	if (tp->state & ISOPEN)
	{
		tp->t_closing++;
		tyd_flush_read_side(tp);
		tyd_start_output(tp);
		tyd_check_if_output_empty(tp);
		if (tp->output_in_progress)
		{
			/* compute time to empty output buf and add seconds*/
#ifdef IOPM
			delay = tyd_compute_delay(tp) + (5*HZ);
#else
			delay = tyd_compute_delay(tp) + (2*HZ);
#endif
			tp->cancel_id = timeout(tyd_cancel_check,tp,delay);
			tp->timer_enable |= CANCEL_CHECK_TIMER;
			sleep(&tp->output_in_progress,STIPRI|PCATCH);
			if (tp->timer_enable & CANCEL_CHECK_TIMER)
				untimeout(tp->cancel_id);
		}
		if ( tp->timer_enable & STEP_TIMER )
		{
			tp->cancel_id = timeout(tyd_cancel_check,tp,1*HZ);
			tp->timer_enable |= CANCEL_CHECK_TIMER;
			sleep(&tp->output_in_progress,STIPRI|PCATCH);
		}
		tyd_flush_write_side(tp);
		tyd_flush_read_side(tp);
	}
 	tp->modpush = 0;
	tyd_shut_down_port(tp);
#ifndef IOPM
	tyd_log_status(tp, 1);
#endif
	u.u_ttyp = NULL;
}/*------------------------------------------------------------------*/
tyd_shut_down_port(tp)
register tyd_tty_t	*tp;
{
	int	cflag;

	if ((cflag = tp->termio.c_cflag) & HUPCL)
	{
		cflag &= ~CBAUD;	/* this call will drop DTR */
		tyd_cflag_program(tp, cflag);	/* program hardware chip*/
#ifndef IOPM
		tyd_set_flag_cmd(tp);
#endif
	}
	tyd_tp_tab[tp->min_dev] = NULL;	/* don't check in tyd_timer */
	if (tp->timer_enable != 0)
	{
		if (tp->timer_enable & ACT_TIMER)
			untimeout(tp->act_id);
		if (tp->timer_enable & DCD_TIMER)
			untimeout(tp->dcd_id);
		if (tp->timer_enable & DSR_TIMER)
			untimeout(tp->dsr_id);
		if (tp->timer_enable & OUTPUT_EMPTY_TIMER)
			untimeout(tp->output_empty_id);
		if (tp->timer_enable & OUTPUT_PENDING_TIMER)
			untimeout(tp->out_pending_id);
		if (tp->timer_enable & STEP_TIMER)
		{
			if ( tp->state & BREAK_IN_PROGRESS )
				tyd_stop_break(tp);
			untimeout(tp->timer_step_id);
		}
		if (tp->timer_enable & CANCEL_CHECK_TIMER)
			untimeout(tp->cancel_id);
	}
	tp->state &= ~TTUSE;		/* flag 'device available */
}/*------------------------------------------------------------------*/
/* unable to close because output drained, so assume output is stuck */
tyd_cancel_check(tp)
register tyd_tty_t	*tp;
{
	tp->timer_enable &= ~CANCEL_CHECK_TIMER;
	tp->output_in_progress = 0;
	wakeup((caddr_t)&tp->output_in_progress);
}/*------------------------------------------------------------------*/
/*
 *	initialize statistics buffers for both read and write queues
 */
tyd_init_stats(tp)
register tyd_tty_t	*tp;
{
	/* read size stats */
	tp->rq_stat = &r_stat[tp->min_dev];
	bzero(&tp->pv_rstat, sizeof(tyd_rstat_t));
	tp->rq_stat->ms_xptr = (char *)&tp->pv_rstat;	/*private read */
	tp->rq_stat->ms_xsize = sizeof(tyd_rstat_t);	/*priv. read size*/

	/* write size stats */
	tp->wq_stat = &w_stat[tp->min_dev];
	bzero(&tp->pv_wstat, sizeof(tyd_wstat_t));
	tp->wq_stat->ms_xptr = (char *)&tp->pv_wstat;	/*private write */
	tp->wq_stat->ms_xsize = sizeof(tyd_wstat_t);	/*priv write size*/
}/*------------------------------------------------------------------*/
/*
 *		READ PUT
 *
 *------------------------------------------------------------------*
 *
 * tyd_rput -- read put routine
 *
 *	should never be called since 'tyd_read()' puts msgs onto the
 *	read queue directly. Included only to scream if anyone trys it.
 *
 * Arguments:
 *	rq	read queue pointer
 *	mp	message pointer
 */
tyd_rput(rq, mp)
queue_t *rq;
mblk_t	*mp;
{
	register tyd_tty_t	*tp;

	tp = (tyd_tty_t *)rq->q_ptr;
	ASSERT(tp != NULL);
	tp->rq_stat->ms_pcnt++;		/* bump count of read put calls */
	if (mp == NULL)
	{
		cmn_err(CE_NOTE, "NULL mp received in tyd_rput, q = %x", rq);
		return;
	}
	BUGCIR_LOG(tp->min_dev, mp->b_datap->db_type, *mp->b_rptr,
							TYD_CIR_RPUT, mp);
	BUGLOG2(NULL,BUG_LERR,"tyd_rput rq=%x mp=%x  ???", rq, mp);
#ifdef NOT_REAL_HARDWARE	
	putnext(rq, mp);		/* send upstream from real driver */
#else
	freemsg(mp);
#endif /* NOT_REAL_HARDWARE */	
}/*------------------------------------------------------------------*/
/*
 *		WRITE PUT
 *
 *------------------------------------------------------------------*
 *
 * tyd_wput -- write put routine
 *
 *	processes M_FLUSH, M_IOCTL, M_BREAK, M_STOP, M_START
 *	pass M_DATA to write service routine
 *
 * Arguments:
 *	wq	write queue pointer
 *	mp	message pointer
 */
tyd_wput(wq, mp)
register queue_t	*wq;
register mblk_t		*mp;
{
	register tyd_tty_t	*tp;
	register type;
	struct	iocblk		*iocbp;
	uint	cmd;
	unchar	arg;
	int	s;

#ifndef IOPM
	ASSERT(is_upkern_lock());
#endif
	tp = (tyd_tty_t *)wq->q_ptr;
	ASSERT(tp != NULL);
	tp->wq_stat->ms_pcnt++;		/* bump count of write put calls */
	if (mp == NULL)
	{
		cmn_err(CE_NOTE, "NULL mp received in tyd_wput, q = %x", wq);
		return;
	}
	if (tp == NULL || !(tp->state & TTUSE))
	{
		freemsg(mp);
		return;
	}
	s = splstr();
	type = mp->b_datap->db_type;
	BUGLOG3(tp,BUG_LGEN,"tyd_wput wq=%x mp=%x type=%x",wq,mp,type);
	switch (type)
	{
	case M_DATA:
		BUGCIR_LOG(tp->min_dev, mp->b_wptr - mp->b_rptr,
					*mp->b_rptr,TYD_CIR_WPUT,(int)mp);
		/* FALL THRU TO DEFAULT */
	default:
		putq(wq, mp);	/* pass most everything to service rtn */
		break;
	case M_FLUSH:		/* flush read and/or write bufs */
		BUGCIR_LOG(tp->min_dev,0,*mp->b_rptr,TYD_CIR_WRITE_FLUSH,mp);
		if (*mp->b_rptr & FLUSHW)
			tyd_flush_write_side(tp);
		if (*mp->b_rptr & FLUSHR)
		{
			tyd_flush_read_side(tp);
			*mp->b_rptr &= ~FLUSHW;
			qreply(wq, mp);
		}
		else	freemsg(mp);
		break;
	case M_IOCTL:
		iocbp = (struct iocblk *)mp->b_rptr;  /*addr of ioctl control */
		cmd = iocbp->ioc_cmd;
		switch(cmd)
		{
		default:
			putq(wq, mp);	/* pass most everything to service rtn*/
			break;
		case TCLDTERM:
			if( tp->modpush & LDTERM )
				goto nak_ioctl;
			else {
				static char ldterm_m[7] = {"ldterm"};

				if( iocbp->ioc_count != sizeof(ldterm_m) ||
					!(mp->b_cont) )
					goto nak_ioctl;
				tp->modpush |= LDTERM;
				bcopy( ldterm_m,
					mp->b_cont->b_rptr,
					iocbp->ioc_count);
				mp->b_cont->b_wptr = mp->b_cont->b_rptr + 
							iocbp->ioc_count;
				mp->b_datap->db_type = M_IOCACK;
				qreply(wq, mp);
				break;
			}
		case TIOCGWINSZ:
		case TIOCSWINSZ:
			goto nak_ioctl;
		case TIOCSETP:
		case TCSBRK:
		case TCSETAW:
		case TCSETAF:
		case TCSETA:
		case GCTIO_SET:
			if (mp->b_cont == NULL)
				goto nak_ioctl;
			else	putq(wq, mp);
			break;
#ifdef SAK
		case TCSAK_SET:
			if (mp->b_cont == NULL || tp->sak.flags) /*already set*/
				goto nak_ioctl;
			else	putq(wq, mp);
			break;
#endif /* SAK */
		case TCFLSH:
			if (mp->b_cont == NULL)
				goto nak_ioctl;
			arg = *(int *)mp->b_cont->b_rptr;
			BUGCIR_LOG(tp->min_dev, cmd,arg,TYD_CIR_WRITE_IOCTL,mp);
			mp->b_datap->db_type = M_IOCACK;
			switch (arg)
			{
			default:
nak_ioctl:
				mp->b_datap->db_type = M_IOCNAK;
				iocbp->ioc_error = EINVAL;
				break;
			case 0:
				tyd_flush_read_side(tp);
				break;
			case 1:
				tyd_flush_write_side(tp);
				break;
			case 2:
				tyd_flush_write_side(tp);
				tyd_flush_read_side(tp);
				break;
			} /* switch (arg) */
			iocbp->ioc_count = 0;
			qreply(wq, mp);
			break;
		case TCXONC:
			if (mp->b_cont == NULL)
				goto nak_ioctl;
			arg = *(int *)mp->b_cont->b_rptr;
			BUGCIR_LOG(tp->min_dev,cmd,arg,TYD_CIR_WRITE_IOCTL, mp);
			mp->b_datap->db_type = M_IOCACK; /* assume good arg */
			switch (arg)
			{
			default:
				goto nak_ioctl;
			case 0:		/* T_SUSPEND */
				tyd_stop_output(tp, T_SUSPEND);
				break;
			case 1:		/* T_RESUME */
				tyd_start_output(tp);
				break;
			case 2:		/* T_BLOCK */
				tp->state |= TBLOCK;	/* stopped by ioctl */
				tyd_send_xoff(tp);
				break;
			case 3:		/* T_UNBLOCK */
				tyd_send_xon(tp);
				break;
			} /* switch (arg) */
			iocbp->ioc_count = 0;
			qreply(wq, mp);
			break;
		case TCCON_ALERT:	/* Alert on connect */
			if (mp->b_cont == NULL)	/* If no arguments, error */
				goto nak_ioctl;

			arg = *(int *)mp->b_cont->b_rptr;
			if (arg == TC_CONNOTIFY_ON) { /* Connect notify on */
				if ((tp->state & ( T_DCD | T_DSR ) ) == 
							(T_DCD | T_DSR)) {
					if (!put_pcproto(tp->rq->q_next, M_PCPROTO))
						cmn_err(CE_WARN,
			"tyd_w_put: connect notify priority message failed\n");
				}
				else tp->state |= T_NON;

			} else if (arg == TC_CONNOTIFY_OFF) /* notify off */
				tp->state &= ~T_NON;
			else
				goto nak_ioctl;

			mp->b_datap->db_type = M_IOCACK;
			iocbp->ioc_count = 0;
			qreply(wq, mp);
			break;
		}	/* switch (cmd) */
	} /* switch (type) */
	splx(s);
}/*------------------------------------------------------------------*/
/*
 *		WRITE SERVICE
 *
 *------------------------------------------------------------------*
 *
 * tyd_wsrv -- write service routine
 *
 * process M_DATA messages from upstream
 *
 * Arguments:
 *	wq	write queue pointer
 */
tyd_wsrv(wq)
register queue_t	*wq;
{
	register mblk_t		*bp;
	register tyd_tty_t	*tp;
	register type;

#ifndef IOPM
	ASSERT(is_upkern_lock());
#endif
	tp = (tyd_tty_t *)wq->q_ptr;
	ASSERT(tp != NULL);
	tp->wq_stat->ms_scnt++;		/* bump count of write service calls */
	while ((bp = getq(wq)) != NULL)
	{
		type = bp->b_datap->db_type;
		BUGLOG2(tp,BUG_LGEN,"tyd_wsrv wq=%x type=%x",wq,type);
		switch (type)
		{
		default:
			BUGCIR_LOG(tp->min_dev,type,*bp->b_rptr,
						TYD_CIR_WSRV, (int)bp);
			freemsg(bp);	/* throw away what we don't understand*/
			break;
		case M_BREAK:		/* tell hardware to send a break */
			BUGCIR_LOG(tp->min_dev,type,0,TYD_CIR_WSRV,(int)bp);
			if ((bp = tyd_process_msg_in_steps(bp, tp)) != NULL)
			{
				putbq(wq, bp);
				return;
			}
			break;
		case M_DELAY:
			BUGCIR_LOG(tp->min_dev,type,0,TYD_CIR_WSRV,
					*(int *)bp->b_rptr);
			if ((bp = tyd_process_msg_in_steps(bp, tp)) != NULL)
			{
				putbq(wq, bp);
				return;
			}
			break;
		case M_STOP:		/* tell ms to stop sending */
			BUGCIR_LOG(tp->min_dev,type,0,TYD_CIR_WSRV,(int)bp);
			tyd_stop_output(tp, T_SUSPEND);
			freemsg(bp);
			break;
		case M_START:		/* tell ms to start sending */
			BUGCIR_LOG(tp->min_dev,type,0,TYD_CIR_WSRV,(int)bp);
			tp->state &= ~T_SUSPEND;
			tyd_see_if_we_can_start_output(tp);
			freemsg(bp);
			break;
		case M_IOCDATA:
			tyd_wsrv_iocdata(bp, tp);
			break;
		case M_IOCTL:
			if ((bp = tyd_wsrv_ioc(wq, bp, tp)) != NULL)
			{
				putbq(wq, bp);
				return;
			}
			break;
		case M_DATA:
			if ((bp = tyd_wsrv_data(bp, tp)) != NULL)
			{
				putbq(wq, bp);
				return;
			}
			break;
		} /* switch (type) */
	} /* while ((bp = getq(wq)) != NULL) */
}/*------------------------------------------------------------------*/
/*
 *		WRITE SERVICE - IOCTL
 *
 *------------------------------------------------------------------
 *
 *	tyd_wsrv_ioc - process ioctl messages from upstream
 *
 * Arguments:
 *	wq	write queue pointer
 *	mp	message pointer
 *
 * 	return	mp	if unable to process
 *		NULL	if processes successfully
 */
mblk_t *
tyd_wsrv_ioc(wq, mp, tp)
register queue_t	*wq;
register mblk_t		*mp;
register tyd_tty_t	*tp;
{
	register struct sgttyb	*gb;
	struct	iocblk		*iocbp;
	register struct termio	*cb;
	uint	cmd;

	ASSERT(mp->b_datap->db_type == M_IOCTL);
	iocbp = (struct iocblk *)mp->b_rptr;	/* addr of ioctl control info */
	cmd = iocbp->ioc_cmd;
	BUGCIR_LOG(tp->min_dev, cmd, 0, TYD_CIR_WRITE_IOCTL, mp);
	switch(cmd)
	{
	default:
		cmd &= IOCTYPE;
		if (cmd == LDIOC || cmd == tIOC || cmd == TIOC)
			mp->b_datap->db_type = M_IOCACK;
		else	mp->b_datap->db_type = M_IOCNAK;
		iocbp->ioc_count = 0;
		qreply(wq, mp);
		break;
	case TIOCSETP:
	case TCSBRK:
	case TCSETAW:
	case TCSETAF:
		return(tyd_process_msg_in_steps(mp, tp));
	case TCSETA:
		if (tp->delay_msg != NULL || tp->timer_enable & STEP_TIMER)
			return(mp);	/*flag 'can't process'*/
		tyd_set_newioctl(mp, tp); /* set new ioctl */
		break;
	/* for GETs, which requires us to send a response back upstream,
	   make sure we have a buffer to put the response into. If not,
	   set 'bufcall' to call us when a buffer becomes available */
	case TCGETA:	/* immediate parm retrieve */
		if (tyd_insure_ioctl_works(mp,tp,(int)sizeof(struct termio)))
			return(mp);	/*flag 'can't process'*/
		cb = (struct termio *)mp->b_cont->b_rptr;
		bcopy(&tp->termio, cb, sizeof(struct termio));
		mp->b_cont->b_wptr += sizeof(struct termio);
		iocbp->ioc_count = sizeof(struct termio);
		mp->b_datap->db_type = M_IOCACK;
		qreply(wq, mp);
		break;
	case TIOCGETP:
		if (tyd_insure_ioctl_works(mp,tp,(int)sizeof(struct sgttyb)))
			return(mp);	/*flag 'can't process'*/
		gb = (struct sgttyb *)mp->b_cont->b_rptr; 
		gb->sg_ospeed = gb->sg_ispeed = tp->termio.c_cflag & CBAUD;
		gb->sg_flags = 0; 
		if (tp->termio.c_cflag & HUPCL)
			gb->sg_flags |= O_HUPCL;
		if (tp->termio.c_cflag & PARODD)
			gb->sg_flags |= O_ODDP;
		else if (tp->termio.c_iflag & INPCK)
			gb->sg_flags |= O_EVENP; 
		else	gb->sg_flags |= O_ODDP | O_EVENP; 

		mp->b_cont->b_wptr += sizeof(struct sgttyb);
		iocbp->ioc_count = sizeof(struct sgttyb);
		mp->b_datap->db_type = M_IOCACK;
		qreply(wq, mp);
		break;
#ifdef SAK
		case TCSAK_SET:
		if (iocbp->ioc_count == TRANSPARENT)
			tyd_ask_for_user_to_copy(mp, sizeof(struct sakioctl));
		else
		{
			tyd_get_sak_from_user(mp, tp);
			mp->b_datap->db_type = M_IOCACK;
			iocbp->ioc_count = 0;
		}
		qreply(wq, mp);
		break;
	case TCSAK_GET:
		return(tyd_send_sak_to_user(mp, tp));
#endif /* SAK */
	case GCTIO_SET:
		if (iocbp->ioc_count == TRANSPARENT)
			tyd_ask_for_user_to_copy(mp, sizeof(struct gctio));
		else
		{
			tyd_get_gctio_from_user(mp, tp);
			mp->b_datap->db_type = M_IOCACK;
			iocbp->ioc_count = 0;
		}
		qreply(wq, mp);
		break;
	case GCTIO_GET:
		return(tyd_send_gctio_to_user(mp, tp));
	} /* switch (cmd) */
	return(NULL);
}/*------------------------------------------------------------------*/
/*
 *	transparent set ioctl receved - ask user to send data
 */
tyd_ask_for_user_to_copy(mp, size)
register mblk_t		*mp;
uint size;
{
	register struct copyreq	*cqp;

	cqp = (struct copyreq *)mp->b_rptr;
	/* get user space structure */
	cqp->cq_addr = (caddr_t) *(long *) mp->b_cont->b_rptr;
	cqp->cq_size = size;		/* size of struct we want */
	cqp->cq_flag = 0;

	freemsg(mp->b_cont);		/* free msg blk, we don't need now */
	mp->b_cont = NULL;

	mp->b_datap->db_type = M_COPYIN;
}/*------------------------------------------------------------------*/
/*
 *	process an ioctl SET command - set new termio and send ACK back
 */
tyd_set_newioctl(mp, tp)
register mblk_t		*mp;
register tyd_tty_t	*tp;
{
	register struct termio	*cb;
	register struct sgttyb	*gb;
	register struct iocblk	*iocbp;		/* ioctl control block */
	register cflag;
	register iflag;
	uint cflag_change;
	uint flag_change = 0;
	int i;
	termio_t *tptr;

	ASSERT(tp != NULL);
	tptr = &tp->termio;
	iocbp = (struct iocblk *)mp->b_rptr;	/* addr of ioctl control info */
	switch(iocbp->ioc_cmd)
	{
	default:
		mp->b_datap->db_type = M_IOCNAK;
		if (mp->b_cont != NULL)
		{
			/* this frees any attached data block quicker */
			freemsg(mp->b_cont);
			mp->b_cont = NULL;
		}
		qreply(tp->wq, mp);
		return;
	case TCSETAW:
	case TCSETAF:
	case TCSETA:
		cb = (struct termio *)mp->b_cont->b_rptr;
		cflag = cb->c_cflag;
		iflag = cb->c_iflag;
		tptr->c_oflag = cb->c_oflag;
		tptr->c_lflag = cb->c_lflag;
		tptr->c_line = cb->c_line;
		for (i = 0; i < NCC; i++)
			tptr->c_cc[i] = cb->c_cc[i];
		break;
	case TIOCSETP:
		gb = (struct sgttyb *)mp->b_cont->b_rptr;
		cflag = (gb->sg_ispeed & CBAUD) | CREAD;
		if ((cflag & CBAUD) == B110)
			cflag |= CSTOPB;
		if (gb->sg_flags & O_HUPCL)
			cflag |= HUPCL;
		if (gb->sg_flags & O_RAW)
			cflag |= CS8;
		else
		{
			iflag |= BRKINT|IGNPAR|ISTRIP|IXON|IXANY;
			cflag |= CS7|PARENB;
		}
		iflag |= INPCK;
		if (gb->sg_flags & O_ODDP)
		{
			if (gb->sg_flags & O_EVENP)
				iflag &= ~INPCK;
			else	cflag |= PARODD;
		}
		break;
	} /* switch(iocbp->ioc_cmd) */
	if (cflag != tptr->c_cflag)
	{
		BUGCIR_LOG(tp->min_dev, 0, 0, TYD_CIR_CFLAG_OLD, tptr->c_cflag);
		BUGCIR_LOG(tp->min_dev, 0, 0, TYD_CIR_CFLAG_NEW, cflag);
		BUGLOG3(tp,BUG_LSIO,"tyd_set_newioctl rq=%x cflag was %x is %x",
					 tp->rq,  tptr->c_cflag, cflag);
		cflag_change = cflag  ^ tptr->c_cflag;
		flag_change = 1;
		tyd_cflag_program(tp, cflag);
		/* if CLOCAL toggled, beter see if CARR_ON will change */
		if (cflag_change & CLOCAL)
			tyd_process_eia_change(tp, tp->state); 
	}
	else	cflag_change = 0;		/* no cflag change */

	/* we must call 'tyd_iflag_program' if the byte size changes in 'cflag'
	   because this could affect the special character table */
	if (iflag != tptr->c_iflag || cflag_change & CSIZE)
	{
		BUGCIR_LOG(tp->min_dev, 0, 0, TYD_CIR_IFLAG_OLD, tptr->c_iflag);
		BUGCIR_LOG(tp->min_dev, 0, 0, TYD_CIR_IFLAG_NEW, iflag);
		BUGLOG3(tp,BUG_LSIO,"tyd_set_newioctl rq=%x iflag was %x is %x",
					 tp->rq,  tptr->c_iflag, iflag);
		tptr->c_iflag = iflag;
		flag_change = 1;
		tyd_iflag_program(tp);
	}
#ifdef IOPM
#ifdef NOT_REAL_HARDWARE	
	putnext(tp->wq, mp);		/* send downstram to real driver */
#else
	mp->b_datap->db_type = M_IOCACK;
	iocbp->ioc_count = 0;
	qreply(tp->wq, mp);
#endif /* NOT_REAL_HARDWARE */	
#else
	if (flag_change)
	{
		tp->state |= NO_OUTPUT_ALLOWED;	 /* don't allow output*/
		tyd_set_flag_cmd(tp);
		tp->delay_msg = mp;
		tp->state |= WAIT_FOR_FLAG_ON_IOCTL;	/* spm already did cmd*/
	}
	else	tyd_ack_to_ioctl(mp, tp);	/* no change, ACK immediately */
#endif /* IOPM */
}/*------------------------------------------------------------------*/
/*
 *		WRITE SERVICE - IOCDATA
 *
 *------------------------------------------------------------------*
 *
 */
tyd_wsrv_iocdata(mp, tp)
register mblk_t		*mp;
register tyd_tty_t	*tp;
{
	register struct	iocblk	*iocbp;		/* addr of ioctl control info */
	register struct copyresp *csp;		/* addr of M_IOCDATA msg */
	ushort	cmd;

	ASSERT(mp->b_datap->db_type == M_IOCDATA);
	iocbp =	(struct iocblk *)mp->b_rptr;
	csp =	(struct copyresp *)mp->b_rptr;
	cmd =   csp->cp_cmd;
	BUGCIR_LOG(tp->min_dev, cmd, 0, TYD_CIR_WRITE_IOCDATA, mp);
	if (csp->cp_rval != 0)
	{
		BUGLOG3(tp,BUG_LSIO,
			 "tyd_wsrv_iocdata wq=%x mp=%x rval=%x",
				      tp->wq,   mp, csp->cp_rval);
		freemsg(mp);
		return;
	}
	switch (cmd)
	{
	default:
		BUGLOG3(tp,BUG_LERR,
			"tyd_wsrv_iocdata bad cmd wq=%x tp=%x mp=%x",
					    tp->wq,   tp,  mp);
		if (mp->b_cont != NULL)
		{
			/* this frees any attached data block quicker */
			freemsg(mp->b_cont);
			mp->b_cont = NULL;
		}
		mp->b_datap->db_type = M_IOCNAK;
		break;
	case  GCTIO_SET:		/* data from user, go set, then ACK */
		tyd_get_gctio_from_user(mp, tp);
		/* FALL THROUGH TO GCTIO_GET */
	case  GCTIO_GET:		/* user got data so ACK to end it */
		mp->b_datap->db_type = M_IOCACK;
		iocbp->ioc_error = 0;		/* may have been overwritten */
		iocbp->ioc_count = 0;		/* may have been overwritten */
		iocbp->ioc_rval  = 0;		/* may have been overwritten */
		break;
#ifdef SAK
	case  TCSAK_SET:		/* data from user, go set, then ACK */
		tyd_get_sak_from_user(mp, tp);
		/* FALL THROUGH TO TCSAK_GET */
	case  TCSAK_GET:		/* user got data so ACK to end it */
		mp->b_datap->db_type = M_IOCACK;
		iocbp->ioc_error = 0;		/* may have been overwritten */
		iocbp->ioc_count = 0;		/* may have been overwritten */
		iocbp->ioc_rval  = 0;		/* may have been overwritten */
		break;
#endif /* SAK */
	} /* switch (csp->cp_cmd) */
	qreply(tp->wq, mp);
}/*------------------------------------------------------------------*/
/*
 *	start up output if possible
 *
 *	start up if output not suspended (M_STOP or ioctl=TCXONC)
 *	and not flow controlled (X-OFF and/or CTS)
 */
tyd_see_if_we_can_start_output(tp)
register tyd_tty_t	*tp;
{
	if (!(tp->state & (T_SUSPEND|T_TXSTOP)))
		tyd_start_output(tp);
}/*------------------------------------------------------------------*/
/*
 *	user sending us new gctio data (time_out values & signals)
 */
tyd_get_gctio_from_user(mp, tp)
register mblk_t		*mp;
register tyd_tty_t	*tp;
{
	register struct gctio	*gt;

	gt = (struct gctio *)mp->b_cont->b_rptr;
	if (tp->t_gctio.gt_tdcd != gt->gt_tdcd)
	{
		BUGCIR_LOG(tp->min_dev, 0, 0, TYD_CIR_GCTIO_DCD,
			(tp->t_gctio.gt_tdcd << 16) + gt->gt_tdcd);
		tp->t_gctio.gt_tdcd = gt->gt_tdcd;
		tyd_stop_dcd_timer(tp);
	}
	if (tp->t_gctio.gt_tdcd_sig != gt->gt_tdcd_sig)
	{
		BUGCIR_LOG(tp->min_dev, 0, 0, TYD_CIR_GCTIO_DCD_SIG,
			(tp->t_gctio.gt_tdcd_sig << 16) + gt->gt_tdcd_sig);
		tp->t_gctio.gt_tdcd_sig = gt->gt_tdcd_sig;
	}
	if (tp->t_gctio.gt_tact != gt->gt_tact)
	{
		BUGCIR_LOG(tp->min_dev, 0, 0, TYD_CIR_GCTIO_ACT,
			(tp->t_gctio.gt_tact << 16) + gt->gt_tact);
		tp->t_gctio.gt_tact = gt->gt_tact;
		tyd_stop_act_timer(tp);
	}
	if (tp->t_gctio.gt_tact_sig != gt->gt_tact_sig)
	{
		BUGCIR_LOG(tp->min_dev, 0, 0, TYD_CIR_GCTIO_ACT_SIG,
			(tp->t_gctio.gt_tact_sig << 16) + gt->gt_tact_sig);
		tp->t_gctio.gt_tact_sig = gt->gt_tact_sig;
	}
}/*------------------------------------------------------------------*/
mblk_t *
tyd_send_gctio_to_user(mp, tp)
register mblk_t		*mp;
register tyd_tty_t	*tp;
{
	register struct gctio	*gt;
	register struct	iocblk	*iocbp;		/* addr of ioctl control info */
	register struct copyreq *cpybp;		/* addr of M_COPYOUYT msg */
	mblk_t *bp;

	iocbp = (struct iocblk *)mp->b_rptr;
	if (iocbp->ioc_count == TRANSPARENT)
	{
		bp = mp->b_cont;	/* save mblk containing copy addr*/
		mp->b_cont = NULL;
		if (tyd_insure_ioctl_works(mp,tp,(int)sizeof(struct gctio)))
		{
			mp->b_cont = bp;	/*restore msg to original form*/
			return(mp);		/*flag 'can't process'*/
		}
		cpybp = (struct copyreq *)mp->b_rptr;
		cpybp->cq_addr = (caddr_t) *(long *)bp->b_rptr;
		cpybp->cq_size = (uint)sizeof(struct gctio);
		mp->b_datap->db_type = M_COPYOUT;
		freemsg(bp);
	}
	else
	{
		if (tyd_insure_ioctl_works(mp,tp,(int)sizeof(struct gctio)))
			return(mp);	/*flag 'can't process'*/
		mp->b_datap->db_type = M_IOCACK;
		iocbp->ioc_count = sizeof(struct gctio);
	}
	gt = (struct gctio *)mp->b_cont->b_rptr;
	gt->gt_tdcd	= tp->t_gctio.gt_tdcd;
	gt->gt_tdcd_sig	= tp->t_gctio.gt_tdcd_sig;
	gt->gt_tact	= tp->t_gctio.gt_tact;
	gt->gt_tact_sig	= tp->t_gctio.gt_tact_sig;
	gt->gt_scc_stat	= tp->state;
	mp->b_cont->b_wptr += sizeof(struct gctio);
	
	qreply(tp->wq, mp);
	return(NULL);
}/*------------------------------------------------------------------*/
#ifdef SAK
/*
 *	user sending us sak ioctl
 */
tyd_get_sak_from_user(mp, tp)
register mblk_t		*mp;
register tyd_tty_t	*tp;
{
	register struct sakioctl	*sk;

	sk = (struct sakioctl *)mp->b_cont->b_rptr;
	BUGCIR_LOG(tp->min_dev, sk->flags, sk->sak_key1, TYD_CIR_SAK_SET,
					(sk->sak_key2 << 8) + sk->sak_key3);
	tp->sak.flags    = sk->flags;
	tp->sak.sak_key1 = sk->sak_key1;
	tp->sak.sak_key2 = sk->sak_key2;
	tp->sak.sak_key3 = sk->sak_key3;
}/*------------------------------------------------------------------*/
mblk_t *
tyd_send_sak_to_user(mp, tp)
register mblk_t		*mp;
register tyd_tty_t	*tp;
{
	register struct sakioctl *sk;
	register struct	iocblk	*iocbp;		/* addr of ioctl control info */
	register struct copyreq *cpybp;		/* addr of M_COPYOUYT msg */
	mblk_t *bp;

	iocbp = (struct iocblk *)mp->b_rptr;
	if (iocbp->ioc_count == TRANSPARENT)
	{
		bp = mp->b_cont;	/* save mblk containing copy addr*/
		mp->b_cont = NULL;
		if (tyd_insure_ioctl_works(mp,tp,(int)sizeof(struct sakioctl)))
		{
			mp->b_cont = bp;	/*restore msg to original form*/
			return(mp);		/*flag 'can't process'*/
		}
		cpybp = (struct copyreq *)mp->b_rptr;
		cpybp->cq_addr = (caddr_t) *(long *)bp->b_rptr;
		cpybp->cq_size = (uint)sizeof(struct sakioctl);
		mp->b_datap->db_type = M_COPYOUT;
		freemsg(bp);
	}
	else
	{
		if (tyd_insure_ioctl_works(mp,tp,(int)sizeof(struct sakioctl)))
			return(mp);	/*flag 'can't process'*/
		mp->b_datap->db_type = M_IOCACK;
		iocbp->ioc_count = sizeof(struct sakioctl);
	}
	sk = (struct sakioctl *)mp->b_cont->b_rptr;
	sk->flags	= tp->sak.flags;
	sk->sak_key1	= tp->sak.sak_key1;
	sk->sak_key2	= tp->sak.sak_key2;
	sk->sak_key3	= tp->sak.sak_key3;
	mp->b_cont->b_wptr += sizeof(struct sakioctl);
	
	qreply(tp->wq, mp);
	return(NULL);
}/*------------------------------------------------------------------*/
#endif /* SAK */
/*
 *	make sure there is a streams msg to put ioctl data into to
 *	send upstream, and that the attached data block is big enougn
 *
 *	return	0 if ok
 *		1 if no msg available
 */
tyd_insure_ioctl_works(mp, tp, size)
register mblk_t		*mp;
register tyd_tty_t	*tp;
register int		size;
{
	register mblk_t	*bp;

	if ((bp = mp->b_cont) != NULL)
	{
		if ((bp->b_datap->db_lim - bp->b_datap->db_base) < size)
		{
			freemsg(bp);	/* discard msg allocated for */
			mp->b_cont = NULL; /* data; not big enough */
		}
	}
	if (mp->b_cont == NULL)
	{
		if ((bp = tyd_alloc(size, BPRI_MED, tyd_wsrv,tp->wq)) == NULL)
		{
			BUGCIR_LOG(tp->min_dev, size,0,TYD_CIR_NOBUF,NULL);
			BUGLOG1(tp,BUG_LERR,"tyd_insure no buf tp=%x",tp);
			return(1);	/*flag 'can't process'*/
		}
		mp->b_cont = bp;
	}
	return(0);
}/*------------------------------------------------------------------*/
/*
 *	ROUTINES USING 'TIMEOUT()'
 *
 *------------------------------------------------------------------*
 *
 *	the following set of routines use the 'timeout()' call to
 *	process a message that can not be executed until the output
 *	is empty.
 *
 *	for example, if we receive a M_DELAY message from upstreamm
 *	this means:
 *	-  wait until there is no data to output,
 *	-  delay for a specified interval, and finally
 *	-  allow output again
 *
 *	to accomplish this, the following routines would be called, in turn,
 *	using 'timeout()':
 *	tyd_check_if_output_empty()	wait for output to empty
 *	tyd_process_msg_output_empty()	process message from upstream
 *	tyd_start_timer_for_step()	start M_DELAY timer
 *	tyd_process_msg_delay_elapsed()	allow output to occur again
 */
mblk_t *
tyd_process_msg_in_steps(mp, tp)
register mblk_t		*mp;
register tyd_tty_t	*tp;
{
	if (tp->delay_msg != NULL || tp->timer_enable & STEP_TIMER)
		return(mp);	/*flag 'can't process'*/
	tp->delay_msg = mp;
	tp->state |= NO_OUTPUT_ALLOWED;
	tyd_check_if_output_empty(tp);
	return(NULL);
}/*------------------------------------------------------------------*/
/*
 *	output is now empty downstream, so process message
 *
 *	all data that came before this message has been output to the
 *	device, since this is now the first message in the queue
 */
tyd_process_msg_output_empty(tp)
register tyd_tty_t	*tp;
{
	register mblk_t	*bp;
	register struct	iocblk		*iocbp;
	int	delay;
	unchar	arg;

	ASSERT(tp != NULL);
	if (tp->output_in_progress == 0 && tp->t_closing)
		wakeup((caddr_t)&tp->output_in_progress);
	if ((bp = tp->delay_msg) != NULL)
		switch(bp->b_datap->db_type)
		{
		default:
			freemsg(bp);	/* throw away what we don't understand*/
			break;
		case M_DELAY:
			tyd_start_timer_for_step(tp, *(int *)bp->b_rptr);
			return;		/* DON'T FALL THROUGH */
		case M_BREAK:
			BUGCIR_LOG(tp->min_dev, 0, 0,TYD_CIR_BRK,
						tp->break_interval);
			tyd_start_break(tp, tp->break_interval);
			return;		/* DON'T FALL THROUGH */
		case M_IOCTL:
			iocbp = (struct iocblk *)bp->b_rptr; /*ioctl info addr*/
			switch(iocbp->ioc_cmd)
			{
			default:
				bp->b_datap->db_type = M_IOCNAK;
				qreply(tp->wq, bp);
				break;
			case TCSBRK:		/* process same as M_BREAK */
				bp->b_datap->db_type = M_IOCACK;
				iocbp->ioc_count = 0;
				qreply(tp->wq, bp);
				tp->delay_msg = NULL;
				if (*(int *)bp->b_cont->b_rptr == 1)
					break;
				if ((arg = *(int *)bp->b_cont->b_rptr) == 0)
					delay = tp->break_interval;
				else	delay = (arg - 1) * 100;
				/* 2 = 100 mills,  3 = 200 mils, etc. */
				BUGCIR_LOG(tp->min_dev,0,0,TYD_CIR_BRK,delay);
				tyd_start_break(tp, delay);  /* start break*/
				return;		/* DON'T FALL THROUGH */
			case TCSETAF:
				/* output is empty, so flush input */
				tyd_flush_read_side(tp);
				tyd_send_flush_upstream(tp->rq); /*flush up*/
				/* FALL THROUGH TO TCSETAW */
			case TCSETAW:
				tyd_set_newioctl(tp->delay_msg, tp);
#ifdef IOPM
				break;
#else
				return;		/* don't allow output yet */
#endif
			} /* switch(iocbp->ioc_cmd) */
		} /* switch(bp->b_datap->db_type) */
	qenable(tp->wq);	/* start output if any pending */
	tp->state &= ~NO_OUTPUT_ALLOWED; /*allow output*/
	tp->delay_msg = NULL;
}/*------------------------------------------------------------------*/
/*
 *	start timer for timer step
 */
tyd_start_timer_for_step(tp, delay)
register tyd_tty_t	*tp;
register delay;			/* no. clock ticks to timeout */
{
	/* if timer_step timer running, turn off before restarting timer*/
	if (tp->timer_enable & STEP_TIMER)
		untimeout(tp->timer_step_id);
	tp->timer_step_id = timeout(tyd_process_msg_delay_elapsed, tp, delay);
	tp->timer_enable |= STEP_TIMER;
}/*------------------------------------------------------------------*/
/*
 *	do timer step after delay timer elapsed
 */
tyd_process_msg_delay_elapsed(tp)
register tyd_tty_t	*tp;
{
	register mblk_t	*bp;
	register struct	iocblk		*iocbp;

	ASSERT(tp != NULL);
	tp->timer_enable &= ~STEP_TIMER;
	qenable(tp->wq);	/* start output if any pending */
	tp->state &= ~NO_OUTPUT_ALLOWED; /*allow output*/
	if (tp->state & BREAK_IN_PROGRESS)
	{
		BUGCIR_LOG(tp->min_dev, 0, 0,TYD_CIR_ENDBRK, 0);
		tyd_stop_break(tp);	/* stop break */
	}
	if (tp->delay_msg != NULL)
	{
		freemsg(tp->delay_msg);
		tp->delay_msg = NULL;
	}
	if ( tp->t_closing )
		wakeup((caddr_t)&tp->output_in_progress);
}/*------------------------------------------------------------------*/
/*
 *	send a flush upstream
 */ 
tyd_send_flush_upstream(rq)
register queue_t	*rq;
{
	register mblk_t	*bp;
	register tyd_tty_t	*tp;

	tp = (tyd_tty_t *)rq->q_ptr;
	if (!(tp->state & TTUSE))
		return;
	if ((bp = tyd_alloc(4,BPRI_HI,tyd_send_flush_upstream,rq)) != NULL)
	{
		bp->b_datap->db_type = M_FLUSH;
		*bp->b_wptr++ = FLUSHR;
		putnext(rq, bp);
	}
	else	BUGLOG1(rq->q_ptr, BUG_LERR, "rflush no buf, rq=%x", rq);
}/*------------------------------------------------------------------*/
/*
 *	send a SIGTACT signal upstream via M_PCSIG because of no activity
 */
tyd_send_tact_sig(tp)
register tyd_tty_t	*tp;
{
	tyd_stop_act_timer(tp);
	putctl1(tp->rq->q_next, M_PCSIG, tp->t_gctio.gt_tact_sig);
}/*------------------------------------------------------------------*/
/*
 *		dcd/dsr processing
 *
 *------------------------------------------------------------------*
 *
 *	we get here on an eia change
 *	we are guaranteed that either DCD or DSR has changed
 */
tyd_process_eia_change(tp, new_status)
register tyd_tty_t	*tp;
register new_status;	/* new port eia status */
{
	uint	change;	/* bits that changed in port status */
	uint	old_state;

	old_state = tp->state;
	new_status &= (T_DCD | T_DSR);	/* only interested in eia */
	tp->state &= ~(CARR_ON | T_DCD | T_DSR); /* turn off eia & carrier */
	tp->state |= new_status; 	/* set new eia bits */
	if (new_status == (T_DCD | T_DSR) || tp->termio.c_cflag & CLOCAL) 
		tp->state |= CARR_ON;
	BUGCIR_LOG(tp->min_dev,0,0,TYD_CIR_EV_EIA,
				(old_state << 16) | tp->state);
	BUGLOG3(tp, BUG_LEIA, "eia change device %x, old = %x, new = %x",
				tp->t_device, old_state, tp->state);
	change = old_state ^ tp->state;  /* find out what changed */

	if (change & (T_DCD | T_DSR)) 
	{
		sysinfo.mdmint++;		/* bump count of eia changes */
		tp->pv_rstat.mdmin++;		/* bump count of eia changes */

		/* If dsr/dcd change indicates a connect and
		 * 'notify on connect' is indicated (T_NON)
		 * then send a high priority message up stream.
		 */
		if ((tp->state & (T_DCD|T_DSR|T_NON)) == (T_DCD|T_DSR|T_NON))
			if (!put_pcproto(tp->rq->q_next, M_PCPROTO))
				cmn_err(CE_WARN,
	"tyd_process_eia_change: connect notify priority message failed\n");

#ifdef SAK
		if ( (tp->state & (T_DCD | T_DSR)) != (T_DCD | T_DSR) &&
		     (tp->sak.flags & SAK_CARR) )
		{
			tyd_sak_event(tp);
			return;
		}
#endif
	}

	if (change & CARR_ON) 
	{
		if (tp->state & CARR_ON)
		{		/* --- carrier now on --- */
			tyd_stop_dcd_timer(tp);
			wakeup((caddr_t)&tp->state);
		}
		else
		{		/* --- lost carrier --- */
			putctl(tp->rq->q_next, M_HANGUP);
			if (tp->t_gctio.gt_tdcd != 0
			 && change & T_DSR
			 && new_status & T_DSR)
				/* DSR just went hi, DCD low so wait for DCD */ 
				tyd_start_dcd_timer(tp);
			else if (!(tp->state & T_DTR_SET)) /* if DTR OFF*/
			/* we get here if DSR or DCD changed but DSR did not
			   just go high, and both are not high;
			   hence DSR or DCD just dropped */
				tyd_turn_on_dtr(tp);
		}
	}
}/*------------------------------------------------------------------*/
/*
 *	start timer waiting for dcd to go hi
 */
tyd_start_dcd_timer(tp)
register tyd_tty_t	*tp;
{
	if (tp->t_gctio.gt_tdcd != 0)
	{
		tp->dcd_id = timeout(tyd_dcd_timeout, tp,
					tp->t_gctio.gt_tdcd * HZ);
		tp->timer_enable |= DCD_TIMER;
	}
}/*------------------------------------------------------------------*/
/*
 *	stop timer waiting for dsr to go lo and dcd to go hi
 */
tyd_stop_dcd_timer(tp)
register tyd_tty_t	*tp;
{
	if (tp->timer_enable & DSR_TIMER)
	{
		untimeout(tp->dsr_id);
		tp->timer_enable &= ~DSR_TIMER;
	}
	if (tp->timer_enable & DCD_TIMER)
	{
		untimeout(tp->dcd_id);
		tp->timer_enable &= ~DCD_TIMER;
	}
}/*------------------------------------------------------------------*/
/*
 *	DCD failed to go high after DSR went high
 *
 *	drop DTR, wait 25 milliseconds, then if DSR is low, raise DTR again
 */
tyd_dcd_timeout(tp)
register tyd_tty_t	*tp;
{
	ASSERT(tp != NULL);
	if (tp->state & T_DTR_SET)	/* if DTR is set, turn DTR off */
	{
		tyd_turn_off_dtr(tp);
		tp->dsr_id = timeout(tyd_wait_for_dsr, tp, (25 * HZ)/1000);
		tp->timer_enable |= DSR_TIMER;
	}
}/*------------------------------------------------------------------*/
tyd_wait_for_dsr(tp)
register tyd_tty_t	*tp;
{
	ASSERT(tp != NULL);
	tp->timer_enable &= ~DSR_TIMER;
	if (tp->state & T_DSR)			/* if DSR is hi */
		tyd_turn_on_dtr(tp);		/* raise DTR */
}/*------------------------------------------------------------------*/
/*
 *		activity on input or output
 *
 *------------------------------------------------------------------*
 *
 *	start timer waiting for activity on input or output
 */
tyd_start_act_timer(tp)
register tyd_tty_t	*tp;
{	/* dcd just went high */
	if (tp->t_gctio.gt_tact != 0)
	{
		tp->act_id =
			timeout(tyd_send_tact_sig, tp, tp->t_gctio.gt_tact*HZ);
		tp->timer_enable |= ACT_TIMER;
	}
}/*------------------------------------------------------------------*/
/*
 *	stop timer waiting for activity on input or output
 */
tyd_stop_act_timer(tp)
register tyd_tty_t	*tp;
{
	if (tp->timer_enable & ACT_TIMER)
	{
		untimeout(tp->act_id);
		tp->timer_enable &= ~ACT_TIMER;
	}
}/*------------------------------------------------------------------*/
/*
 *		MISC
 *
 *------------------------------------------------------------------*
 *
 * tyd_alloc -- try to allocb a mblk size long at pri
 *
 *	if it fails it calls bufcall and returns NULL
 *	also returns NULL if bufcall can't store buffer request
 *
 *	input
 *		size	number of bytes to alloc
 *		pri	allocation priority
 *		func	addr of function passed bufcall if no buffers available
 *		arg	argument to pass to bufcall if no buffers available
 *
 *	return	address of buffer if one available
 *		NULL if no buffer available
 */
mblk_t *
tyd_alloc(size, pri, func, arg)
int	size, pri;
int	(*func)();
queue_t	*arg;
{
	mblk_t		*mp;

	if ((mp = allocb(size, pri)) == NULL)
	{
		if ((bufcall(size, pri, func, arg)) == 0)
		{
			BUGLOG3(NULL,BUG_LERR,
		 	 "tyd_alloc bufcall failed pri<<16+size=%x rtn=%c q=%x",
						  (pri << 16)+size, func, arg);
			BUGCIR_LOG(pri, size, 0,TYD_CIR_NOBUF,(int)arg);
		}
	}
	return (mp);
}/*------------------------------------------------------------------*/
tyd_fail(a, f, l)
char *a, *f;
int	l;
{
	cmn_err(CE_NOTE, "assertion failed: %s, file: %s, line: %d", a, f, l);
}/*------------------------------------------------------------------*/
tyd_log(tp, level, fmt, ARGS)
tyd_tty_t	*tp;
int		level;
char		*fmt;
unsigned	ARGS;
{
	int p_level;
	dev_t	dev;
	int	s;

	s = splstr();
	if (tp == NULL)
		dev = 255;
	else	dev = tp->min_dev;

	p_level = (tyd_debug & 0xf0000) >> 16;
	if (tyd_debug != 0 && (p_level == 0 || level <= p_level))
	{
		if ((tyd_debug & 0xfff) == 0x123)
			tyd_print_no_log(fmt, ARGS);
		else	strlog(TYD_MOD,dev,level, SL_TRACE, fmt, ARGS);
	}
	splx(s);
}/*------------------------------------------------------------------*/
tyd_print_no_log(fmt, ARGS)
char		*fmt;
unsigned	ARGS;
{
	printf(fmt, ARGS);
	if ((tyd_debug & 0x1000) == 0)
		printf("\n");
	else	printf(" ** ");
}/*------------------------------------------------------------------*/
tyd_cir_logit(dev, parm3, parm4, log_type, parm)
unchar dev, parm4, log_type;
ushort parm3;
int parm;
{
	register struct cir_log *ptr;	 /* ptr to next tyd_cir_log_buf entry */
	int	s;

#ifndef IOPM
	ASSERT(is_upkern_lock());
#endif
	if (dev != tyd_num_ports)
	{
		if (tyd_cir_log_only_one_port && dev != tyd_cir_port_to_log)
			return;
	}
	s = splstr();
	ptr = tyd_cir_ptr;
	ptr->rel_time	= (dev << 24) + (lbolt & 0x7fffff);
	ptr->log_type	= log_type;
	ptr->parm_1	= parm >> 16;
	ptr->parm_2	= parm & 0xffff;
	ptr->parm_3	= parm3;
	ptr->parm_4	= parm4;
	if (++tyd_cir_index >= TYD_CIR_LOG_SIZE)
	{
		tyd_cir_index = 0;
		tyd_cir_ptr = tyd_cir_log_buf;
	}
	else	tyd_cir_ptr++;
	tyd_cir_ptr->log_type = -1;	/* end of buf */
	splx(s);
}/*------------------------------------------------------------------*/
/*
 *	initialize the circular log buffer
 *
 *	- set the 'next entry pointer' to the start of the buffer
 *	- flag that entry as the 'end of buffer' entry (log_type = -1)
 */
tyd_cir_log_init()
{
	if (tyd_cir_log_init_flag == 0)
	{
		tyd_cir_log_init_flag++;	/* flag 'initialization done */
		tyd_cir_index = 0;		/* next entry pointer */
		tyd_cir_ptr   = tyd_cir_log_buf; /* next entry pointer */
		tyd_cir_ptr->log_type = -1;	/* end of buf */
	}
}/*------------------------------------------------------------------*/
tyd_init()
{
	tyd_cir_log_init();		/* init circular logging */
	tyd_msinit();			/* init microsequencer program */
}/*------------------------------------------------------------------*/

/*
 * Create and put a message on queue.
 */
put_pcproto(q, type)
queue_t *q;
{
	register mblk_t *bp;

	if (!(bp = allocb(0, BPRI_HI)))
		return(0);

	bp->b_datap->db_type = type;

	(*q->q_qinfo->qi_putp)(q, bp);

	return(1);
}/*------------------------------------------------------------------*/
