/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) pm_iomap.c: version 25.1 created on 11/27/91 at 14:59:09	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)pm_iomap.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
/*	Copyright (c) 1984 AT&T	*/
/*	  All Rights Reserved  	*/

/*
 * iomap -- all routines that deal with PM's iomap
 */

#include "sys/types.h"
#include "sys/lio.h"
#include "sys/vmem.h"
#include "sys/kmem.h"
#include "sys/sbus.h"
#include "sys/sbus_iom.h"
#include "sys/debug.h"
#include "sys/own.h"
#include "sys/pm_iomap.h"
#include "sys/spm_mem.h"
#include "sys/sbus.h"
#include "sys/sbus_pm.h"
#include "sys/synch.h"
#include "sys/cmn_err.h"
#include "sys/ints.h"

uint	*pm_int_req_regs[SBUS_MAX_NUM_PM];	/* array of I/O space pointers
						 * to interrupt request register
						 * of pm's in css slots
						 */


/*
 * This file contains routines that allocate and manipulate the PM's
 * mapping registers.
 *
 * iomap and sbus_map all have the same interface:
 *	Given a SBUS slot and an addr on that slot, they return a pointer
 *	that can be used to access the item(s) at that slot/addr.
 */

/*
 * iomap_init
 *
 *	clear all mapping registers.
 *
 *	intialize first 16 entries of map.
 *
 * Notes:
 *	
 *	The first 16 IOMAP entries will be mapped to the top mbyte of
 *	each CSS slot.
 *	
 *	Entry number 16 will be dedicated for use by iomap().
 */

iomap_init()
{
	register uint	map_reg;
	extern unchar	*iom_base[];
	register uint	sbus_slot;

	/* Clear all IOMAP registers */

	/* Init first 16 entries to map the upper mbyte of each CSS slot */
	for (map_reg = 0; map_reg < SBUS_NUM_SLOT; map_reg++) {
		if (spm_mem.sbus_slot_id[map_reg] != SBUS_NO_BOARD)
			PM_IO_MAPPER[map_reg] = (map_reg << PM_MapSlotShft) |
			  Upper_Mbyte | IOMAP_VALID;
		else
			PM_IO_MAPPER[map_reg] = IOMAP_INVALID;
	}

	/*
	 * Initialize a fixed location entry to our own slot for use
	 * in clearing pending interrupts in our interrupt request
	 * register.
	 */

	PM_IO_MAPPER[PM_Int_Req_Reg_Index] = 
	  (own.o_slot << PM_MapSlotShft) | Upper_Mbyte | IOMAP_VALID;

	/*
	 * Initialize a fixed location entry to use for writing and
	 * acking the interrupt dispatcher.
	 */

	PM_IO_MAPPER[Int_Disp_Index] = 
	  (spm_mem.spm_slot << PM_MapSlotShft) | Upper_Mbyte | IOMAP_VALID;
}

static uint	piomap_index = PIOMAP_START_INDEX;
static uint	piomap_val;

/*
 * piomap -- permanent map of system bus area
 *
 *	Return a kernel virtual address which is mapped to the passed in offset
 *	behind the passed in sbus_slot;
 *
 *	Warnings: 
 *
 *		This routine may not be called from interrupt service.
 *
 *		The kernel virtual returned is only valid up to the next
 *		1 Mbyte boundary.
 *
 *		Running out of maps causes a panic.
 */

unsigned char *
piomap(sbus_slot, offset)
uint	sbus_slot;
uint	offset;
{
	register uint		low_part, high_part;
	uint			index;

	static suspend_lock_t	piomap_susp_lock = SUSPEND_INIT(PZERO);

	ASSERT(! own.o_in_int_service);

	suspend_lock(&piomap_susp_lock);

	ASSERT(piomap_index < USEABLE_IOMAPS);

	/* find next available PM_IO_MAPPER register */

	while (PM_IO_MAPPER[piomap_index] & IOMAP_VALID)
		if (++piomap_index == USEABLE_IOMAPS)
			cmn_err(CE_PANIC,"piomap: out of iomap registers");

	ASSERT(! (PM_IO_MAPPER[piomap_index] & IOMAP_VALID));

	low_part = LowBits(PM_UnmappedBits, offset);
	high_part = HighBits(PM_UnmappedBits, offset);
	high_part |= (sbus_slot << (UintBits - PM_UnmappedBits));

	/*
	 * high_part now contains the upper part of the combined address
	 * low_part now contains the lower part of the combined address
	 */

	piomap_val = (high_part << PM_MapOutShft) | IOMAP_VALID;

	update_all_iomaps();
	index = piomap_index;
	suspend_unlock(&piomap_susp_lock);

	return(PM_IO_SPACE + (index << PM_UnmappedBits) + low_part);
}



/*
 * iomap -- temporary map of system bus area
 *
 * WARNING: iomap programs a single mapping register.
 *	iomap_save and iomap_restore must be used.
 *
 *	Typical use:
 *
 *	uint xxx = iomap_save();
 *
 *	abc_reg = iomap(slot, offset);
 *	printf("abc_reg contains\n", *abc_reg);
 *
 *	iomap_restore(xxx);
 */


unsigned char	*
iomap(sbus_slot, addr)
uint	sbus_slot;
uint	addr;
{
	register uint	low_part, high_part;

	low_part = LowBits(PM_UnmappedBits, addr);
	high_part = HighBits(PM_UnmappedBits, addr);
	high_part |= (sbus_slot << (UintBits - PM_UnmappedBits));

	/*
	 * high_part now contains the upper part of the combined address
	 * low_part now contains the lower part of the combined address
	 */

	PM_IO_MAPPER[PM_MyMap] = (high_part << PM_MapOutShft) | IOMAP_VALID;
	return(PM_IO_SPACE + (PM_MyMap << PM_UnmappedBits) + low_part);
}

/*
 * iomap_save -- return current map value
 */

uint
iomap_save()
{
	return(PM_IO_MAPPER [PM_MyMap]);
}

/*
 * iomap_restore -- restore value returned from earlier call to iomap_save 
 */

iomap_restore(value)
uint	value;
{
	PM_IO_MAPPER [PM_MyMap] = value;
}


/*
 * is_iomap -- return TRUE (i.e. 1) if address is in mapped i/o area
 */

is_iomap(vaddr)
unsigned char	*vaddr;
{
	uint	map;

	if (vaddr < PM_IO_SPACE)
		return(0);

	map = (vaddr - PM_IO_SPACE) >> PM_UnmappedBits;

	if ( PM_IO_MAPPER[map] & IOMAP_VALID )
		return(1);

	return(0);
}


/*
 * iomap_to_slot --  given an virtual address in iomap return to system
 *		bus slot currently being mapped to
 */

uint
iomap_to_slot(vaddr)
unsigned char	*vaddr;
{
	uint	map;

	ASSERT(is_iomap(vaddr));

	map = HighBits(PM_UnmappedBits, (vaddr - PM_IO_SPACE));

#ifdef M68040
	ASSERT(iomap_is_valid(map));
#endif

	return(HighBits((UintBits - PM_UnmappedBits), PM_IO_MAPPER[map]) &
	       (SBUS_NUM_SLOT - 1));
}

/*
 * iomap_to_offset --  given an virtual address in iomap return to system
 *		offset currently being mapped to
 */

uint
iomap_to_offset(vaddr)
unsigned char	*vaddr;
{
	register uint	map;
	register uint	high_part;

	ASSERT(is_iomap(vaddr));

	map = HighBits(PM_UnmappedBits, (vaddr - PM_IO_SPACE));
#ifdef M68040
	ASSERT(iomap_is_valid(map));
#endif
	high_part = LowBits((UintBits - PM_UnmappedBits), PM_IO_MAPPER[map]);

	return ((high_part << PM_UnmappedBits) |
	  LowBits(PM_UnmappedBits, (uint)vaddr));
}


/*
 * sbus_map -- map to control portion of sbus module
 *		return a pointer to a value in the mapped area
 *
 * Paramters:
 *
 *	sbus_slot - value from 0 to 0xf
 *
 *	sbus_offset - value from 0xfff00000 to 0xffffffff
 *
 *	return a pointer into the permanently mapped CSS slots (done in
 *	iomap_init), only with an offset within the upper Mbyte of an sbus slot.
 */

unchar *
sbus_map(sbus_slot, sbus_offset)
uint	sbus_slot;
uint	sbus_offset;
{
	uint	low_part;

	ASSERT(sbus_slot < SBUS_NUM_SLOT);
	ASSERT(sbus_offset >= 0xfff00000);
	ASSERT(iomap_is_valid(sbus_slot));

	return((unchar *)PM_IO_SPACE + (sbus_slot << PM_UnmappedBits) +
		LowBits(PM_UnmappedBits, sbus_offset));
}

/*
 * Temporarily map CSS expansion chassis address space.
 *
 * same usage, same warnings as iomap
 */
unchar *
iosba_map(sbus_slot, expansion_slot, offset)
uint  sbus_slot;
uint  expansion_slot;
uint  offset;
{
	extern unchar  *piomap();

	ASSERT(sbus_slot < SBUS_NUM_SLOT);
	ASSERT(spm_mem.sbus_slot_id[sbus_slot] == SBUS_NIOM);
	ASSERT(expansion_slot < SBUS_NUM_SLOT);

	return iomap(sbus_slot, expansion_slot << IOSBA_SubSlotShift |
				LowBits(IOSBA_SubSlotShift, offset));
}

/*
 * Permanently map an expansion bus address.
 * Calls piomap. All warning for piomap apply.
 */
unchar *
piosba_map(sbus_slot, expansion_slot, offset)
uint  sbus_slot;
uint  expansion_slot;
uint  offset;
{
	extern unchar  *piomap();

	ASSERT(sbus_slot < SBUS_NUM_SLOT);
	ASSERT(spm_mem.sbus_slot_id[sbus_slot] == SBUS_NIOM);
	ASSERT(expansion_slot < SBUS_NUM_SLOT);

	return piomap(sbus_slot, expansion_slot << IOSBA_SubSlotShift |
				 LowBits(IOSBA_SubSlotShift, offset));
}

/*
 * initiliaze a global array of pointers to the interrupt requeust
 * registers for PMs on a css slot basis.
 */

pm_int_req_reg_init()
{
	register uint	num_pm = spm_mem.num_pm;
	register uint	pm_id;
	register own_t	**o;

	/* Multiple pm's may have the same slot, but will have unique patterns
	 * sent, on interrupt, according to their pm value on the xpm board.
	 */
	o = spm_mem.pm_own;
	for (pm_id = 0; pm_id < num_pm; pm_id++, o++)
		pm_int_req_regs[pm_id] =
		  (uint *)sbus_map((*o)->o_slot, PM_INT_REQ_REG);
}

static disp_int_t	iomap_int_data;

update_iomap_init()
{
	iomap_int_data.fields.vector = DO_IOMAP_UPDATE;
	iomap_int_data.fields.level = MOT_LEVEL_ONE;
	iomap_int_data.fields.directed = NON_DIRECTED;
}

update_all_iomaps()
{
	register own_t	**opp, *o;
	register int	n;

	/* tell all pms to update their iomaps */
	for (opp = spm_mem.pm_own, n = spm_mem.num_pm; --n >= 0; opp++) {
		/* since cpu A and cpu B share the same iomap on the 040 board,
		 * we need to interrupt A only.
		 */
		o = *opp;
		if (o->o_pm_id == o->o_base_pm_id)  {
			o->o_update_iomap = 1;
			queue_level_one(o, iomap_int_data);
		}
	} /* for */

	/* wait for all pms to update their iomaps */
	for (opp = spm_mem.pm_own, n = spm_mem.num_pm; --n >= 0; opp++) {
		/* since we interrupted cpu A only, we needn't wait for cpu B */
		o = *opp;
		if (o->o_pm_id == o->o_base_pm_id)  {
			while (o->o_update_iomap)
				delay(1);
		}
	} /* for */
}


void
update_my_iomap()
{
	ASSERT(piomap_index < USEABLE_IOMAPS);
	ASSERT(! (PM_IO_MAPPER[piomap_index] & IOMAP_VALID));
	PM_IO_MAPPER[piomap_index] = piomap_val;
	own.o_update_iomap = 0;
}
