/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) lpmod.c: version 25.5 created on 3/6/92 at 21:55:11	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)lpmod.c	25.5	3/6/92 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
/*------------------------------------------------------------------*
 *	lpmod.c - streams lp module
 *
 *	10-22-88 - gil hunt - original version
 *	07-20-90 - gh	    - fix processing of LPRGET to return correct cq_addr
 *		            - LPMODVER2
 *
 *	this is a streams lp module for async processing.
 *	it handles almost all of the processing of each character sent
 *	either upstream or downstream, i.e., it handles all lp processing
 *	except for the following:
 *
 *	- cflag is handled entirely handled by the driver
 *
 *	- the only values in iflag handled by this modlue are:
 *	  INLCR, IGNCR, ICRNL, IUCLC, and ISWITCH
 *	  all remaining iflag values are handled by the driver
 *
 *	- the generation of signals when the VINTR, VQUIT, or VSWTCH
 *	  character is received are done by this module
 *
 *	- all eia handling done by the driver (e.g., HUP signal sent by
 *	  the driver when carrier drops)
 *
 *	- the ARIX add-on timer functions of dcd timeout and activity timeout
 *	  are handled by the driver
 *
 *------------------------------------------------------------------*/
#include "sys/types.h"
#include "sys/param.h"
#include "sys/errno.h"
#include "sys/sysmacros.h"
#include "sys/systm.h"
#include "sys/stream.h"
#include "sys/stropts.h"
#include "sys/signal.h"
#include "sys/debug.h"
#include "sys/cmn_err.h"
#include "sys/termio.h"
#include "sys/lprio.h"
#include "sys/lpm.h"
#include "sys/lpm_clog.h"
#include "sys/cir_log.h"
#include "sys/strlog.h"
#if	defined(IOPM)
#include "sys/str_conf.h"
#endif	/* IOPM */
/*----------------------------------------------------------------------*/
int	LPMODVER2;	/* version of stream lp module			*/
/*----------------------------------------------------------------------*/
extern	long	time;
int	lpm_cir_log_debug = 0;	/* set non-zero to enable lpm_cir_log */
int	lpm_cir_log_init_flag = 0; /* non-zero if cur_log initialized */

int	lpm_cir_index;		/* number of next lpm_cir_log_buf entry */

struct  cir_log	*lpm_cir_ptr;  /* ptr to next lpm_cir_log_buf entry */
struct  cir_log	lpm_cir_log_buf[LPM_CIR_LOG_SIZE];
/*------------------------------------------------------------------*/
int	lpm_debug = 0;		/* set non-zero to enable lpm_log */
/*------------------------------------------------------------------*/
#define LOG_DEBUG	lpm_debug
#define	LOG_RTN		lpm_log

#define CIRLOG_DEBUG	lpm_cir_log_debug
#define	CIRLOG_RTN	lpm_cir_logit

#include "sys/debuglog.h"
/*------------------------------------------------------------------*/
#define LPM_MOD		130

int lpm_open(), lpm_close(), lpm_rput(), lpm_wsrv(), lpm_wput();

struct module_info	lpm_minfo =
{
/*			 packet size
   mod id   module name  min    max	hi-water  lo_water
   ------   -----------  -------------  --------  --------*/
   LPM_MOD, "lpmod",     0,     INFPSZ, 4096,     1024 };
struct qinit lpm_ri =
{
/* put       service   open      close      unused  mod_info   stats
   ---       -------   ----      -----      ------  --------   ----- */
   lpm_rput, NULL,     lpm_open, lpm_close, NULL,  &lpm_minfo, NULL
};
struct qinit lpm_wi =
{
/* put       service   open      close      unused  mod_info   stats
   ---       -------   ----      -----      ------  --------   ----- */
   lpm_wput, lpm_wsrv, NULL,    NULL,       NULL,  &lpm_minfo, NULL
};

struct streamtab	 lpmodinfo = { &lpm_ri, &lpm_wi, NULL, NULL };
/*------------------------------------------------------------------*/
typedef struct	lpm lpm_t;

mblk_t *lpm_alloc();
mblk_t *lpm_wsrv_data();
mblk_t *lpm_send_lprio_to_user();
mblk_t *lpm_wsrv_ioc();
/*------------------------------------------------------------------*/
/*
 *	types of characters that can be sent downstream
 *
 *	this table is used to quickly process each data char sent downstream
 *	there is an entry for each character and the basic entry types are:
 *
 *	value of 0	= pass through as is
 *	control chars	= special processing
 *	a-z		= case conversion
 *	others		= look up in lpm_cap_tab for conversion
 */
unchar lpm_outchar_tab[] =
{
	/* 0x00 - 0x0f		control characters */
	       0,       0,       0,       0,       0,       0,       0,       0,
	LPM_Ox08,LPM_Ox09,LPM_Ox0a,       0,LPM_Ox0c,LPM_Ox0d,       0,       0,

	/* 0x10 - 0x1f		control characters */
	       0,       0,       0,       0,       0,       0,       0,       0,
	       0,       0,       0,       0,       0,       0,       0,       0,

	/* 0x20 - 0x2f */
	LPM_Oblank,     0,       0,       0,       0,       0,       0,       0,
	       0,       0,       0,       0,       0,       0,       0,       0,

	/* 0x30 - 0x3f 		digits 0 - 9 */
	       0,       0,       0,       0,       0,       0,       0,       0,
	       0,       0,       0,       0,       0,       0,       0,       0,

	/* 0x40 - 0x4f  	letters A - O */
	       0,       0,       0,       0,       0,       0,       0,       0,
	       0,       0,       0,       0,       0,       0,       0,       0,

	/* 0x50 - 0x5f  	letters P - Z  plus [\]^_ */
	       0,       0,       0,       0,       0,       0,       0,       0,
	       0,       0,       0,       0,       0,       0,       0,       0,

	/* 0x60 - 0x6f 		letters a - o */
	LPM_Ox60,LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,
	LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,

	/* 0x70 - 0x7f 		letters p - z */
	LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,
	LPM_Oa_z,LPM_Oa_z,LPM_Oa_z,LPM_Ox7b,LPM_Ox7c,LPM_Ox7d,LPM_Ox7e,       0,
};
/*------------------------------------------------------------------*/
/*
 *	conversion table for output
 *
 *	if CAP and not LPRAW set in flag, then some characters
 *	get replaced by the 2-character sequence 'x-' (x, then backspace -)
 *	where 'x' is the non-zero value in lpm_cap_tab that is derived
 *	using the original character as an index into the table.
 *
 *	for entries in the table that are zero, there is no conversion
 *	(not because the code checks for a non-zero entry, but because it
 *	only does the conversion for certain characters, based on the value
 *	in lpm_outchar_tab
 *
 *	convert	y		to	\x
 *		---------		--------
 *		` (0x60)	to	' (0x27) -
 *		{ (0x7b)	to	( (0x28) -
 *		| (0x7c)	to	! (0x21) -
 *		} (0x7d)	to	) (0x29) -
 *		~ (0x7e)	to	^ (0x5e) -
 */
unchar lpm_cap_out_tab[] =
{
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,			/* 0x00 - 0x0f*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,			/* 0x10 - 0x1f*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,			/* 0x20 - 0x2f*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,			/* 0x30 - 0x3f*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,			/* 0x40 - 0x4f*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,			/* 0x50 - 0x5f*/
	0x27,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,			/* 0x60 - 0x6f*/
	0,0,0,0,0,0,0,0,0,0,0,'(','!',')','^',0,		/* 0x70 - 0x7f*/
};
/*------------------------------------------------------------------*/
/*
 *		OPEN
 *
 *------------------------------------------------------------------*
 *
 * lpm_open -- open stream lp module
 *
 * Arguments:
 *	rq	if stream open, read queue pointer
 *	dev	major/minor device number
 *		regular open		- major/minor
 *		clone driver open	- major only
 *	flag	flags device was opened with
 *		0		- module open
 *	sflag	open type flag
 *		0		- driver open
 *		MODOPEN		- normal module open
 *		CLONEOPEN	- clone driver open
 *
 * Returns minor device number or OPENFAIL
 */
lpm_open(rq, dev, flag, sflag)
queue_t		*rq;
dev_t	dev;
int	flag, sflag;
{
	register lpm_t	*lp;
	register queue_t *wq;
	mblk_t	*bp_lp;		/* addr of msg used to get lp struct */
	int min_dev;		/* minor device number */

#if !defined(IOPM)
	lpm_cir_log_init();
#endif	/* IOPM */
	min_dev = minor(dev);
	BUGCIR_LOG(min_dev, flag, sflag, LPM_CIR_OPEN, (int)rq);
	if (sflag != MODOPEN)
	{
		BUGLOG1(NULL,BUG_LERR,"lpm_open bad sflag=%x", sflag);
		return(OPENFAIL);		/* error if not module open */
	}
	if (rq->q_ptr == NULL)
	{	/* first open for this port */
		if ((bp_lp = allocb(sizeof(lpm_t), BPRI_HI)) == NULL)
		{
			BUGCIR_LOG(min_dev,1,0, LPM_CIR_ALLOCB_FAIL, (int)rq);
			BUGLOG0(NULL,BUG_LERR,"unable to allocate tty struct");
			return (OPENFAIL);
		}
		lp = (lpm_t *)bp_lp->b_rptr;
		bzero(lp, sizeof(lpm_t));
		lp->myname[0] = 'L';
		lp->myname[1] = 'P';
		lp->myname[2] = 'M';
		lp->myname[3] = min_dev;

		lp->ind	= 4;
		lp->max_col = 132;
		lp->line = 66;

		lp->bp_lp = bp_lp;	/* addr of msg containing lp struct */
		lp->t_device		= dev;		/* device no */
		lp->min_dev		= min_dev;	/* minor device no. */

		/* get a msg for output so as to always have one on hand */
		lpm_get_msg(lp);
	
		/* link queues to lp struct and vice versa */
		wq = OTHERQ(rq);
		rq->q_ptr = (char *)lp;
		wq->q_ptr = (char *)lp;
		lp->rq = rq;
		lp->wq = wq;
	}
	BUGLOG3(lp, BUG_LGEN, "lpm_open rq=%x dev=%x flag=%x", rq, dev, flag);
	return(min_dev);
}/*------------------------------------------------------------------*/
/*
 *		CLOSE
 *
 *------------------------------------------------------------------*
 *
 * lpm_close -- close stream lp module
 *
 * Arguments:
 *	rq	read queue pointer
 *	flag	flags device opened with, 0 for modules
 */
lpm_close(rq, flag)
queue_t	*rq;
int	flag;
{
	register lpm_t	*lp;

	lp = (lpm_t *)rq->q_ptr;
	ASSERT(lp != NULL);
	BUGCIR_LOG(lp->min_dev, flag, 0, LPM_CIR_CLOSE, (int)rq);
	BUGLOG2(lp, BUG_LGEN, "lpm_close rq=%x flag=%x", rq, flag);

	if (lp->out_bp)
		freemsg(lp->out_bp);
 	freemsg(lp->bp_lp);		/* return lp structure to pool */
	rq->q_ptr = NULL;		/* unlink queues from lp struct */
	OTHERQ(rq)->q_ptr = NULL;
}/*------------------------------------------------------------------*/
/*
 *			READ PUT
 *
 *------------------------------------------------------------------*
 *
 * lpm_rput -- read put routine - read data from downstream
 *
 *
 * Arguments:
 *	rq	read queue pointer
 *	mp	message pointer
 */
lpm_rput(rq, mp)
register queue_t	*rq;
register mblk_t		*mp;
{
	register lpm_t	*lp;
	register type;			/* msg type */
	register unchar flag;

	lp = (lpm_t *)rq->q_ptr;
	ASSERT(lp != NULL);
	if (mp == NULL)
	{
		cmn_err(CE_NOTE, "NULL mp received in lpm_rput, q = %x", rq);
		return;
	}
	if (lp == NULL)
	{
		freemsg(mp);
		return;		/* port is not open */
	}
	type = mp->b_datap->db_type;
	BUGLOG3(lp, BUG_LPUT, "lpm_rput rq=%x mp=%x type=%x", rq, mp, type);
	switch (type)
	{
	default:
		BUGCIR_LOG(lp->min_dev,type,*mp->b_rptr,LPM_CIR_READ,mp);
		/* pass anything we don't understand upstream*/
		putnext(rq, mp);
		break;
	case M_DATA:
		if ( lp->flag & PASS )
			putnext(rq, mp);
		else
			/* by default we throw any stuff FROM the printer out */
			freemsg(mp);
		break;
	case M_FLUSH:
		flag = *mp->b_rptr;
		BUGCIR_LOG(lp->min_dev,0,flag,LPM_CIR_READ_FLUSH,(int)mp);
		if (flag & FLUSHW)
			lpm_flush_write_side(lp);
		/* ignore read flush, since nothing ever put on read queue */
		putnext(rq, mp);
		break;
	} /* switch (type) */
}/*------------------------------------------------------------------*/
/*
 *		WRITE PUT
 *
 *------------------------------------------------------------------*
 *
 * lpm_wput -- write put routine
 *
 *	processes M_FLUSH, M_BREAK, and M_IOCTL
 *	pass other high priority messages downstream
 *	pass low priority messages to write service routine
 *
 * Arguments:
 *	wq	write queue pointer
 *	mp	message pointer
 */
lpm_wput(wq, mp)
register queue_t	*wq;
register mblk_t	*mp;
{
	register lpm_t	*lp;
	register type;

	lp = (lpm_t *)wq->q_ptr;
	ASSERT(lp != NULL);
	if (mp == NULL)
	{
		cmn_err(CE_NOTE, "NULL mp received in lpm_wput, q = %x", wq);
		return;
	}
	if (lp == NULL)
	{
		freemsg(mp);
		return;		/* port is not open */
	}
	type = mp->b_datap->db_type;
	BUGLOG3(lp, BUG_LPUT, "lpm_wput wq=%x, mp=%x, type=%x",
					wq,    mp,    mp->b_datap->db_type);
	switch (type)
	{
	default:
		BUGCIR_LOG(lp->min_dev,type,*mp->b_rptr,LPM_CIR_WRITE,(int)mp);
		if ( type & QPCTL )
			putnext(wq, mp);
		else
			putq(wq, mp);
		break;
	case M_DATA:
		BUGCIR_LOG(lp->min_dev, mp->b_wptr - mp->b_rptr,
	 				*mp->b_rptr,LPM_CIR_WRITE_DATA,(int)mp);
		putq(wq, mp);		/* pass data to service routine */
		break;
	case M_FLUSH:
		BUGCIR_LOG(lp->min_dev,0,*mp->b_rptr,LPM_CIR_WRITE_FLUSH,mp);
		if (*mp->b_rptr & FLUSHW)
			lpm_flush_write_side(lp);
		/* ignore read flush, since nothing ever put on read queue */
		putnext(wq, mp);
		break;
	case M_IOCTL:
		lpm_wput_ioc(wq, mp, lp);
		break;
	case M_IOCDATA:
		lpm_wput_iocdata(wq, mp, lp);
		break;
	} /* switch (type) */
}/*------------------------------------------------------------------*/
/*
 *		WRITE PUT - IOCTL
 *
 *------------------------------------------------------------------*/
lpm_wput_ioc(wq, mp, lp)
register queue_t	*wq;
register mblk_t		*mp;
register lpm_t	*lp;
{
	register struct iocblk *iocbp;
	int	cmd;		/* ioctl command */
	unchar	arg;
	
	iocbp = (struct iocblk *)mp->b_rptr;
	cmd = iocbp->ioc_cmd;
	BUGLOG3(lp, BUG_LIOC, "lpm_wput_ioc wq=%x mp=%x cmd=%x", wq, mp, cmd);
	switch(cmd)
	{
	default:
		putq(wq, mp);
		break;
	case LPRSET:
		if (mp->b_cont == NULL && iocbp->ioc_count != TRANSPARENT)
			goto nak_ioctl;
		else	putq(wq, mp);		/* pass to service routine */
		break;
	case LPRGET:
		putq(wq, mp);		/* pass to service routine */
		break;
	case TCFLSH:
		if (mp->b_cont == NULL)
			goto nak_ioctl;
		arg = *(int *)mp->b_cont->b_rptr;
		BUGCIR_LOG(lp->min_dev, cmd, arg, LPM_CIR_WRITE_IOCTL, mp);
		switch (arg)
		{
		default:
nak_ioctl:
			mp->b_datap->db_type = M_IOCNAK;	/* bad arg */
			iocbp->ioc_error = EINVAL;
			qreply(wq, mp);
			return;
		case 0:
		/* ignore read flush, since nothing ever put on read queue */
			break;
		case 1:
			lpm_flush_write_side(lp);	/* flush write side */
			break;
		case 2:
		/* ignore read flush, since nothing ever put on read queue */
			lpm_flush_write_side(lp);	/* flush write side */
			break;
		} /* switch (*(int *)mp->b_cont->b_rptr) */
		putnext(wq, mp);
		break;
	} /* switch(cmd) */
}/*------------------------------------------------------------------*/
/*
 *		WRITE PUT - IOCDATA
 *
 *------------------------------------------------------------------*
 *
 *		so far, only GCTIO_SET & GCTIO_GET recognized
 */
lpm_wput_iocdata(wq, mp, lp)
register queue_t	*wq;
register mblk_t		*mp;
register lpm_t		*lp;
{
	register struct	iocblk	*iocbp;		/* addr of ioctl control info */
	register struct copyresp *csp;		/* addr of M_IOCDATA msg */

	ASSERT(mp->b_datap->db_type == M_IOCDATA);
	iocbp =	(struct iocblk *)mp->b_rptr;
	csp =	(struct copyresp *)mp->b_rptr;
	BUGCIR_LOG(lp->min_dev, csp->cp_cmd, 0, LPM_CIR_WRITE_IOCDATA, mp);
	if (csp->cp_rval != 0)
	{
		BUGLOG3(lp,BUG_LSIO,
			 "lpm_wput_iocdata wq=%x mp=%x rval=%x",
					   wq,   mp, csp->cp_rval);
		freemsg(mp);
		return;
	}
	switch (csp->cp_cmd)
	{
	default:
		putnext(wq, mp);	/* pass all unknown iocdata downstream*/
		break;
	case  LPRSET:		/* data from user, go set, then ACK */
		lpm_get_lpset_from_user(mp, lp);
		/* FALL THROUGH TO LPRGET */
	case  LPRGET:		/* user got data so ACK to end it */
		mp->b_datap->db_type = M_IOCACK;
		iocbp->ioc_error = 0;		/* may have been overwritten */
		iocbp->ioc_count = 0;		/* may have been overwritten */
		iocbp->ioc_rval  = 0;		/* may have been overwritten */
		qreply(wq, mp);
		break;
	} /* switch (csp->cp_cmd) */
}/*------------------------------------------------------------------*/
/*
 *		WRITE SERVICE
 *
 *------------------------------------------------------------------*
 *
 *	lpm_wsrv -- write service routine
 *
 *	process M_DATA and M_IOCTL messages from upstream
 *
 */
lpm_wsrv(wq)
register queue_t	*wq;	/* write queue pointer */
{
	register mblk_t		*bp;
	register lpm_t	*lp;
	register type;

	lp = (lpm_t *)wq->q_ptr;
	ASSERT(lp != NULL);
	while ((bp = getq(wq)) != NULL)
	{
		type = bp->b_datap->db_type;
		BUGLOG2(lp, BUG_LGEN, "lpm_wsrv wq=%x, type=%x",
						wq,    type);
		switch (type)
		{
		default:
			putnext(wq, bp);
			break;
		case M_IOCTL:
			if ((bp = lpm_wsrv_ioc(wq, bp, lp)) != NULL)
			{
				putbq(wq, bp);
				return;	/* unable to process any more data */
			}
			break;
		case M_DATA:
			if ((bp = lpm_wsrv_data(wq, bp, lp)) != NULL)
			{
				putbq(wq, bp);
				return;	/* unable to process any more data */
			}
			break;
		} /* switch (type) */
	} /* while ((bp = getq(wq)) != NULL) */
}/*------------------------------------------------------------------*/
/*
 *		WRITE SERVICE - DATA
 *
 *------------------------------------------------------------------*
 *
 *	process write data - called from service routine
 *
 * 	return	mp	if unable to process and putbq called
 *		NULL	if processes successfully
 */
mblk_t *
lpm_wsrv_data(wq, mp, lp)
register queue_t	*wq;
register mblk_t		*mp;
register lpm_t	*lp;
{
	mblk_t		*next;

	if (!(canput(wq->q_next)))
	{
		BUGCIR_LOG(lp->min_dev, 0, 0, LPM_CIR_WCANPUT, (int)wq);
		return(mp);	/* 'unable to process data */
	}
	if (lp->flag & LPRAW)
	{
		/* no output processing required, pass msg downstream as is */
		putnext(wq, mp);
		return(NULL);
	}
	BUGLOG2(lp, BUG_LGEN, "lpm_canon_cap_and_opost mp=%x lp=%x", mp, lp);
	while (mp != NULL)
	{
		while (mp->b_rptr < mp->b_wptr)
		{
			if (lp->out_bp == NULL)
				return(mp);	/* waiting for msg */
			lpm_output(lp, *mp->b_rptr++);
		}
		/* free up message block as soon as it is processed */
		next = mp->b_cont;
		mp->b_cont = NULL;
		BUGLOG1(lp, BUG_LGEN, "lpm_opost_only freeb call mp=%x", mp);
		freeb(mp);
		mp = next;
	}
	return(NULL);
}/*------------------------------------------------------------------*/
lpm_output(lp, c)
register lpm_t	*lp;
register unchar c;
{
	register c_type;		/* type of byte in message */ 
	int	tmp;

	if (lp->flag & CAP)
	{
		c_type = lpm_outchar_tab[c];
		switch(c_type)
		{
		default:
			break;
		case LPM_Oa_z:		/* map lower to upper case if CAP set */
			c -= 'a' - 'A';
			break;
		case LPM_Ox60:		/* map '`' to "'-"	   if CAP set */
		case LPM_Ox7b:		/* map '{' to "(-" 	   if CAP set */
		case LPM_Ox7c:		/* map '|' to "!-" 	   if CAP set */
		case LPM_Ox7d:		/* map '}' to ")-" 	   if CAP set */
		case LPM_Ox7e:		/* map '~' to "^-" 	   if CAP set */
			lpm_output(lp, lpm_cap_out_tab[c]);
			lp->col_count--;	/* count 2 chars as 1 */
			c = '-';
			break;
		}
	}
	c_type = lpm_outchar_tab[c];
	BUGCIR_LOG(lp->min_dev, c, c_type, LPM_CIR_TEST, NULL);
	switch (c_type)
	{
	default:
		if (lp->col_count < lp->out_index)
		{
			/* don't reset column count */
			tmp = lp->col_count;
			lpm_output_end_of_line(lp, '\r');
			lp->col_count = tmp;
		}
		if (lp->col_count < lp->max_col)
		{
			lpm_blanks_out(lp);
			lp->outbuf[lp->out_index++] = c;
			lp->col_count++;
		}
		break;
	case LPM_Ox08:		/*--- Backspace ---*/
		if (lp->col_count > lp->ind)
			lp->col_count--;
		break;
	case LPM_Ox09:		/*--- \t (tab) ---*/
		lp->col_count = ((lp->col_count + 8 - lp->ind) & ~7) + lp->ind;
		break;
	case LPM_Ox0a:		/*--- \n (line feed) ---*/
		lp->line_count++;
		if (lp->line != 0 && lp->line_count >= lp->line )
		{
			c = LP_FORM;
			goto form_feed;
		}
		else if (!(lp->flag & NOCR))
		{
			if (lp->col_count != lp->ind)
				lpm_blanks_out(lp);	/* pad with blanks */
			lp->outbuf[lp->out_index++] = c;
			c = '\r';
		}
		lpm_output_end_of_line(lp, c);
		break;
form_feed:
	case LPM_Ox0c:		/* form feed */
		if (lp->line != 0 && lp->line_count != 0)
		{
			if (lp->col_count != lp->ind)
				lpm_blanks_out(lp);	/* pad with blanks */
			if (lp->out_index) /* if not at start of line */
				lp->outbuf[lp->out_index++] = '\r';
			lpm_output_end_of_line(lp, c);
		}
		lp->line_count = 0;
		break;
	case LPM_Ox0d:		/* --- \r (carriage return) */
		lpm_output_end_of_line(lp, c);
		break;
	case LPM_Oblank:	/* blank */
		if (lp->col_count < lp->max_col)
			lp->col_count++;
		break;
	} /* ------------- switch(c_type) ------------- */
}/*------------------------------------------------------------------*/
/*
 *	move blanks in to output buffer until software column count reached
 */
lpm_blanks_out(lp)
register lpm_t	*lp;
{
	if (lp->col_count < lp->max_col)
	{
		while (lp->col_count > lp->out_index)
			lp->outbuf[lp->out_index++] = ' ';
	}
}/*------------------------------------------------------------------*/
/*
 *	send complete line downstream - line contained in outbuf
 */
lpm_output_end_of_line(lp, c)
register lpm_t	*lp;
unchar	c; 
{
	register size, i;
	register unchar *from_ptr, *to_ptr;
	mblk_t	*bp;

	if (lp->col_count != lp->ind)
		lpm_blanks_out(lp);
	lp->outbuf[lp->out_index++] = c;
	if ((bp = allocb(lp->out_index, BPRI_LO)) == NULL)
	{
		bp = lp->out_bp;
		lp->out_bp = NULL;
		lpm_get_msg(lp);
	}
	if (bp != NULL)
	{
		from_ptr = lp->outbuf;
		to_ptr = bp->b_wptr;
		size = lp->out_index;
		for (i = 0; i < size; i++)
			*to_ptr++ = *from_ptr++;
		bp->b_wptr = to_ptr;
		BUGCIR_LOG(lp->min_dev, size, *bp->b_rptr, LPM_CIR_XPOST, bp);
		putnext(lp->wq, bp);
	}
	lp->col_count = lp->ind;
	lp->out_index = 0;
}/*------------------------------------------------------------------*/
/*
 *		IOCTL WRITE SERVICE
 *
 *------------------------------------------------------------------*/
mblk_t *
lpm_wsrv_ioc(wq, mp, lp)
register queue_t	*wq;
register mblk_t		*mp;
register lpm_t		*lp;
{
	register struct iocblk *iocbp;
	struct copyreq	*cqp;
	int	cmd;		/* ioctl command */
	
	iocbp = (struct iocblk *)mp->b_rptr;
	cmd = iocbp->ioc_cmd;
	BUGCIR_LOG(lp->min_dev, cmd, 0, LPM_CIR_WRITE_IOCTL, mp);
	BUGLOG3(lp, BUG_LSIO, "lpm_wsrv_ioc wq=%x mp=%x cmd=%x", wq, mp, cmd);
	switch(cmd)
	{
	default:
		putnext(wq, mp);	/* pass all other ioctls downstream */
		break;
	case LPRSET:
		if (iocbp->ioc_count == TRANSPARENT)
		{
			cqp = (struct copyreq *)mp->b_rptr;
			/* get user space structure */
			cqp->cq_addr = (caddr_t) *(long *) mp->b_cont->b_rptr;
			cqp->cq_size = (uint)sizeof(struct lprio);
			cqp->cq_flag = 0;
			freemsg(mp->b_cont);
			mp->b_cont = NULL;
			mp->b_datap->db_type = M_COPYIN;
			qreply(wq, mp);
			break;
		}
		else
		{
			lpm_get_lpset_from_user(mp, lp);
			mp->b_datap->db_type = M_IOCACK;
			iocbp->ioc_count = 0;
			qreply(wq, mp);
			break;
		}
	case LPRGET:
		return(lpm_send_lprio_to_user(mp, lp));
	} /* switch(cmd) */
	return(NULL);
}/*------------------------------------------------------------------*/
/*
 *	set lprio flags
 */
lpm_get_lpset_from_user(mp, lp)
register mblk_t		*mp;
register lpm_t	*lp;
{
	register struct lprio *cb;

	cb = (struct lprio *)mp->b_cont->b_rptr;
	if (cb->ind != lp->ind)
	{
		BUGCIR_LOG(lp->min_dev, 0, 0, LPM_CIR_IND,
					(lp->ind << 16) | cb->ind);
		BUGLOG3(lp,BUG_LSIO,"lpm_get_lpset rq=%x ind was %x is %x",
					       lp->rq, lp->ind, cb->ind);
		lp->ind = cb->ind;
	}
	if (cb->col != lp->max_col)
	{
		BUGCIR_LOG(lp->min_dev, 0, 0, LPM_CIR_COL,
					(lp->max_col << 16) | cb->col);
		BUGLOG3(lp,BUG_LSIO,"lpm_set_newioctl rq=%x col was %x is %x",
						  lp->rq, lp->max_col, cb->col);
		if (cb->col > MAX_OUT_SIZE)
			lp->max_col = MAX_OUT_SIZE;
		else	lp->max_col = cb->col;
	}
	if (cb->line != lp->line)
	{
		BUGCIR_LOG(lp->min_dev, 0, 0, LPM_CIR_LINE,
					(lp->line << 16) | cb->line);
		BUGLOG3(lp,BUG_LSIO,"lpm_set_newioctl rq=%x line was %x is %x",
						  lp->rq, lp->line, cb->line);
		lp->line = cb->line;
	}
	if (cb->flag != lp->flag)
	{
		BUGCIR_LOG(lp->min_dev, 0, 0, LPM_CIR_FLAG,
					(lp->flag << 16) | cb->flag);
		BUGLOG3(lp,BUG_LSIO,"lpm_set_newioctl rq=%x flag was %x is %x",
						  lp->rq, lp->flag, cb->flag);
		lp->flag = cb->flag;
	}
}/*------------------------------------------------------------------*/
mblk_t *
lpm_send_lprio_to_user(mp, lp)
register mblk_t		*mp;
register lpm_t	*lp;
{
	register struct lprio	*lpr;
	register struct	iocblk	*iocbp;		/* addr of ioctl control info */
	register struct copyreq *cpybp;		/* addr of M_COPYOUYT msg */
	mblk_t		*bp;

	iocbp = (struct iocblk *)mp->b_rptr;
	if (iocbp->ioc_count == TRANSPARENT)
	{
		bp = mp->b_cont;	/*	LPMODVER2 changes */
		mp->b_cont = NULL;
		if (lpm_insure_ioctl_works(mp,lp,(int)sizeof(struct lprio)))
		{
			mp->b_cont = bp;
			return(mp);	/*flag 'can't process'*/
		}
		cpybp = (struct copyreq *)mp->b_rptr;
		cpybp->cq_addr = (caddr_t) *(long *) bp->b_rptr;
		cpybp->cq_size = (uint)sizeof(struct lprio);
		mp->b_datap->db_type = M_COPYOUT;
		freemsg(bp);
	}
	else
	{
		if (lpm_insure_ioctl_works(mp,lp,(int)sizeof(struct lprio)))
			return(mp);	/*flag 'can't process'*/
		mp->b_datap->db_type = M_IOCACK;
		iocbp->ioc_count = sizeof(struct lprio);
	}
	lpr = (struct lprio *)mp->b_cont->b_rptr;
	lpr->ind   = lp->ind;
	lpr->col   = lp->max_col;
	lpr->line  = lp->line;
	lpr->flag  = lp->flag;
	mp->b_cont->b_wptr += sizeof(struct lprio);
	
	qreply(lp->wq, mp);
	return(NULL);
}/*------------------------------------------------------------------*/
/*
 *	make sure there is a streams msg to put ioctl data into to
 *	send upstream, and that the attached data block is big enougn
 *
 *	return	0 if ok
 *		1 if no msg available
 */
lpm_insure_ioctl_works(mp, lp, size)
register mblk_t	*mp;
register lpm_t	*lp;
register int	size;
{
	register mblk_t	*bp;

	if ((bp = mp->b_cont) != NULL)
	{
		if ((bp->b_datap->db_lim - bp->b_datap->db_base) < size)
		{
			freemsg(bp);	/* discard msg allocated for */
			mp->b_cont = NULL; /* data; not big enough */
		}
	}
	if (mp->b_cont == NULL)
	{
		if ((bp = lpm_alloc(size, BPRI_MED, lpm_wsrv, lp->wq)) == NULL)
		{
			BUGLOG1(lp,BUG_LERR,"send_lprio no buf wq=%x", lp->wq);
			return(1);	/*flag 'can't process'*/
		}
		mp->b_cont = bp;
	}
	return(0);
}/*------------------------------------------------------------------*/
/*
 *		MISC
 *
 *------------------------------------------------------------------*
 *
 *	flush write side
 */
lpm_flush_write_side(lp)
register lpm_t	*lp;
{
	flushq(lp->wq, FLUSHALL);
	lp->col_count = lp->ind;
	lp->out_index = 0;
}/*------------------------------------------------------------------*/
lpm_get_msg(lp)
register lpm_t	*lp;
{
	if (lp->out_bp == NULL)
		lp->out_bp = lpm_alloc(MAX_OUT_SIZE,BPRI_LO,lpm_get_msg,lp->wq);
	if (lp->out_bp != NULL)
		lp->state &= ~WAITING_FOR_MSG;
	else	lp->state |= WAITING_FOR_MSG;
}/*------------------------------------------------------------------*/
/*
 * lpm_alloc -- try to allocb a mblk size long at pri
 *
 *	if it fails it calls bufcall and returns NULL
 *	also returns NULL if bufcall can't store buffer request
 *
 *	input
 *		size	number of bytes to alloc
 *		pri	allocation priority
 *		func	addr of function passed bufcall if no buffers available
 *		arg	argument to pass to bufcall if no buffers available
 *
 *	return	address of buffer if one available
 *		NULL if no buffer available
 */
mblk_t *
lpm_alloc(size, pri, func, arg)
int	size, pri;
int	(*func)();
queue_t	*arg;
{
	register mblk_t		*mp;

	if ((mp = allocb(size, pri)) == NULL)
	{

		if ((bufcall(size, pri, func, arg)) == 0)
		{
			BUGLOG3(NULL,BUG_LERR, "lpm_alloc bufcall failed, (pri<<16+size=%x func=%c q=%x", (pri << 16) + size, func, arg);
		}
		BUGCIR_LOG(pri, size, 0, LPM_CIR_NOBUF, (int)arg);
		return (NULL);
	}
	BUGLOG3(NULL,BUG_LERR,"lpm_alloc: rtn=%x size=%x mp=%x",func, size, mp);
	return (mp);
}/*------------------------------------------------------------------*/
lpm_fail(a, f, l)
register char *a, *f;
{
	cmn_err(CE_NOTE, "assertion failed: %s, file: %s, line: %d", a, f, l);
}/*------------------------------------------------------------------*/
lpm_log(lp, level, fmt, ARGS)
lpm_t	*lp;
int		level;
char		*fmt;
unsigned	ARGS;
{
	int p_level;
	dev_t	dev;

	if (lp == NULL)
		dev = 255;
	else	dev = lp->min_dev;

	p_level = (lpm_debug & 0xf0000) >> 16;
	if (lpm_debug != 0 && (p_level == 0 || level <= p_level))
	{
		if ((lpm_debug & 0xfff) == 0x123)
			lpm_print_no_log(fmt, ARGS);
		else	strlog(LPM_MOD,dev,level, SL_TRACE, fmt, ARGS);
	}
}/*------------------------------------------------------------------*/
lpm_print_no_log(fmt, ARGS)
char		*fmt;
unsigned	ARGS;
{
	printf(fmt, ARGS);
	if ((lpm_debug & 0x1000) == 0)
		printf("\n");
	else	printf(" ** ");
}/*------------------------------------------------------------------*/
lpm_cir_logit(dev, parm3, parm4, log_type, parm)
unchar dev, parm4, log_type;
ushort parm3;
int parm;
{
	register struct cir_log	*ptr;  /* ptr to next lpm_cir_log_buf entry */
	int s;

	s = splstr();
	ptr = lpm_cir_ptr;
	ptr->rel_time	= (dev << 24) + (lbolt & 0x7fffff);
	ptr->log_type	= log_type;
	ptr->parm_1	= parm >> 16;
	ptr->parm_2	= parm;
	ptr->parm_3	= parm3;
	ptr->parm_4	= parm4;

	if (++lpm_cir_index >= LPM_CIR_LOG_SIZE)
	{
		lpm_cir_index = 0;		/* wrap */
		lpm_cir_ptr = lpm_cir_log_buf;
	}
	else	lpm_cir_ptr++;
	lpm_cir_ptr->log_type = -1;	/* flag next entry as end of data */
	splx(s);
}/*------------------------------------------------------------------*/
/*
 *	initialize the circular log buffer
 *
 *	- set the 'next entry pointer' to the start of the buffer
 *	- flag that entry as the 'end of buffer' entry (log_type = -1)
 */
lpm_cir_log_init()
{
	if (lpm_cir_log_init_flag == 0)
	{
		lpm_cir_log_init_flag++;	/* flag 'initialization done */
		lpm_cir_index = 0;		/* next entry number */
		lpm_cir_ptr   = lpm_cir_log_buf; /* next entry pointer */
		lpm_cir_ptr->log_type = -1;	/* end of buf */
	}
}/*------------------------------------------------------------------*/
#if	defined(IOPM)
/*
 *	initialize IOPM configuration information
 */
lpmodinit()
{
	static struct str_mod_config strmodconf;

	strmodconf.smc_version = SMC_VERSION;
	strmodconf.smc_strtab = &lpmodinfo;
	attach_streams_module(&strmodconf);

	lpm_cir_log_init();
}/*------------------------------------------------------------------*/
#endif	/* IOPM */
