/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) ldterm.c: version 25.1 created on 11/27/91 at 14:58:24	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)ldterm.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
/*------------------------------------------------------------------*
 *	ldterm.c - ARIX streams tty module
 *
 *	10-22-88 - gil hunt - original version
 *	10-06-89 - gh - POSIX support
 *	11-06-89 - gh - add M_START, M_STOP support
 *	11-29-89 - gh - add stats - LDTERMVER2
 *	01-31-90 - gh - pass thru '\r' as is if ICRNL not set
 *	02-26-90 - gh - do flush before sending signal upstream, vice after
 * 	02-28-90 - gh - make tym_outchar_tab 256 bytes instead of 128 so
 *			that bytes >= 0x80 are passed downstream as is
 *		      - add 'sar' stats
 *		      - correct M_START/M_STOP support
 *		      - LDTERMVER3
 *	03-08-90 - gh - pass up our complete copy of iflag on TCGETA
 *			since 'ptem' doesn't set anything in that field
 *		      - stop processing current msg and discard if we
 *			receive key which caused us to send a signal
 *			upstream, and NOFLSH is not set (see 'check_noflsh')
 *	03-22-90 - gh - if switching from RAW to ICANON, discard any raw
 *			data waiting to be sent upstream.
 *			if switching from ICANON to RAW, check for ICANON
 *			in 'tym_rsrv_canon' so any messages put on the
 *			queue while in ICANON mode are now treated as RAW
 *	04-01-90 - gh - fix dropping of msgs if in raw mode
 *		      - enable sar stats
 *		      - clear raw_message ptr before calling 'putnext'
 *			since, if msg is 'delete' key, could rcv M_FLUSH
 *			message before returning from 'putnext'
 *		      - LDTERMVER4
 *	04-12-90 - gh - if in raw mode, only pass as much data upstream
 *			as is requested by M_READ so a lot of data does
 *			accumulate above this module and use up all of
 *			the streams resources
 *		      - LDTERMVER5
 *	05-02-90 - gh - fix 'tym_send_raw_up' to only ask for as many bytes
 *			as we need from 'tym_alloc' instead of the count
 *			in m_read_request (possibly larger than biggest msg)
 *		      - LDTERMVER6
 *	05-31-90 - gh -	fix calculation of 'wq_stat' so it doesn't point
 *			outside stats block (and clobber other msgs)
 *		      - if in raw mode, put M_DATA back onto read queue
 *			if raw_count exceeds MAX_INPUT (instead of just
 *			tossing excess data)
 *		      - LDTERMVER7
 *  06-12-90 - clear M_READ_RECVD when flushing pending input
 *			 - don't send down IOCTL of TCXONC for arg 1 and 2
 *		      - LDTERMVER8
 *
 *	this is a streams tty module for async processing.
 *	it handles almost all of the processing of each character sent
 *	either upstream or downstream, i.e., it handles all tty processing
 *	except for the following:
 *
 *	- cflag is handled entirely handled by the driver
 *
 *	- the only values in iflag handled by this modlue are:
 *	  INLCR, IGNCR, ICRNL, IUCLC, and ISWITCH
 *	  all remaining iflag values are handled by the driver
 *
 *	- the generation of signals when the VINTR, VQUIT, or VSWTCH
 *	  character is received are done by this module
 *
 *	- all eia handling done by the driver (e.g., HUP signal sent by
 *	  the driver when carrier drops)
 *
 *	- the ARIX add-on timer functions of dcd timeout and activity timeout
 *	  are handled by the driver
 *
 *------------------------------------------------------------------*/
#include "sys/types.h"
#include "sys/user.h"
#include "sys/proc.h"
#include "sys/param.h"
#include "sys/errno.h"
#include "sys/sysmacros.h"
#include "sys/systm.h"
#include "sys/stream.h"
#include "sys/stropts.h"
#include "sys/strstat.h"
#include "sys/signal.h"
#include "sys/ttold.h"
#include "sys/debug.h"
#include "sys/cmn_err.h"
#include "sys/termio.h"
#include "sys/tym_tty.h"
#include "sys/tym_clog.h"
#include "sys/cir_log.h"
#include "sys/strlog.h"
#include "sys/sysinfo.h"
#if	defined(IOPM)
#include "sys/str_conf.h"
#endif	/* IOPM */
/*----------------------------------------------------------------------*/
int	LDTERMVER8;		/* version of stream tty module			*/
/*----------------------------------------------------------------------*/
/*	debugging stuff	*/
/*------------------------------------------------------------------*/
extern	long	time;
int	tym_debug = 0;		/* set non-zero to enable tym_log */
int	tym_debug_opost = 0;	/* set non-zero to log each opost char */

int	tym_cir_log_debug = 1;	/* set non-zero to enable tym_cir_log */
int	tym_cir_log_init_flag = 0; /* non-zero if cur_log initialized */
int	tym_cir_log_only_one_port = 0; /* set non-zero to log only one port */
int	tym_cir_port_to_log = 0; /* single port no. to log */

int	tym_cir_index;		/* number of next tym_cir_log_buf entry */

struct  cir_log	*tym_cir_ptr;  /* ptr to next tym_cir_log_buf entry */
struct  cir_log	tym_cir_log_buf[TYM_CIR_LOG_SIZE];

#define LOG_DEBUG	tym_debug
#define	LOG_RTN		tym_log

#define CIRLOG_DEBUG	tym_cir_log_debug
#define	CIRLOG_RTN	tym_cir_logit

#include "sys/debuglog.h"
/*------------------------------------------------------------------*/
#define TYM_MOD		124

#define INPUT_OVERFLOW	MAX_INPUT*2	/* max to accept in raw mode */
#define INPUT_UNDERFLOW	MAX_INPUT/2	/* lo-water mark for streams flow ctrl*/
#define RAW_BUF_SIZE	16		/* size of input buffer for copying */

int tym_open(), tym_close(), tym_rsrv(), tym_rput(), tym_wsrv(), tym_wput();

struct module_info	tym_minfo =
{
/*			 packet size
   mod id   module name  min    max	hi-water	lo_water
   ------   -----------  -------------  --------  	--------*/
   TYM_MOD, "ldterm",     0,    INFPSZ, INPUT_OVERFLOW, INPUT_UNDERFLOW };
struct qinit tym_ri =
{
/* put       service   open      close      unused  mod_info   stats
   ---       -------   ----      -----      ------  --------   ----- */
   tym_rput, tym_rsrv, tym_open, tym_close, NULL,  &tym_minfo, NULL};
struct qinit tym_wi =
{
/* put       service   open      close      unused  mod_info   stats
   ---       -------   ----      -----      ------  --------   ----- */
   tym_wput, tym_wsrv, NULL,    NULL,       NULL,  &tym_minfo, NULL};

struct streamtab	 ldterminfo = { &tym_ri, &tym_wi, NULL, NULL };
/*------------------------------------------------------------------*/
typedef struct	termio  termio_t;
typedef struct	module_stat  mod_s_t;

int	tym_send_raw_up();
mblk_t *tym_alloc();
mblk_t *tym_rsrv_canon();
mblk_t *tym_rsrv_raw();
mblk_t *tym_wsrv_data();
mblk_t *tym_canon_xcase_only();
mblk_t *tym_opost_only();
mblk_t *tym_canon_xcase_and_opost();
/*------------------------------------------------------------------*/
/*
 *	types of characters that can be sent downstream
 *
 *	this table is used to quickly process each data char sent downstream
 *	there is an entry for each character and the basic entry types are:
 *
 *	value of 0	= pass through as is
 *	TYM_ONOP	= pass through as is but don't bump column position
 *	control chars	= special processing
 *	a-z		= case conversion
 *	others		= look up in tym_xcase_tab for conversion
 */
unchar tym_outchar_tab[] =
{
	/* 0x00 - 0x0f		control characters */
	TYM_ONOP,TYM_ONOP,TYM_ONOP,TYM_ONOP,TYM_ONOP,TYM_ONOP,TYM_ONOP,TYM_ONOP,
	TYM_Ox08,TYM_Ox09,TYM_Ox0a,TYM_Ox0b,TYM_Ox0c,TYM_Ox0d,TYM_ONOP,TYM_ONOP,

	/* 0x10 - 0x1f		control characters */
	TYM_ONOP,TYM_ONOP,TYM_ONOP,TYM_ONOP,TYM_ONOP,TYM_ONOP,TYM_ONOP,TYM_ONOP,
	TYM_ONOP,TYM_ONOP,TYM_ONOP,TYM_ONOP,TYM_ONOP,TYM_ONOP,TYM_ONOP,TYM_ONOP,

	/* 0x20 - 0x2f */
	       0,       0,       0,       0,       0,       0,       0,       0,
	       0,       0,       0,       0,       0,       0,       0,       0,

	/* 0x30 - 0x3f 		digits 0 - 9 */
	       0,       0,       0,       0,       0,       0,       0,       0,
	       0,       0,       0,       0,       0,       0,       0,       0,

	/* 0x40 - 0x4f  	letters A - O */
	       0,TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,
	TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,

	/* 0x50 - 0x5f  	letters P - Z */
	TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,
	TYM_OA_Z,TYM_OA_Z,TYM_OA_Z,TYM_Ox5c,       0,       0,       0,       0,

	/* 0x60 - 0x6f 		letters a - o */
	TYM_Ox60,TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,
	TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,

	/* 0x70 - 0x7f 		letters p - z */
	TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,
	TYM_Oa_z,TYM_Oa_z,TYM_Oa_z,TYM_Ox7b,TYM_Ox7c,TYM_Ox7d,TYM_Ox7e,TYM_ONOP,

	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,		/* 0x80 - 0x8f*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,		/* 0x90 - 0x9f*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,		/* 0xa0 - 0xaf*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,		/* 0xb0 - 0xbf*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,		/* 0xc0 - 0xcf*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,		/* 0xd0 - 0xdf*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,		/* 0xe0 - 0xef*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,		/* 0xf0 - 0xff*/
};
/*------------------------------------------------------------------*/
/*
 *	conversion table for input
 *
 *	if both ICANON and XCASE set in lflag, then some characters are input
 *	by the 2-character sequence '\x' (back slash, then x). What is passed
 *	to the kernel is the character 'y', where 'y' is the non-zero value
 *	in tym_xcase_in_tab that is derived the original character 'x' as an
 *	index into the table.
 *
 *	for entries in the table that are zero, there is no conversion
 *	(not because the code checks for a non-zero entry, but because it
 *	only does the conversion for certain characters, based on the value
 *	in tp->char_tab
 *
 *	convert	\x		to	y
 *		---------		--------
 *		\a - \z		to	A - Z
 *		\! (0x21)	to	| (0x7c)
 *		\' (0x27)	to	` (0x60)
 *		\( (0x28)	to	{ (0x7b)
 *		\) (0x29)	to	} (0x7d)
 *		\\ (0x5c)	to	\ (0x5c)
 *		\^ (0x5e)	to	~ (0x7e)
 */
unchar tym_xcase_in_tab[] =
{
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,			/* 0x00 - 0x0f*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,			/* 0x10 - 0x1f*/
	0,'|',0,0,0,0,0,'\`','{','}',0,0,0,0,0,0,		/* 0x20 - 0x2f*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,			/* 0x30 - 0x3f*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,			/* 0x40 - 0x4f*/
	0,0,0,0,0,0,0,0,0,0,0,0,'\\',0,'~',0,			/* 0x50 - 0x5f*/
	0,'A','B','C','D','E','F','G','H','I','J','K','L','M','N','O', /*60-6f*/
	'P','Q','R','S','T','U','V','W','X','Y','Z',0,0,0,0,0,	/* 0x70 - 0x7f*/
};
/*------------------------------------------------------------------*/
/*
 *	conversion table for output
 *
 *	if both ICANON and XCASE set in lflag, then some characters
 *	get replaced by the 2-character sequence '\x' (back slash, then x)
 *	where 'x' is the non-zero value in tym_xcase_tab that is derived
 *	using the original character as an index into the table.
 *
 *	for entries in the table that are zero, there is no conversion
 *	(not because the code checks for a non-zero entry, but because it
 *	only does the conversion for certain characters, based on the value
 *	in tym_outchar_tab
 *
 *	convert	y		to	\x
 *		---------		--------
 *		A - Z		to	\A - \Z
 *		\ (0x5c)	to	\\ (0x5c)
 *		` (0x60)	to	\' (0x27)
 *		{ (0x7b)	to	\( (0x28)
 *		| (0x7c)	to	\! (0x21)
 *		} (0x7d)	to	\) (0x29)
 *		~ (0x7e)	to	\^ (0x5e)
 */
unchar tym_xcase_out_tab[] =
{
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,			/* 0x00 - 0x0f*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,			/* 0x10 - 0x1f*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,			/* 0x20 - 0x2f*/
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,			/* 0x30 - 0x3f*/
	0,'A','B','C','D','E','F','G','H','I','J','K','L','M','N','O', /*40-4f*/
	'P','Q','R','S','T','U','V','W','X','Y','Z',0,'\\',0,0,0,/*0x50 - 0x5f*/
	0x27,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,			/* 0x60 - 0x6f*/
	0,0,0,0,0,0,0,0,0,0,0,'(','!',')','^',0,		/* 0x70 - 0x7f*/
};
/*------------------------------------------------------------------*/
/*
 *		OPEN
 *
 *------------------------------------------------------------------*
 *
 * tym_open -- open stream tty module
 *
 * Arguments:
 *	rq	if stream open, read queue pointer
 *	dev	major/minor device number
 *		regular open		- major/minor
 *		clone driver open	- major only
 *	flag	flags device was opened with
 *		0		- module open
 *	sflag	open type flag
 *		0		- driver open
 *		MODOPEN		- normal module open
 *		CLONEOPEN	- clone driver open
 *
 * Returns minor device number or OPENFAIL
 */
tym_open(rq, dev, flag, sflag)
queue_t		*rq;
dev_t	dev;
int	flag, sflag;
{
	register tym_tty_t	*tp;
	register queue_t	*wq;
	mblk_t		*bp_tp;		/* addr of msg used to get tty struct */
	mblk_t		*bp_tab;	/* addr of msg used to get char tab */
	dev_t		min_dev;	/* minor device number */

#if !defined(IOPM) 
	tym_cir_log_init();
#endif	/* IOPM */
	min_dev = minor(dev);
	BUGCIR_LOG(min_dev, flag, sflag, TYM_CIR_OPEN, (int)rq);
	if (sflag != MODOPEN)
	{
		BUGLOG1(NULL,BUG_LERR,"tym_open bad sflag=%x", sflag);
		u.u_error = EINVAL;
		return(OPENFAIL);	/* error if not module open */
	}
	if (rq->q_ptr == NULL)
	{	/* first open for this port */
		if ((bp_tp = allocb(sizeof(tym_tty_t), BPRI_LO)) == NULL)
		{
			BUGCIR_LOG(min_dev,1,0, TYM_CIR_ALLOCB_FAIL, (int)rq);
			BUGLOG0(NULL,BUG_LERR,"unable to allocate tty struct");
			u.u_error = EAGAIN;
			return (OPENFAIL);
		}
		if ((bp_tab = allocb(256, BPRI_LO)) == NULL)
		{
			BUGCIR_LOG(min_dev,2,0, TYM_CIR_ALLOCB_FAIL, (int)rq);
			BUGLOG0(NULL,BUG_LERR,"unable to allocate char. table");
			u.u_error = EAGAIN;
			freemsg(bp_tp);
			return (OPENFAIL);
		}
		tp = (tym_tty_t *)bp_tp->b_rptr;
		bzero(tp, sizeof(tym_tty_t));
		tp->myname[0] = 'T';		/* 'myname' for memory dumps */
		tp->myname[1] = 'Y';
		tp->myname[2] = 'M';
		tp->myname[3] = min_dev;	/* use lower byte of minor no.*/
		tp->bp_tp = bp_tp;	/* addr of msg containing tp struct */
		tp->bp_tab = bp_tab;	/* addr of msg containing char_tab */
		tp->termio.c_lflag	= ISIG | ICANON | ECHO;
		tp->termio.c_cc[VERASE]	= CERASE;
		tp->termio.c_cc[VKILL] 	= CKILL;
		tp->termio.c_cc[VINTR] 	= CINTR;
		tp->termio.c_cc[VQUIT] 	= CQUIT;
		tp->termio.c_cc[VEOF] 	= CEOF;
		tp->termio.c_cc[VEOL] 	= 0;
		tp->termio.c_cc[VEOL2] 	= 0;
		tp->termio.c_cc[VSWTCH] = CSWTCH;
		tp->t_device		= dev;		/* device no */
		tp->min_dev		= min_dev;	/* minor device no. */
	
		tym_alloc_stats(tp);		/* allocate stats buffers */
		if (tp->rq_stat != NULL)	/* general read stats*/
			tp->rq_stat->ms_ocnt++;	/* count of open calls */

		/* initialize tp->char_tab based on initial ioctl settings */
		tp->char_tab = bp_tab->b_rptr;
		tym_init_input_char_tab(tp);
	
		/* link queues to tp struct and vice versa */
		wq = OTHERQ(rq);
		rq->q_ptr = (char *)tp;
		wq->q_ptr = (char *)tp;
		tp->rq = rq;
		tp->wq = wq;

		tym_setopt_2_head(rq);		/* set stream head options */
	}
	BUGLOG3(tp, BUG_LGEN, "tym_open rq=%x dev=%x flag=%x", rq, dev, flag);
	return(min_dev);
}/*------------------------------------------------------------------*/
/*
 *		CLOSE
 *
 *------------------------------------------------------------------*
 *
 * tym_close -- close stream tty module
 *
 * Arguments:
 *	rq	read queue pointer
 *	flag	flags device opened with, 0 for modules
 */
tym_close(rq, flag)
queue_t	*rq;
int	flag;
{
	register tym_tty_t	*tp;
	register mblk_t		*bp;
	struct stroptions	*sop;

	tp = (tym_tty_t *)rq->q_ptr;
	ASSERT(tp != NULL);
	BUGCIR_LOG(tp->min_dev, flag, 0, TYM_CIR_CLOSE, (int)rq);
	BUGLOG2(tp, BUG_LGEN, "tym_close rq=%x flag=%x", rq, flag);
	tp->state = 0;
	tym_flush_read_side(tp);	/* flush read side */
	tym_flush_write_side(tp);	/* flush_write_side */
	freemsg(tp->bp_tab);		/* return character table to pool */
	freemsg(tp->bp_canon);		/* return character table to pool */
	freemsg(tp->bp_stat);		/* return statistics buffers to pool */
	/* reset stream head options for this stream */
	if ((bp = allocb(sizeof(struct stroptions), BPRI_HI)) != NULL)
	{
		bp->b_datap->db_type = M_SETOPTS;
		bp->b_wptr += sizeof (struct stroptions);
		sop = (struct stroptions *)bp->b_rptr;
		sop->so_flags = SO_READOPT|SO_NDELOFF|SO_MREADOFF;
		sop->so_readopt = RNORM;	/* byte stream, no ICANON */
		BUGCIR_LOG(tp->min_dev, 0, 0, TYM_CIR_HEADOPT,
				(sop->so_readopt << 16) + sop->so_flags);
		putnext (rq, bp);
	}
	else	BUGCIR_LOG(tp->min_dev,4,0, TYM_CIR_ALLOCB_FAIL, (int)rq);
	if (tp->timer_enable & RAW_TIMER)
		untimeout(tp->raw_id);
 	freemsg(tp->bp_tp);		/* return tty structure to pool */
	rq->q_ptr = NULL;		/* unlink queues from tp struct */
	tp->wq->q_ptr = NULL;
}/*------------------------------------------------------------------*/
/*
 *	allocate space for module statistics for both read and write queues
 *
 *	get a data block that will have room for general stats ('module_stat')
 *	for both the read side and the write side, plus room for private stats,
 *	again, both for the read side and the write side.
 */
tym_alloc_stats(tp)
register tym_tty_t	*tp;
{
	register mblk_t *bp;

	bp = tp->bp_stat = allocb(sizeof(mod_s_t) * 2, BPRI_LO);
	if (bp != NULL)
	{
		bzero(bp->b_rptr, sizeof(mod_s_t) * 2);

		/* read size stats */
		tp->rq_stat = (mod_s_t *)bp->b_rptr;	/* general read stats*/
		tp->rq_stat->ms_xptr = (char *)&tp->pv_rstat; /*private read */
		tp->rq_stat->ms_xsize = sizeof(tym_rstat_t); /*priv. read size*/

		/* write size stats */
		tp->wq_stat = tp->rq_stat + 1;	/* general write stats*/
		tp->wq_stat->ms_xptr = (char *)&tp->pv_wstat; /*private write */
		tp->wq_stat->ms_xsize = sizeof(tym_wstat_t); /*priv write size*/
	}
	else	tp->rq_stat = tp->wq_stat = NULL;

	/* zero out private read and write stats counts */
	bzero(&tp->pv_rstat, sizeof(tym_rstat_t));
	bzero(&tp->pv_wstat, sizeof(tym_wstat_t));
}/*------------------------------------------------------------------*/
/*
 *			READ PUT
 *
 *------------------------------------------------------------------*
 *
 * tym_rput -- read put routine - read data from downstream
 *
 *
 * Arguments:
 *	rq	read queue pointer
 *	mp	message pointer
 */
tym_rput(rq, mp)
register queue_t	*rq;
register mblk_t		*mp;
{
	register tym_tty_t	*tp;
	register type;			/* msg type */
	register unchar flag;
	struct iocblk *iocbp;
	termio_t *tptr;

	tp = (tym_tty_t *)rq->q_ptr;
	ASSERT(tp != NULL);
	if (tp == NULL)
	{
		freemsg(mp);
		return;		/* port is not open */
	}
	if (mp == NULL)
	{
		cmn_err(CE_NOTE, "NULL mp received in tym_rput, q = %x", rq);
		return;
	}
	if (tp->rq_stat != NULL)	/* general read stats*/
		tp->rq_stat->ms_pcnt++;	/* count of put calls */
	tptr = &tp->termio;
	type = mp->b_datap->db_type;
	flag = *mp->b_rptr;
	BUGLOG3(tp,BUG_LPUT,"tym_rput rq=%x mp=%x type=%x",rq,mp,type);
	switch (type)
	{
	default:
		BUGCIR_LOG(tp->min_dev,*mp->b_rptr, type,TYM_CIR_READ,(int)mp);
		putnext(rq, mp);/* pass anything we don't understand upstream*/
		break;
	case M_START:
	case M_STOP:
		BUGCIR_LOG(tp->min_dev,*mp->b_rptr, type,TYM_CIR_READ,(int)mp);
		freemsg(mp);
		break;
	case M_BREAK:
		BUGCIR_LOG(tp->min_dev,0,0,TYM_CIR_READ_BREAK,mp);
		if (!(tptr->c_iflag & IGNBRK) && (tptr->c_iflag & BRKINT))
			/* do NOT check for NOFLSH, as we do for a VINTR key */
			tym_send_sig_up(tp, SIGINT, 0);
		freemsg(mp);
		break;
	case M_FLUSH:
		BUGCIR_LOG(tp->min_dev,0,flag,TYM_CIR_READ_FLUSH,(int)mp);
		if (*mp->b_rptr & FLUSHW)
			tym_flush_write_side(tp);
		if (flag & FLUSHR)
			tym_flush_read_side(tp);
		putnext(rq, mp);
		break;
	case M_IOCACK:
		tym_rput_ioc(rq, mp, tp);	/* process or put on queue */
		break;
	case M_IOCNAK:
		/* this module both processes and sends downstream the
		 * ioctl command TCXONC. However, only a TCXONC with a valid
		 * argument is passed downstream, so if we get a NAK back
		 * from downstream, it means that what is below us did NOT
		 * understand TCXONC. Since we did understand, convert
		 * the NAK to an ACK */
		BUGCIR_LOG(tp->min_dev,*mp->b_rptr, type,TYM_CIR_READ,(int)mp);
		iocbp = (struct iocblk *)mp->b_rptr;
		if (iocbp->ioc_cmd == TCXONC)
		{
			mp->b_datap->db_type = M_IOCACK;
			iocbp->ioc_count = 0;
		}
		putnext(rq, mp);
		break;
	case M_DATA:
		BUGCIR_LOG(tp->min_dev, mp->b_wptr - mp->b_rptr,
	 				*mp->b_rptr, TYM_CIR_READ_DATA,(int)mp);
		if (tptr->c_lflag & ICANON)
			tym_rput_canon(rq, mp, tp);
		else if (tptr->c_iflag & (IXON|IGNCR|INLCR|ICRNL|IUCLC)
						 || tptr->c_lflag & ISIG)
			tym_rput_noncanon(rq, mp, tp);
		else if (tptr->c_lflag & ECHO)
			tym_rput_echo(mp, tp);
		else	putq(rq, mp);
		break;
	} /* switch (type) */
}/*------------------------------------------------------------------*/
/*
 *		READ PUT - IOCTL
 *
 *------------------------------------------------------------------*
 *
 *	process the GET IOCTLs, pass al others upstream
 */
tym_rput_ioc(rq, mp, tp)
register queue_t	*rq;
register mblk_t		*mp;
register tym_tty_t	*tp;
{
	register struct sgttyb	*gb;
	register struct termio	*cb, *tptr;
	register struct iocblk *iocbp;
	register lflag;
	register oflag;
	register flag;	
	int	cmd;		/* ioctl command */

	tptr = &tp->termio;
	iocbp = (struct iocblk *)mp->b_rptr;
	cmd = iocbp->ioc_cmd;
	BUGCIR_LOG(tp->min_dev,cmd, 0, TYM_CIR_READ_IOCACK, mp);
	BUGLOG3(tp,BUG_LIOC,"tym_rput_ioc rq=%x mp=%x cmd=%x",rq,mp,cmd);
	switch(cmd)
	{
	default:			/* pass all other ioctls upstream */
		break;
	case TIOCGETP:
		if (tym_insure_ioctl_works(mp,tp,(int)sizeof(struct sgttyb)))
		{
			mp->b_datap->db_type = M_IOCNAK;
			iocbp->ioc_error = EAGAIN;
			break;
		}
		gb = (struct sgttyb *)mp->b_cont->b_rptr; 
		/* ispeed, ospeed, and cflag variables set by the driver */
		gb->sg_erase = tptr->c_cc[VERASE]; 
		gb->sg_kill = tptr->c_cc[VKILL]; 
		flag = 0; 
		if (!((lflag = tptr->c_lflag) & ICANON))
		    flag |= O_RAW; 
		if (lflag & XCASE)
		    flag |= O_LCASE; 
		if (lflag & ECHO)
		    flag |= O_ECHO; 
		if (!(lflag & ECHOK))
		    flag |= O_NOAL; 
		if ((oflag = tptr->c_oflag) & ONLCR)
		{
			flag |= O_CRMOD; 
			if (oflag & CR1)
				flag |= O_CR1; 
			if (oflag & CR2)
			    	flag |= O_CR2; 
		}
		else
		{
			if (oflag & CR1)
			    	flag |= O_NL1; 
		    	if (oflag & CR2)
			   	 flag |= O_NL2; 
		}
		if ((oflag & TABDLY) == TAB3)
		    	flag |= O_XTABS; 
	    	else if (oflag & TAB1)
		    flag |= O_TBDELAY; 
	    	if (oflag & FFDLY)
		    flag |= O_VTDELAY; 
	    	if (oflag & BSDLY)
		    flag |= O_BSDELAY; 
	    	gb->sg_flags |= flag; 
		break;
	case TCGETA:
		if (tym_insure_ioctl_works(mp,tp,(int)sizeof(struct termio)))
		{
			mp->b_datap->db_type = M_IOCNAK;
			iocbp->ioc_error = EAGAIN;
			break;
		}
		cb = (struct termio *)mp->b_cont->b_rptr;

		/* c_cflag is provided entirely by the driver
		 * c_iflag used to be provided partially by the driver and
		 * partially by this module.
		cb->c_iflag &= ~(ISTRIP|INLCR|IGNCR|ICRNL|IUCLC|ISWITCH);
		cb->c_iflag |= tptr->c_iflag
				   & (ISTRIP|INLCR|IGNCR|ICRNL|IUCLC|ISWITCH);
		 */
		 /*
		 * c_iflag is not provided totally by us, since some
 		 * driver code does not provide iflag (e.g., 'ptem')
		 */
		cb->c_iflag =	tptr->c_iflag;
		cb->c_oflag =	tptr->c_oflag;
		cb->c_lflag =	tptr->c_lflag;
		cb->c_line  =	tptr->c_line;	
		BUGLOG3(tp, BUG_LGET, "tym_rput_ioc rq=%x c_flag=%x o_flag=%x",
				 		rq, cb->c_cflag, cb->c_iflag);
		BUGLOG3(tp, BUG_LGET, "tym_rput_ioc rq=%x o_flag=%x l_flag=%x",
				 		rq, cb->c_oflag, cb->c_lflag);
		bcopy(tptr->c_cc, cb->c_cc, NCC);
		iocbp->ioc_count = sizeof(struct termio);
		iocbp->ioc_error = 0;
		iocbp->ioc_rval = 0;
		break;
	} /* switch (cmd) */
	putnext(rq, mp);
}/*------------------------------------------------------------------*/
/*
 *			READ PUT - DATA
 *
 *------------------------------------------------------------------*
 *
 *	the processing of data in the read put routine is handled by
 *	the following:
 *
 *	if (tp->termio.c_lflag & ICANON)
 *		tym_rput_canon()
 *	else
 *	{
 *		if (tp->termio.c_iflag & (IGNCR|INLCR|ICRNL|IUCLC)
 *		      || tp->termio.c_lflag & ISIG)
 *			tym_rput_noncanon()
 *		else if (tp->termio.c_lflag & ECHO)
 *			tym_rput_echo()
 *	}
 *
 *	this division of labor minimizes the redundant checking of flags
 *	and allows each type of processing to be customized to be as fast
 *	as possible
 *------------------------------------------------------------------*
 *
 *			READ PUT - DATA - ICANON
 *
 *------------------------------------------------------------------*
 *
 * tym_rput_canon -- read put service routine for M_DATA if ICANON set
 */
tym_rput_canon(rq, mp, tp)
register queue_t	*rq;
register mblk_t		*mp;
register tym_tty_t	*tp;
{
	register mblk_t	*bp;
	register mblk_t	*ebp;		/* msg containing echoed characters*/
	register unchar *rptr;		/* next place to read char from */
	unchar *wptr;			/* last place to read char from (+1)*/
	register unchar *nwptr;		/* next place to store processed char */
	unchar *ewptr;			/* next place to store echo char */
	register last_was_escape;	/* non-zero = last char was '\' */
	register state, mask;
	uint lflag;			/* copy of c_lflag */
	unchar *char_tab_ptr;		/* addr of input char table this tty*/
	unchar c;			/* next character to process */
	unchar ce;			/* echo of next character */
	unchar c_type;			/* type of char (char_tab_ptr+c) */
	int echo_size;			/* no. bytes left in echo msg blk */
	ushort echo_is_enabled;		/* non-zero if ECHO set in lflag */

	BUGLOG2(tp, BUG_LPUT, "tym_rput_canon rq=%x mp=%x", rq, mp);
	lflag = tp->termio.c_lflag;
	if (lflag & (ECHO | ECHOE | ECHOK | ECHONL))
	{
		/* + 3 for expansion*/
		ebp = allocb(mp->b_wptr - mp->b_rptr + 3, BPRI_MED);
		if (ebp != NULL)
		{
			ewptr = ebp->b_wptr;
			echo_size = (int)(ebp->b_datap->db_lim - ewptr);
		}
	}
	else	ebp = NULL;
	/* NOTE: for canon-processing, we have to always check if there is
	   enough room left in the echo buffer, since we can sometimes echo
	   more bytes than we receive (e.g., ERASE), so we can't tell in
	   advance how much echo buffer we need */
	bp = mp;
	char_tab_ptr = tp->char_tab;	/* addr of character process table,
					   indexed by value of character */
	state = tp->state;
	if (tp->termio.c_iflag & IXANY)
		mask = 0;
	else	mask = TTSTOP;	/* don't clear TTSTOP via 'mask'*/
next_msg:
	wptr = bp->b_wptr;
	if (tp->termio.c_iflag & ISTRIP)
		/* strip everything except ff followed by ff or */
		/* ff followed by 0 followed by something */
		for (rptr = bp->b_rptr; rptr < wptr; rptr++)
			if ( rptr[0] != 0xff ||
			     rptr + 1 >= wptr ||
			     (rptr[1] != 0x00 && rptr[1] != 0xff) ||
			     (rptr[1] == 0x00 && rptr + 2 >= wptr) )
				*rptr &= 0x7f;

	nwptr = rptr = bp->b_rptr;
	sysinfo.rawch += (wptr - rptr);
	tp->pv_rstat.rawch += (wptr - rptr);
	while (rptr < wptr)
	{
		echo_is_enabled = tp->termio.c_lflag & ECHO;
		last_was_escape = tp->state & LCLESC;
		tp->state &= ~LCLESC;
		ce = c = *(rptr++);
		c_type = *(char_tab_ptr + c);
		switch (c_type)
		{
		default:
			state &= mask;
			break;
		case TYM_NEWLINE_1:	/*--- new-line && (iflag & INLCR) ---*/
			state &= mask;
			ce = c = '\r';
			if (lflag & ECHONL && ebp != NULL)
				echo_is_enabled++;
			break;
		case TYM_NEWLINE_2:	/*--- new-line && !(iflag & INLCR) ---*/
			state &= mask;
			if (lflag & ECHONL && ebp != NULL)
				echo_is_enabled++;
			break;
		case TYM_RETURN_1:	/* carriage ret. && (iflag & IGNCR)---*/
			state &= mask;
			continue;	/* don't put \r in msg */
		case TYM_RETURN_2:	/* carriage ret. && (iflag & ICRNL)---*/
			state &= mask;
			ce = c = '\n';
			if (lflag & ECHONL && ebp != NULL)
				echo_is_enabled++;
			break;
		case TYM_A_Z:		/*--- upper case && (iflag & IUCLC)---*/
			state &= mask;
			ce = c += 'a' - 'A';	/* convert A to a */
			break;
		case TYM_EOF:		/*--- EOF && ICANON ---*/
			state &= mask;
			if (last_was_escape)
				last_was_escape = 0;
			else if (ebp != NULL)
				echo_is_enabled = 0;	/* dont echo EOF char */
			break;		/* let the server discard this EOF */
		case TYM_INTR:
			state &= mask;
			tym_send_sig_up(tp, SIGINT, 1);
			goto check_noflsh;	/* see if we continue */
		case TYM_QUIT:
			state &= mask;
			tym_send_sig_up(tp, SIGQUIT, 1);
check_noflsh:
			if (!(tp->termio.c_lflag & NOFLSH))
			{	/* we are flushing so discard current msg */
				freemsg(bp);		/* free received msg */
				if (ebp != NULL)
					freemsg(ebp);	/* free echo buf */
				tp->state &= ~LCLESC;
				return;
			}
			continue;	/* don't put special char in msg */
#ifdef POSIX
		case TYM_SUSP:		/*--  SUSP char && lflag&ISIG && */
			state &= mask;
			tym_send_sig_up(tp, SIGTSTP, 1);
			goto check_noflsh;	/* see if we continue */
#else
		case TYM_SWTCH:		/*-- ^z && lflag&ISIG && iflag&ISWITCH*/
			state &= mask;
			putctl1(rq->q_next, M_CTL, 'Z');/* send ^Z up as M_CTL*/
			continue;	/* don't put SWTCH in msg */
#endif /* POSIX */
		case TYM_ERASE:		/*--- ^h ---*/
			state &= mask;
			if (!last_was_escape && ebp != NULL && (lflag & ECHOE))
			{
				if (echo_is_enabled)
				{
					*(ewptr++) = '\010';
					echo_size--;
				}
				*(ewptr++) = '\040';
				echo_size--;
				ce = '\010';
				echo_is_enabled++;
			}
			break;
		case TYM_KILL:		/*--- ^u ---*/
			state &= mask;
			if (!last_was_escape && ebp != NULL && (lflag & ECHOK))
			{
			/* Echo NL after KILL char if not escaped. */
				if (echo_is_enabled)
					*(ewptr++) = c;
				ce = '\n';
				echo_is_enabled++;
			}
			break;
		case TYM_a_z:		/*--- lower case ---*/
		case TYM_0x21:		/*--- \!  ->  | ---*/
		case TYM_0x27:		/*--- \'  ->  ` ---*/
		case TYM_0x28:		/*--- \(  ->  { ---*/
		case TYM_0x29:		/*--- \)  ->  } ---*/
		case TYM_0x5e:		/*--- \^  ->  ~ ---*/
			state &= mask;
			if (last_was_escape)
			{
				ce = c = tym_xcase_in_tab[c];   /*convert char*/
				last_was_escape = 0;
			}
			break;
		case TYM_0x5c:		/*--- \\  ->  \ ---*/
			state &= mask;
			if (last_was_escape)
			{
			/* if last was \, don't do anything special and
			   one \ will get stored in canon buf */
				last_was_escape = 0;
				break;
			}
			else
			{
				tp->state |= LCLESC;
				continue;	/* don't put \ in msg maybe */
			}
		case TYM_CSTOP:		/* x-off and IXON set in iflag*/
			if (tp->termio.c_iflag & IXANY)
				state ^= TTSTOP; /*if stopped, start else stop*/
			else	state |= TTSTOP;	/* stop */
			continue;
		case TYM_CSTART:	/* x-on  and IXON set in iflag*/
			state = 0;
			continue;	/* discard */
		} /* switch (c_type) */
		if (last_was_escape)
			*(nwptr++) = '\\';
		*(nwptr++) = c;
		if (ebp != NULL && echo_is_enabled)
		{
			*(ewptr++) = ce;	/* put echo char in echo buf */
			/* must have room for at least 3 more bytes in echo
			   buffer since the cc[VERASE] could require up to
			   3 echo bytes */
			if (--echo_size < 3)
			{
				ebp->b_wptr = ewptr;
	/* put in tp->echobuf rather than sending to put, so that the output
	 * rtn will know to not echo a '\' as '\\' if XCASE. This eliminates
	 * setting the sign bit to distinguish between a '\' from the kernel
	 * and one being echoed
	 */
				if (tp->echobuf == NULL)
					tp->echobuf = ebp;
				else	linkb(tp->echobuf, ebp);
				ebp = allocb(wptr - rptr + 3, BPRI_MED);
				if (ebp != NULL)
				{
					ewptr = ebp->b_wptr;
					echo_size = (int)(ebp->b_datap->db_lim - ewptr);
				}
			}
		}
	} /* while (rptr < wptr) */
	/* set 'new end of message' since we may have discarded some bytes */
	bp->b_wptr = nwptr;
	/* loop on chained msg blocks within each message */
	if ((bp = bp->b_cont) != NULL)
		goto next_msg;
	putq(rq, mp);		/* now pass to service routine (tym_rsrv) */
	if ((state ^ tp->state) & TTSTOP)
	{
		/* if we were stopped and we should not be stopped now
		 * then send M_START downstream
		 */
		if (tp->state & TTSTOP)
			tym_send_start_down(tp->wq);

		/* if we should be stopped, send M_STOP downstream */
		else	tym_send_stop_down(tp->wq);
	}
	if (ebp != NULL)
	{
		if (ebp->b_wptr != ewptr)
		{
			/* data in echo buffer */
			ebp->b_wptr = ewptr;
			if (tp->echobuf == NULL)
				tp->echobuf = ebp;
			else	linkb(tp->echobuf, ebp);
		}
		else	freemsg(ebp);		/* echo buffer empty */
	}
	if (tp->echobuf != NULL)
		qenable(tp->wq);	/*get output service to pass echo data*/
}/*------------------------------------------------------------------*/
/*
 *		READ PUT - DATA - NON-ICANON
 *
 *------------------------------------------------------------------*
 *
 * tym_rput_noncanon -- read put routine - ICANON not set
 *					   but either ISIG set and/or
 *					   IXON|IGNCR|INLCR|ICRNL|IUCLC set
 */
tym_rput_noncanon(rq, mp, tp)
register queue_t	*rq;
register mblk_t		*mp;
register tym_tty_t	*tp;
{
	register mblk_t	*bp, *tmp, *tbp;
	register unchar *rptr;		/* next place to read char from */
	register unchar *nwptr;		/* next place to store processed char */
	mblk_t	*ebp;			/* msg containing echoed characters*/
	unchar *wptr;			/* last place to read char from (+1)*/
	unchar *ewptr;			/* next place to store echo char */
	unchar *char_tab_ptr;		/* addr of input char table this tty*/
	register state, mask;
	unchar c;			/* next character to process */
	unchar c_type;			/* type of char (char_tab_ptr+c) */

	BUGLOG2(tp, BUG_LPUT, "tym_rput_noncanon rq=%x mp=%x", rq, mp);
	bp = mp;
	char_tab_ptr = tp->char_tab;	/* addr of character process table,
					   indexed by value of character */
	state = tp->state;
	if (tp->termio.c_iflag & IXANY)
		mask = 0;
	else	mask = TTSTOP;	/* don't clear TTSTOP via 'mask'*/
next_msg:
	wptr = bp->b_wptr;
	if (tp->termio.c_iflag & ISTRIP)
		/* strip everything except ff followed by ff or */
		/* ff followed by 0 followed by something */
		for (rptr = bp->b_rptr; rptr < wptr; rptr++)
			if ( rptr[0] != 0xff ||
			     rptr + 1 >= wptr ||
			     (rptr[1] != 0x00 && rptr[1] != 0xff) ||
			     (rptr[1] == 0x00 && rptr + 2 >= wptr) )
				*rptr &= 0x7f;

	nwptr = rptr = bp->b_rptr;
	/* NOTE: for non-icanon processing, we don't have to worry about
	   whether or not there is enough room in the echo buffer, since
	   non-icanon processing will never echo more bytes than it receives */
	if (tp->termio.c_lflag & ECHO)
	{
		ebp = allocb(wptr - rptr, BPRI_MED);
		if (ebp != NULL)
			ewptr = ebp->b_wptr;
	}
	else	ebp = NULL;
	while (rptr < wptr)
	{
		c = *(rptr++);
		c_type = *(char_tab_ptr + c);
		switch (c_type)
		{
		default:
			state &= mask;
			break;
		case TYM_NEWLINE_1:	/*--- new-line && (iflag & INLCR) ---*/
			state &= mask;
			c = '\r';
			break;
		case TYM_RETURN_1:	/*--- carriage ret. && (iflag & IGNCR)*/
			state &= mask;
			continue;	/* don't put \r in msg */
		case TYM_RETURN_2:	/*--- carriage ret. && (iflag & ICRNL)*/
			state &= mask;
			c = '\n';
			break;
		case TYM_A_Z:		/*--- upper case && (iflag & IUCLC)---*/
			state &= mask;
			c += 'a' - 'A';		/* convert A to a */
			break;
		case TYM_INTR:
			state &= mask;
			tym_send_sig_up(tp, SIGINT, 1);
			goto check_noflsh;	/* see if we continue */
		case TYM_QUIT:
			state &= mask;
			tym_send_sig_up(tp, SIGQUIT, 1);
check_noflsh:
			if (!(tp->termio.c_lflag & NOFLSH))
			{	/* we are flushing so discard current msg */
				freemsg(bp);		/* free received msg */
				if (ebp != NULL)
					freemsg(ebp);	/* free echo buf */
				tp->state &= ~LCLESC;
				return;
			}
			continue;	/* don't put cc[VQUIT] in msg */
#ifdef POSIX
		case TYM_SUSP:		/*--  SUSP char && lflag&ISIG && */
			state &= mask;
			tym_send_sig_up(tp, SIGTSTP, 1);
			goto check_noflsh;	/* see if we continue */
#else
		case TYM_SWTCH:		/*-- ^z && lflag&ISIG && iflag&ISWITCH*/
			state &= mask;
			putctl1(rq->q_next, M_CTL, 'Z');/* send ^Z up as M_CTL*/
			continue;	/* don't put SWTCH in msg */
#endif /* POSIX */
		case TYM_CSTOP:		/* x-off and IXON set in iflag*/
			if (tp->termio.c_iflag & IXANY)
				state ^= TTSTOP; /*if stopped, start else stop*/
			else	state |= TTSTOP;	/* stop */
			continue;
		case TYM_CSTART:	/* x-on  and IXON set in iflag*/
			state = 0;
			continue;	/* discard */
		} /* switch (c_type) */
		*(nwptr++) = c;
		if (ebp != NULL && (tp->termio.c_lflag & ECHO))
			*(ewptr++) = c;		/* put char in echo buffer */
	} /* while (rptr < wptr) */
	if (ebp != NULL)
	{
		ebp->b_wptr = ewptr;
	/* put in tp->echobuf rather than sending to put, so that the output
	 * rtn will know to not echo a '\' as '\\' if XCASE. This eliminates
	 * setting the sign bit to distinguish between a '\' from the kernel
	 * and one being echoed
	 */
		if (tp->echobuf == NULL)
			tp->echobuf = ebp;
		else
		{	/* find end of echo chain of buffers */
			tmp = tp->echobuf;
			while ((tbp = tmp->b_cont) != NULL)
				tmp = tbp;
			/* we get here when tmp points to last msg */
			tmp->b_cont = ebp;
		}
	}
	/* set 'new end of message' since we may have discarded some bytes */
	bp->b_wptr = nwptr;
	/* loop on chained msg blocks within each message */
	if ((bp = bp->b_cont) != NULL)
		goto next_msg;
	if ((state ^ tp->state) & TTSTOP)
	{
		/* if we were stopped and we should not be stopped now
		 * then send M_START downstream
		 */
		if (tp->state & TTSTOP)
			tym_send_start_down(tp->wq);

		/* if we should be stopped, send M_STOP downstream */
		else	tym_send_stop_down(tp->wq);
	}
	if (tp->echobuf != NULL)
		qenable(tp->wq);	/*get output service to pass echo data*/
	putq(rq, mp);
}/*------------------------------------------------------------------*/
/*
 *		READ PUT - DATA - RAW & ECHO
 *
 *------------------------------------------------------------------*
 *
 *	tym_rput_echo -      ICANON not set and ISIG not set and
 *			     IGNCR|INLCR|ICRNL|IUCLC not set
 *			     but ECHO is set
 *
 *	no processing required so echo as is
 */
tym_rput_echo(mp, tp)
register mblk_t		*mp;
register tym_tty_t	*tp;
{
	register mblk_t		*dupebp;
	register mblk_t		*tbp, *tmp;

	BUGLOG2(tp,BUG_LPUT,"tym_rput_echo rq=%x mp=%x",tp->rq,mp);
	if ((dupebp = dupmsg(mp)) != NULL)
	{
	/* since the echo buffer will be an exact duplicate of the original
	   msg, use 'dupmsg' to generate the echo msg rather than copying
	   the data into a separate group of data block */

	/* put in tp->echobuf rather than sending to put, so that the output
	   rtn will know to not echo a '\' as '\\' if XCASE. This eliminates
	   setting the sign bit to distinguish between a '\' from the kernel
	   and one being echoed */

		if (tp->echobuf == NULL)
			tp->echobuf = dupebp;
		else
		{	/* find end of echo chain of messages */
			tmp = tp->echobuf;
			while ((tbp = tmp->b_next) != NULL)
				tmp = tbp;
			/* we get here when tmp points to last msg */
			tmp->b_next = dupebp;
		}
		qenable(tp->wq);	/* make sure output active for echo */
	}
	putq(tp->rq, mp);
}/*------------------------------------------------------------------*/
/*
 *		READ SERVICE
 *
 *------------------------------------------------------------------*
 *
 * tym_rsrv -- read service routine
 */
tym_rsrv(rq)
register queue_t	*rq;
{
	register mblk_t		*bp;
	register tym_tty_t	*tp;

	tp = (tym_tty_t *)rq->q_ptr;
	ASSERT(tp != NULL);
	BUGLOG1(tp, BUG_LGEN, "tym_rsrv rq=%x", rq);
	if (tp->rq_stat != NULL)	/* general read stats*/
		tp->rq_stat->ms_scnt++;	/* count of read service calls */

	while ((bp = getq(rq)) != NULL)
	{
		if (bp->b_datap->db_type != M_DATA)
		{
			BUGLOG2(tp, BUG_LERR,"tym_rsrv unknown msg rq=%x bp=%x",
								   rq,   bp);
			putnext(rq, bp);	/* should never get here */
			continue;
		}
		if (!canput(rq->q_next))
		{
			putbq(rq, bp);
			return;
		}
		if (tp->termio.c_lflag & ICANON)
			bp = tym_rsrv_canon(bp, tp);	/* process cooked msg */
		else	bp = tym_rsrv_raw(bp, tp);	/* process raw msg */
		if (bp != NULL)
		{
			putbq(rq, bp);
			return;
		}
	} /* while ((bp = getq(rq)) != NULL) */
	if (tp->termio.c_lflag & ICANON)
		tym_rsrv_canon_wrapup(tp);
	else if (tp->raw_message != NULL)
		tym_check_raw_state(tp);	/* see what raw mode we are in*/
}/*------------------------------------------------------------------*/
/* 
 *		READ SRV - DATA - ICANON NOT SET (RAW)
 *
 *	this routine is called when we receive some raw data from downstream.
 *	as messages are received, we try to copy the data into larger
 *	message blocks to preserve resources
 *
 *	then, depending on:
 *	1. the number of bytes assembled,
 *	2. the setting of V_MIN and V_TIME, and
 *	3. whether or not a M_READ msg has been received
 *	try to send the data upstream immediately, or start a timer,
 *	then try to send the data up when the timer expires
 *------------------------------------------------------------------*
 */
mblk_t *
tym_rsrv_raw(mp, tp)
register mblk_t		*mp;
register tym_tty_t	*tp;
{
	register	count;
	mblk_t		*bp;		/* current msg block in received msg */
	mblk_t		*bpr;		/* last msg in raw msg chain */
	mblk_t		*bp_temp, *bcont;
	int	bytes_in_msg;		/* no. of bytes in received msg */
	int	room_left;		/* room left for data in new msg */
	int	bytes_to_move, bytes_2_ask_4; 

	BUGLOG2(tp,BUG_LGEN,"tym_rsrv_raw rq=%x mp=%x", tp->rq, mp);

	if ((count = msgdsize(mp)) == 0)		/* count bytes in msg */
	{
		freemsg(mp);
		return(NULL);
	}
	if (tp->raw_count > MAX_INPUT)		/* max to accept in raw mode */
		return(mp);			/* flag "can't process" */

	/* update statistics */
	sysinfo.rawch += count;
	tp->pv_rstat.rawch += count;
	tp->raw_count += count;

	/* try to copy raw data into bigger msgs if possible to save
	 * smaller msg blocks for other uses */
	bcont = mp;
	while ((bp = bcont) != NULL)
	{
		bcont = bp->b_cont;
		bytes_in_msg = bp->b_wptr - bp->b_rptr;
		if (tp->raw_message == NULL && bytes_in_msg < RAW_BUF_SIZE)
			tp->raw_message = allocb(RAW_BUF_SIZE, BPRI_MED);
		if ((bpr = tp->raw_message) == NULL)
		{
			tp->raw_message = bp;	/*if allocb fails, use rcvd bp*/
			continue;
		}
		/* get the last msg block in the chain we already have on hand*/
		for ( ; bpr->b_cont != NULL; bpr = bpr->b_cont)
			;

		while (bytes_in_msg != 0)
		{
			room_left = bpr->b_datap->db_lim - bpr->b_wptr;
			if (room_left == 0)
			{
				bytes_2_ask_4 = max(bytes_in_msg, RAW_BUF_SIZE);
				bp_temp = allocb(bytes_2_ask_4, BPRI_MED);
				if (bp_temp != NULL)
				{
					/* attach empty msg to end of chain */
					bpr->b_cont = bp_temp;
					bpr = bp_temp;
					room_left = bpr->b_datap->db_lim - bpr->b_wptr;
				}
				else	/* no new msg available, use recv'd bp*/
				{
					/* if we can't copy the data into last
					 * msg block in existing raw_message
					 * chain, then just attach bp to chain*/
					bpr->b_cont = bp;
					goto end_for_loop;
				}
			}
			bytes_to_move = min(room_left, bytes_in_msg);
			bcopy(bp->b_rptr, bpr->b_wptr, bytes_to_move);
			bp->b_rptr += bytes_to_move;
			bpr->b_wptr += bytes_to_move;
			bytes_in_msg -= bytes_to_move;
		}
		freeb(bp);
end_for_loop: ;
	}
	return(NULL);
}/*------------------------------------------------------------------*/
/*
 *	we get here when either a M_READ message is received from upstream
 *	(indicating the user has done a READ system call), or when
 *	a data message is received from downstream and ICANON is not set
 *	and a M_READ message has been previously received
 *
 *	this routine will send data upstream only if the VMIN/VTIME
 *	criteria is satisfied 
 *
 *	see termio(BA_ENV) for detailed definition of cases 1 thru 4
 *	   1. MIN > 0, TIME > 0
 *	      TIME = interval to wait between characters
 *	      start timer after 1st character, restart each character
 *	      send data upstream if TIME elapses or VMIN reached
 *	   2. MIN > 0, TIME = 0
 *	      no timer involved
 *	      send data upstream when at least VMIN characters received
 *	   3. MIN = 0, TIME > 0
 *	      start timer when M_READ msg received from upstream
 *	      send data upstream if TIME elapses or first char received
 *	   3. MIN = 0, TIME = 0
 *	      no timer involved
 *	      send data upstream when 1st char received or when M_READ received
 */
tym_check_raw_state(tp)
register tym_tty_t	*tp;
{
	termio_t *tptr = &tp->termio;

	if (tptr->c_cc[VMIN] != 0 && tptr->c_cc[VTIME] != 0)
	{	/* CASE 1, MIN > 0, TIME > 0 */
		/* start timer once the first character is received */
		if (tp->raw_count == 0)
			return;
		/* we need at least VMIN bytes to send them upstream */
		if (tp->raw_count < tptr->c_cc[VMIN])
		{	/* less than VMIN bytes, restart timer which,
			   in this case, is an inter-character timer */
			/* if RAW timer running, turn off before on*/
			if (tp->timer_enable & RAW_TIMER)
				untimeout(tp->raw_id);
			/*flag 'RAW timer on'*/
			tp->timer_enable |= RAW_TIMER;
			tp->raw_id = timeout(tym_send_raw_up, tp->rq,
					(tptr->c_cc[VTIME]*HZ)/10);
			return;
		}
		else	tym_send_raw_up(tp->rq);
	}
	else if (tptr->c_cc[VMIN] != 0 && tptr->c_cc[VTIME] == 0)
	{	/* CASE 2, MIN > 0, TIME = 0 */
		/* we need at least VMIN bytes to send them upstream */
		if (tp->raw_count >= tptr->c_cc[VMIN])
			tym_send_raw_up(tp->rq);
	}
	else if (tptr->c_cc[VMIN] == 0 && tptr->c_cc[VTIME] != 0)
	{	/* CASE 3, MIN = 0, TIME > 0 */
		if (tp->raw_count == 0)
		{
			if (!(tp->timer_enable & RAW_TIMER))
			{
				tp->timer_enable |= RAW_TIMER;
				tp->raw_id = timeout(tym_send_raw_up, tp->rq,
						(tptr->c_cc[VTIME]*HZ)/10);
			}
			return;
		}
		else	tym_send_raw_up(tp->rq);
	}
		/* CASE 4, MIN = 0, TIME = 0	send what we have upstream */
	else	tym_send_raw_up(tp->rq);
}/*------------------------------------------------------------------*/
/*
 *	tym_send_raw_up - send data upstream in non-ICANON
 *
 *	we get here either when we have at least VMIN bytes to send, or
 *	when the inter-character timer or the M_READ timer has expired
 *
 *	If we have no bytes, we send up an empty message
 */
tym_send_raw_up(rq)
register queue_t	*rq;
{
	register tym_tty_t	*tp;
	register mblk_t		*bp;

	if ((tp = (tym_tty_t *)rq->q_ptr) == NULL)
		return;		/* port is closed or module was popped */

	/* if RAW timer running, turn off before restarting timer*/
	if (tp->timer_enable & RAW_TIMER)
	{
		untimeout(tp->raw_id);
		tp->timer_enable &= ~RAW_TIMER;  /*flag 'M_READ timer off'*/
	}
	if (tp->raw_message == NULL)
	{
		/* send up empty msg*/
		bp = allocb(1, BPRI_MED);
		if (bp == NULL)
		{
			BUGCIR_LOG(tp->min_dev,-1, 0, TYM_CIR_RAW_WRAP, 0);
			tp->state &= ~M_READ_RECVD;
			return;
		}
		BUGLOG2(tp, BUG_LGEN, "tym_send_raw_up msg=%x cnt=%d", bp, 0);
	}
	else
	{
		bp = tp->raw_message;
		tp->raw_message = NULL;
		BUGLOG2(tp, BUG_LGEN, "tym_send_raw_up msg=%x cnt=%d",
		  bp, tp->raw_count);
		tp->raw_count = 0;
	}
	tp->state &= ~M_READ_RECVD;
	qenable(rq);		/* start read of any data backed up on queue */
	if (!(canput(rq->q_next)))
	{
		BUGCIR_LOG(tp->min_dev,-1, *bp->b_rptr, TYM_CIR_RAW_WRAP, bp);
		freemsg(bp);
		return;
	}
	putnext(rq, bp);
}/*------------------------------------------------------------------*/
/*
 *		READ SERVICE - DATA - ICANON SET
 *
 *------------------------------------------------------------------*
 *
 * tym_rsrv_canon -- read service routine for M_DATA if ICANON set
 */
tym_qenable(q)
queue_t *q;
{
	qenable(q);
}

mblk_t *
tym_rsrv_canon(mp, tp)
register mblk_t		*mp;
register tym_tty_t	*tp;
{
	register mblk_t		*bp, *next;
	register unchar *rptr;		/* next place to read char from */
	register last_was_escape;	/* non-zero = last char was '\' */
	int	count;			/* no. bytes in canon buffer */
	unchar	*char_tab_ptr;		/* addr of input char table this tty*/
	unchar	c;			/* next character to process */
	int	c_type;			/* type of byte in message */ 

	BUGLOG2(tp, BUG_LGEN, "tym_rsrv_canon rq=%x mp=%x", tp->rq, mp);
	char_tab_ptr = tp->char_tab;	/* addr of char. process table,
					   indexed by value of char. */
	while (mp != NULL)
	{
		for (rptr = mp->b_rptr; rptr < mp->b_wptr; rptr++)
		{
			if (tp->state & END_OF_LINE)
			{	/* send what we have in canon buf upstream*/
				count = tp->canptr - tp->canon_base;
				BUGLOG1(tp, BUG_LGEN,
					"tym_rsrv_canon count=%x", count);
				bp = tym_alloc(count, BPRI_MED, tym_qenable,tp->rq);
				if (bp == NULL)
				{ 	/* dont process next line until
					   last line is sent upstream */
					mp->b_rptr = rptr;
					return(mp);
				}
				else
				{	/* send line upstream and reset
					   canon buf pointer to start */
					if (count != 0)
						bcopy(tp->canon_base,bp->b_wptr,
									count);
					bp->b_wptr += count;
					putnext(tp->rq, bp);
					tp->canptr = tp->canon_base;
					tp->state &= ~END_OF_LINE;
				}
			}
			last_was_escape = tp->state & LCLESC;
			tp->state	&= ~LCLESC;
			c	= *rptr;
			c_type = *(char_tab_ptr + c);
			switch (c_type)
			{
			default:
				break;
			case TYM_NEWLINE_1:	/*---new-line && (iflag&INLCR)*/
			case TYM_NEWLINE_2:	/*---new-line &&!(iflag&INLCR)*/
			case TYM_EOL:		/*--- ^d ---*/
			case TYM_EOL2:
				tp->state |= END_OF_LINE;
				break;
			case TYM_ERASE	:	/*--- ^h ---*/
				if (last_was_escape)
				{
					last_was_escape = 0;
					break;	/* store cc[] but not escape*/
				}
				else	/* last char was not an escape char */
				{
					if (tp->canptr > tp->canon_base)
						tp->canptr--;
					continue;
				}
			case TYM_KILL:		/*--- ^u ---*/
				if (last_was_escape)
				{
					last_was_escape = 0;
					break;
				}
				else	/* last char was not an escape char */
				{
					tp->canptr = tp->canon_base;
					continue;   /* dont store in canon buf*/
				}
			case TYM_EOF:
				if (last_was_escape)
				{
					last_was_escape = 0;
					break;
				}
				else	/* last char was not an escape char */
				{
					tp->state |= END_OF_LINE;
					continue;   /* dont store in canon buf*/
				}
			case TYM_0x5c:		/*--- \\  ->  \ ---*/
				/* by the time we get here, the only escapes (\)
				   left in the data stream are those which do
				   nothing, or those that precede KILL, ERASE,
				   or EOF */
				tp->state |= LCLESC;
				if (last_was_escape)
				{	/* store the previous escape */
					last_was_escape = 0;
					break;
				}
				else	continue;
			} /* switch (c_type) */
			if (last_was_escape)
			{
				*(tp->canptr++) = '\\';
				if (tp->canptr > (tp->canon_base + MAX_CANON))
					tp->canptr--;
			}
			*(tp->canptr++) = c;
			if (tp->canptr > (tp->canon_base + MAX_CANON))
				tp->canptr--;
		} /* for (rptr = mp->b_rptr; rptr < mp->b_wptr; rptr++) */

		/* loop on chained msg blocks within each message */
		next = mp->b_cont;
		mp->b_cont = NULL;
		freemsg(mp);
		mp = next;
	} /* while (mp != NULL) */
	return(NULL);
}/*------------------------------------------------------------------*/
/*
 * tym_rsrv_canon_wrapup -- all M_DATA processed and ICANOB set
 */
tym_rsrv_canon_wrapup(tp)
register tym_tty_t	*tp;
{
	register mblk_t		*bp;
	int count;			/* no. bytes in canon buffer */

	/* we get here after all messages processed */
	count = tp->canptr - tp->canon_base;
	BUGLOG2(tp,BUG_LGEN,"tym_rsrv_canon_wrapup rq=%x cnt=%x",tp->rq,count);
	if (tp->state & END_OF_LINE)	/* was last byte end of line? */
	{	/* send what we have in canon buf upstream*/
		if ((bp = tym_alloc(count,BPRI_MED,tym_qenable,tp->rq)) != NULL)
		{	/* send line upstream and reset canon buf ptr to start*/
			if (count != 0)
			{
				bcopy(tp->canon_base,bp->b_wptr, count);
				bp->b_wptr += count;
			}
			tp->pv_rstat.canch += count;
			sysinfo.canch += count;
			BUGCIR_LOG(tp->min_dev, count, *bp->b_rptr,
					TYM_CIR_CANON_WRAP, (int)bp);
			putnext(tp->rq, bp);
			tp->canptr = tp->canon_base;
			tp->state &= ~END_OF_LINE;
		}
	}
}/*------------------------------------------------------------------*/
/*
 *		WRITE PUT
 *
 *------------------------------------------------------------------*
 *
 * tym_wput -- write put routine
 *
 *	processes M_FLUSH, M_READ, and M_IOCTL
 *	pass other high priority messages downstream
 *	pass low priority messages to write service routine
 *
 * Arguments:
 *	wq	write queue pointer
 *	mp	message pointer
 */
tym_wput(wq, mp)
register queue_t *wq;
register mblk_t	*mp;
{
	register tym_tty_t	*tp;
	register type;
	termio_t *tptr;

	tp = (tym_tty_t *)wq->q_ptr;
	ASSERT(tp != NULL);
	if (tp == NULL)
	{
		freemsg(mp);
		return;		/* port is not open */
	}
	if (tp->wq_stat != NULL)	/* general write stats*/
		tp->wq_stat->ms_pcnt++;	/* count of write put calls */
	if (mp == NULL)
	{
		cmn_err(CE_NOTE, "NULL mp received in tym_wput, q = %x", wq);
		return;
	}
	tptr = &tp->termio;
	type = mp->b_datap->db_type;
	BUGLOG3(tp, BUG_LPUT, "tym_wput wq=%x mp=%x type=%x", wq, mp, type);
	switch (type)
	{
	default:
		BUGCIR_LOG(tp->min_dev,type,*mp->b_rptr,TYM_CIR_WRITE,(int)mp);
		putnext(wq, mp);  /* pass what we don't understand downstream*/
		break;
	case M_DATA:
		BUGCIR_LOG(tp->min_dev, mp->b_wptr - mp->b_rptr,
	 				*mp->b_rptr,TYM_CIR_WRITE_DATA,(int)mp);
		if (canput(wq->q_next)		/* we can send data downstream*/
		 && tp->canon_xcase == 0	/*no ICANON nor XCASE in lflag*/
		 && tp->oflag_type == 0		/*no OPOST processing */
		 && wq->q_first == NULL)	/*no previos msgs on queue */
			putnext(wq, mp);
		else	putq(wq, mp);	/* pass data to service routine */
		break;
	case M_READ:
		/* if M_READ timer running, turn off before restarting timer*/
		BUGCIR_LOG(tp->min_dev, tptr->c_cc[VMIN], tptr->c_cc[VTIME],
					 TYM_CIR_WRITE_MREAD, tptr->c_lflag);
		if (!(tp->termio.c_lflag & ICANON))
		{
			tp->state |= M_READ_RECVD;	/* user issued read */
			tp->m_read_request = *(uint *)mp->b_rptr;
			tym_check_raw_state(tp);
		}
		putnext(wq, mp); /* send downstream in case other module needs*/
		break;
	case M_FLUSH:
		BUGCIR_LOG(tp->min_dev,0,*mp->b_rptr,TYM_CIR_WRITE_FLUSH,mp);
		if (*mp->b_rptr & FLUSHW)
			tym_flush_write_side(tp);
		if (*mp->b_rptr & FLUSHR)
			tym_flush_read_side(tp);
		putnext(wq, mp);
		break;
	case M_IOCTL:
		tym_wput_ioc(wq, mp, tp);
		break;
	} /* switch (type) */
}/*------------------------------------------------------------------*/
/*
 *		WRITE PUT - IOCTL
 *
 *------------------------------------------------------------------*/
tym_wput_ioc(wq, mp, tp)
register queue_t	*wq;
register mblk_t		*mp;
register tym_tty_t	*tp;
{
	register struct iocblk *iocbp;
	int	cmd;		/* ioctl command */
	unchar	arg;
	
	iocbp = (struct iocblk *)mp->b_rptr;
	cmd = iocbp->ioc_cmd;
	BUGLOG3(tp, BUG_LIOC, "tym_wput_ioc wq=%x mp=%x cmd=%x", wq, mp, cmd);

	/* b_cont could be NULL if the user sends down an ioctl
	 * via the I_STR ioctl command, and fails to set the
	 * address of the data (ic_dp)
	 */
	if (mp->b_cont != NULL)
		arg = *(int *)mp->b_cont->b_rptr;
	else	arg = 0;
	BUGCIR_LOG(tp->min_dev, cmd, arg, TYM_CIR_WRITE_IOCTL, mp);
	switch(cmd)
	{
	default:
		putnext(wq, mp);	/* pass all other ioctls downstream */
		break;
	case TCXONC:
		if (mp->b_cont == NULL)
			goto nak_ioctl;
		switch (arg)
		{
		default:
nak_ioctl:
			mp->b_datap->db_type = M_IOCNAK;
			iocbp->ioc_error = EINVAL;
			qreply(wq, mp);
			return;
		case 0:		/* T_SUSPEND */
			tym_send_stop_down(wq);
			mp->b_datap->db_type = M_IOCACK;
			iocbp->ioc_count = 0;
			qreply(wq, mp);
			return;
		case 1:		/* T_RESUME */
			tym_send_start_down(wq);
			mp->b_datap->db_type = M_IOCACK;
			iocbp->ioc_count = 0;
			qreply(wq, mp);
			return;
		case 2:		/* T_BLOCK */
		case 3:		/* T_UNBLOCK */
			/* let driver handle case 2 and 3 - stop/start input */
			break;
		}
		putnext(wq, mp);
		break;
	case TCFLSH:		/* if valid arg, convert TCFLSH to M_FLUSH */
		if (mp->b_cont == NULL)
			goto nak_ioctl;
		mp->b_datap->db_type = M_IOCACK;	/* assume arg ok */
		switch (arg)
		{
		default:
			goto nak_ioctl;
		case 0:
			tym_flush_read_side(tp);	/* flush read side */
			putctl1(wq->q_next, M_FLUSH, FLUSHR);
			break;
		case 1:
			putctl1(tp->rq->q_next, M_FLUSH, FLUSHW);
			break;
		case 2:
			tym_flush_read_side(tp);	/* flush read side */
			putctl1(tp->rq->q_next, M_FLUSH, FLUSHW);
			putctl1(wq->q_next, M_FLUSH, FLUSHR);
			break;
		} /* switch (*(int *)mp->b_cont->b_rptr) */
		tym_delay_tcflsh_ack(tp, mp);
		break;
	case TCSETA:
		if (mp->b_cont == NULL)
			goto nak_ioctl;
		tym_set_newioctl(mp, tp);
		putnext(wq, mp);
		break;
	case TCSETAF:
	case TCSETAW:
	case TIOCSETP:
		if (mp->b_cont == NULL)
			goto nak_ioctl;
		else	putq(wq, mp);
		break;
	} /* switch(cmd) */
}/*------------------------------------------------------------------*/
/*
 *	wait a bit before acking TCFLSH command
 *
 *	when we receive the TCFLSH ioctl, we send:
 *	- a M_FLUSH with FLUSHR downstream for the driver to send upstream
 *	  to flush the read side
 *	- a M_FLUSH with FLUSHW upstream for the stream head to send downstream
 *	  to flush the write side
 *
 *	if we ack the TCFLSH ioctl immediately, we allow the user to do
 *	other stuff before the flushing action is complete;
 *	in particular, if the user immediately sends down another msg,
 *	that msg could get flushed
 *
 *	so this solution is to wait a bit before acking the TCFLSH ioctl,
 *	thus allowing time for the flushing action to complete
 */
tym_ack_tcflsh(tp)
register tym_tty_t	*tp;
{
	register mblk_t		*bp;
	register struct iocblk	*iocbp;

	if ((bp = tp->tcflsh_mp) != NULL)
	{
		tp->tcflsh_mp = NULL;
		iocbp = (struct iocblk *)bp->b_rptr;
		iocbp->ioc_count = 0;
		qreply(tp->wq, bp);
	}
}/*------------------------------------------------------------------*/
tym_delay_tcflsh_ack(tp, mp)
register tym_tty_t	*tp;
register mblk_t		*mp;
{
	tp->tcflsh_mp = mp;
	timeout(tym_ack_tcflsh, tp, 3);	/* wait 3 clock ticks to ack TCFLSH */
}/*------------------------------------------------------------------*/
/*
 *		WRITE SERVICE
 *
 *------------------------------------------------------------------*
 *
 * tym_wsrv -- write service routine
 *
 * process M_DATA  and M_IOCTL messages from upstream
 *
 */
tym_wsrv(wq)
register queue_t	*wq;	/* write queue pointer */
{
	register mblk_t		*bp;
	register tym_tty_t	*tp;
	register type;

	tp = (tym_tty_t *)wq->q_ptr;
	ASSERT(tp != NULL);
	if (tp->wq_stat != NULL)	/* general write stats*/
		tp->wq_stat->ms_scnt++;	/* count of write service calls */
	if ((bp = tp->echobuf) != NULL && canput(wq))
	{
		/* we have one or more echo msgs we can send downstream */

		BUGCIR_LOG(tp->min_dev, 0, *bp->b_rptr, TYM_CIR_ECHO, bp);
		if (tp->canon_xcase == 0)
		{	/* no XCASE (ICANON must be set to echo)*/
			if (tp->oflag_type == 0)
			{	/* no OPOST or OPOST only in oflag */
				/* no output processing required */

				int count = bp->b_wptr - bp->b_rptr;

				sysinfo.outch += count;
				tp->pv_wstat.postch += count;
				tp->echobuf = NULL;  /* flag 'msg processed' */
				putnext(wq, bp);
			}
			else	tp->echobuf = tym_opost_only(bp, tp);
		}
		else		/* ICANON and XCASE set */
		{
			if (tp->oflag_type == 0)
				tp->echobuf =tym_canon_xcase_only(     bp,tp,1);
			else	tp->echobuf =tym_canon_xcase_and_opost(bp,tp,1);
		}
	}
	while ((bp = getq(wq)) != NULL)
	{
		type = bp->b_datap->db_type;
		BUGLOG2(tp, BUG_LGEN, "tym_wsrv wq=%x, type=%x", wq, type);
		BUGCIR_LOG(tp->min_dev, 0, *bp->b_rptr, TYM_CIR_WSRV, bp);
		switch (type)
		{
		default:
			BUGLOG2(tp, BUG_LERR,"tym_wsrv unknown msg wq=%x bp=%x",
								   wq,   bp);
			putnext(wq, bp);	/*should never get here */
			break;
		case M_IOCTL:
			tym_wsrv_ioc(wq, bp, tp);
			break;
		case M_DATA:
			if (!(canput(wq->q_next)) || tp->state & TTSTOP)
			{
				BUGCIR_LOG(tp->min_dev,0,0,TYM_CIR_WCANPUT,wq);
				putbq(wq, bp);
				return;		/* 'unable to process data */
			}
 			/*	 there are 4 possibilities:
			 *   1. ICANON & XCASE are not set, OPOST is not set
			 *   2. ICANON & XCASE are set,     OPOST is not set
			 *   3. ICANON & XCASE are not set, OPOST is set
			 *   4. ICANON & XCASE are set,     OPOST is set
			 * (note: 'OPOST set' means at OPOST and at least
			 * 	  one other bit in oflag is set)
			 */
			if (tp->canon_xcase == 0) /* no ICANON or XCASE */
			{
				if (tp->oflag_type == 0)
				{	/* no OPOST or OPOST only in oflag */
					/* no output processing required */

					int count = bp->b_wptr - bp->b_rptr;

					sysinfo.outch += count;
					tp->pv_wstat.outch += count;
					tp->pv_wstat.postch += count;
					putnext(wq, bp);
					bp = NULL;  /* flag 'msg processed' */
				}
				else	bp = tym_opost_only(bp, tp);
			}
			else		/* ICANON and XCASE set */
			{
				if (tp->oflag_type == 0)
					bp = tym_canon_xcase_only(     bp,tp,0);
				else	bp = tym_canon_xcase_and_opost(bp,tp,0);
			}
			if (bp != NULL)
			{
				putbq(wq, bp);
				return;	/* unable to process any more data */
			}
			else	break;
		} /* switch (type) */
	} /* while ((bp = getq(wq)) != NULL) */
}/*------------------------------------------------------------------*/
/*
 *		WRITE SERVICE - DATA - OPOST
 *
 *------------------------------------------------------------------*
 *
 *	tym_opost_only
 *
 *	process output data with OPOST set, but ICANON and XCASE not set
 *
 *	return: NULL if entire message processed
 *		mp if part or all of message is to be put back on the queue
 *		   (mp points to first msg block to put back on queue)
 *
 */
mblk_t *
tym_opost_only(mp, tp)
register mblk_t		*mp;
register tym_tty_t	*tp;
{
	mblk_t	*next;			/* addr of next msg block in msg */
	mblk_t	*bp;
	register mblk_t *omp;		/*addr of output message */
	register unchar	*rptr;		/* next byte in msg to get */
	register unchar	*wptr;		/* last byte + 1 in msg */
	register unchar	*owptr;		/*next place to put byte in output buf*/
	register oflag;			/* output processing flag */
	unchar	*owend;			/* max end of output buffer */
	int	size;		/* no. bytes needed beyond bytes left in msg */
	int	count;			/* no. bytes to fill */
	int	delay;			/* no. milliseconds to delay */
	int i, tmp;
	register unchar c, c_type;		/* next byte in message */

	BUGLOG2(tp, BUG_LGEN, "tym_opost_only mp=%x tp=%x", mp, tp);
	oflag	= tp->termio.c_oflag;	/* output processing flag */
	omp	= NULL;		/*addr of output message */
	owend = owptr = NULL;	/* do so 'lint' doesn't complain */
	size	= 0;		/* no. bytes needed beyond bytes left in msg */
	count	= 0;		/* no. bytes to fill */
	delay	= 0;		/* no. milliseconds to delay */
more:
	rptr = mp->b_rptr;
	wptr = mp->b_wptr;
	sysinfo.outch += (wptr - rptr);
	tp->pv_wstat.outch += (wptr - rptr);
	while (rptr < wptr || count != 0)
	{
		/* if no room in output buf, get next output buffer */
		if (omp == NULL || owptr >= owend)
		{
			if (omp != NULL)   /* send previous output buf*/
			{
				omp->b_wptr = owptr; 
				if (omp->b_wptr == omp->b_rptr)
					freemsg(omp);
				else
				{
					BUGCIR_LOG(tp->min_dev,
						omp->b_wptr - omp->b_rptr,
		 				*omp->b_rptr,TYM_CIR_OPOST,omp);
					tp->pv_wstat.postch +=
						omp->b_wptr - omp->b_rptr;
					putnext(tp->wq, omp);
				}
			}
			/* get new output buffer */
			omp = tym_alloc((int)(wptr - rptr) + size,
					BPRI_MED, qenable, tp->wq);
			BUGLOG3(tp, BUG_LGEN,
			     "tym_opost_only alloc omp=%x wptr-rptr=%x size=%x",
						   omp,   wptr-rptr,   size); 
			if (omp == NULL)
			{	/* no buffer available, wait awhile */
				mp->b_rptr = rptr;
				return(mp);
			}
			/* got new buffer, set up first loc to store into */
			owptr = omp->b_wptr;
			/* set up last place + 1 we can store into */
			owend = (unchar *)(omp->b_datap->db_lim);
			size = 0;
			if (count != 0)
				goto fill;    /*move fill chars to output buf*/
		} /* if (owptr >= owend) */

		c = *rptr++;	/* Get next byte from input msg block */
		c_type = tym_outchar_tab[c];	/* get character type */
		if (tym_debug_opost != 0)
		{
			BUGLOG2(tp, BUG_LGEN, "tym_opost_only c=%x c_type=%x",
							      c,   c_type);
		}
		switch (c_type)
		{
		default:		/* default value of c_type is 0 */
			tp->col++;	/* bump the column position */
			break;
		case TYM_Oa_z:	/*---Map lower to upper case if OLCUC set---*/
			if (oflag & OLCUC)
				c -= 'a' - 'A';
			tp->col++;	/* bump the column position */
			break;
		case TYM_ONOP:		/*---dont bump col---*/
			break;
		case TYM_Ox08:		/*--- Backspace ---*/
			if (tp->col != 0)	/* if not in column 1 */
				tp->col--;	/* then move back one col */
			if ((oflag & BSDLY) == BS1)
			{
				if (oflag & OFILL)
					count = 1;    /*fill 1 char*/
				else	delay = 50;   /*delay 50 mills*/
			}
			break;
		case TYM_Ox0a:		/*--- line feed ---*/
			if (oflag & ONLRET)
				goto cr;	/* go do the function of cr */
			if (oflag & ONLCR)
			{
			/* convert '\n' to '\r \n' */
				if (!(tp->col == 0 && oflag & ONOCR))
				{
					/* output '\r' if room */
					if ((int)(owend - owptr) < 2)
					{     /*not room in output for 2 bytes*/
						owend = owptr;
						size = 2;
						rptr--;	/*leave \n unprocessed*/
						continue;
					}
					*owptr++ = '\r';
				}
				goto cr;
			}
nl:
			if ((oflag & NLDLY) == NL1)
			{
				if (oflag & OFILL)
					count = 2;	/*fill 2 chars*/
				else	delay = 100; /*delay 100 mills*/
			}
			tp->col++;	/* bump the column position */
			break;
		case TYM_Ox0d:
			if (oflag & OCRNL)
			{
				c = '\n';	/* map \r to \n */
				goto nl;	/*process as line feed*/
			}
cr:
			/* Ignore <CR> in column 0 if ONOCR flag set*/
			if (tp->col == 0 && oflag & ONOCR)
				continue;
			if (oflag & OFILL)
			{
				if ((oflag & CRDLY) == CR1)
					count = 2;	/*fill 2 chars*/
				else if ((oflag & CRDLY) == CR2)
					count = 4;	/*fill 4 chars*/
				else if ((oflag & CRDLY) == CR3)
					count = 6;	/*fill 6 chars*/
			}
			else
			{
				if ((oflag & CRDLY) == CR1)
					delay = max((tp->col<<3)+150,300);
				else if ((oflag & CRDLY) == CR2)
					delay = 100;  /*delay 100 mill*/
				else if ((oflag & CRDLY) == CR3)
					delay = 150;	/* delay 150 mills */
			}
			tp->col = 0;
			break;
		case TYM_Ox09:		/*--- tab ---*/
			tmp = 8 - (tp->col & 07);
			/* Map \t to spaces if TAB3 set, ignore OFILL*/
			if ((oflag & TABDLY) == TAB3)
			{
				if (tmp > (owend - owptr))
				{
					/*no room in output for spaces*/
					owend = owptr;
					rptr--;
					size = tmp;
					continue;
				}
				/* replace tab with 0 to 7 spaces */
				for (i = 0; i < tmp; i++)
					*owptr++ = ' ';
				tp->col += tmp;
				continue;
			}
			/* we get here if TABDLY != TAB3 */
			if (oflag & OFILL)
			{
				if ((oflag & TABDLY) == TAB1 ||
				    (oflag & TABDLY) == TAB2)
					count = 2;	/* fill 2 chars */
			}
			else if ((oflag & TABDLY) == TAB1)
				delay = tmp * 50;  /* delay 50 mills each col*/
			else if ((oflag & TABDLY) == TAB2)
				delay = 100;	/* delay 100 mills */
			tp->col += tmp;		/* point to next tab position */
			break;
		case TYM_Ox0b:		/*--- vertical tab ---*/
 			if ((oflag & VTDLY) == VT1)
			{
				if (oflag & OFILL)
					count = 4;	/* fill 4 chars */
				else	delay = 2000;	/* delay 2000 mills */
			}
			break;
		case TYM_Ox0c:		/*--- form feed ---*/
			if ((oflag & FFDLY) == FF1)
			{
				if (oflag & OFILL)
					count = 4;	/*fill 4 chars*/
				else	delay = 2000;	/* delay 2 sec*/
			}
			break;
		} /* switch (c_type) */

		*owptr++ = c;	/* store byte in output buffer*/
fill:
		if (count != 0)		/* one or more fill characters needed */
		{
			if (count > (int)(owend - owptr))
			{
			/* leave count non-zero so when we get the next output
			 * buffer, we jump right to 'fill'
			 */
				size = count;
				owend = owptr;
				continue;
			}
			if (oflag & OFDEL)
				c = 0x7f;
			else	c = CNUL;
			for (; count != 0; count--)
				*owptr++ = c;
		}
		else if (delay != 0)
		{
			omp->b_wptr = owptr; 
			BUGCIR_LOG(tp->min_dev, omp->b_wptr - omp->b_rptr,
	 				*omp->b_rptr,TYM_CIR_OPOST,(int)omp);
			tp->pv_wstat.postch += omp->b_wptr - omp->b_rptr;
			putnext(tp->wq, omp);	/*send data we have downstream*/
			/* now tell driver to delay after last byte of
			   previous message output to terminal */
			BUGLOG0(tp, BUG_LGEN, "tym_opost_only M_DELAY");
			if ((bp = allocb(4, BPRI_HI)) != NULL)
			{
				bp->b_datap->db_type = M_DELAY;
				/* convert milliseconds to machine ticks */
				*((int *)bp->b_wptr) = (delay * HZ)/1000;
				putnext(tp->wq, bp);
			}
			omp = NULL;
			delay = 0;
		}
	} /* while (rptr < wptr) */
	/* free up message block as soon as it is processed */
	next = mp->b_cont;
	mp->b_cont = NULL;
	BUGLOG1(tp, BUG_LGEN, "tym_opost_only freeb call mp=%x", mp);
	freeb(mp);
	if ((mp = next) != NULL)
	{
		BUGLOG1(tp, BUG_LGEN, "tym_opost_only more mp=%x", mp);
		goto more;	/* process next message block in msg */
	}
	if (omp != NULL)
	{
		omp->b_wptr = owptr;
		BUGCIR_LOG(tp->min_dev, omp->b_wptr - omp->b_rptr,
				*omp->b_rptr,TYM_CIR_OPOST,(int)omp);
		tp->pv_wstat.postch += omp->b_wptr - omp->b_rptr;
		putnext(tp->wq, omp);
	}
	return(NULL);
}/*------------------------------------------------------------------*/
/*
 *		WRITE SERVICE - DATA - CANON & XCASE
 *
 *------------------------------------------------------------------*
 *
 *	tym_canon_xcase_only
 *
 *	process output data with ICANON and XCASE set, but OPOST not set
 *
 *	return: NULL if entire message processed
 *		mp if part or all of the message put back on the queue
 *
 */
mblk_t *
tym_canon_xcase_only(mp, tp, data_is_echo)
register mblk_t		*mp;
register tym_tty_t	*tp;
int	data_is_echo;			/* if non-zero, data is from echobuf */
{
	mblk_t	*next;			/* addr of next msg block in msg */
	register mblk_t *omp;		/*addr of output message */
	register unchar	*rptr;		/* next byte in msg to get */
	register unchar	*wptr;		/* last byte + 1 in msg */
	register unchar	*owptr;		/*next place to put byte in output buf*/
	unchar		*owend;		/* max end of output buffer */
	int	size;		/* no. bytes needed beyond bytes left in msg */
	unchar	c;			/* next byte in message */
	int	c_type;			/* type of byte in message */ 

	BUGLOG3(tp, BUG_LGEN, "tym_canon_xcase_only mp=%x tp=%x echo=%x",
						    mp,   tp, data_is_echo);
	omp	= NULL;		/*addr of output message */
	owend = owptr = NULL;	/* do so 'lint' doesn't complain */
	size	= 0;		/* no. bytes needed beyond bytes left in msg */
more:
	rptr = mp->b_rptr;
	wptr = mp->b_wptr;
	sysinfo.outch += (wptr - rptr);
	tp->pv_wstat.outch += (wptr - rptr);
	while (rptr < wptr)
	{
		if (omp == NULL || owptr >= owend)
		/* if no room in output buffer, get new output buf */
		{
			if (omp != NULL)   /* send previous output buf*/
			{
				omp->b_wptr = owptr;
				if (omp->b_wptr == omp->b_rptr)
					freemsg(omp);
				else
				{
					BUGCIR_LOG(tp->min_dev,
						omp->b_wptr - omp->b_rptr,
	 					*omp->b_rptr,TYM_CIR_XCASE,omp);
					tp->pv_wstat.postch += 
						omp->b_wptr - omp->b_rptr;
					putnext(tp->wq, omp);
				}
			}
			/* now get new output buffer */
			omp = tym_alloc((int)(wptr - rptr) + size,
					BPRI_MED, qenable, tp->wq);
			if (omp == NULL)
			{	/* no buffers available right now, wait awhile*/
				mp->b_rptr = rptr;
				return(mp);
			}
			/* we have a buffer, set up the next loc to store into*/
			owptr = omp->b_wptr;
			/* set up the last place + 1 we can store into */
			owend = (unchar *)(omp->b_datap->db_lim);
			size = 0;
		}
		c = *rptr++;	/* Get next byte from input msg block */
		c_type = tym_outchar_tab[c];	/* get character type */
		switch (c_type)
		{
		default:
			*owptr++ = c;	/* store byte in output buf */
			break;
		case TYM_Ox5c:		/*--- map '\' to "\\" ---*/
			if (data_is_echo)
			{	/* dont map \ to \\ if echoing \ */
				*owptr++ = c;	/* store byte in output buf */
				break;
			}
			/* else FALL THROUGH */
		case TYM_OA_Z:		/*--- precede with '\' ---*/
		case TYM_Ox60:		/*--- map '`' to "\'" ---*/
		case TYM_Ox7c:		/*--- map '|' to "\!" ---*/
		case TYM_Ox7e:		/*--- map '~' to "\^" ---*/
		case TYM_Ox7b:		/*--- map '{' to "\(" ---*/
		case TYM_Ox7d:		/*--- map '}' to "\)" ---*/
			if ((int)(owend - owptr) < 2)
			{	/* not room for 2 more bytes, get new buf */
				owend = owptr;	/*ask 4 new output buf*/
				size = 2;	/*get at least 2 bytes*/
				rptr--;	    /*point to same byte again*/
			}
			else
			{
				*owptr++ = '\\';
				*owptr++ = tym_xcase_out_tab[c];
			}
			break;
		} /* switch (c_type) */
	} /* while (rptr < wptr) */
	/* free up message block as soon as it is processed */
	next = mp->b_cont;
	mp->b_cont = NULL;
	freeb(mp);
	/* get next message block in message */
	if ((mp = next) != NULL)
		goto more;
	if (omp != NULL)
	{
		omp->b_wptr = owptr;
		BUGCIR_LOG(tp->min_dev, omp->b_wptr - omp->b_rptr,
				*omp->b_rptr,TYM_CIR_XCASE,omp);
		tp->pv_wstat.postch += omp->b_wptr - omp->b_rptr;
		putnext(tp->wq, omp);
	}
	return(NULL);
}/*------------------------------------------------------------------*/
/*
 *		WRITE SERVICE -	DATA - CANON/XCASE & OPOST
 *
 *------------------------------------------------------------------*
 *
 *	tym_canon_xcase_and_opost
 *
 *	process output data with OPOST, ICANON, and XCASE set
 *
 *	return: NULL if entire message processed
 *		mp if part or all of the message put back on the queue
 *
 */
mblk_t *
tym_canon_xcase_and_opost(mp, tp, data_is_echo)
register mblk_t		*mp;
register tym_tty_t	*tp;
int	data_is_echo;			/* if non-zero, data is from echobuf */
{
	mblk_t	*next;			/* addr of next msg block in msg */
	mblk_t	*bp;
	register mblk_t *omp;		/*addr of output message */
	register unchar	*rptr;		/* next byte in msg to get */
	register unchar	*wptr;		/* last byte + 1 in msg */
	register unchar	*owptr;		/*next place to put byte in output buf*/
	register oflag;			/* output processing flag */
	unchar	*owend;			/* max end of output buffer */
	int	size;		/* no. bytes needed beyond bytes left in msg */
	int	count;			/* no. bytes to fill */
	int	delay;			/* no. milliseconds to delay */
	unchar c;			/* next byte in message */
	int	c_type;			/* type of byte in message */ 
	int	i, tmp;

	BUGLOG3(tp, BUG_LGEN, "tym_canon_xcase_and_opost mp=%x tp=%x echo=%x",
							 mp, tp, data_is_echo);
	oflag	= tp->termio.c_oflag;	/* output processing flag */
	omp	= NULL;		/*addr of output message */
	owend = owptr = NULL;	/* do so 'lint' doesn't complain */
	size	= 0;		/* no. bytes needed beyond bytes left in msg */
	count	= 0;		/* no. bytes to fill */
more:
	rptr = mp->b_rptr;
	wptr = mp->b_wptr;
	sysinfo.outch += (wptr - rptr);
	tp->pv_wstat.outch += (wptr - rptr);
	while (rptr < wptr || count != 0)
	{
		/* if no room in output buf, get next output buffer */
		if (omp == NULL || owptr >= owend)
		{
			if (omp != NULL)	/* send previous output buf*/
			{
				omp->b_wptr = owptr; 
				if (omp->b_wptr == omp->b_rptr)
					freemsg(omp);
				else
				{
					BUGCIR_LOG(tp->min_dev,
						omp->b_wptr - omp->b_rptr,
		 				*omp->b_rptr,TYM_CIR_XPOST,omp);
					tp->pv_wstat.postch += 
						omp->b_wptr - omp->b_rptr;
					putnext(tp->wq, omp);
				}
			}
			omp = tym_alloc((int)(wptr - rptr) + size,
					BPRI_MED, qenable, tp->wq);
			if (omp == NULL)
			{
				mp->b_rptr = rptr;
				return(mp);
			}
			owptr = omp->b_wptr;
			/* set up the last place + 1 we can store into */
			owend = (unchar *)(omp->b_datap->db_lim);
			size = 0;
			if (count != 0)
				goto fill;
		} /* if (owptr >= owend) */

		c = *rptr++;	/* Get a character from the input msg block */
		c_type = tym_outchar_tab[c];
		switch (c_type)
		{
		default:
			tp->col++;	/* bump the column position */
			break;
		case TYM_Ox5c:		/*--- map '\' to "\\" ---*/
			if (data_is_echo)
			{	/* dont map \ to \\ if echoing \ */
				tp->col++;	/* bump the column position */
				break;
			}
			/* else FALL THROUGH */
		case TYM_OA_Z:		/*--- precede with '\' ---*/
		case TYM_Ox60:		/*--- map '`' to "\'" ---*/
		case TYM_Ox7c:		/*--- map '|' to "\!" ---*/
		case TYM_Ox7e:		/*--- map '~' to "\^" ---*/
		case TYM_Ox7b:		/*--- map '{' to "\(" ---*/
		case TYM_Ox7d:		/*--- map '}' to "\)" ---*/
			if ((int)(owend - owptr) < 2)
			{
				owend = owptr;	/* ask for new output buf*/
				size = 2;	/* get at least 2 bytes */
				rptr--;		/* point to same byte again*/
			}
			else
			{
				*owptr++ = '\\';
				*owptr++ = tym_xcase_out_tab[c];
			}
			continue;
		case TYM_Oa_z:	/*--- Map lower to upper case if OLCUC set---*/
			if (oflag & OLCUC)
				c -= 'a' - 'A';
			tp->col++;	/* bump the column position */
			break;
		case TYM_ONOP:	/*--- Non-printing chars, dont bump col pos---*/
			break;
		case TYM_Ox08:		/*--- Backspace ---*/
			if (tp->col != 0)
				tp->col--;	/* move back one column */
			if ((oflag & BSDLY) == BS1)
			{
				if (oflag & OFILL)
					count = 1;	/* fill 1 char */
				else	delay = 50;	/* delay 50 mills */
			}
			break;
		case TYM_Ox0a:		/*--- line feed ---*/
			if (oflag & ONLRET)
				goto cr;
			if (oflag & ONLCR)
			{
			/* convert '\n' to '\r\n' */
				if (!(tp->col == 0 && oflag & ONOCR))
				{
				/* output '\r' if room */
					if ((int)(owend - owptr) < 2)
					{
						owend = owptr;
						size = 2;
						rptr--;
						continue;
					}
					*owptr++ = '\r';
				}
				goto cr;
			}
nl:
			if ((oflag & NLDLY) == NL1)
			{
				if (oflag & OFILL)
					count = 2;	/* fill 2 chars */
				else	delay = 100;	/* delay 100 mills */
			}
			tp->col++;	/* bump the column position */
			break;
		case TYM_Ox0d:		/*--- carriage return ---*/
			if (oflag & OCRNL)
			{
				c = '\n';	/* map \r to /n */
				goto nl;	/* process as line feed*/
			}
cr:
		/* Ignore <CR> in column 0 if ONOCR flag set.	*/
			if (tp->col == 0 && oflag & ONOCR)
				continue;
			if (oflag & OFILL)
			{
				if ((oflag & CRDLY) == CR1)
					count = 2;	/* fill 2 chars */
				else if ((oflag & CRDLY) == CR2)
					count = 4;	/* fill 4 chars */
				else if ((oflag & CRDLY) == CR3)
					count = 6;	/* fill 6 chars */
			}
			else
			{
				if ((oflag & CRDLY) == CR1)
					delay = max((tp->col<<3) +150, 300);
				else if ((oflag & CRDLY) == CR2)
					delay = 100;	/* delay 100 mills */
				else if ((oflag & CRDLY) == CR3)
					delay = 150;	/* delay 150 mills */
			}
			tp->col = 0;
			break;
		case TYM_Ox09:		/*--- tab ---*/
			tmp = 8 - (tp->col & 07);
			/* Map '\t' to spaces if TAB3 flag set, ignore OFILL*/
			if ((oflag & TABDLY) == TAB3)
			{
				if (tmp > (int)(owend - owptr))
				{
					/*no room in output buf for tmp spaces*/
					owend = owptr;
					size = tmp;
					rptr--;
					continue;
				}
				for (i = 0; i < tmp; i++)
					*owptr++ = ' ';
				tp->col += tmp;
				continue;
			}
			/* we get here if TABDLY != TAB3 */
			if (oflag & OFILL)
			{
				if ((oflag & TABDLY) == TAB1 ||
				    (oflag & TABDLY) == TAB2)
					count = 2;	/* fill 2 chars */
			}
			else if ((oflag & TABDLY) == TAB1)
				delay = tmp * 50;  /* delay 50 mills each col*/
			else if ((oflag & TABDLY) == TAB2)
				delay = 100;	/* delay 100 mills */
			tp->col += tmp;		/* point to next tab position */
			break;
		case TYM_Ox0b:		/*--- vertical tab ---*/
 			if ((oflag & VTDLY) == VT1)
			{
				if (oflag & OFILL)
					count = 4;	/* fill 4 chars */
				else	delay = 2000;	/* delay 2000 mills */
			}
			break;
		case TYM_Ox0c:		/*--- form feed ---*/
			if ((oflag & FFDLY) == FF1)
			{
				if (oflag & OFILL)
					count = 4;	/* fill 4 chars */
				else	delay = 2000;	/* delay 2000 mills */
			}
			break;
		} /* switch (c_type) */

		*owptr++ = c;	/* store byte in output buf */
fill:
		if (count != 0)
		{
			if (count > (int)(owend - owptr))
			{
				owend = owptr;
				size = count;
				/* leave count non-zero so that when we get the
				   next output buffer, we jump right to 'fill'*/
				continue;
			}
			if (oflag & OFDEL)
				c = 0x7f;
			else	c = CNUL;
			for (; count != 0; count--)
				*owptr++ = c;
		}
		else if (delay != 0)
		{
			omp->b_wptr = owptr; 
			BUGCIR_LOG(tp->min_dev, omp->b_wptr - omp->b_rptr,
	 				*omp->b_rptr,TYM_CIR_OPOST,(int)omp);
			tp->pv_wstat.postch += omp->b_wptr - omp->b_rptr;
			putnext(tp->wq, omp);
			if ((bp = allocb(4, BPRI_HI)) != NULL)
			{
				bp->b_datap->db_type = M_DELAY;
				/* convert milliseconds to machine ticks */
				*((int *)bp->b_wptr) = (delay * HZ)/1000;
				putnext(tp->wq, bp);
			}
			omp = NULL;
			delay = 0;
		}
	} /* while (rptr < wptr) */
	/* free up message block as soon as it is processed */
	next = mp->b_cont;
	mp->b_cont = NULL;
	freeb(mp);
	if ((mp = next) != NULL)
		goto more;	/* process next message block in msg */
	if (omp != NULL)
	{
		omp->b_wptr = owptr;
		BUGCIR_LOG(tp->min_dev, omp->b_wptr - omp->b_rptr,
				*omp->b_rptr,TYM_CIR_XPOST,(int)omp);
		tp->pv_wstat.postch += omp->b_wptr - omp->b_rptr;
		putnext(tp->wq, omp);
	}
	return(NULL);
}/*------------------------------------------------------------------*/
/*
 *		IOCTL WRITE SERVICE
 *
 *------------------------------------------------------------------*/
tym_wsrv_ioc(wq, mp, tp)
register queue_t	*wq;
register mblk_t		*mp;
register tym_tty_t	*tp;
{
	register struct iocblk *iocbp;
	struct sgttyb	*gb;
	register flag;
	termio_t *tptr = &tp->termio;
	int	cmd;		/* ioctl command */
	
	iocbp = (struct iocblk *)mp->b_rptr;
	cmd = iocbp->ioc_cmd;
	BUGLOG3(tp, BUG_LSIO, "tym_wsrv_ioc wq=%x mp=%x cmd=%x", wq, mp, cmd);
	switch(cmd)
	{
	default:
		break;
	case TIOCSETP:
		gb = (struct sgttyb *)mp->b_cont->b_rptr;
		tptr->c_iflag = 0;
		tptr->c_oflag = 0;
		tptr->c_lflag = 0;
		tptr->c_cc[VERASE] = gb->sg_erase;
		tptr->c_cc[VKILL] = gb->sg_kill;
		flag = gb->sg_flags;
		if (flag & O_XTABS)
			tptr->c_oflag |= TAB3;
		else if (flag & O_TBDELAY)
			tptr->c_oflag |= TAB1;
		if (flag & O_LCASE)
		{
			tptr->c_iflag |= IUCLC;
			tptr->c_oflag |= OLCUC;
			tptr->c_lflag |= XCASE;
		}
		if (flag & O_ECHO)
			tptr->c_lflag |= ECHO;
		if (!(flag & O_NOAL))
			tptr->c_lflag |= ECHOK;
		if (flag & O_CRMOD)
		{
			tptr->c_iflag |= ICRNL;
			tptr->c_oflag |= ONLCR;
			if (flag & O_CR1)
				tptr->c_oflag |= CR1;
			if (flag & O_CR2)
				tptr->c_oflag |= ONOCR | CR2;
		}
		else
		{
			tptr->c_oflag |= ONLRET;
			if (flag & O_NL1)
				tptr->c_oflag |= CR1;
			if (flag & O_NL2)
				tptr->c_oflag |= CR2;
		}
		if (flag & O_RAW)
		{
			tptr->c_cc[VEOL] = 1;
			tptr->c_cc[VEOF] = 6;
			tptr->c_iflag &= ~(ICRNL | IUCLC);
		}
		else
		{
			tptr->c_cc[VEOF] = CEOF;
			tptr->c_cc[VEOL] = 0;
			tptr->c_cc[VEOL2] = 0;
			tptr->c_oflag |= OPOST;
			tptr->c_lflag |= ICANON | ISIG;
		}
		if (flag & O_VTDELAY)
			tptr->c_oflag |= FFDLY;
		if (flag & O_BSDELAY)
			tptr->c_oflag |= BSDLY;
		iocbp->ioc_count = sizeof(struct termio);
		iocbp->ioc_error = 0;
		iocbp->ioc_rval = 0;
		tym_set_newioctl(mp, tp);
		break;
	case TCSETA:			/* should never see this one */
		BUGLOG2(tp, BUG_LERR, "tym_wsrv_ioc TCSETA wq=%x mp=%x",wq,mp);
		/* FALL THROUGH */
	case TCSETAF:
	case TCSETAW:
		tym_set_newioctl(mp, tp);
		break;
	} /* switch(cmd) */
	putnext(wq, mp);	/* pass all ioctls downstream */
}/*------------------------------------------------------------------*/
/*
 *	set ioctl flags
 *
 *	some kind of ioctl set command has been received, so change
 *	our ioctl flags, change the character look-up table for input
 *	processing, and change (if necessary) the stream head options
 */
tym_set_newioctl(mp, tp)
register mblk_t		*mp;
register tym_tty_t	*tp;
{
	register struct termio *cb;
	register i;
	termio_t *tptr = &tp->termio;

	cb = (struct termio *)mp->b_cont->b_rptr;
	tptr->c_cflag = cb->c_cflag;	/* info only - driver provides */

	/* c_iflag is provided partially by the driver and
	   partially by this module. */
	if (cb->c_iflag != tptr->c_iflag)
	{
		BUGCIR_LOG(tp->min_dev, 0, 0, TYM_CIR_IFLAG_OLD, tptr->c_iflag);
		BUGCIR_LOG(tp->min_dev, 0, 0, TYM_CIR_IFLAG_NEW, cb->c_iflag);
		BUGLOG3(tp,BUG_LSIO,"tym_set_newioctl rq=%x iflag was %x is %x",
					 tp->rq,  tptr->c_iflag, cb->c_iflag);
		tptr->c_iflag = cb->c_iflag;
	}
	if (cb->c_oflag != tptr->c_oflag)
	{
		BUGCIR_LOG(tp->min_dev, 0, 0, TYM_CIR_OFLAG_OLD, tptr->c_oflag);
		BUGCIR_LOG(tp->min_dev, 0, 0, TYM_CIR_OFLAG_NEW, cb->c_oflag);
		BUGLOG3(tp,BUG_LSIO,"tym_set_newioctl rq=%x oflag was %x is %x",
					 tp->rq,  tptr->c_oflag, cb->c_oflag);
		tptr->c_oflag = cb->c_oflag;
		/* don't do opost processing even if OPOST is set
		   unless at least one other oflag bit is also set */
		if ((tp->termio.c_oflag & OPOST) != 0
		 && (tp->termio.c_oflag & ~OPOST) != 0)
			tp->oflag_type = 1;
		else	tp->oflag_type = 0;
	}
	if (cb->c_lflag != tptr->c_lflag)
	{
		BUGCIR_LOG(tp->min_dev, 0, 0, TYM_CIR_LFLAG_OLD, tptr->c_lflag);
		BUGCIR_LOG(tp->min_dev, 0, 0, TYM_CIR_LFLAG_NEW, cb->c_lflag);
		BUGLOG3(tp,BUG_LSIO,"tym_set_newioctl rq=%x lflag was %x is %x",
					 tp->rq,  tptr->c_lflag, cb->c_lflag);
		tptr->c_lflag = cb->c_lflag;
		if ((tp->termio.c_lflag & (ICANON|XCASE)) == (ICANON|XCASE))
			tp->canon_xcase = 1;
		else	tp->canon_xcase = 0;
	}
	if (cb->c_line != tptr->c_line)
	{
		BUGCIR_LOG(tp->min_dev, 0, 0, TYM_CIR_LINE,
					(tptr->c_line << 16) | cb->c_line);
		BUGLOG3(tp,BUG_LSIO,"tym_set_newioctl rq=%x line was %x is %x",
			 tp->rq,  tptr->c_line, cb->c_line);
		tptr->c_line = cb->c_line;	
	}
	for (i = 0; i < NCC; i++)
		if (cb->c_cc[i] != tptr->c_cc[i])
		{
			BUGCIR_LOG(tp->min_dev, 0, 0, TYM_CIR_CC0 + i,
					(tptr->c_cc[i] << 16) | cb->c_cc[i]);
			tptr->c_cc[i] = cb->c_cc[i];
		}
	/* initialize tp->char_tab based on ioctl settings */
	tym_init_input_char_tab(tp);
	tym_setopt_2_head(tp->rq);	/* send M_SETOPTS to head if change*/

	/* now pass down only the ioctl setting which will put the driver
	   below us in raw mode, but pass ISTRIP downstream */
	cb->c_iflag &= ~(INLCR|IGNCR|ICRNL|IUCLC|ISWITCH);
	cb->c_lflag = 0;
	cb->c_oflag = 0;
	cb->c_cc[VMIN] = 1;

	if (tp->rq->q_first != NULL)
		qenable(tp->rq);
}/*------------------------------------------------------------------*/
/*
 *	initialize tp->char_tab based on initial ioctl settings
 *
 *	this table is used to process data from downstream
 */
tym_init_input_char_tab(tp)
register tym_tty_t	*tp;
{
	register mblk_t	*bp;		/* msg used to send up canon data */
	register mblk_t	*bp_canon;	/* msg used to allocate canon buffer */
	register unchar *char_tab_ptr;	/* addr of input char table this dev*/
	register i;
	termio_t *tptr = &tp->termio;
	int count;

	char_tab_ptr = tp->char_tab;
	/* start out with default settings */
	bzero(char_tab_ptr, 256);

	if (tptr->c_iflag & IXON)
	{
		*(char_tab_ptr + CSTOP)  = TYM_CSTOP;
		*(char_tab_ptr + CSTART) = TYM_CSTART;
	}
	if (tptr->c_iflag & INLCR)
		*(char_tab_ptr + 0xa) = TYM_NEWLINE_1;
	else	*(char_tab_ptr + 0xa) = TYM_NEWLINE_2;
	if (tptr->c_iflag & IGNCR)
		*(char_tab_ptr + 0xd) = TYM_RETURN_1;	/* ignore */
	else if (tptr->c_iflag & ICRNL)
		*(char_tab_ptr + 0xd) = TYM_RETURN_2;	/* change to \n */
	/* else pass \r up as is */

	if (tptr->c_iflag & IUCLC)
		for (i = 0x41; i <= 0x5a; i++)	/* 0x41 = 'A', 0x5a = 'Z' */
			*(char_tab_ptr + i) = TYM_A_Z;

	/* if ICANON and we don't have a canon buffer yet, try to get
	   one (if not successful, clear ICANON flag)
	   if not ICANON and we have a canon buffer, return it to pool */
	if (tptr->c_lflag & ICANON && tp->canon_base == NULL)
	{
		if ((bp_canon = allocb(MAX_CANON, BPRI_LO)) == NULL)
		{
			BUGLOG0(tp, BUG_LERR, "can't allocate canon buffer");
			tptr->c_lflag &= ~ICANON; /* no ICANON if no buf avail*/
		}
		else
		{
			tym_flush_pending_input(tp);	/* toss any raw data */
			tp->bp_canon	= bp_canon;
			tp->canon_base	= bp_canon->b_rptr;
			tp->canptr	= tp->canon_base;
			tp->state	&= ~END_OF_LINE;
		}
	}
	else if (!(tptr->c_lflag & ICANON) && tp->canon_base != NULL)
	{
		if ((count = (tp->canptr - tp->canon_base)) != 0)
		{
		/* if M_IOCACK is TCSETxx and the caller required RAW mode
		 * and the canonical buffer has some data, then send the
		 * data upstream; otherwise, data will hang around the 
		 * canonical buffer until returning to canonical mode; */
		/* Get a block to put data into */
			if ((bp = allocb(count, BPRI_LO)) == NULL)
			{
				BUGLOG0(tp, BUG_LERR,"tym_rsrv_ioc no bufs");
			}
			else
			{
				bcopy(tp->canon_base, bp->b_wptr,count);
			    	bp->b_wptr += count;
			    	putnext(tp->rq, bp); 
			}
		}
		freemsg(tp->bp_canon);
		tp->bp_canon = NULL;
		tp->canon_base = NULL;
		tp->canptr = NULL;
		tp->state &= ~(END_OF_LINE | M_READ_RECVD);
	}
	if ((tptr->c_lflag & (ICANON|XCASE)) == (ICANON|XCASE))
	{
		for (i = 0x61; i <= 0x7a; i++)	/* 0x61 = 'a', 0x7a = 'z' */
			*(char_tab_ptr + i) = TYM_a_z;	/* a - z */
		*(char_tab_ptr + 0x21) = TYM_0x21;	/* ! */
		*(char_tab_ptr + 0x27) = TYM_0x27;	/* ' */
		*(char_tab_ptr + 0x28) = TYM_0x28;	/* ( */
		*(char_tab_ptr + 0x29) = TYM_0x29;	/* ) */
		*(char_tab_ptr + 0x5c) = TYM_0x5c;	/* \ */
		*(char_tab_ptr + 0x5e) = TYM_0x5e;	/* ^ */
	}
	if (tptr->c_lflag & ICANON)
	{
#ifdef POSIX
		/* don't put entry in table if cc char is POSIX disable char */
		if (tptr->c_cc[VERASE] != VDISABLE)
			*(char_tab_ptr + tptr->c_cc[VERASE]) = TYM_ERASE;
		if (tptr->c_cc[VKILL] != VDISABLE)
			*(char_tab_ptr + tptr->c_cc[VKILL])  = TYM_KILL;/*kill*/
		if (tptr->c_cc[VEOF] != VDISABLE)
			*(char_tab_ptr + tptr->c_cc[VEOF])   = TYM_EOF;
		if (tptr->c_cc[VEOL] != VDISABLE)
			*(char_tab_ptr + tptr->c_cc[VEOL])   = TYM_EOL;
		if (tptr->c_cc[VEOL2] != VDISABLE)
			*(char_tab_ptr + tptr->c_cc[VEOL2])  = TYM_EOL2;
#else
		*(char_tab_ptr + tptr->c_cc[VERASE]) = TYM_ERASE;  /*backspace*/
		*(char_tab_ptr + tptr->c_cc[VKILL])  = TYM_KILL;  /* kill */
		*(char_tab_ptr + tptr->c_cc[VEOF])   = TYM_EOF;	/* 0x04 */
		*(char_tab_ptr + tptr->c_cc[VEOL])   = TYM_EOL;
		*(char_tab_ptr + tptr->c_cc[VEOL2])  = TYM_EOL2;
#endif	/* POSIX */
	}
	if (tptr->c_lflag & ISIG)
	{
#ifdef POSIX
		/* don't put entry in table if cc char is POSIX disable char */
		if (tptr->c_cc[VINTR] != VDISABLE)
			*(char_tab_ptr + tptr->c_cc[VINTR])  = TYM_INTR; 
		if (tptr->c_cc[VQUIT] != VDISABLE)
			*(char_tab_ptr + tptr->c_cc[VQUIT])  = TYM_QUIT;
		if (tptr->c_cc[VSUSP] != VDISABLE)
			*(char_tab_ptr + tptr->c_cc[VSUSP]) = TYM_SUSP;
#else
		*(char_tab_ptr + tptr->c_cc[VINTR])  = TYM_INTR;  /* 0x7f */
		*(char_tab_ptr + tptr->c_cc[VQUIT])  = TYM_QUIT;  /* 0x1c */
		if (tptr->c_iflag & ISWITCH)	/* recognize ^Z */
			*(char_tab_ptr + tptr->c_cc[VSWTCH]) = TYM_SWTCH;/*1a*/
#endif	/* POSIX */
	}
}/*------------------------------------------------------------------*/
/*
 *	tym_setopt_2_head - send M_SETOPTS msg to stream head
 *
 * set options in the stream head to always receive M_READ messages
 */
tym_setopt_2_head(rq)
register queue_t	*rq;
{
	register tym_tty_t	*tp;
	register mblk_t		*bp_set;
	register struct stroptions *sop;
	register readopt;		/* read mode option */
	register flags;			/* option flags */

	if ((tp = (tym_tty_t *)rq->q_ptr) == NULL)
		return;		/* port is closed or module was popped */
	/* always ask for M_READ msgs */
	flags = SO_READOPT | SO_NDELON | SO_MREADON;
#ifdef POSIX
	if (tp->termio.c_lflag & TOSTOP)
		flags |= SO_TOSTOPON;
	else	flags |= SO_TOSTOPOFF;
#endif /* POSIX */
	if (tp->termio.c_lflag & ICANON)
		readopt = RMSGN;  	/* each line has its own msg boundary */
	else	readopt = RNORM;	/* byte stream, no lines */
	if (readopt != tp->so_readopt || flags != tp->so_flags)
	{
		/* only send M_SETOPTS to head if there is a change */
		bp_set = tym_alloc((int)sizeof(struct stroptions),
				   BPRI_MED, tym_setopt_2_head, rq);
		if (bp_set == NULL)
		{
			BUGLOG1(tp,BUG_LERR,
				"tym_setopt_2_head no buf avail, rq=%x",rq);
			return;
		}
		bp_set->b_datap->db_type = M_SETOPTS;
		bp_set->b_wptr += sizeof(struct stroptions);
		sop = (struct stroptions *)bp_set->b_rptr;
		tp->so_readopt = sop->so_readopt = readopt;
		tp->so_flags   = sop->so_flags   = flags;
		BUGLOG2(tp,BUG_LGEN,"tym_setopt_2head tp=%x flags=%x",tp,flags);
		BUGCIR_LOG(tp->min_dev, 0, 0, TYM_CIR_HEADOPT,
				(readopt << 16) + flags);
		putnext(rq, bp_set); 
	}
}/*------------------------------------------------------------------*/
/*
 *	flush routines
 *
 *------------------------------------------------------------------*/
tym_flush_read_side(tp)
register tym_tty_t	*tp;
{
	flushq(tp->rq, FLUSHALL);	/* flush streams read queue */
	tym_flush_pending_input(tp);	/* flush any pending msgs */
	tym_flush_echobuf(tp);		/* flush any data in output echo buf */
	qenable(tp->rq);		/* do this to set QWANTR in case
					   there is data on the queue */
}/*------------------------------------------------------------------*/
tym_flush_write_side(tp)
register tym_tty_t	*tp;
{
	flushq(tp->wq, FLUSHALL);	/* flush streams write queue */
	tym_flush_echobuf(tp);		/* flush any data in output echo buf */
	if (tp->state & TTSTOP)
		tym_send_start_down(tp->wq);
}/*------------------------------------------------------------------*/
/*
 *	flush any canon and non-icanon data waiting to be sent upstream
 */
tym_flush_pending_input(tp)
register tym_tty_t	*tp;
{
	if (tp->raw_message != NULL)
	{
		BUGLOG2(tp,BUG_LFSH, "tym_flush_pending_input tp=%x raw_msg=%x",
							 tp, tp->raw_message);
		freemsg(tp->raw_message);
	}
	tp->state &= ~M_READ_RECVD;
	tp->raw_message = NULL;
	tp->raw_count = 0;
	tp->canptr = tp->canon_base;	/* flush canon buffer */
}/*------------------------------------------------------------------*/
/*
 *	flush any echo data
 */
tym_flush_echobuf(tp)
register tym_tty_t	*tp;
{
	register mblk_t		*tbp;

	while (tp->echobuf != NULL)
	/* we have one or more echo msgs we can free */
	{
		BUGLOG2(tp, BUG_LFSH, "tym_flush_echobuf tp=%x tp->echobuf=%x",
							 tp,   tp->echobuf);
		tbp = tp->echobuf->b_next;	/* remainder of msg chain */
		tp->echobuf->b_next = NULL;	/* isolate msg we want to send*/
		freemsg(tp->echobuf);		/* pass echo data downstream */
		tp->echobuf = tbp;		/* point to remainder of chain*/
	} /* while (tp->echobuf != NULL) */
}/*------------------------------------------------------------------*/
/*
 *		signal routines
 *
 *------------------------------------------------------------------*
 *
 *	send a SIGINT signal upstream via M_PCSIG
 *
 *	I used to do the flush first, then send the signal upstream.
 *	However, if the user did an 'ls -l', then 'ctrl-s', then hit
 *	the delete key, 'strwrite' (which was sleeping because the write
 *	queue was full), would wake up because of the flush and start
 *	transfering data downstream again before the user process 
 *	had a chance to stop the write.
 *
 *	So I changed the code to send the signal upstream first, then
 *	do the flush, to give the user process a chance respond to the
 *	signal and cancel any pending writes. But this caused a problem
 *	when changing the password: when the user typed 'passwd',
 *	the 'passwd' program would turn off echoing via an 'ioctl' command.
 *	If the user then hit the delete key, sending the signal up
 *	would cause 'passwd' to turn on echoing via another ioctl, bug
 *	this ioctl would sometimes be flushed by the flush command
 *	coming back down from the stream head, and hence echoing was not
 *	turned on.
 *
 *	So I am now back to sending the flush first, then the signal.
 */ 
tym_send_sig_up(tp, sig, lflag_type)
register tym_tty_t	*tp;
int	sig;
int	lflag_type;		/* 0 to ignore c_lflag (BREAK call) */
{
	if (!(tp->termio.c_lflag & NOFLSH) || lflag_type == 0)
	{
		tym_flush_read_side(tp);	/* flush read side */

		/* instead of flushing our own write side here,
		 * wait until we receive the FLUSHW from upstream.
		 * Otherwise, if the write side is backed up, flushing
		 * our write queue here will start data flowing downstream
		 * again, before the FLUSHW arrives from upstream,
		 * causing pending data (data the user process wrote which
		 * is sitting in a write queue upstream of us) 
		 * to be sent to the terminal after this signal event occurs
		 */ 

		/* tell stream head to send FLUSHW downstream, to flush
		 * everyone's write queue
		 */
		putctl1(tp->rq->q_next, M_FLUSH, FLUSHW);

		/* tell driver to send FLUSHR upstream, to flush
		 * everyone's read queue
		 */
		putctl1(tp->wq->q_next, M_FLUSH, FLUSHR);
	}
	putctl1(tp->rq->q_next, M_PCSIG, sig);
	BUGCIR_LOG(tp->min_dev,sig, 0,TYM_CIR_SIG,(int)tp);
}/*------------------------------------------------------------------*/
/*
 *	send M_START msg downstream to tell driver to start output
 */
tym_send_start_down(wq)
register queue_t	*wq;
{
	register tym_tty_t	*tp;
	register mblk_t		*bp;

	if ((tp = (tym_tty_t *)wq->q_ptr) == NULL)
		return;		/* port is closed or module was popped */
	if ((bp = tym_alloc(1, BPRI_HI, tym_send_start_down, wq)) != NULL)
	{
		bp->b_datap->db_type = M_START;
		putnext(wq, bp);
		tp->state &= ~TTSTOP;	/* clear 'output not allowed' */
		qenable(wq);	/* start output processing again */
		BUGCIR_LOG(tp->min_dev, 0, M_START, TYM_CIR_SEND_MISC, tp);
	}
}/*------------------------------------------------------------------*/
/*
 *	send M_STOP msg downstream to tell driver to stop output
 */
tym_send_stop_down(wq)
register queue_t	*wq;
{
	register tym_tty_t	*tp;
	register mblk_t		*bp;

	if ((tp = (tym_tty_t *)wq->q_ptr) == NULL)
		return;		/* port is closed or module was popped */
	if ((bp = tym_alloc(1, BPRI_HI, tym_send_stop_down, wq)) != NULL)
	{
		bp->b_datap->db_type = M_STOP;
		putnext(wq, bp);
		tp->state |= TTSTOP;
		BUGCIR_LOG(tp->min_dev, 0, M_STOP, TYM_CIR_SEND_MISC, tp);
	}
}/*------------------------------------------------------------------*/
/*
 *	make sure there is a streams msg to put ioctl data into to
 *	send upstream, and that the attached data block is big enougn
 *
 *	return	0 if ok
 *		1 if no msg available
 */
tym_insure_ioctl_works(mp, tp, size)
register mblk_t		*mp;
register tym_tty_t	*tp;
register int		size;
{
	register mblk_t	*bp;

	if ((bp = mp->b_cont) != NULL)
	{
		if ((bp->b_datap->db_lim - bp->b_datap->db_base) < size)
		{
			freemsg(bp);	/* discard msg allocated for */
			mp->b_cont = NULL; /* data; not big enough */
		}
	}
	if (mp->b_cont == NULL)
	{
		if ((bp = allocb(size, BPRI_MED)) == NULL)
		{
			BUGCIR_LOG(tp->min_dev, size,0,TYM_CIR_NOBUF,NULL);
			BUGLOG1(tp,BUG_LERR,"tym_insure no buf tp=%x",tp);
			return(1);	/*flag 'can't process'*/
		}
		mp->b_cont = bp;
		bzero(bp->b_rptr, size);
	}
	return(0);
}/*------------------------------------------------------------------*/
/*
 *		MISC
 *
 *------------------------------------------------------------------*
 *
 * tym_alloc -- try to allocb a mblk size long at pri
 *
 *	if it fails it calls bufcall and returns NULL
 *	also returns NULL if bufcall can't store buffer request
 *
 *	input
 *		size	number of bytes to alloc
 *		pri	allocation priority
 *		func	addr of function passed bufcall if no buffers available
 *		arg	argument to pass to bufcall if no buffers available
 *
 *	return	address of buffer if one available
 *		NULL if no buffer available
 */
mblk_t *
tym_alloc(size, pri, func, arg)
int	size, pri;
int	(*func)();
queue_t	*arg;
{
	register mblk_t		*mp;

	if ((mp = allocb(size, pri)) == NULL)
	{

		if ((bufcall(size, pri, func, arg)) == 0)
		{
			BUGLOG3(NULL,BUG_LERR, "tym_alloc bufcall failed, (pri<<16+size=%x func=%c q=%x", (pri << 16) + size, func, arg);
			/* wait 2 seconds then try again */
			timeout(func, arg, HZ*2);
		}
		BUGCIR_LOG(pri, size, 0,TYM_CIR_NOBUF, (int)arg);
		return (NULL);
	}
	BUGLOG3(NULL,BUG_LERR,"tym_alloc: rtn=%x size=%x mp=%x",func, size, mp);
	return (mp);
}/*------------------------------------------------------------------*/
tym_fail(a, f, l)
register char *a, *f;
{
	cmn_err(CE_NOTE, "assertion failed: %s, file: %s, line: %d", a, f, l);
}/*------------------------------------------------------------------*/
tym_log(tp, level, fmt, ARGS)
tym_tty_t	*tp;
int		level;
char		*fmt;
unsigned	ARGS;
{
	int p_level;
	dev_t	dev;

	if (tp == NULL)
		dev = 255;
	else	dev = tp->min_dev;

	p_level = (tym_debug & 0xf0000) >> 16;
	if (tym_debug != 0 && (p_level == 0 || level <= p_level))
	{
		if ((tym_debug & 0xfff) == 0x123)
			tym_print_no_log(fmt, ARGS);
		else	strlog(TYM_MOD,dev,level, SL_TRACE, fmt, ARGS);
	}
}/*------------------------------------------------------------------*/
tym_print_no_log(fmt, ARGS)
char		*fmt;
unsigned	ARGS;
{
	printf(fmt, ARGS);
	if ((tym_debug & 0x1000) == 0)
		printf("\n");
	else	printf(" ** ");
}/*------------------------------------------------------------------*/
tym_cir_logit(dev, parm3, parm4, log_type, parm)
unchar dev, parm4, log_type;
ushort parm3;
int parm;
{
	register struct cir_log *ptr;	/* ptr to next tym_cir_log_buf entry */
	int s;

	if (dev != 0xff)
	{
		if (tym_cir_log_only_one_port && dev != tym_cir_port_to_log)
			return;
	}
	s = splstr();
	ptr = tym_cir_ptr;
	ptr->rel_time	= (dev << 24) + (lbolt & 0x7fffff);
	ptr->parm_1	= parm >> 16;
	ptr->parm_2	= parm;
	ptr->parm_3	= parm3;
	ptr->parm_4	= parm4;
	ptr->log_type	= log_type;

	if (++tym_cir_index >= TYM_CIR_LOG_SIZE)
	{
		tym_cir_index = 0;		/* wrap */
		tym_cir_ptr = tym_cir_log_buf;
	}
	else	tym_cir_ptr++;
	tym_cir_ptr->log_type = 0xff;	/* flag next entry as end of data */
	splx(s);
}/*------------------------------------------------------------------*/
/*
 *	initialize the circular log buffer
 *
 *	- set the 'next entry pointer' to the start of the buffer
 *	- flag that entry as the 'end of buffer' entry (log_type = -1)
 */
tym_cir_log_init()
{
	if (tym_cir_log_init_flag == 0)
	{
		tym_cir_log_init_flag++;	/* flag 'initialization done */
		tym_cir_index = 0;		/* next entry number */
		tym_cir_ptr   = tym_cir_log_buf; /* next entry pointer */
		tym_cir_ptr->log_type = -1;	/* end of buf */
	}
}/*------------------------------------------------------------------*/
#if	defined(IOPM)
/*
 *	initialize IOPM configuration information
 */
ldterminit()
{
	static struct str_mod_config strmodconf;

	strmodconf.smc_version = SMC_VERSION;
	strmodconf.smc_strtab = &ldterminfo;
	attach_streams_module(&strmodconf);

	tym_cir_log_init();
}/*------------------------------------------------------------------*/
#endif	/* IOPM */
