/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) iopms.c: version 25.1 created on 11/27/91 at 14:58:10	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)iopms.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
#include "sys/types.h"
#include "sys/errno.h"
#include "sys/param.h"
#include "sys/user.h"
#include "sys/proc.h"
#include "sys/iopmcomm.h"
#include "sys/stream.h"
#include "sys/iopmstream.h"
#include "sys/debug.h"
#include "sys/cmn_err.h"
#include "sys/synch.h"
#include "sys/sysmacros.h"
#include "sys/stropts.h"
#include "sys/iopmsltbl.h"
#include "sys/spm_mem.h"
#include "sys/iopmdebug.h"
#include "sys/var.h"
#include "sys/sysinfo.h"

#define STHIPRI PZERO

struct iopm_ctl  *init_ctl();
static void  set_list_avail(), teardwn_ctl();
static void  mux_wput(), mux_rput(), cmd_wput(), cmd_rput();
static void  send_flow_msg(), returncpmsg(), clear_kdownq(), sendmptoiopm();
void         sigiopm(), init_iopmscb();
static int   iopms_open(), iopms_close();
static int   iopms_rput(), iopms_rsrv(), iopms_wput(), iopms_wsrv();

/* driver accepts one message then goes full */
static struct module_info  iopswminfo = { 1, "iopm_driver", 0, INFPSZ, 1, 0 };

static struct module_info  iopsrminfo = {
	1, "iopm_driver", 0, INFPSZ, DEFAULT_HIWAT, 0
};

static struct qinit  rinit = {
	iopms_rput, iopms_rsrv, iopms_open, iopms_close, NULL, &iopsrminfo, NULL
};

static struct qinit  winit = {
	iopms_wput, iopms_wsrv, NULL, NULL, NULL, &iopswminfo, NULL
};

struct streamtab  iopms_info = { &rinit, &winit, NULL, NULL };

extern struct iopm_comm  *iopm_base[];
struct iopmsltbl         iopmstbl[SLTBLSZ];

mblk_t  *itoks;
mblk_t  *sig_mblk_hook;
mblk_t  *close_mblk_hook;

/******************************************************************************/
init_iopms(slot)
{
	iopm_base[slot]->itoksp = &itoks;
	iopm_base[slot]->ksysinfop = &sysinfo;
}

/******************************************************************************/
iopms_start()
{
	register mblk_t  *mp;

	ASSERT(!sig_mblk_hook);
	ASSERT(!close_mblk_hook);

	if ( !(mp = allocb(sizeof(int), BPRI_HI)) )
		cmn_err(CE_PANIC, "init_mblk_hooks: no msg blk for signal");

	mp->b_datap->db_type = M_KILLS;
	sig_mblk_hook = mp;

	if ( !(mp = allocb(sizeof(int), BPRI_HI)) )
		cmn_err(CE_PANIC, "init_mblk_hooks: no msg blk for close");

	mp->b_datap->db_type = M_CLOSES;
	close_mblk_hook = mp;
}

/******************************************************************************/
/* open & close (cmd layer) */
/******************************************************************************/
/* iopms_open gets a mblk to keep a local struct.
/* It uses q_ptr in the wq to point to this struct.
/* iopms_open gets a mblk for use by the IOPM to request mblk's to pass data
/* upstream.  It sets QREMOTE in the write
/* queue and REMOTE in the read stream data structure.
/* protect from multiproc. ie only one read mp
*/
#define IOPMSLOT iopmstbl[minor(dev) >> MNPGSHFT].iopmslot
static
iopms_open(rqp, dev, flag, sflag)
register queue_t  *rqp;
dev_t             dev;
{
	register queue_t          *wqp = WR(rqp);
	register struct iopm_ctl  *iopmctl;
	register mblk_t           *mp;
	register struct iopmscb   *iopmscbp;
	int                       iopmslot = IOPMSLOT;
	uint                      first_open = 0;
	int                       savpri;
	uint                      slpdisp;

	PRINT3(SOPEN,"iopms_open: krq = %x, kwq = %x, dev = %x\n",
	  rqp, wqp, dev);

	ASSERT(valid_rq(rqp));
	iopmctl = (struct iopm_ctl *)wqp->q_ptr;
	ASSERT(dev & UNMINOR_BIT);

	if ( iopmstbl[minor(dev) >> MNPGSHFT].valid == 0 )
	{
		/* no IOPM streams driver loaded for this group of minors */
		u.u_error = ENXIO;
		return OPENFAIL;
	}

	ASSERT(is_IOPM(iopmslot));
	ASSERT(iopm_base[iopmslot]->iopm_version == IOPM_VERSION);

	/* The first streams open for this IOPM will init the iopmcomm struct */
	/* This can't be done at kernel init since the IOPMs arn't loaded */
	if ( !iopm_base[iopmslot]->itoksp )
		init_iopms(iopmslot);

	/*
	** Get control structures and attach them to the write q_ptr
	*/
	if ( !wqp->q_ptr )	/* no IOPM control struct, ie new stream */
	{
		first_open = 1;
		wqp->q_flag |= QREMOTE;
		setsremote(rqp);

		if ( !(iopmctl = init_ctl(wqp, dev)) )
			return OPENFAIL;
	}

	/*
	** Get a mblk to send the open command to the IOPM
	*/
	while ( !(mp = allocb(sizeof(struct iopmscb), BPRI_MED)) )
		if ( strwaitbuf(sizeof(struct iopmscb), BPRI_MED) )
		{
			cmn_err(CE_WARN,
			  "iopms_open: no msg blk to send to IOPM");

			if ( first_open )
				teardwn_ctl(wqp);

			return OPENFAIL;
		}

	PRINT1(SOPEN,"iopms_open: mp = %x\n",mp);

	/*
	** fill in the open msg and send it to the IOPM
	*/
	mp->b_prev = (mblk_t *)iopmctl;
	mp->b_datap->db_type = M_OPEN;
	iopmscbp = (struct iopmscb *)mp->b_wptr;
	iopmscbp->iopms_dev = minor(dev);
	init_iopmscb(iopmscbp);
	iopmscbp->iopms_oflag = flag;
	iopmscbp->iopms_sflag = sflag;

	iopmctl->ic_kflags &= ~PM_DONE;
	sendmptoiopm(mp);

	/*
	** wait for response from IOPM
	*/
	slpdisp = STOPRI | PCATCH;
	savpri = splstr();
	while ( !(iopmctl->ic_kflags & PM_DONE) )
		if ( sleep(mp, slpdisp) )
		{				/* received signal */
			splx(savpri);
			sigiopm(WR(iopmctl->ic_krq));
			slpdisp = STHIPRI;
			savpri = splstr();
		}

	if ( iopmscbp->iopms_dev != (dev_t)OPENFAIL )
	{				/* open succeded */
		if ( iopmscbp->iopms_ttyp != u.u_ttyp )
		{
			iopmctl->ic_pgrp = iopmscbp->iopms_pgrp;
			u.u_ttyp = &iopmctl->ic_pgrp;
			iopmctl->ic_iopmttyp = iopmscbp->iopms_ttyp;
		}

		dev = iopmscbp->iopms_dev;	/* get minor dev back */
		freemsg(mp);
		PRINT2(SOPEN,"iopms_open: dev = %x, ttyp = %x\n",dev,u.u_ttyp);
		return dev;
	}
	else
	{				/* open failed */
		u.u_error = iopmscbp->iopms_err;

		if ( first_open )
			teardwn_ctl(wqp);

		freemsg(mp);
		return OPENFAIL;
	}
}

/******************************************************************************/
static
iopms_close(rqp, flag)
queue_t  *rqp;
{
	register queue_t          *wqp = WR(rqp);
	register struct iopm_ctl  *iopmctl;
	mblk_t                    *mp;
	struct iopmscb            *iopmscbp;
	int                       id;
	queue_t                   *qp;
	queue_t                   *lastqp;
	int                       savpri;
	int                       slpdisp;
	extern                    setrun();

	PRINT2(SCLOSE,"iopms_close: krq = %x, flag = %x,",rqp,flag);

	ASSERT(valid_rq(rqp));
	iopmctl = (struct iopm_ctl *)wqp->q_ptr;
	ASSERT(valid_iopmctl(iopmctl));

	iopmctl->ic_kflags |= CLOSING;
	if ( !(mp = allocb(sizeof(struct iopmscb), BPRI_HI)) )
	{
		while ( !(mp = close_mblk_hook) )
			sleep(&close_mblk_hook, STHIPRI);

		close_mblk_hook = 0;
		ASSERT(mp->b_datap->db_type == M_CLOSES);
	}

	PRINT1(SCLOSE," mp = %x\n",mp);

	mp->b_prev = (mblk_t *)iopmctl;
	if ( mp->b_datap->db_type != M_CLOSES )
		mp->b_datap->db_type = M_CLOSE;
	iopmscbp = (struct iopmscb *)mp->b_wptr;
	init_iopmscb(iopmscbp);
	iopmscbp->iopms_oflag = flag;

	iopmctl->ic_kflags &= ~PM_DONE;
	((struct copy_msg *)iopmctl->ic_hicopy_mp->b_wptr)->cm_close = 1;
	((struct copy_msg *)iopmctl->ic_locopy_mp->b_wptr)->cm_close = 1;
	sendmptoiopm(mp);

	/* The drivers in the IOPM !MUST! return from a close. */
	/* This driver blindly assumes that the IOPM os will return the close */
	/* message and the IOPM os blindly assumes that the drivers will */
	/* return when it calls their close routines. */
	slpdisp = STOPRI | PCATCH;
	savpri = splstr();
	while ( !(iopmctl->ic_kflags & PM_DONE) )
		if ( sleep(mp, slpdisp) )
		{
			splx(savpri);
			sigiopm(wqp);
			slpdisp = STHIPRI;
			savpri = splstr();
		}

	splx(savpri);

	if ( u.u_ttyp == &iopmctl->ic_pgrp || !iopmscbp->iopms_ttyp )
		u.u_ttyp = 0;

	if ( iopmscbp->iopms_err )
		u.u_error = iopmscbp->iopms_err;

	if ( mp->b_datap->db_type == M_CLOSES )
	{
		close_mblk_hook = mp;
		wakeup(&close_mblk_hook);
	}
	else
		freemsg(mp);

	teardwn_ctl(wqp);
}

/******************************************************************************/
static
init_mblk_hooks()
{
	register mblk_t  *mp;

	ASSERT(!sig_mblk_hook);
	ASSERT(!close_mblk_hook);

	while ( !(mp = allocb(sizeof(int), BPRI_LO)) )
		if ( strwaitbuf(sizeof(int), BPRI_LO) )
		{
			cmn_err(CE_WARN,
			  "init_mblk_hooks: no msg blk for signal");
			return 0;
		}

	sig_mblk_hook = mp;

	while ( !(mp = allocb(sizeof(int), BPRI_LO)) )
		if ( strwaitbuf(sizeof(int), BPRI_LO) )
		{
			cmn_err(CE_WARN,
			  "init_mblk_hooks: no msg blk for close");
			freemsg(sig_mblk_hook);
			sig_mblk_hook = 0;
			return 0;
		}

	close_mblk_hook = mp;

	return 1;
}

/******************************************************************************/
static struct iopm_ctl *
init_ctl(wqp, dev)
queue_t  *wqp;
dev_t    dev;
{
	register mblk_t           *mp;
	register struct iopm_ctl  *iopmctl;

	while ( !(mp = allocb(sizeof(struct iopm_ctl), BPRI_LO)) )
		if ( strwaitbuf(sizeof(struct iopm_ctl), BPRI_LO) )
		{
			cmn_err(CE_WARN,
			  "init_ctl: can't allocate IOPM ctl blk");
			return NULL;
		}

	bzero(mp->b_wptr, sizeof(struct iopm_ctl));
	iopmctl = (struct iopm_ctl *)mp->b_wptr;
	wqp->q_ptr = (caddr_t)iopmctl;
	iopmctl->ic_mblkp = mp;
	iopmctl->ic_version = CTL_VERSION;
	iopmctl->ic_krq = RD(wqp);
	iopmctl->ic_strhd = (struct stdata *)RD(wqp)->q_next->q_ptr;
	ASSERT(iopmctl->ic_strhd->sd_wrq == WR(RD(wqp)->q_next));
	iopmctl->ic_iopmslot = IOPMSLOT;
	iopmctl->ic_iopmdev = iopmstbl[minor(dev) >> MNPGSHFT].iopmdev +
	  (minor(dev) & MNPG - 1);
	set_list_avail(RD(wqp));

	/* get hi pri copy mblk for this stream */
	while ( !(mp = allocb(sizeof(struct copy_msg), BPRI_LO)) )
		if ( strwaitbuf(sizeof(struct copy_msg), BPRI_LO) )
		{
			cmn_err(CE_WARN,
			  "init_ctl: no msg blk for hicopy requests");
			teardwn_ctl(wqp);
			return NULL;
		}

	mp->b_datap->db_type = M_HICOPY;
	iopmctl->ic_hicopy_mp = mp;
	((struct copy_msg *)mp->b_wptr)->cm_use = 0;
	((struct copy_msg *)mp->b_wptr)->cm_close = 0;

	/* get lo pri copy mblk for this stream */
	while ( !(mp = allocb(sizeof(struct copy_msg), BPRI_LO)) )
		if ( strwaitbuf(sizeof(struct copy_msg), BPRI_LO) )
		{
			cmn_err(CE_WARN,
			  "init_ctl: no msg blk for locopy requests");
			teardwn_ctl(wqp);
			return NULL;
		}

	mp->b_datap->db_type = M_LOCOPY;
	iopmctl->ic_locopy_mp = mp;
	((struct copy_msg *)mp->b_wptr)->cm_use = 0;
	((struct copy_msg *)mp->b_wptr)->cm_close = 0;

	/* get write flow mblk for this stream */
	while ( !(mp = allocb(sizeof(int), BPRI_LO)) )
		if ( strwaitbuf(sizeof(int), BPRI_LO) )
		{
			cmn_err(CE_WARN,
			  "init_ctl: no msg blk for wflow requests");
			teardwn_ctl(wqp);
			return NULL;
		}

	mp->b_datap->db_type = M_WFLOW;
	iopmctl->ic_wflow_mp = mp;

	/* get read flow mblk for this stream */
	while ( !(mp = allocb(sizeof(int), BPRI_LO)) )
		if ( strwaitbuf(sizeof(int), BPRI_LO) )
		{
			cmn_err(CE_WARN,
			  "init_ctl: no msg blk for rflow requests");
			teardwn_ctl(wqp);
			return NULL;
		}

	mp->b_datap->db_type = M_RFLOW;
	iopmctl->ic_rflow_mp = mp;

	return iopmctl;
}

/******************************************************************************/
static void
teardwn_ctl(wqp)
queue_t  *wqp;
{
	struct iopm_ctl  *iopmctl;
	register mblk_t  *cmp;
	int              savpri;

	ASSERT(valid_wq(wqp));
	iopmctl = (struct iopm_ctl *)wqp->q_ptr;
	ASSERT(valid_iopmctl(iopmctl));
	iopmctl->ic_version = 0;
	/* if the copy blk's are not in use, free them now */
	/* else mark the mblk so that in returncpmsg it will be freed */
	savpri = splstr();
	if ( cmp = iopmctl->ic_hicopy_mp )
		if ( !((struct copy_msg *)cmp->b_rptr)->cm_use )
			freemsg(cmp);
		else
			((struct copy_msg *)cmp->b_rptr)->cm_use = 0;

	if ( cmp = iopmctl->ic_locopy_mp )
		if ( !((struct copy_msg *)cmp->b_rptr)->cm_use )
			freemsg(cmp);
		else
			((struct copy_msg *)cmp->b_rptr)->cm_use = 0;
	splx(savpri);

	freemsg(iopmctl->ic_wflow_mp);
	freemsg(iopmctl->ic_rflow_mp);
	freemsg(iopmctl->ic_mblkp);	/* free control struct */
	wqp->q_ptr = 0;
}

/******************************************************************************/
void
init_iopmscb(iopmscbp)
struct iopmscb  *iopmscbp;
{
	iopmscbp->iopms_pid = u.u_procp->p_pid;
	iopmscbp->iopms_pgrp = u.u_procp->p_pgrp;
	iopmscbp->iopms_session_id = u.u_procp->p_session_id;
	iopmscbp->iopms_uid = u.u_uid;
	iopmscbp->iopms_gid = u.u_gid;
	iopmscbp->iopms_ruid = u.u_ruid;
	iopmscbp->iopms_rgid = u.u_rgid;
	iopmscbp->iopms_err = 0;
	iopmscbp->iopms_ttyp = u.u_ttyp;
	iopmscbp->iopms_savsig = 0;
}

/******************************************************************************/
/* Send a kill signal to the IOPM.
*/
void
sigiopm(wqp)
queue_t  *wqp;
{
	mblk_t  *mp;
	int     savpri;

	if ( !(mp = allocb(sizeof(short), BPRI_HI)) )
	{
		while ( !(mp = sig_mblk_hook) )
			sleep(&sig_mblk_hook, STHIPRI);

		sig_mblk_hook = 0;
		ASSERT(mp->b_datap->db_type == M_KILLS);
	}

	mp->b_prev = (mblk_t *)wqp->q_ptr;	/* iopmctl struct */
	if ( mp->b_datap->db_type != M_KILLS )
		mp->b_datap->db_type = M_KILL;

	*(short *)mp->b_wptr = u.u_procp->p_pid;
	((struct iopm_ctl *)wqp->q_ptr)->ic_kflags &= ~SIGDONE;
	sendmptoiopm(mp);

	savpri = splstr();
	while ( !(((struct iopm_ctl *)wqp->q_ptr)->ic_kflags & SIGDONE) )
		sleep(mp, STHIPRI);
	splx(savpri);

	if ( mp->b_datap->db_type == M_KILLS )
	{
		sig_mblk_hook = mp;
		wakeup(&sig_mblk_hook);
	}
	else
		freemsg(mp);
}

/******************************************************************************/
/* driver layers - puts and service */
/******************************************************************************/
/* The code is structured as a series of layers. Two of the layers are not
/* real streams modules, but instead only have 'put' routines. The other layer
/* is a real streams module with queues. The layers from the top toward the
/* kernel/IOPM boundary are:
/*	flow layer - real streams module; flow control across kernel/IOPM
/*	mux layer - associate mp with stream using b_prev
/*	cmd layer - deal with stream head stuff, eg open, ioctl
*/

/******************************************************************************/
/* FLOW LAYER */
#define WFLOW_STATE  ((struct iopm_ctl *)wqp->q_ptr)->ic_wflow_state
/******************************************************************************/
/* If the IOPM write queue is not full send the mblk there (thru cmd_wput).
/* If it is full, put the mblk on the driver write queue. Since the hi water
/* mark is 1, the queue will immediately be full.
*/
static int
iopms_wput(wqp, mp)
register queue_t  *wqp;
register mblk_t   *mp;
{
	ASSERT(valid_wq(wqp));
	ASSERT(valid_mblk(mp));
	ASSERT(valid_iopmctl((struct iopm_ctl *)wqp->q_ptr));

	/* if hi pri or canput then putnext else putq */
	if ( mp->b_datap->db_type >= QPCTL ||
	     WFLOW_STATE == EMPTY && icanput(wqp) )
	{
		if ( mp->b_datap->db_type == M_FLUSH && *mp->b_rptr & FLUSHW )
			flushq(wqp, FLUSHALL);

		mux_wput(wqp, mp);
	}
	else
	{
		send_flow_msg(wqp);
		WFLOW_STATE = CLOGGED;
		putq(wqp, mp);
	}
}

/******************************************************************************/
static int
iopms_wsrv(wqp)
register queue_t  *wqp;
{
	register mblk_t  *mp;
	int              savpri;

	ASSERT(valid_wq(wqp));
	ASSERT(valid_iopmctl((struct iopm_ctl *)wqp->q_ptr));

	WFLOW_STATE = DRAINING;

	while ( (mp = getq(wqp)) && icanput(wqp) )
		mux_wput(wqp, mp);

	if ( mp )
	{
		send_flow_msg(wqp);
		WFLOW_STATE = CLOGGED;
		ASSERT(mp->b_datap->db_type < QPCTL);
		putbq(wqp, mp);
	}
	else
	{
		savpri = splstr();
		if ( WFLOW_STATE == DRAINING )
			WFLOW_STATE = EMPTY;
		splx(savpri);
	}
}

/******************************************************************************/
static int
iopms_rput(rqp, mp)
register queue_t  *rqp;
register mblk_t   *mp;
{
	int  savmap;

	ASSERT(valid_rq(rqp));
	ASSERT(valid_mblk(mp));

	PRINT(SREAD,"iopms_rput: ");
	SPRINT_TYPE(SREAD, mp);
	PRINT(SREAD,"\n");

	if ( mp->b_datap->db_type >= QPCTL )
	{
		if ( mp->b_datap->db_type == M_FLUSH && *mp->b_rptr & FLUSHR )
			flushrq(rqp);

		putnext(rqp, mp);
	}
	else
	{
		putq(rqp, mp);
	}

	set_list_avail(rqp);
}

/******************************************************************************/
static int
iopms_rsrv(rqp)
register queue_t  *rqp;
{
	register mblk_t  *mp;

	ASSERT(valid_rq(rqp));

	while ( (mp = getq(rqp)) && canput(rqp->q_next) )
		if ( mp->b_datap->db_type == M_RFLOW )
		{
			set_list_avail(rqp);
			ASSERT(valid_iopmctl((struct iopm_ctl *)WR(rqp)->q_ptr));
			mp->b_prev = (mblk_t *)WR(rqp)->q_ptr;
			sendmptoiopm(mp);
		}
		else
			putnext(rqp, mp);

	if ( mp )
	{
		ASSERT(mp->b_datap->db_type < QPCTL);
		putbq(rqp, mp);
	}

	set_list_avail(rqp);
}

/******************************************************************************/
static void
set_list_avail(rqp)
queue_t  *rqp;
{
	register queue_t  *q;
	register uint     avail;
	register uint     count;
	register struct iopm_ctl  *iopmctl;

	ASSERT(valid_rq(rqp));
	iopmctl = (struct iopm_ctl *)WR(rqp)->q_ptr;
	ASSERT(valid_iopmctl(iopmctl));
	q = rqp->q_next;
	ASSERT(valid_rq(q));

	while ( q->q_next && !q->q_qinfo->qi_srvp )
	{	ASSERT(valid_rq(q));
		q = q->q_next;
	}

#ifndef FLOWTEST
	if ( q->q_qinfo->qi_srvp ||
	     WR(q) == iopmctl->ic_strhd->sd_wrq &&	/* got to the head */
	     !(iopmctl->ic_strhd->sd_flag & STPLEX) )	/* not multuplexed */
		avail = q->q_hiwat + 1;
	else
		avail = DEFAULT_HIWAT;
#else
	avail = q->q_hiwat + 1;
#endif

	count = q->q_count + rqp->q_count;
	if ( avail < count )
	{
		PRINT2(SFLOW, "set_list_avail: avail %d, hd/mod rq count %d, ",
		  avail, q->q_count);
		PRINT1(SFLOW, "flow rq count %d\n", rqp->q_count);

		avail = 0;
	}
	else
		avail -= q->q_count + rqp->q_count;

	iopmctl->ic_ravail = avail;
}

/******************************************************************************/
static
icanput(wqp)
queue_t  *wqp;
{
	register struct iopm_ctl  *iopmctl = (struct iopm_ctl *)wqp->q_ptr;

	ASSERT(valid_wq(wqp));
	ASSERT(valid_iopmctl(iopmctl));

#if defined ( IOPMDEBUG )
	if ( iopmctl->ic_wavail <=
	     iopmctl->ic_wproduced - iopmctl->ic_wconsumed )
	{
		PRINT3(SFLOW, "wavail %d, wproduced %d wconsumed %d\n",
		  iopmctl->ic_wavail, iopmctl->ic_wproduced,
		  iopmctl->ic_wconsumed);
	}
#endif

	return iopmctl->ic_wavail >
	       iopmctl->ic_wproduced - iopmctl->ic_wconsumed;
}

/******************************************************************************/
static
flushrq(rqp)
register queue_t  *rqp;
{
	register int     savspl;
	register mblk_t  *bp, *tmp;
	int              wantw;

	savspl = splstr();
	bp = rqp->q_first;
	rqp->q_first = rqp->q_last = NULL;
	rqp->q_count = 0;
	wantw = rqp->q_flag & QWANTW;
	rqp->q_flag &= ~(QFULL | QWANTW);
	splx(savspl);

	while ( bp )
	{
		tmp = bp->b_next;
		if ( bp->b_datap->db_type == M_RFLOW )
		{
			set_list_avail(rqp);
			ASSERT(valid_iopmctl((struct iopm_ctl *)WR(rqp)->q_ptr));
			bp->b_prev = (mblk_t *)WR(rqp)->q_ptr;
			sendmptoiopm(bp);
		}
		else
			freemsg(bp);

		bp = tmp;
	}

	savspl = splstr();
	if ( wantw )
	{
		/* find nearest back queue with service proc */
		for ( rqp = backq(rqp); rqp && !rqp->q_qinfo->qi_srvp;
		      rqp = backq(rqp) )
			;
		if (rqp)
			qenable(rqp);
	}
	splx(savspl);
}

/******************************************************************************/
/* MUX LAYER */
/******************************************************************************/
static void
mux_wput(wqp, mp)
queue_t  *wqp;
mblk_t   *mp;
{
	ASSERT(valid_wq(wqp));
	ASSERT(valid_mblk(mp));
	ASSERT(valid_iopmctl((struct iopm_ctl *)wqp->q_ptr));

	mp->b_prev = (mblk_t *)wqp->q_ptr;
	cmd_wput(mp);
}

/******************************************************************************/
static void
mux_rput(mp)
mblk_t  *mp;
{
	struct iopm_ctl  *iopmctl;

	ASSERT(valid_mblk(mp));
	iopmctl = (struct iopm_ctl *)mp->b_prev;
	ASSERT(valid_iopmctl(iopmctl));

	if ( iopmctl->ic_kflags & CLOSING )
	{
		freemsg(mp);
		return;
	}

	ASSERT(valid_rq(iopmctl->ic_krq));

	mp->b_prev = 0;
	iopms_rput(iopmctl->ic_krq, mp);
}

/******************************************************************************/
/* CMD LAYER */
/******************************************************************************/
static void
cmd_wput(mp)
mblk_t   *mp;
{
	ASSERT(valid_mblk(mp));
	ASSERT(!(mp->b_datap->db_type & QREMCMD));

	if ( mp->b_datap->db_type == M_IOCTL )
		switch ( ((struct iocblk *)mp->b_rptr)->ioc_cmd )
		{

		    case I_LINK:
		    case I_FIND:
		    case I_LOOK:
		    case I_PUSH:
		    case I_POP:
		    case I_UNLINK:
		    case I_FDINSERT:
			mp->b_datap->db_type |= QREMCMD;
			break;
		}

	sendmptoiopm(mp);
}

/******************************************************************************/
static void
cmd_rput(mp)
register mblk_t  *mp;
{
	register queue_t          *rqp;
	register struct iopm_ctl  *iopmctl;
	int                       savpri;

	ASSERT(valid_mblk(mp));
	iopmctl = (struct iopm_ctl *)mp->b_prev;
	ASSERT(valid_iopmctl(iopmctl));
	rqp = iopmctl->ic_krq;
	ASSERT(valid_rq(rqp));

	if ( !(mp->b_datap->db_type & QREMCMD) )
	{
		mux_rput(mp);
		return;
	}

	switch ( mp->b_datap->db_type )
	{
	    case M_RFLOW:	/* IOPM request to be back enabled */
		ASSERT(M_RFLOW < QPCTL);
		mux_rput(mp);
		return;

	    case M_WFLOW:	/* IOPM back enables kernel */
		ASSERT(iopmctl->ic_wflow_mp == NULL);
		iopmctl->ic_wflow_mp = mp;
		qenable(WR(rqp));
		return;

	    case M_HICOPY:	/* IOPM wants to pass data upstream */
	    case M_LOCOPY:
	    {
		if ( ((struct copy_msg *)mp->b_rptr)->cm_close )
		{
			((struct copy_msg *)mp->b_rptr)->cm_use = 0;
			return;
		}

		if ( !bufcall(((struct copy_msg *)mp->b_rptr)->cm_count,
		  BPRI_LO, returncpmsg, mp) )
			timeout(returncpmsg, mp, HZ);
		return;
	    }

	    case M_OPEN:
	    case M_CLOSE:
	    case M_CLOSES:
		iopmctl->ic_kflags |= PM_DONE;
		wakeup(mp);
		return;

	    case M_KILL:
	    case M_KILLS:
		iopmctl->ic_kflags |= SIGDONE;
		wakeup(mp);
		return;

	    case M_IOCACK | QREMCMD:
	    {
		struct iocblk  *iocbp = (struct iocblk *)mp->b_rptr;

		mp->b_datap->db_type &= ~QREMCMD;
		switch ( iocbp->ioc_cmd )
		{
			/* The IOPM returns the modified ttyp in ioc_rval */
			/* The ioc_rval points to a private driver pgrp in
			   the IOPM. */
			/* That pgrp is moved to main memory (ic_pgrp) and
			   ioc_rval is set to point to &ic_pgrp */
			/* strdoioctl will move iocbp_rval to u.u_rval */
			/* strioctl will mov u.u_rval to u.u_ttyp */
		    case I_PUSH:
		    {
			struct iopmscb  *iopmscb;

			iopmscb = (struct iopmscb *)mp->b_cont->b_rptr;
			if ( iopmscb->iopms_ttyp != (short *)iocbp->ioc_rval )
			{
				iopmctl->ic_pgrp = iopmscb->iopms_pgrp;
				iocbp->ioc_rval = (int)&iopmctl->ic_pgrp;
			}
			break;
		    }

		    case I_LINK:
		    {
			struct iopm_ctl  *lower_ctl;
			queue_t          *kdrq;

			lower_ctl = iopmctl->ic_savctl;
			ASSERT(valid_iopmctl(lower_ctl));
			kdrq = lower_ctl->ic_krq;
			ASSERT(valid_rq(kdrq));

			clear_kdownq(kdrq);
			break;
		    }

		    default:
			ASSERT(0);
		}

		mux_rput(mp);
		return;
	    }

	    default:
		ASSERT(0);
	}
}

/******************************************************************************/
static void
send_flow_msg(wqp)
queue_t  *wqp;
{
	register struct iopm_ctl  *iopmctl = (struct iopm_ctl *)wqp->q_ptr;
	register mblk_t            *mp;
	int                        savpri;

	savpri = splstr();

	if ( mp = iopmctl->ic_wflow_mp )
		iopmctl->ic_wflow_mp = NULL;

	splx(savpri);

	if ( mp )
	{
		ASSERT(mp->b_datap->db_type == M_WFLOW);
		mp->b_prev = (mblk_t *)iopmctl;
		sendmptoiopm(mp);
	}
}

/******************************************************************************/
/* if the uint at b_rptr is 0 then close has been called, don't send the mblk
/* to the IOPM. Close did not free the mblk, it must be freed here.
*/
static void
returncpmsg(mp)
register mblk_t  *mp;
{
	if ( ((struct copy_msg *)mp->b_rptr)->cm_use )
		if ( !((struct copy_msg *)mp->b_rptr)->cm_close )
			sendmptoiopm(mp);
		else
			((struct copy_msg *)mp->b_rptr)->cm_use = 0;
	else
	{
		ASSERT(((struct copy_msg *)mp->b_rptr)->cm_close);
		freemsg(mp);
	}
}

/******************************************************************************/
/* This code clears out the kernel portion of the "lower" stream.
/* It is a half hearted attempt at best. A truly malevolent program that writes
/* to a stream head while linking the stream under another will break things.
/* This code catches messages flowing up stream right before the stream is
/* linked under anouther stream.
*/
static void
clear_kdownq(rq)
queue_t  *rq;
{
	ASSERT(valid_rq(rq));
	ASSERT(WR(rq)->q_flag & QREMOTE);

	/* clear the kernel IOPM driver queues */
	if ( rq->q_first )
	{
		cmn_err(CE_WARN, "messages discarded from kdrqdn");
		flushq(rq, FLUSHALL);
	}

	if ( WR(rq)->q_first )
	{
		cmn_err(CE_WARN, "messages discarded from kdwqdn");
		flushq(WR(rq), FLUSHALL);
	}

	/* take the driver off runqueues, bufcall and timeout lists */
	chk_runqueue(rq);
	chk_bufcall(rq);
	chk_timeout(rq, WR(rq));

	rq = rq->q_next;
	ASSERT(valid_rq(rq));
	/* clear the kernel head queues */
	if ( rq->q_first )
	{
		cmn_err(CE_WARN, "messages discarded from khrqdn");
		flushq(rq, FLUSHALL);
	}

	if ( WR(rq)->q_first )
	{
		cmn_err(CE_WARN, "messages discarded from khwqdn");
		flushq(WR(rq), FLUSHALL);
	}

	/* take the head off runqueues, bufcall and timeout lists */
	chk_runqueue(rq);
	chk_bufcall(rq);
	chk_timeout(rq, WR(rq));
}

/******************************************************************************/
/* MSG LAYER */
/******************************************************************************/
static void
sendmptoiopm(mp)
register mblk_t  *mp;
{
	register struct iopm_ctl  *iopmctl;
	register uint             slot;
	register mblk_t           *tmp;
	extern ushort             bsize[];

	ASSERT(valid_mblk(mp));
	iopmctl = (struct iopm_ctl *)mp->b_prev;
	ASSERT(valid_iopmctl(iopmctl));

	PRINT3(SSEND,"sendmptoiopm: kwq %x, iwq %x, mp %x ",
	  WR(iopmctl->ic_krq), iopmctl->ic_iwq, mp);
	SPRINT_TYPE(SSEND, mp);
	PRINT(SSEND,"\n");

	slot = iopmctl->ic_iopmslot;
	ASSERT(is_IOPM(slot));

	for (tmp = mp; tmp; tmp = tmp->b_cont)
	{
		ASSERT(valid_mblk(tmp));
		iopmctl->ic_wproduced += bsize[tmp->b_datap->db_class];
	}

	if ( !add_chain(&iopm_base[slot]->ktois, mp, &mp->b_next) )
		intriop(slot);
}

/******************************************************************************/
iopms_intr()
{
	register mblk_t           *mp;
	register mblk_t           *next_mp;
	register mblk_t           *tmp;
	register struct iopm_ctl  *iopmctl;
	extern mblk_t             *rev_mblk_chain();

	mp = (mblk_t *)remove_chain(&itoks);

	ASSERT(valid_mblk(mp) || mp == 0);

	if ( mp && mp->b_next )
		mp = rev_mblk_chain(mp);

	while ( mp )
	{
		ASSERT(valid_mblk(mp));

		next_mp = mp->b_next;
		mp->b_next = 0;

		iopmctl = (struct iopm_ctl *)mp->b_prev;
		ASSERT(valid_pm_msg(iopmctl));

		for (tmp = mp; tmp; tmp = tmp->b_cont)
		{
			ASSERT(valid_mblk(tmp));
			iopmctl->ic_rconsumed += bsize[tmp->b_datap->db_class];
		}

		PRINT3(SRECV,"iopms_intr: krq %x, irq %x, mp %x ",
		  iopmctl->ic_krq, RD(iopmctl->ic_iwq), mp);
		SPRINT_TYPE(SRECV, mp);
		PRINT(SRECV,"\n");

		cmd_rput(mp);

		mp = next_mp;
	}
	upkern_dec();
}

/******************************************************************************/
/* misc routines */
/******************************************************************************/
static mblk_t *
rev_mblk_chain(mp)
register mblk_t  *mp;
{
	register mblk_t  *lastp = 0;
	register mblk_t  *savforw;

	ASSERT(valid_mblk(mp));
	ASSERT(valid_mblk(mp->b_next));

	do {
		savforw = mp->b_next;
		mp->b_next = lastp;
		lastp = mp;
		mp = savforw;
		ASSERT(valid_mblk(mp) || mp == 0);
	} while ( mp );

	return lastp;
}

/******************************************************************************/
valid_iopm_addr(addr, slot)
uint addr;
uint slot;
{
	ASSERT(is_IOPM(slot));

	return addr >= IOPM_RAM_START &&
	  addr < IOPM_RAM_START + iopm_base[slot]->memsize;
}

valid_pm_addr(addr)
uint addr;
{
	extern uint  physmem;

	return addr >= MAINSTORE && addr < MAINSTORE + ctob(physmem);
}

valid_queue(addr)
uint addr;
{
	return addr >= (uint)queue && addr < (uint)&queue[v.v_nqueue] &&
	  !((addr - (uint)queue) % sizeof(queue_t));
}

valid_rq(q)
queue_t  *q;
{
	if ( !valid_queue((uint)q) )
		return 0;

	return (q->q_flag & (QUSE | QREADR)) == (QUSE | QREADR);
}

valid_wq(q)
queue_t  *q;
{
	if ( !valid_queue((uint)q) )
		return 0;

	return (q->q_flag & (QUSE | QREADR)) == QUSE;
}

valid_pm_mblk(addr)
uint addr;
{
	return addr >= (uint)mblock && addr < (uint)&mblock[nmblock] &&
	  !((addr - (uint)mblock) % sizeof(mblk_t));
}

valid_mblk(mp)
mblk_t  *mp;
{
	if ( !valid_pm_mblk((uint)mp) )
		return 0;

	return mp->b_datap->db_ref;
}

valid_pm_msg(addr)
uint addr;
{
	extern caddr_t  strmsgbase;
	extern caddr_t  strmsgend;

	return addr >= (uint)strmsgbase && addr < (uint)&strmsgend;
}

static
valid_iopmctl(iopmctl)
struct iopm_ctl  *iopmctl;
{
	if ( !valid_pm_msg((uint)iopmctl) )
		return 0;

	return iopmctl->ic_version == CTL_VERSION;
}
