/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) config.c: version 4.1 created on 5/2/90 at 19:58:33	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)config.c	4.1	5/2/90 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/

#ident	"@(#)uts/cf:config.c	4.1"

/*
 *	config.c -- configure a kernel
 */

#include <sys/types.h>
#include <sys/sysmacros.h>
#include <sys/conf.h>
#include "in_data.h"

/* defines and typedefs for the device, linesw, filesw, and fmodsw tables */

#define LSIZE		256	/* max linesw table			*/

#define MAX_CONTROLLER	80	/* maximum number of controllers / system*/
#define MAX_PHYSICAL	4096	/* max # of physical devices / controller*/
#define MAX_LOGICAL	4096	/* max # of logical devices / phys dev	*/

/* name arrays and defines */

char		*driver_funcs[] = {
#define DF_INIT			0
	"init",
#define DF_START		1
	"start",
#define DF_POWER		2
	"power",
#define DF_OPEN			3
	"open",
#define DF_CLOSE		4
	"close",
#define DF_READ			5
	"read",
#define DF_WRITE		6
	"write",
#define DF_IOCTL		7
	"ioctl",
#define DF_STRATEGY		8
	"strategy",
#define DF_PRINT		9
	"print",
	NULL
};
#define NUM_DRIVER_FUNCS	(NEL(driver_funcs) - 1)


char		*filesys_funcs[] = {
#define FF_INIT			0
	"init",
#define FF_IPUT			1
	"iput",
#define FF_IREAD		2
	"iread",
#define FF_IUPDATE		3
	"iupdat",
#define FF_READI		4
	"readi",
#define FF_WRITEI		5
	"writei",
#define FF_ITRUNC		6
	"itrunc",
#define FF_STATF		7
	"statf",
#define FF_NAMEI		8
	"namei",
#define FF_MOUNT		9
	"mount",
#define FF_UMOUNT		10
	"umount",
#define FF_GETINODE		11
	"getinode",
#define FF_OPENI		12
	"openi",
#define FF_CLOSEI		13
	"closei",
#define FF_UPDATE		14
	"update",
#define FF_STATFS		15
	"statfs",
#define FF_ACCESS		16
	"access",
#define FF_GETDENTS		17
	"getdents",
#define FF_ALLOCMAP		18
	"allocmap",
#define FF_FREEMAP		19
	"freemap",
#define FF_READMAP		20
	"readmap",
#define FF_SETATTR		21
	"setattr",
#define FF_NOTIFY		22
	"notify",
#define FF_FCNTL		23
	"fcntl",
#define FF_FSINFO		24
	"fsinfo",
#define FF_IOCTL		25
	"ioctl",
	NULL
};
#define NUM_FILESYS_FUNCS	(NEL(filesys_funcs) - 1)


char		*linedis_funcs[] = {
#define LF_OPEN			0
	"open",
#define LF_CLOSE		1
	"close",
#define LF_READ			2
	"read",
#define LF_WRITE		3
	"write",
#define LF_IOCTL		4
	"ioctl",
#define LF_IN			5
	"in",
#define LF_OUT			6
	"out",
#define LF_MODEM		7
	"modem",
	NULL
};
#define NUM_LINEDIS_FUNCS	(NEL(linedis_funcs) - 1)

#define new_device()	((device_t *)alloc((int)sizeof(device_t)))

typedef struct device {
	struct device	*next;
	char		name[10];	/* device name			*/
	char		prefix[10];	/* function prefix		*/
	unsigned	blockdev : 1;	/* is a block device		*/
	unsigned	chardev : 1;	/* is a character device	*/
	unsigned	streamdev : 1;	/* is a stream device		*/
	unsigned	linedis : 1;	/* uses a line discipline	*/
	unsigned	required : 1;	/* always include this driver	*/
	unsigned	mirrorable : 1;	/* device can be mirrored	*/
	unsigned char	funcs[NUM_DRIVER_FUNCS];
	unsigned char	mpflags[NUM_DRIVER_FUNCS];
	unsigned short	xoffset;	/* extended minor device offset	*/
	char		maj;		/* major device number		*/
	char		parent;		/* major device number of parent*/
	char		next_maj;	/* next major number in group	*/
	short		mult;		/* controller multiplier	*/
	short		controller;	/* max number of controllers	*/
	short		phys;		/* max # of physical devices / ctrlr */
	short		log;		/* max # of logical devices / phys */
	short		line;		/* linesw entry if linedis true	*/
} device_t;

#define new_linedis()	((linedis_t *)alloc((int)sizeof(linedis_t)))

typedef struct linedis {
	char		name[10];	/* line discipline name		*/
	short		number;		/* line discipline number	*/
	unsigned	config : 1;	/* configured flag		*/
	unsigned	required : 1;	/* required flag		*/
	unsigned char	funcs[NUM_LINEDIS_FUNCS];
} linedis_t;

#define new_filesys()	((filesys_t *)alloc((int)sizeof(filesys_t)))

typedef struct filesys {
	struct filesys	*next;		/* linked list			*/
	char		name[10];	/* file system name		*/
	char		prefix[10];	/* file system function prefix	*/
	char		pipedev[32];	/* filesys pipedev mount point	*/
	char		fsflags[32];	/* file system flags		*/
	char		notifyflags[32];/* notify function flags	*/
	short		number;		/* file system number		*/
	unsigned	config : 1;	/* configured flag		*/
	unsigned	required : 1;	/* required flag		*/
	unsigned int	mpflags[NUM_FILESYS_FUNCS];
	unsigned char	funcs[NUM_FILESYS_FUNCS];
} filesys_t;

#define new_strmod()	((strmod_t *)alloc((int)sizeof(strmod_t)))

typedef struct strmod {
	struct strmod	*next;		/* linked list			*/
	short		number;		/* stream module number		*/
	char		name[9];	/* module name	*/
	char		prefix[10];	/* stream module prefix		*/
	unsigned	config : 1;	/* configured flag		*/
	unsigned	required : 1;	/* required flag		*/
} strmod_t;

#define new_param()	((param_t *)alloc((int)sizeof(param_t)))

typedef struct param {
	struct param	*next;
	char		name[32];	/* parameter's name		*/
	char		value[64];	/* actual parameter value	*/
} param_t;


device_t	*devices[DSIZE];	/* pointers to devices		*/
linedis_t	*linedis[LSIZE];	/* pointers to line disciplines	*/
device_t	*devicehead;		/* head of devices linked list	*/
filesys_t	*filesyshead;		/* head of file sys linked list	*/
strmod_t	*strmodhead;		/* head of stream modules list	*/
param_t		*paramhead;		/* parameters linked list head	*/

device_t	*mirror;		/* mirror device		*/

#define DEF_C_FILE	"conf.c"	/* default .c, .h, and master	*/
#define DEF_H_FILE	"config.h"
#define DEF_MASTER_DIR	"/etc/master.d"

char		*c_file;		/* name of .c file		*/
char		*h_file;		/* name of .h file		*/
char		*master_dir;		/* master directory		*/
char		*s_file;		/* name of system file		*/
int		debugflag;		/* debug flag			*/
FILE		*c_fp;			/* c_file file pointer		*/
FILE		*h_fp;			/* h_file file pointer		*/
FILE		*out_fp;		/* outstr output file pointer	*/

unsigned char	nodev_funcs[NUM_FILESYS_FUNCS];
unsigned char	mp_dummy[NUM_FILESYS_FUNCS] = {
	1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 
	21, 22, 23, 24, 25, 26
};

int		errcount;		/* error counter		*/
char		*progname;		/* program name			*/

/* some local functions */

FILE		*doopen();
char		*pathend();
linedis_t	*get_linedis();

/* misc. external declarations */

extern int	errno;
extern char	*sys_errlist[];
extern char	*optarg;
extern int	optind;

extern char	*strcpy(), *strncpy(), *strchr(), *strrchr(), *memcpy();
extern char	*ctime();
extern void	exit(), qsort();
extern long	time();
char vers[] = "25.1";	/* the sccs version of this file */


main(argc, argv)
int	argc;
char	**argv;
{
	initialize(argc, argv);

	parse_system();
	dump_results();

	die(errcount);
	return (0);
}


initialize(argc, argv)
int	argc;
char	**argv;
{
	register int	i;
	register char	*tyme;
	long		clock;

	progname = pathend(*argv);

	while ((i = getopt(argc, argv, "h:c:lm:d")) != EOF) {
		switch (i) {
		case 'h':	/* specify .h filename */
			h_file = optarg;

		CASE 'c':	/* specify .c filename */
			c_file = optarg;

		CASE 'l':	/* specify machine dep .s filename (DEC only) */
			printf("%s: -l option not needed on ARIX computers\n",
			  progname);
			die(1);

		CASE 'm':	/* specify master directory */
			master_dir = optarg;

		CASE 'd':	/* turn on debug mode */
			++debugflag;

		DEFAULT:	/* bad argument */
			usage(CNULL);
		}
	}

	if (optind >= argc)
		usage("You must specify a system file.");
	s_file = argv[optind];

	if (c_file == NULL)
		c_file = DEF_C_FILE;
	if (h_file == NULL)
		h_file = DEF_H_FILE;
	if (master_dir == NULL)
		master_dir = DEF_MASTER_DIR;

	if (debugflag) {
		printf("%s debug mode on\n\n", progname);
		printf("cfile:\t%s\n", c_file);
		printf("hfile:\t%s\n", h_file);
		printf("master:\t%s\n\n", master_dir);
	}
	c_fp = doopen(c_file, "w");
	h_fp = doopen(h_file, "w");
	out_fp = c_fp;

	/* get the current date and time */
	clock = time((long *)0);
	tyme = ctime(&clock);
	tyme[strlen(tyme)-1] = '\0'; /* get rid of the new-line in string  */

	/*
	 * Print configuration file headings.
	 */
	fprintf(h_fp, "/*\n *  Configuration header information\n */\n\n");
	fprintf(h_fp, "#ident\t\"%s:  %s. VER: %s\"\n\n", h_file, tyme, vers);
	fprintf(h_fp, "#define _5000_60_80      1\n\n");

	fprintf(c_fp, "/*\n *  Configuration table information\n */\n\n");
	fprintf(c_fp, "#ident\t\"%s:  %s. VER: %s\"\n\n", c_file, tyme, vers);
	fprintf(c_fp, "#include\t\"%s\"\n\n", h_file);
}


/*
 * usage -- print a usage message and exit
 */

usage(msg)
char	*msg;
{
	if (msg != NULL)
		printf("%s\n\n", msg);

	printf("Usage: config [-c file][-h file][-m dir][-d] system_file\n");
	printf("\nWhere:\n");
	printf("-c file		Sets the filename of the produced .c file.\n");
	printf("-h file		Sets the filename of the produced .h file.\n");
	printf("-m dir		Sets the master file directory.\n");
	printf("-d		Turns on debugging.\n");
	printf("system_file	Contains the system configuration info.\n");
	die(1);
}


/*
 * die -- close files and exit
 */

die(exval)
int	exval;
{
	if (c_fp)
		fclose(c_fp);
	if (h_fp)
		fclose(h_fp);
	fflush(stdout);
	exit(exval);
}


/*
 * nextline -- get the next line of input, returns EOF or 0
 */

nextline(inp)
register in_t	*inp;
{
	if (fgets(inp->in_buf, (int)sizeof(inp->in_buf), inp->in_fp) == NULL) {
		inp->in_ptr = NULL;			/* EOF: zap last line */
		return (EOF);
	}
	++inp->in_lineno;
	inp->in_ptr = inp->in_buf;
	return (0);
}


/*
 * parse_system -- parses the system file
 */

parse_system()
{
	register in_t	*inp;
	in_t		input;
	char		word[MAXLEN];
	static char	*system_sections[] = {	/* system file sections	*/
		"%INCLUDE",
		"%PARAM",
		NULL
	};

	if (debugflag)
		printf("parse_system\n");

	inp = &input;
	openfile(inp, s_file);

	while (findword(inp, word) != EOF) {
		switch (selectlist(word, system_sections)) {
		case 0:		/* %INCLUDE */
			parse_include(inp);
		CASE 1:		/* %PARAM */
			parse_param(inp);
		DEFAULT:
			err(inp, "Not a system section header: '%s'", word);
			printf("\nValid headers:  ");
			printlist(system_sections);
		}
	}

	fclose(inp->in_fp);
}


/*
 * parse_master -- opens and parses the named master file in the master dir
 */

parse_master(file, conf)
char	*file;
int	conf;
{
	register in_t	*inp;
	in_t		input;
	char		word[MAXLEN];
	static char	*master_sections[] = {	/* master file sections	*/
		"%CONFIG",
		"%DRIVER",
		"%FILESYS",
		"%LINEDIS",
		"%PARAM",
		"%STRMODULE",
		NULL
	};

	sprintf(word, "%s/%s", master_dir, file);
	if (debugflag)
		printf("parse_master: %s (%d)\n", word, conf);
	inp = &input;
	openfile(inp, word);

	while (findword(inp, word) != EOF) {
		switch (selectlist(word, master_sections)) {
		case 0:		/* %CONFIG */
			parse_config(inp);
		CASE 1:		/* %DRIVER */
			parse_driver(inp, conf);
		CASE 2:		/* %FILESYS */
			parse_filesys(inp, conf);
		CASE 3:		/* %LINEDIS */
			parse_linedis(inp, conf);
		CASE 4:		/* %PARAM */
			parse_param(inp);
		CASE 5:		/* %STRMODULE */
			parse_strmodule(inp, conf);
		DEFAULT:
			err(inp, "Not a master section header: '%s'", word);
			printf("Valid headers:  ");
			printlist(master_sections);
		}
	}

	fclose(inp->in_fp);
}


/*
 * parse_include -- parse the include section in a system file
 *	Parse each master file named in the section.  One name per line.
 *
 *	Format:	<master_file_name>	[<num_controllers_or_flag>]
 */

parse_include(inp)
register in_t	*inp;
{
	register int	num;
	char		name[MAXLEN];

	if (debugflag)
		printf("parse_include: %s\n", inp->in_filename);

	while (findcmd(inp, name) == 0) {
		num = get_number(inp, 0, MAX_CONTROLLER,
		  (char *)NULL, "Invalid number of controllers");
		parse_master(name, num);
	}
}


/*
 * parse_param -- parse the param section of a file
 *	Format:  param_name [value ....]
 */

parse_param(inp)
register in_t	*inp;
{
	char	name[MAXLEN], value[MAXLEN];

	if (debugflag)
		printf("parse_param: %s\n", inp->in_filename);

	while (findcmd(inp, name) == 0) {
		*value = NULL;
		strbuild(inp, value, STRSIZE(paramhead->value), "Param Value");
		param_set(name, value);
	}
}


/*
 * get_param -- return a pointer to the named parameter, or NULL
 */

param_t *
get_param(name)
register char	*name;
{
	register param_t	*pap;

	for (pap = paramhead; pap != NULL; pap = pap->next) {
		if (strncmp(pap->name, name, STRSIZE(pap->name)) == 0)
			return (pap);
	}
	return (NULL);
}


/*
 * param_set -- set value to the named parameter; will create a new param
 *	if one doesn't already exist.
 */

param_set(name, value)
char	*name, *value;
{
	register param_t	*pap;

	if ((pap = get_param(name)) == NULL) {
		pap = new_param();			/* create new param */
		strncpy(pap->name, name, STRSIZE(pap->name));
		pap->next = paramhead;
		paramhead = pap;
		if (debugflag)
			printf("[NEW] ");
	}
	strncpy(pap->value, value, STRSIZE(pap->value));
	if (debugflag)
		printf("Param %s=%s\n", pap->name, pap->value);
}


/*
 * parse_config -- parse the config section of a master file
 *	All lines to either the end of the file or the next section
 *	header are added to the c_file.
 *
 *	NOTE:  neither the '*' or '#' comments are honored in this section.
 *		Use C style comments instead.
 */

parse_config(inp)
register in_t	*inp;
{
	char	word[MAXLEN];

	if (debugflag)
		printf("parse_config: %s\n", inp->in_filename);

	fprintf(c_fp, "\n/* %%CONFIG from %s */\n\n", inp->in_filename);

	while (nextline(inp) == 0) {
		if (nextword(inp, word) == 0 && *word == '%') {
			ungetline(inp);			/* unget this line */
			return;				/* next section header*/
		}

		fputs(inp->in_buf, c_fp);		/* send line to c_file*/
	}

	inp->in_ptr = NULL;				/* EOF: zap last line */
}


/*
 * parse_driver -- parse the driver section of a master file
 *	Format:
 *		Type:		[driver_type . . .]
 *		Prefix:		function_prefix
 *		Funcs:		[function_name . . .]
 *		Major:		[major_number . . .]
 *		Controller:	max_num_of_controllers
 *		Physical:	num_physical_devices_per_controller
 *		Logical:	num_logical_devices_per_physical_device
 *		LineDis:	line_discipline_name
 *		Multiproc:	[function(s) . . .]
 *		Options:	[option . . .]
 */

parse_driver(inp, numctrlr)
register in_t	*inp;
int		numctrlr;
{
	register device_t	*devp, *ndp, **dpp;
	register int		i, maj, nummajors;
	char			word[MAXLEN];
	unsigned char		majors[DSIZE];
	int			ucharcomp();
	linedis_t		*ldp;
	static char		*driver_fields[] = {
		"Type:",
		"Prefix:",
		"Funcs:",
		"Major:",
		"Controller:",
		"Physical:",
		"Logical:",
		"LineDis:",
		"Options:",
		"Multiproc:",
		NULL
	};
	static char		*driver_types[] = {
		"block",
		"character",
		"stream",
		NULL
	};
	static char		*driver_options[] = {
		"multiproc",
		"required",
		"mirrorable",
		NULL
	};

	if (debugflag)
		printf("parse_driver: %s\n", inp->in_filename);

	devp = new_device();
	devp->controller = devp->phys = devp->log = 1;
	ndp = NULL;
	nummajors = 0;

	if (get_name(inp, inp->in_filename, devp->name, STRSIZE(devp->name),
	  "Invalid driver name", "Driver name too long"))
		return;

	/*
	 * avoid duplicate driver names
	 */
	for (i = DSIZE, dpp = devices; --i >= 0; dpp++)
		if ((ndp = *dpp) != NULL && EQ(devp->name, ndp->name)) {
			err(inp, "Driver name '%s' already exists!",
			  devp->name);
			return;
		}

	while (findcmd(inp, word) == 0) {
		switch (selectlist(word, driver_fields)) {
		case 0:		/* Type: */
			while (nextword(inp, word) == 0) {
				switch (selectlist(word, driver_types)) {
				case 0:	/* block */
					devp->blockdev = 1;
				CASE 1:	/* character */
					devp->chardev = 1;
				CASE 2:	/* stream */
					devp->streamdev = 1;
					devp->chardev = 1;
				DEFAULT:
					err(inp, "Unknown driver type '%s'",
					  word);
					printf("Use one or more of: ");
					printlist(driver_types);
				}
			}

		CASE 1:		/* Prefix: */
			if (get_string(inp, devp->prefix, STRSIZE(devp->prefix),
			  "Missing driver prefix from Prefix:",
			  "Driver prefix already set to",
			  "Driver prefix too long"))
				return;
			if (legal_name(devp->prefix)) {
				err(inp, "Invalid prefix");
				return;
			}

		CASE 2:		/* Funcs: */
			while (nextword(inp, word) == 0) {
				if ((i = selectlist(word, driver_funcs)) >= 0) {
					devp->funcs[i] = 1;
				}
				else {
					err(inp, "Unknown function '%s'", word);
					printf("Use one of: ");
					printlist(driver_funcs);
				}
			}

		CASE 3:		/* Major: */
			while ((i = get_number(inp, 0, DSIZE-1, CNULL,
			  "Invalid major device number")) != EOF) {
				if (nummajors > DSIZE-1) {
					err(inp,
					  "Too many major numbers (max %d)",
					  DSIZE);
					break;
				}
				if (i == MAJ_UNKNOWN) {
					err(inp, "Unassigned major number '-'");
					printf("\tAssign using drvinstall\n");
					break;
				}
				if (devices[i]) {
					err(inp, "Major number %d is already",
					  i);
					printf("in use by driver '%s'\n",
					  devices[i]->name);
					break;
				}
				majors[nummajors++] = i;
			}

		CASE 4:		/* Controller: */
			devp->controller = get_number(inp, 0, MAX_CONTROLLER,
			  "Missing number of controllers",
			  "Invalid number of controllers");

		CASE 5:		/* Physical: */
			devp->phys = get_number(inp, 0, MAX_PHYSICAL,
			  "Missing number of physical devices per controller",
			  "Invalid number of physical devices per controller");

		CASE 6:		/* Logical: */
			devp->log = get_number(inp, 0, MAX_LOGICAL,
			  "Missing number of logical devices per physical dev",
			  "Invalid number of logical devices per physical dev");

		CASE 7:		/* LineDis: */
			*word = 0;
			if (get_string(inp, word, STRSIZE(linedis[0]->name),
			  "Missing line discipline name for LineDis:",
			  "already_set impossible!",
			  "Line discipline name too long"))
				return;
			if (legal_name(word))
				break;

			/* if the line dis isn't already loaded, load it */
			if ((ldp = get_linedis(word)) == NULL) {
				parse_master(word, 1);
				if ((ldp = get_linedis(word)) == NULL) {
					err(inp, "Can't load line dis '%s'",
					  word);
					break;
				}
			}
			devp->linedis = 1;
			devp->line = ldp->number;

		CASE 8:		/* Options: */
			while (nextword(inp, word) == 0) {
				switch (selectlist(word, driver_options)) {
				case 0:	/* multiproc */
					devp->mpflags[DF_READ]	   = 1;
					devp->mpflags[DF_WRITE]	   = 1;
					devp->mpflags[DF_STRATEGY] = 1;
				CASE 1:	/* required */
					devp->required = 1;
				CASE 2:	/* mirrorable */
					devp->mirrorable = 1;
				DEFAULT:
					err(inp, "Unknown option '%s'", word);
					printf("Use one or more of: ");
					printlist(driver_options);
				}
			}

		CASE 9:		/* Multiproc: */
			while (nextword(inp, word) == 0) {
				if ((i = selectlist(word, driver_funcs)) >= 0)
					devp->mpflags[i] = 1;
				else {
					err(inp, "Unknown function '%s'", word);
					printf("Use one of: ");
					printlist(driver_funcs);
				}
			}

		DEFAULT:
			err(inp, "Unknown field '%s'", word);
			printf("Use one of: ");
			printlist(driver_fields);
		}
	}

	if (devp->blockdev == 0 && devp->chardev == 0) {
		err(inp, "Driver requires a Type: {block|character|stream}");
		return;
	}
	if (!(*devp->prefix)) {
		err(inp, "Driver requires a Prefix:");
		return;
	}
	if (nummajors == 0) {
		err(inp, "Driver requires a Major: with one or more unique");
		printf("major device numbers.");
		return;
	}

	if (nummajors > 1)
		qsort((char *)majors, (unsigned)nummajors, sizeof(*majors),
		  ucharcomp);

	if (devp->controller && numctrlr > devp->controller) {
		err(inp, "Can't set controller to %d (max=%d)",
		  numctrlr, devp->controller);
		return;
	}
	devp->mult = numctrlr;
	if (numctrlr < 0) {
		printf("Driver '%s' had no controller multiplier specified on ",
		  devp->name);
		printf("its line in the system file, so it was turned off.\n");
	}
	devp->mpflags[DF_INIT]	= 1;
	devp->mpflags[DF_POWER]	= 1;
	devp->mpflags[DF_START]	= 1;
	devp->mpflags[DF_PRINT]	= 1;

	for (i = 0; i < nummajors; i++) {
		if (i == 0) {
			ndp = devp;
		}
		else {
			ndp = new_device();
			*ndp = *devp;
		}
		if (i == nummajors - 1)
			ndp->next_maj = -1;
		else
			ndp->next_maj = majors[i + 1];
		ndp->maj = maj = majors[i];
		ndp->xoffset = TO_MAJOR(maj) - TO_MAJOR(i);
		ndp->parent = devp->maj;

		devices[maj] = ndp;
		devp->next = devicehead;		/* link into list */
		devicehead = devp;
	}
}


/*
 * get_string -- gets the next one word string, checks it and copies it to str
 *	Uses several error strings for different problems.
 */

get_string(inp, str, size, missing, already_set, too_long)
register in_t	*inp;
char		*str;
int		size;
char		*missing, *already_set, *too_long;
{
	char	word[MAXLEN];

	if (nextword(inp, word)) {
		err(inp, missing);
		return (EOF);
	}
	if (*str) {
		err(inp, "%s '%s'", already_set, str);
		return (EOF);
	}
	if (strlen(word) > size) {
		err(inp, "%s (max %d)", too_long, size);
		return (EOF);
	}
	strcpy(str, word);
	return (0);
}


/*
 * get_name -- checks the given name and copy it to str
 *	Uses several error strings for different problems.
 */

get_name(inp, name, str, size, invalid, too_long)
register in_t	*inp;
register char	*name;
char		*str;
int		size;
char		*invalid, *too_long;
{
	name = pathend(name);

	if (strlen(name) > size) {
		err(inp, "%s '%s' (max %d)",
		  (too_long != NULL ? too_long : "Name too long"), name, size);
		return (EOF);
	}
	if (legal_name(name)) {
		err(inp, "%s '%s'",
		  (invalid != NULL ? invalid : "Invalid name"), name);
		return (EOF);
	}
	strcpy(str, name);
	return (0);
}


/*
 * legal_name -- checks for legal C names, returns non-zero on error, else zero
 */

legal_name(name)
register char	*name;
{
	register int	c;
	register char	*p;

	if (name == NULL || *name == '\0') {
		printf("Null names are invalid\n");
		return (1);
	}

	for (p = name; c = *p++; ) {
		if (!isalnum(c) && c != '_') {
			printf("Invalid character '%c' in '%s'\n", c, name);
			return (1);
		}
		if (p == name && isdigit(c)) {
			printf("'%s' Must start with a letter or underscore",
			  name);
			return (1);
		}
	}
	return (0);
}


/*
 * get_linedis -- returns a pointer to the named line discipline, or NULL
 */

linedis_t *
get_linedis(name)
register char	*name;
{
	register linedis_t	*ldp;
	register int		i;

	for (i = 0; i < NEL(linedis); i++) {
		if ((ldp = linedis[i]) == NULL)
			continue;
		if (EQ(ldp->name, name))
			return (ldp);
	}
	return (NULL);
}


/*
 * ucharcomp -- unsigned char comparison function for parse_driver
 */

ucharcomp(a, b)
unsigned char	*a, *b;
{
	return ((int)*a - (int)*b);
}


/*
 * parse_filesys -- parse the file system switch section of a master file
 *	Format:
 *		Prefix:		filesys_function_prefix
 *		FSFlags:	filesys_flags
 *		NotifyFlags:	notify_func_flags
 *		PipeDev:	filesys_pipedev_mount_point
 *		Funcs:		[function(s) . . .]
 *		Multiproc:	[function(s) . . .]
 *		Options:	[option . . .]
 */

parse_filesys(inp, conf)
register in_t	*inp;
int		conf;
{
	register filesys_t	*fsp, *lfp;
	register int		i;
	char			word[MAXLEN];
	static char		*filesys_fields[] = {
		"Prefix:",
 		"FSFlags:",
 		"NotifyFlags:",
 		"PipeDev:",
		"Funcs:",
		"Options:",
		"Multiproc:",
		NULL
	};
	static char		*filesys_options[] = {
		"required",
		NULL
	};

	if (debugflag)
		printf("parse_filesys: %s\n", inp->in_filename);

	fsp = new_filesys();
	if (get_name(inp, inp->in_filename, fsp->name, STRSIZE(fsp->name),
	  "Invalid filesys name", "Filesys name too long"))
		return;

	for (lfp = filesyshead; lfp != NULL; lfp = lfp->next) {
		if (EQ(lfp->name, fsp->name)) {
			err(inp, "Filesys name '%s' already used", fsp->name);
			return;
		}
	}

	while (findcmd(inp, word) == 0) {
		switch (selectlist(word, filesys_fields)) {
		case 0:		/* Prefix: */
			if (get_string(inp, fsp->prefix, STRSIZE(fsp->prefix),
			  "Missing prefix name from Prefix:",
			  "Filesys prefix already set to",
			  "Filesys prefix too long"))
				return;
			if (legal_name(fsp->prefix)) {
				err(inp, "Invalid prefix");
				return;
			}

		CASE 1:		/* FSFlags: */
			strbuild(inp, fsp->fsflags, STRSIZE(fsp->fsflags),
			  "FSFlags");

		CASE 2:		/* NotifyFlags: */
			strbuild(inp, fsp->notifyflags,
			  STRSIZE(fsp->notifyflags), "NotifyFlags");

		CASE 3:		/* PipeDev: */
			strbuild(inp, fsp->pipedev, STRSIZE(fsp->pipedev),
			  "PipeDev");

		CASE 4:		/* Funcs: */
			while (nextword(inp, word) == 0) {
				if ((i = selectlist(word, filesys_funcs)) >= 0){
					fsp->funcs[i] = 1;
				}
				else {
					err(inp, "Unknown function '%s'", word);
					printf("Use one of: ");
					printlist(filesys_funcs);
				}
			}

		CASE 5:		/* Options: */
			while (nextword(inp, word) == 0) {
				switch (selectlist(word, filesys_options)) {
				case 0:	/* required */
					fsp->required = 1;
				DEFAULT:
					err(inp, "Unknown option '%s'", word);
					printf("Use one or more of: ");
					printlist(filesys_options);
				}
			}

		CASE 6:		/* Multiproc: */
			while (nextword(inp, word) == 0) {
				if ((i = selectlist(word, filesys_funcs)) >= 0){
					fsp->mpflags[i] |= FS_MULTIPROC;
				}
				else {
					err(inp, "Unknown function '%s'", word);
					printf("Use one of: ");
					printlist(filesys_funcs);
				}
			}

		DEFAULT:
			err(inp, "Unknown field '%s'", word);
			printf("Use one of: ");
			printlist(filesys_fields);
		}
	}

	if (!(*fsp->prefix)) {
		err(inp, "Filesys requires a Prefix:");
		return;
	}

	if ((lfp = filesyshead) == NULL) {
		filesyshead = fsp;			/* install it */
	}
	else {
		for ( ; lfp != NULL && lfp->next != NULL; lfp = lfp->next)
			;
		lfp->next = fsp;			/* install it */
	}

	fsp->config = (conf != 0);
}


/*
 * strbuild -- add words onto a string, up to a maximum size
 */

strbuild(inp, str, size, name)
register in_t	*inp;
register char	*str;
int		size;
char		*name;
{
	register int	roomleft;
	char		word[MAXLEN];

	roomleft = size;
	while (nextword(inp, word) == 0) {
		if ((roomleft -= strlen(word)) < 0) {
			err(inp, "%s too long (max %d)", name, size);
			return;
		}
		str = stradd(str, word);
	}
}


/*
 * parse_linedis -- parse the file discipline section of a master file
 *	Format:
 *		Number:		linedis_table_number
 *		Funcs:		[function(s) . . .]
 *		Options:	[option . . .]
 */

parse_linedis(inp, conf)
register in_t	*inp;
int		conf;
{
	register linedis_t	*lnp;
	register int		i;
	char			word[MAXLEN];
	static char		*linedis_fields[] = {
		"Number:",
		"Funcs:",
		"Options:",
		NULL
	};
	static char		*linedis_options[] = {
		"required",
		NULL
	};

	if (debugflag)
		printf("parse_linedis: %s\n", inp->in_filename);

	lnp = new_linedis();
	if (get_name(inp, inp->in_filename, lnp->name, STRSIZE(lnp->name),
	  "Invalid linedis name", "LineDis name too long"))
		return;

	for (i = 0; i < NEL(linedis); i++)
		if (linedis[i] != NULL && EQ(lnp->name, linedis[i]->name)) {
			err(inp, "Line discipline name '%s' already used",
			  lnp->name);
			return;
		}

	while (findcmd(inp, word) == 0) {
		switch (selectlist(word, linedis_fields)) {
		case 0:		/* Number: */
			i = get_number(inp, 0, LSIZE - 1,
			  "Missing linedis number from Number:",
			  "Invalid linedis number");
			if (linedis[i]) {
				err(inp, "Linedis %d is already used by '%s'",
				  i, linedis[i]->name);
				break;
			}
			lnp->number = i;

		CASE 1:		/* Funcs: */
			while (nextword(inp, word) == 0) {
				if ((i = selectlist(word, linedis_funcs)) >= 0){
					lnp->funcs[i] = 1;
				}
				else {
					err(inp, "Unknown function '%s'", word);
					printf("Use one of: ");
					printlist(linedis_funcs);
				}
			}

		CASE 2:		/* Options: */
			while (nextword(inp, word) == 0) {
				switch (selectlist(word, linedis_options)) {
				case 0:	/* required */
					lnp->required = 1;
				DEFAULT:
					err(inp, "Unknown option '%s'", word);
					printf("Use one or more of: ");
					printlist(linedis_options);
				}
			}

		DEFAULT:
			err(inp, "Unknown field '%s'", word);
			printf("Use one of: ");
			printlist(linedis_fields);
		}
	}

	/* check again, just to be sure */
	if (linedis[i = lnp->number] != NULL) {
		err(inp, "LineDis # %d is being used by '%s'",
		  i, linedis[i]->name);
		return;
	}
	linedis[i] = lnp;			/* install it */

	lnp->config = (conf != 0);
}


/*
 * parse_strmodule -- parse the stream module section of a master file
 *	Format:
 *		Prefix:		strmod_prefix
 *		Options:	[option . . .]
 */

parse_strmodule(inp, conf)
register in_t	*inp;
int		conf;
{
	register strmod_t	*smp, *lsmp;
	char			word[MAXLEN];
	static char		*strmod_fields[] = {
		"Prefix:",
		"Options:",
		NULL
	};
	static char		*strmod_options[] = {
		"required",
		NULL
	};

	if (debugflag)
		printf("parse_strmodule: %s\n", inp->in_filename);

	smp = new_strmod();
	if (get_name(inp, inp->in_filename, smp->name, STRSIZE(smp->name),
	  "Invalid strmod name", "StrModule name too long"))
		return;

	for (lsmp = strmodhead; lsmp != NULL; lsmp = lsmp->next)
		if (EQ(lsmp->name, smp->name)) {
			err(inp, "Stream module name '%s' already used",
			 smp->name);
			return;
		}

	while (findcmd(inp, word) == 0) {
		switch (selectlist(word, strmod_fields)) {
		case 0:		/* Prefix: */
			if (get_string(inp, smp->prefix, STRSIZE(smp->prefix),
			  "Missing strmod prefix from Prefix:",
			  "StrModule prefix already set to",
			  "StrModule prefix too long"))
				return;
			if (legal_name(smp->prefix)) {
				err(inp, "Invalid prefix");
				return;
			}

		CASE 1:		/* Options: */
			while (nextword(inp, word) == 0) {
				switch (selectlist(word, strmod_options)) {
				case 0:	/* required */
					smp->required = 1;
				DEFAULT:
					err(inp, "Unknown option '%s'", word);
					printf("Use one or more of: ");
					printlist(strmod_options);
				}
			}

		DEFAULT:
			err(inp, "Unknown field '%s'", word);
			printf("Use one of: ");
			printlist(strmod_fields);
		}
	}

	if (!(*smp->prefix)) {
		err(inp, "StrModule requires a Prefix:");
		return;
	}

	if ((lsmp = strmodhead) == NULL) {
		strmodhead = smp;				/* install it */
	}
	else {
		for ( ; lsmp->next != NULL; lsmp = lsmp->next)
			;
		lsmp->next = smp;				/* install it */
	}

	smp->config = (conf != 0);
}


/*
 * upshift -- returns a pointer to a static upper-case copy of str
 *	Observe the usual precautions in dealing with pointers to static areas.
 */

char *
upshift(str)
char	*str;
{
	register char	*p, *q;
	register int	c;
	static char	buf[64];

	p = buf;
	q = str;
	while (c = *q++) {
		if (islower(c))
			c = TOUPPER(c);
		*p++ = c;
		if (p >= &buf[sizeof(buf)]) {
			printf("upshift:  string too big '%s'\n", str);
			die(1);
		}
	}
	*p = NULL;
	return (buf);
}


/*
 * dump_results -- build the different tables, etc.
 */

dump_results()
{
	dump_params();
	dump_devices();
	dump_init();
	dump_linedis();
	dump_strmod();
	dump_filesys();
	dump_upkern_stubs();
}


/*
 * dump_params -- read out the param list into h_file
 */

dump_params()
{
	register param_t	*pap;

	for (pap = paramhead; pap != NULL; pap = pap->next) {
		fprintf(h_fp, "#define %s\t%s\n", pap->name, pap->value);
	}
}


/*
 * dump_devices -- build the device tables -- sets bmax and cmax
 */

dump_devices()
{
	conf_scan();

	fprintf(c_fp, "\n/* system tables */\n\n");
	fprintf(c_fp, "/*\n * the print func is used only by prdev, which\n");
	fprintf(c_fp, " * checks for NULL\n */\n");
	fprintf(c_fp, "#define NULLPRINT\t(int(*)())0\n\n");
	fprintf(c_fp, "extern\tnodev(), nulldev(), nostrat();\n");
	dev_extern();

	mirror_scan();

	build_bdev();
	build_cdev();
	build_minor();

	build_mirror();
}


/*
 * conf_scan -- scan the device table and deconfigure those devices that:
 *	1) aren't required, and 2) have a controller multiplier less than one
 *
 *	Print the defines for the devices that are left, except that all
 *	devices get the [BCL]_ defines.
 */

conf_scan()
{
	register device_t	*devp, **devpp;
	register char		*up;
	register int		i, num;

	for (i = 0, devpp = devices; i < NEL(devices); i++, devpp++) {
		if ((devp = *devpp) == NULL)
			continue;				/* nodev */
		if (devp->parent != i)
			continue;				/* no dups */

		/* deconfig the devices that aren't needed */

		if (devp->mult <= 0 && !devp->required)
			*devpp = NULL;

		/* do the major/linesw defines for all devices */

		fprintf(h_fp, "\n");
		up = upshift(devp->name);
		if (devp->blockdev)
			fprintf(h_fp, "#define B_%s\t%d\n", up, devp->maj);
		if (devp->chardev)
			fprintf(h_fp, "#define C_%s\t%d\n", up, devp->maj);
		if (devp->linedis)
			fprintf(h_fp, "#define L_%s\t%d\n", up, devp->line);

		if (*devpp == NULL)
			continue;				/* deconfiged */

		/* do the other defines for the survivors */

		num = 1;
		if (devp->mult)		/* use a multiplier over a controller */
			num = devp->mult;
		else if (devp->controller)
			num = devp->controller;
		fprintf(h_fp, "#define %s_0\t%d\n", up, num);
		if (devp->phys)
			num *= devp->phys;
		if (devp->log)
			num *= devp->log;

		fprintf(h_fp, "#define %s_0_PHYS\t%d\n", up, devp->phys);
		fprintf(h_fp, "#define %s_0_LOG\t%d\n", up, devp->log);
		fprintf(h_fp, "#define %s_0_DEV\t%d\n", up, num);
	}
}

/*
 * mirror_scan -- find the mirror driver and delete it from configured devices
 */

mirror_scan()
{
	register int		n;
	register device_t	**devpp, *devp;

	/* first, find the mirror driver */

	for (n = NEL(devices), devpp = devices; --n >= 0; devpp++)
		if ((devp = *devpp) && EQ(devp->name, "mirror")) {
			mirror = devp;
			*devpp = NULL;			/* deconfigure	*/
			return;				/* got it	*/
		}

	/* else, no mirroring */
	mirror = NULL;
	for (n = NEL(devices), devpp = devices; --n >= 0; devpp++)
		if (devp = *devpp)
			devp->mirrorable = 0;
}

/*
 * build_mirror -- build the mirror tables:  mr_devsw, and MR_MINOR
 */

build_mirror()
{
	register device_t	*devp;
	register int		f, maj, cnt, max;
	register char		*fmt;
	register uchar		*multi;
	static char		open_close_str[] =
"static _mr_%s%s(_minor_, _flags_, _op_typ_)\n\
dev_t _minor_; int _flags_, _op_typ_;\n\
{\n\
%s\
\t%s%s(B_%s, _minor_, _flags_, _op_typ_);\n\
%s\
}\n\n",
				read_write_str[] =
"static _mr_%s%s(_minor_)\n\
dev_t _minor_;\n\
{\n\
%s\
\t%s%s(B_%s, _minor_);\n\
%s\
}\n\n",
				ioctl_str[] =
"static _mr_%s%s(_minor_, _commnd_, _arg_, _flags_)\n\
dev_t _minor_; int _commnd_, _arg_, _flags_;\n\
{\n\
%s\
\t%s%s(B_%s, _minor_, _commnd_, _arg_, _flags_);\n\
%s\
}\n\n",
				print_str[] =
"static _mr_%s%s(_dev_, _str_)\n\
dev_t _dev_; char *_str_;\n\
{\n\
%s\
\t%s%s(B_%s, _dev_, _str_);\n\
%s\
}\n\n";

	/*
	 * Note that the values in mrdev_funcs must follow the order of the
	 * fields in struct mr_devsw.
	 */
	static int		mrdev_funcs[] = {
		DF_OPEN, DF_CLOSE, DF_READ, DF_WRITE, DF_IOCTL, DF_STRATEGY,
		DF_PRINT
	};

	if (!mirror)
		return;

	for (max = NEL(devices); --max >= 0; )
		if ((devp = devices[max]) && devp->mirrorable)
			break;

	outstr("\n/* mirror tables */\n\n");

	for (maj = 0; maj <= max; maj++) {
		if ((devp = devices[maj]) == NULL)
			continue;
		if (!devp->mirrorable)
			continue;
		if (devp->parent != maj)
			continue;

		do_extern(NUM_DRIVER_FUNCS, devp->funcs, mp_dummy,
		  devp->prefix, driver_funcs, 0);
	}

	outstr("\nstruct mr_devsw\tmr_devsw[] = {\n");

	for (maj = cnt = 0; maj <= max; maj++) {
		if ((devp = devices[maj]) == NULL || !devp->mirrorable) {
			do_struct(NEL(mrdev_funcs), nodev_funcs, mp_dummy,
			  "ERR!", driver_funcs, mrdev_funcs, maj, 0);
		}
		else {
			do_struct(NEL(mrdev_funcs), devp->funcs, mp_dummy,
			  devp->prefix, driver_funcs, mrdev_funcs, maj, 0);
			++cnt;
		}
		outstr("},\n");
	}
	if (cnt == 0)
		outstr("\t0\n");
	outstr("};\n");
	outstr("uint\tmr_devcnt = sizeof(mr_devsw) / sizeof(*mr_devsw);\n\n");

	outstr("/* mirror stubs */\n\n");

	for (maj = 0; maj <= max; maj++) {
		if ((devp = devices[maj]) == NULL || !devp->mirrorable)
			continue;
		if (devp->parent != maj)
			continue;		/* no need to redeclare dup. */

		multi = devp->mpflags;
		for (f = 0; f < NUM_DRIVER_FUNCS; f++) {
			if (!devp->funcs[f])
				continue;	/* no func */

			switch (f) {
			case DF_OPEN:
			case DF_CLOSE:
				fmt = open_close_str;
			CASE DF_READ:
			case DF_WRITE:
				fmt = read_write_str;
			CASE DF_IOCTL:
				fmt = ioctl_str;
			CASE DF_PRINT:
				fmt = print_str;
			DEFAULT:
				continue;
			}
			out(fmt, devp->prefix, driver_funcs[f],
			  (multi[f] ? "" : "\tupkern_lock();\n"),
			  mirror->prefix, driver_funcs[f], upshift(devp->name),
			  (multi[f] ? "" : "\tupkern_unlock();\n"));
		}
	}

	/*
	 * restore the mirror device in case it has any init or start routines
	 */
	devices[mirror->maj] = mirror;
}


/*
 * dev_extern -- declare the external functions for a device
 */

dev_extern()
{
	register int		i;
	register device_t	*devp, **devpp;

	for (i = 0, devpp = devices; i < NEL(devices); i++, devpp++) {
		if ((devp = *devpp) == NULL)
			continue;		/* nodev		*/
		if (devp->parent != i)
			continue;		/* no need to redeclare dup. */

		do_extern(NUM_DRIVER_FUNCS, devp->funcs, devp->mpflags,
		  devp->prefix, driver_funcs, (int)devp->mirrorable);
		if (devp->streamdev)
			out("extern struct streamtab\t%sinfo;\n", devp->prefix);
	}
}

/*
 * do_extern -- declare external functions
 */

do_extern(num_funcs, funcs, multi, prefix, func_names, mirrored)
register unsigned char	*funcs, *multi;
register char		*prefix, **func_names;
register int		num_funcs, mirrored;
{
	register int		f, cnt;

	for (f = cnt = 0; f < num_funcs; f++) {
		if (funcs[f]) {
			outstr(cnt == 0 ? "extern\t" : ", ");
			/*
			 * Note: only devices should be mirrored
			 */
			if (f == DF_STRATEGY) {
				switch ((mirrored != 0) + 2 * !multi[f]) {
				case 0:		/* no stubs */
					out("%sstrategy", prefix);
				CASE 1:		/* mirror stub */
					out("mrstrategy");
				CASE 2:		/* upkern stub */
					out("_upkern_%sstrategy", prefix);
				CASE 3:		/* mirror and upkern stub */
					out("_upkern_mrstrategy");
				}
			}
			else {
				out("%s%s%s",
				  (mirrored ? "_mr_" :
				   (multi[f] ? "" : "_upkern_")),
				  prefix, func_names[f]);
			}
			outstr("()");
#if 0
			out("%s%s%s%s()", (cnt == 0 ? "extern\t" : ", "),
			  (mirrored ? (f == DF_STRATEGY ? "" : "_mr_") :
			   (multi[f] ? "" : "_upkern_")),
			  prefix, func_names[f]);
#endif
			++cnt;
		}
	}
	if (cnt > 0)
		outstr(";\n");
}


/*
 * build_bdev -- build the bdevsw array
 */

build_bdev()
{
	register device_t	*devp;
	register int		i, max;
	static int		bdev_funcs[] = {
		DF_OPEN, DF_CLOSE, DF_STRATEGY, DF_PRINT
	};

	for (max = NEL(devices); --max >= 0; )
		if ((devp = devices[max]) && devp->blockdev)
			break;

	fprintf(c_fp, "\nstruct bdevsw	bdevsw[] = {\n");

	for (i = 0; i <= max; i++) {
		devp = devices[i];

		if (devp && devp->blockdev) {
			do_struct(NEL(bdev_funcs), devp->funcs, devp->mpflags,
			  devp->prefix, driver_funcs, bdev_funcs, i,
			  devp->mirrorable);
		}
		else {
			do_struct(NEL(bdev_funcs), nodev_funcs, mp_dummy,
			  "ERR!", driver_funcs, bdev_funcs, i, 0);
		}
		outstr("},\n");
	}

	outstr("};\n");
	outstr("bdevcnt = sizeof(bdevsw) / sizeof(*bdevsw);\n\n");
}


/*
 * build_cdev -- build the cdevsw array
 */

build_cdev()
{
	register device_t	*devp;
	register int		i, max;
	static int		cdev_funcs[] = {
		DF_OPEN, DF_CLOSE, DF_READ, DF_WRITE, DF_IOCTL
	};

	for (max = NEL(devices); --max >= 0; )
		if ((devp = devices[max]) && devp->chardev)
			break;

	fprintf(c_fp, "\n#define STR\t(struct streamtab *)0\n");
	fprintf(c_fp, "#define TTy(x)\t((struct tty *)(x))\n");
	fprintf(c_fp, "#define TTY\tTTy(-1)\n");
	fprintf(c_fp, "\nstruct cdevsw	cdevsw[] = {\n");

	for (i = 0; i <= max; i++) {
		devp = devices[i];

		if (devp && devp->chardev) {
			do_struct(NEL(cdev_funcs), devp->funcs, devp->mpflags,
			  devp->prefix, driver_funcs, cdev_funcs, i,
			  devp->mirrorable);
		}
		else {
			do_struct(NEL(cdev_funcs), nodev_funcs, mp_dummy,
			  "ERR", driver_funcs, cdev_funcs, i, 0);
			outstr(", TTY, STR },\n");
			continue;
		}
		if (devp->linedis) {
			out(", TTy(L_%s)", upshift(devp->name));
		}
		else {
			outstr(", TTY");
		}
		if (devp->streamdev) {
			out(", &%sinfo", devp->prefix);
		}
		else {
			outstr(", STR");
		}
		outstr("},\n");
	}

	outstr("};\n");
	fprintf(c_fp, "cdevcnt = sizeof(cdevsw) / sizeof(*cdevsw);\n\n");
}


/*
 * do_struct -- print one structure in a structure array
 */

do_struct(num_funcs, funcs, multi, prefix, func_names, func_list, num, mirrored)
register unsigned char	*funcs;
unsigned char		*multi;
register char		*prefix, **func_names;
register int		num_funcs, *func_list;
int			num;
uint			mirrored;
{
	register int	i, f, funcs_max, func_cnt;

	out("/*%2d*/ {", num);

	/*
	 * if there are no functions at all, use nodev instead of nulldev
	 */
	for (i = func_cnt = 0; i < num_funcs; i++)
		if (funcs[func_list[i]])
			++func_cnt;

	for (i = 0, funcs_max = num_funcs - 1; i < num_funcs; i++) {
		f = func_list[i];

		if (funcs[f]) {
			/*
			 * Note: only devices should be mirrored
			 */
			if (f == DF_STRATEGY) {
				switch ((mirrored != 0) + 2 * !multi[f]) {
				case 0:		/* no stubs */
					out("%sstrategy", prefix);
				CASE 1:		/* mirror stub */
					out("mrstrategy");
				CASE 2:		/* upkern stub */
					out("_upkern_%sstrategy", prefix);
				CASE 3:		/* mirror and upkern stub */
					out("_upkern_mrstrategy");
				}
			}
			else {
				out("%s%s%s",
				  (mirrored ? "_mr_" :
				   (multi[f] ? "" : "_upkern_")),
				  prefix, func_names[f]);
			}
#if 0
			out("%s%s%s",
			  (mirrored ? (f == DF_STRATEGY ? "mr" : "_mr_") :
			   (multi[f] ? "" : "_upkern_")),
			  (mirrored && f == DF_STRATEGY ? "" : prefix),
			  func_names[f]);
#endif
		}
		else if (f == DF_STRATEGY)
			outstr("nostrat");
		else if (f == DF_PRINT)
			outstr("NULLPRINT");
		else
			outstr(func_cnt ? "nulldev" : "nodev");

		if (i < funcs_max)
			outstr(", ");
	}
}


/*
 * build_minor -- build the MINOR array for extended minor devices
 */

build_minor()
{
	register device_t	*devp;
	register int		i, maj, cnt;

	fprintf(c_fp, "\n/* extended minor device offset array */\n");
	fprintf(c_fp, "\n#define UM(maj)\tnotminored(maj * 256)\n");
	fprintf(c_fp, "\ndev_t\tMINOR[] = {\n");

	for (i = cnt = 0; i < NEL(devices); i++, cnt++) {
		if (devp = devices[i])
			maj = major(devp->xoffset);
		else
			maj = i;

		if (cnt >= 8) {
			fprintf(c_fp, ",\n");			/* line wrap */
			cnt = 0;
		}
		if (cnt == 0)
			fprintf(c_fp, "/*%3d*/ ", i);
		else
			fprintf(c_fp, ", ");

		fprintf(c_fp, "UM(%3d)", maj);
	}
	fprintf(c_fp, "\n};\n");

	fprintf(c_fp, "\n/* major device group leader array */\n");
	fprintf(c_fp, "\nchar\tGMAJOR[] = {\n");

	for (i = cnt = 0; i < NEL(devices); i++, cnt++) {
		if (devp = devices[i])
			maj = devp->parent;
		else
			maj = i;

		if (cnt >= 8) {
			fprintf(c_fp, ",\n");			/* line wrap */
			cnt = 0;
		}
		if (cnt == 0)
			fprintf(c_fp, "/*%3d*/ ", i);
		else
			fprintf(c_fp, ", ");

		fprintf(c_fp, "%3d", maj);
	}
	fprintf(c_fp, "\n};\n");

	fprintf(c_fp, "\n/* next major device in group array (-1 = end)*/\n");
	fprintf(c_fp, "\nchar\tNMAJOR[] = {\n");

	for (i = cnt = 0; i < NEL(devices); i++, cnt++) {
		if (devp = devices[i])
			maj = devp->next_maj;
		else
			maj = -1;

		if (cnt >= 8) {
			fprintf(c_fp, ",\n");			/* line wrap */
			cnt = 0;
		}
		if (cnt == 0)
			fprintf(c_fp, "/*%3d*/ ", i);
		else
			fprintf(c_fp, ", ");

		fprintf(c_fp, "%3d", maj);
	}
	fprintf(c_fp, "\n};\n");
}

/*
 * dump_upkern_stubs -- make the device functions that do auto-upkern
 */

dump_upkern_stubs()
{
	register device_t	*devp;
	register char		*fmt, *prefix;
	register int		f, i;
	uint			mr_upk_cnt = 0;

	static char		open_close_str[] =
"static _upkern_%s%s(_minor_, _flags_, _op_typ_)\n\
dev_t _minor_; int _flags_, _op_typ_;\n\
{\n\
\tupkern_lock();\n\
\t%s%s(_minor_, _flags_, _op_typ_);\n\
\tupkern_unlock();\n\
}\n\n",
				read_write_str[] =
"static _upkern_%s%s(_minor_)\n\
dev_t _minor_;\n\
{\n\
\tupkern_lock();\n\
\t%s%s(_minor_);\n\
\tupkern_unlock();\n\
}\n\n",
				ioctl_str[] =
"static _upkern_%s%s(_minor_, _commnd_, _arg_, _flags_)\n\
dev_t _minor_; int _commnd_, _arg_, _flags_;\n\
{\n\
\tupkern_lock();\n\
\t%s%s(_minor_, _commnd_, _arg_, _flags_);\n\
\tupkern_unlock();\n\
}\n\n",
				strategy_str[] =
"static _upkern_%s%s(_buf_ptr_)\n\
buf_t _buf_ptr_;\n\
{\n\
\textern int\t%s%s();\n\n\
\tupkern_strategy(%s%s, _buf_ptr_);\n\
}\n\n";

	outstr("\n/* upkern device functions */\n\n");

	for (i = 0; i < NEL(devices); i++) {
		if ((devp = devices[i]) == NULL)
			continue;		/* nobody home	*/
		if (devp->parent != i)
			continue;		/* no need to redeclare dup. */

		prefix = devp->prefix;
		for (f = 0; f < NUM_DRIVER_FUNCS; f++) {
			if (!devp->funcs[f] || devp->mpflags[f])
				continue;	/* no func or multi	*/
			if (devp->mirrorable) {
				if (f != DF_STRATEGY)
					continue; /* mirror stub did upkern */
				if (mr_upk_cnt++)
					continue; /* one per system */
				prefix = mirror->prefix;
			}

			switch (f) {
			case DF_OPEN:
			case DF_CLOSE:
				fmt = open_close_str;
			CASE DF_READ:
			case DF_WRITE:
				fmt = read_write_str;
			CASE DF_IOCTL:
				fmt = ioctl_str;
			CASE DF_STRATEGY:
				fmt = strategy_str;
			DEFAULT:
				fmt = NULL;
			}
			if (fmt)
				out(fmt,
				  prefix, driver_funcs[f],
				  prefix, driver_funcs[f],
				  prefix, driver_funcs[f]);
		}
	}
}


/*
 * dump_init -- build the init, start, and power fail handler tables
 */

dump_init()
{
	fprintf(c_fp, "\n/* system init, start, and powerfail tables */\n");
	fprintf(c_fp, "\nint\t(*pwr_clr[])() = {\n");
	list_dfuncs(driver_funcs[DF_POWER], DF_POWER);
	fprintf(c_fp, "\t(int (*)())0\n};\n");

	fprintf(c_fp, "\nint\t(*io_init[])() = {\n");
	list_dfuncs(driver_funcs[DF_INIT], DF_INIT);
	fprintf(c_fp, "\t(int (*)())0\n};\n");

	fprintf(c_fp, "\nint\t(*io_start[])() = {\n");
	list_dfuncs(driver_funcs[DF_START], DF_START);
	fprintf(c_fp, "\t(int (*)())0\n};\n");
}

/*
 * list_dfuncs -- scan the given device table and produce
 *	a list of all the instances of the specified function
 */

list_dfuncs(func_name, func)
register int	func;
register char	*func_name;
{
	register device_t	*devp, **devpp;
	register int		i;

	for (i = NEL(devices), devpp = devices; --i >= 0; devpp++) {
		if ((devp = *devpp) == NULL)
			continue;			/* nobody home	*/
		if (devp->funcs[func] != 1)
			continue;			/* no func	*/
		if (devp->maj != devp->parent)
			continue;			/* don't duplicate */

		fprintf(c_fp, "\t%s%s,\n", devp->prefix, func_name);
		devp->funcs[func] = 3;			/* mark it	*/
	}
}


/*
 * dump_linedis -- build the line discipline table
 */

dump_linedis()
{
	register int		i, max;
	register linedis_t	*lnp;
	static int		line_funcs[] = {
		LF_OPEN, LF_CLOSE, LF_READ, LF_WRITE,
		LF_IOCTL, LF_IN, LF_OUT, LF_MODEM
	};

	/* remove deconfigured and not required line discs & find max slot */

	for (i = max = 0; i < NEL(linedis); i++) {
		if ((lnp = linedis[i]) == NULL)
			continue;

		if (lnp->config || lnp->required) {
			max = i;
			continue;
		}

		linedis[i] = NULL;
	}

	/* now, print the extern declarations */

	outstr("\n/* line disciplines */\n\n");
	for (i = 0; i <= max; i++) {
		if ((lnp = linedis[i]) == NULL)
			continue;		/* nodev		*/

		do_extern(NUM_LINEDIS_FUNCS, lnp->funcs, mp_dummy,
		  lnp->name, linedis_funcs, 0);
	}

	/* finally print the structure array */

	outstr("\nstruct linesw	linesw[] = {\n");

	for (i = 0; i <= max; i++) {
		lnp = linedis[i];

		if (lnp) {
			do_struct(NEL(line_funcs), lnp->funcs, mp_dummy,
			  lnp->name, linedis_funcs, line_funcs, i, 0);
		}
		else {
			do_struct(NEL(line_funcs), nodev_funcs, mp_dummy,
			  "ERR!", linedis_funcs, line_funcs, i, 0);
		}
		outstr("},\n");
	}

	outstr("};\n");
	outstr("linecnt = sizeof(linesw) / sizeof(*linesw);\n\n");
}

/*
 * dump_strmod -- print out the fmodsw table
 */

dump_strmod()
{
	register strmod_t	*smp, *lsp;
	register int		i;

	/* remove deconfigured and not required stream modules */
	
	for (smp = strmodhead, lsp = NULL; smp != NULL; smp = smp->next) {
		if ( ! smp->config && ! smp->required ) {
			if (lsp == NULL) {
				strmodhead = smp->next;
				continue;		/* avoid changing lsp */
			}
			else
				lsp->next = smp->next;
		}
		lsp = smp;
	}

	/* declare streamtabs */

	outstr("\n/* stream modules */\n\n");

	for (smp = strmodhead; smp != NULL; smp = smp->next) {
		fprintf(c_fp, "extern struct streamtab\t%sinfo;\n",
		  smp->prefix);
	}

	/* do the structure and defines */

	fprintf(h_fp, "\n");
	fprintf(c_fp, "\nstruct fmodsw\tfmodsw[] = {\n");
	if (strmodhead == NULL)
		fprintf(c_fp, "\t{ \"\" }\n");		/* dummy entry */
	for (i = 0, smp = strmodhead; smp != NULL; smp = smp->next, i++) {
		fprintf(c_fp, "/*%2d*/ { \"%s\", &%sinfo },\n",
		  i, smp->name, smp->prefix);
		fprintf(h_fp, "#define %s_0_STR\n", upshift(smp->name));
	}
	fprintf(c_fp, "};\n");
	fprintf(c_fp, "int\tfmodcnt = sizeof(fmodsw) / sizeof(*fmodsw);\n");
}


/*
 * dump_filesys -- build the fsinfo and fstypsw tables
 */

dump_filesys()
{
	register int		i, var, f, cnt;
	register filesys_t	*fsp, *hfs;
	register unsigned char	*funcs;
	char			*ext;
	static char		*var_types[] = {
		"int",
		"inode_t"
	};
	static char		fs_multi[] = "FS_MULTIPROC";

	/* unlink deconfigured and not required file systems */

	for (fsp = filesyshead, hfs = NULL; fsp != NULL; fsp = fsp->next) {
		if ( ! fsp->config && ! fsp->required ) {
			if (hfs == NULL) {
				filesyshead = fsp->next;
				continue;		/* avoid changing hsp */
			}
			else
				hfs->next = fsp->next;
		}
		hfs = fsp;
	}

	/* now, print the extern declarations */

	outstr("\n/* file system switch externs */\n\n");
	outstr("extern int\tfsstray();\n");
	outstr("extern inode_t\t*fsinstray();\n");

	outstr("\n");
	for (fsp = filesyshead; fsp != NULL; fsp = fsp->next) {
		for (var = 0; var < NEL(var_types); var++) {
			ext = var_types[var];

			for (f = cnt = 0; f < NUM_FILESYS_FUNCS; f++) {
				switch (f) {
				/* int functions */
				case FF_INIT:		case FF_IPUT:
				case FF_IUPDATE:	case FF_READI:
				case FF_WRITEI:		case FF_ITRUNC:
				case FF_STATF:		case FF_NAMEI:
				case FF_MOUNT:		case FF_UMOUNT:
				case FF_OPENI:		case FF_CLOSEI:
				case FF_UPDATE:		case FF_STATFS:
				case FF_ACCESS:		case FF_GETDENTS:
				case FF_ALLOCMAP:	case FF_FREEMAP:
				case FF_READMAP:	case FF_SETATTR:
				case FF_NOTIFY:		case FF_FCNTL:
				case FF_FSINFO:		case FF_IOCTL:
					if (var != 0)
						continue;

				/* inode pointer functions */
				CASE FF_GETINODE:	case FF_IREAD:
					if (var != 1)
						continue;
				DEFAULT:
					printf("dump_filesys: impossible!\n");
					die(1);
				}

				if (fsp->funcs[f]) {
					if (cnt == 0)
						out("extern %s ", ext);
					out("%s%s%s%s()",
					  (cnt > 0 ? ", " : ""),
					  (var == 1 ? "*" : ""),
					  fsp->prefix, filesys_funcs[f]);
					++cnt;
				}
			}
			if (cnt > 0)
				outstr(";\n");
		}
	}

	/* print the file system info table and defines */

	hfs = new_filesys();			/* filesys 0 is always stray */
	hfs->next = filesyshead;

	fprintf(h_fp, "\n");
	outstr("\nstruct fsinfo\tfsinfo[] = {\n");
	for (i = 0, fsp = hfs; fsp != NULL; fsp = fsp->next, i++) {
		out("/*%2d*/ { %s, %s, \"%s\", %s },\n", i,
		  (*fsp->fsflags ? fsp->fsflags : "0"),
		  (*fsp->pipedev ? fsp->pipedev : "(struct mount *)0"),
		  fsp->name,
		  (*fsp->notifyflags ? fsp->notifyflags : "0"));
		if (fsp != hfs)
			fprintf(h_fp, "#define %s_0_FS\n", upshift(fsp->name));
	}
	outstr("};\n");

	/* print the fstypsw table */

	outstr("\nstruct fstypsw\tfstypsw[] = {\n");
	for (i = 0, fsp = hfs; fsp != NULL; fsp = fsp->next, i++) {
		funcs = fsp->funcs;

		out("/*%2d*/ { ", i);

		for (f = 0; f < NUM_FILESYS_FUNCS; f++) {
			/* kludge to insert stray f_filler before f_iupdat */
			if (f == FF_IUPDATE)
				outstr("fsstray, 0, ");

			if (funcs[f]) {
				out("%s%s, %s, ", fsp->prefix, filesys_funcs[f],
				  (fsp->mpflags[f] ? fs_multi : "0"));
			}
			else {
				switch (f) {
				/* int functions */
				case FF_INIT:		case FF_IPUT:
				case FF_IUPDATE:	case FF_READI:
				case FF_WRITEI:		case FF_ITRUNC:
				case FF_STATF:		case FF_NAMEI:
				case FF_MOUNT:		case FF_UMOUNT:
				case FF_OPENI:		case FF_CLOSEI:
				case FF_UPDATE:		case FF_STATFS:
				case FF_ACCESS:		case FF_GETDENTS:
				case FF_ALLOCMAP:	case FF_FREEMAP:
				case FF_READMAP:	case FF_SETATTR:
				case FF_NOTIFY:		case FF_FCNTL:
				case FF_FSINFO:		case FF_IOCTL:
					ext = "fsstray";

				/* inode pointer functions */
				CASE FF_GETINODE:	case FF_IREAD:
					ext = "fsinstray";
				DEFAULT:
					printf("dump_filesys: IMPOSSIBLE!\n");
					die(1);
				}
				out("%s, %s, ", ext,
				  (fsp->mpflags[f] ? fs_multi : "0"));
			}
		}
		/* add five more fill entries */
		outstr("fsstray, fsstray, fsstray, fsstray, fsstray");
		outstr(" },\n");
	}
	outstr("};\n");
	outstr("short\tnfstyp = sizeof(fstypsw) / sizeof(*fstypsw);\n");
}


/*
 *	out.c -- word wrap output functions
 *
 *	Copyright (C) 1986, James Cleverdon and David Stone
 */

#include <varargs.h>

#define ISWHITE(c)	(c == ' ' || c == '\t')


extern char	*strchr();

/*
 * curx == the cursor's x coordinate.
 */
int	curx = 0;

/*
 * maxx == maximum x value allowed
 */
int	maxx = 79;		/* note the default value implies 80 columns */

/*
 * outbuf == the line buffer
 */
char	outbuf[BUFSIZ];

/*
 * outptr == the pointer into outbuf
 */
char	*outptr;

/*
 * out -- printf-like function that word wraps text
 */

/*VARARGS0*/
out(va_alist)
va_dcl
{
	char	*p;
	char	buf[BUFSIZ];
	va_list	args;

	va_start(args);				/* start varargs */

	p = va_arg(args, char *);		/* get the format string */

	/* are there any arguments in the format string ? */

	if (strchr(p, '%')) {
		vsprintf(buf, p, args);		/* sprintf the args */
		p = buf;
	}
	/* else, don't bother.  use p directly */

	va_end(args);				/* end varargs */

	outstr(p);
}

/*
 * outstr -- builds a line in outbuf until its full, or a newline is seen, then
 *	calls do_outstr
 */

outstr(src)
register char	*src;
{
	register char	*dest;
	register int	c, n;

	if ((dest = outptr) == NULL)
		dest = outbuf;

	n = sizeof(outbuf) - 1 - (dest - outbuf);
	while (*dest = c = *src++)
		if (--n < 0 || c == '\n') {
			dest[1] = '\0';
			do_outstr(outbuf);
			dest = outbuf;
			n = sizeof(outbuf);
		}
		else
			++dest;

	outptr = dest;
}


/*
 * do_outstr -- do all the word wrapping, etc.
 */

do_outstr(str)
char	*str;
{
	register char	*p, *head;
	register int	c;
	char		*wrapper();

	p = str;

	/* scan through the string, prepared to word wrap */

	for (head = p; c = *p; p++) {
		if (c >= ' ' && c <= 0177)		/* ordinary char */
			++curx;
		else if (c == '\n') {			/* newline	*/
			lineout(head, p, 0);
			head = p + 1;		/* head = start of next line */
		}
		else if (c == '\t')			/* tab */
			curx += 8 - (curx & 7);

		if (curx >= maxx) {		/* time to word wrap?	*/
			head = p = wrapper(head, p);
			--p;
			do_outstr("\t");	/* indent one tab	*/
		}
	}

	if (*--p != '\n')
		lineout(head, p, 0);		/* send last line	*/
}


/*
 *	lineout -- output a line from head to tail
 */

lineout(head, tail, nl)
char	*head, *tail;
int	nl;
{
	if (fwrite(head, 1, tail - head + 1, out_fp) == EOF) 
		die(1);

	if (nl || *tail == '\n') {		/* do newline bookkeeping? */
		if (nl)
			fputc('\n', out_fp);

		curx = 0;
	}
}


/*
 *	wrapper(head, p) -- do a word wrap from the head of the line to p.
 *		Print out the line from head to the end of the previous
 *		word (q), then return the pointer to the next word (p).
 */

char *
wrapper(head, p)		/* do the actual word wrap */
register char	*head, *p;
{
	register char	*q, *ret;
	register int	c = *p;

	/* first, back q up to the end of previous word, then advance p */

	if ( ! ISWHITE(c) ) {

		/* p is in the middle of a word, ret = head of this word */

		for (q = p - 1; (c = *q), ! ISWHITE(c); --q)
			if (q <= head) {
				if (p - head >= maxx) {
					q = p;	/* word too long, break it */
					ret = p + 1;	/* in half */

					goto doit;
				}
				q = head - 1;	/* else, string too short */
				break;		/* to wrap properly */
			}

		ret = q + 1;		/* remember start of this word */
	}
	else {
		/* p is in some white space, ret = next word */

		for (q = p + 1; (c = *q) && ISWHITE(c); ++q)
			;

		ret = q;		/* remember start of this word */

		q = p - 1;		/* set q to in front of p */
	}

	/* now, back q up to end of previous word */

	for ( ; (c = *q), ISWHITE(c); --q)
		if (q <= head) {
			q = head - 1;		/* all white */
			break;
		}
doit:
	lineout(head, q, 1);

	return (ret);
}
