/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) synch.c: version 25.1 created on 11/27/91 at 14:49:32	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)synch.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
#include "sys/spm_mem.h"
#include "sys/types.h"
#include "sys/iopmcomm.h"
#include "sys/debug.h"
#include "sys/synch.h"

/*
** dizzy_lock is motivated by the desire to have lockers w/o
** spying caches (IOPM) spin in their local memory, not on the CSS. Even
** among lockers with spying caches (PM) this new lock forces FIFO behaviour
** on requestors instead of the free-for-all behaviour with sem_lock.
**
** The dizzy_lock, spin and lock_req_array structures are manipulated
** by dizzy_lock and dizzy_unlock routines.
** There is one dizzy_lock struct per lockable resource.
** There is one lock req per locker (eg PM20, IOPM).
** For PM's the lock req is the lock_req_array element indexed by iopm_lock_id.
** For IOPM's part of the lock req is the lock_req_array element indexed
** by iopm_lock_id and part is "spin" in local memory for the IOPM.
**
** The lock is free if "head_lock_req" is zero.
** To aquire the lock, the requester adds its lock req to the head of the
** list of requesters. If the list transistioned from empty, the requester
** is the new owner, otherwise the requester waits.
** When releasing the lock, the owner passes the lock to the next requester.
** The next_lock_req entry gives the address of the array portion of the lock
** req as a link to the previous requestor.
** Requestors add themselves
** to the head of the list (near dizzy_lock). The owner of the lock removes
** themself from the tail of the list (far from dizzy_lock) and
** passes the lock to the next requestor (new tail of the list).
** Requesters stay on the list until they give up the lock.
** Requesters can be on more than one list. They may own one lock and be waiting
** for one other lock.
** This gives rise to lock lists that logically look like this:
**
**	LOCK A -> req 1 -> req 2 -> req 3
**	LOCK B -> req 3 -> req 4
**
** Note that req 3 owns LOCK A and is waiting for LOCK B. Note also that
** even though req 3 points to req 4, req 4 is not on the LOCK A list and
** is not the owner of LOCK A. This is not apparent to an outside observer
** traveling down the list for LOCK A without knowlegde of the LOCK B list.
** But req 3 knows it owns LOCK A and can give it to req 2 at will.
*/

valid_lock_req_array(lrap)
register struct lock_req_array  *lrap;
{
	if ( lrap >= spm_mem.lock_req_arrayp &&
	  lrap < spm_mem.lock_req_arrayp + LLOCK_MAX &&
	  ((uint)lrap - (uint)spm_mem.lock_req_arrayp) %
	  sizeof(struct lock_req_array) )
		return 0;

	return 1;
}

/*****************************************************************************/
dizzy_llock_init()
{
	extern uint                   iopm_lock_id;	/* in build_addr */
	extern struct lock_req_array  *lock_req_ptr;	/* in build_addr */
	extern uint                   dizzy;		/* in build_addr */

	iopm_lock_id = atom_add(&spm_mem.next_lock_id, 1);
	ASSERT(iopm_lock_id < LLOCK_MAX);
	while ( !spm_mem.lock_req_arrayp );
	lock_req_ptr = spm_mem.lock_req_arrayp + iopm_lock_id;
	lock_req_ptr->remote_spin = 1;
	lock_req_ptr->slot = iopmcomm.slot;
	lock_req_ptr->lra.addr = &dizzy;
}

/*****************************************************************************/
/* If dizzy_unlock calls dizzy_unlock_c we know there is at least one requestor
/* on the list. Requestors never come off the list except by being taken off by
/* the owner of the lock (us).
/* Walk down the list to the last requestor and give the lock to requestor.
*/

dizzy_unlock_c(lockp)
register struct dizzy_lock  *lockp;
{
	register struct lock_req_array  *lrap;
	register uint                   chain_len = 1;
	uint                            savmap;
	extern uint                     iopm_lock_id;	/* in build_addr */
	extern struct lock_req_array    *lock_req_ptr;	/* in build_addr */

	ASSERT(lockp->lock_owner == iopm_lock_id);	/* belongs to us */

	lrap = lockp->head_lock_req;	/* must be atomic, hence aligned */
	ASSERT(lrap != lock_req_ptr);	/* someone waiting*/
	ASSERT(valid_lock_req_array(lrap));

	/* Travel down the list of dizzy requestors and remove the last req */
	while ( lrap->next_lock_req != lock_req_ptr )
	{
		chain_len++;
		lrap = lrap->next_lock_req;
		ASSERT(valid_lock_req_array(lrap));
	}

	/* lrap is the lock req pointing to us. It will be the new lock owner*/
	/* lrap->next_lock_req is us. We are the end of the request chain. */
	ASSERT(lock_req_ptr->next_lock_req == 0);
	lockp->max_chain = max(lockp->max_chain, chain_len);

	/* remove our req from list */
	lrap->next_lock_req = 0;

	lockp->lock_owner = lrap - spm_mem.lock_req_arrayp;

	ASSERT(lrap->lra.addr);		/* either spin is set or addr is set */
	/* alert next lock owner */
	if ( lrap->remote_spin )
	{
		ASSERT(lrap->lra.addr != (ulong *)1);	/* valid IOPM addr */
		savmap = iomap_save();
		*(ulong *)iosba_map(lrap->slot, lrap->lra.addr) = 0;
		iomap_restore(savmap);
	}
	else
		lrap->lra.spin = 0;
}
