/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) stream.c: version 25.1 created on 11/27/91 at 14:49:08	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)stream.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
#include "sys/types.h"
#include "sys/param.h"
#include "sysmacros.h"
#include "sys/cmn_err.h"
#include "sys/iopmdebug.h"
#include "sys/debug.h"
#include "sys/map.h"
#include "sys/immu.h"
#include "sys/sbus_iopm.h"
#include "devsw.h"
#include "stream.h"
#include "sys/strstat.h"	/* after stream (NCLASS) */

/*
 * queue scheduling control variables 
 */
char qrunflag;		/* set to enable queues */
char queueflag;		/* set iff inside queuerun() */
char strbcflag;		/* bufcall functions ready to go */

/*
 * arrays to map block classes to real size and weighted size,
 * the latter being used for flow control calculations.
 */
ushort	rbsize[] = { 4, 16, 64, 128, 256, 512, 1024, 2048, 4096 };
ushort	bsize[]  = { 4, 16, 64, 100, 200, 300, 600, 1200, 2400 };

extern uint      nqueue;	/* in iopmcomm */
extern iqueue_t  *queuep;	/* in iopmcomm */
extern caddr_t   mblkp;		/* in iopmcomm */
extern caddr_t   dblkp;		/* in iopmcomm */
extern uint      nblk[];	/* in iopmcomm */

struct strevent         *sefreelist;	/* stream event cell freelist */
extern struct strevent  *streventp;	/* ptr to table of stream event cells */
extern int              nstrevent;	/* max number of stream event cells */

extern char strlofrac;		/* low priority cutoff percentage */
extern char strmedfrac;		/* medium priority cutoff percentage */

extern nmblock;

extern uint  memalloc();

/******************************************************************************/
/*
 * Allocate a message and data block.
 * Tries to get a block large enough to hold 'size' bytes.
 * A message block and data block are allocated together, initialized,
 * and a pointer to the message block returned.  Data blocks are
 * always allocated in association with a message block, but there
 * may be several message blocks per data block (see dupb).
 * If no message blocks or data blocks (of the required size)
 * are available, NULL is returned.
 */
mblk_t *
allocb(size, pri)
register size;
uint pri;
{
	register dblk_t *databp;
	register mblk_t *bp;
	register s;
	register class;

	if ((class = getclass((uint)size)) >= NCLASS) return(NULL);
	ASSERT(pri_ok(pri));
	s = splstr();

	/*
	 * get buffer - if can't in class then try one class higher
	 */
	if ((dballoc[class].dba_cnt < bcmax(class,pri)) && 
	    (databp = dbfreelist[class]))  goto gotdp;

	PRINT3(SRESRC, "allocb: failed %d/%d used, size %d\n",
	  dballoc[class].dba_cnt, nblk[NCLASS-1-class], size);

	strst.dblk[class].fail++;
	strst.dblock.fail++;
	if (class++ == NCLASS-1) {
		splx(s);
		if ( nblk[NCLASS-class] == 0 )
			cmn_err(CE_WARN,
			  "No configured mblks of size %d to satisfy request.",
			  rbsize[class]);

		return(NULL);
	}
	if ((dballoc[class].dba_cnt < bcmax(class,pri) ) &&
	    (databp = dbfreelist[class])) goto gotdp;

	PRINT2(SRESRC, "allocb: failed to get next size, %d/%d used\n",
	  dballoc[class].dba_cnt, nblk[NCLASS-1-class] );

	strst.dblk[class].fail++;
	strst.dblock.fail++;
	splx(s);
	if ( nblk[NCLASS-1-class] == 0 && nblk[NCLASS-2-class] == 0 )
		cmn_err(CE_WARN,
		  "No configured mblks of size %d or %d to satisfy request.",
		  rbsize[class - 1], rbsize[class]);
	return(NULL);

gotdp:
	/*
	 * The data buffer is in hand, try for a message block.
	 */
	if (!(bp = mbfreelist)) {
		PRINT(SRESRC, "allocb: no messages blocks");
		strst.mblock.fail++;
		splx(s);
		return(NULL);
	}

	dbfreelist[class] = databp->db_freep;
	dballoc[class].dba_cnt++;
	databp->db_freep = NULL;
	mbfreelist = bp->b_next;
	BUMPUP(strst.mblock);
	BUMPUP(strst.dblock);
	BUMPUP(strst.dblk[class]);
	splx(s);

	/*
	 * initialize message block and
	 * data block descriptors
	 */
	bp->b_next = NULL;
	bp->b_prev = NULL;
	bp->b_cont = NULL;
	bp->b_datap = databp;
	bp->b_rptr = databp->db_base;
	bp->b_wptr = databp->db_base;
	databp->db_type = M_DATA;
	/*
	 * set reference count to 1 (first use)
	 */
	databp->db_ref = 1;
	return(bp);
}

/******************************************************************************/
/*
 * test if block of given size can be allocated with a request of
 * the given priority.
 */
testb(size, pri)
register size;
uint pri;
{
	register class;

	if ((class = getclass((uint)size)) >= NCLASS) return(0);
	ASSERT(pri_ok(pri));
	if ((dballoc[class].dba_cnt < bcmax(class,pri)) && dbfreelist[class]) 
		return(1);

	if (class++ == NCLASS-1) return(0);
	if ((dballoc[class].dba_cnt < bcmax(class,pri) ) && dbfreelist[class]) 
		return(1);
	return(0);
}

/******************************************************************************/
/*
 * Get class of buffer.  Returns NCLASS if size is greater than the
 * the largest block class.
 */
getclass(size)
register uint size;
{
	register class;

	for (class = 0; (class < NCLASS) && (size > rbsize[class]) ; class++);
	return(class);
}

/******************************************************************************/
/*
 * Call function 'func' with argument 'arg' when there is a reasonably
 * good chance that a block of size 'size' can be allocated with
 * priority 'pri'.  
 */

bufcall(size, pri, func, arg)
uint size;
int pri;
int (*func)();
long arg;
{
	register s;
	register class;
	struct strevent *sep, **tmp;
	struct dbalcst *dbp;

	ASSERT(size <= rbsize[NCLASS-1]);
	ASSERT(pri_ok(pri));
	ASSERT(valid_iopm_vaddr(func));

	class = getclass(size);

	/*
	 * fail bufcall if not enough configured buffers for given
	 * priority allocation to ever succeed.
	 */
	if (bcmax(class, pri) == 0) return(0);	/* if not BPRI_HI */
	if ((pri == BPRI_HI) && (*(nblk + (NCLASS - 1 - class)) == 0)) 
		return(0);

	if (!(sep = sealloc())) {
		cmn_err(CE_WARN, "bufcall: could not allocate stream event.");
		return(0);
	}
	dbp = &dballoc[class];
	s = splstr();
	tmp = ( (pri == BPRI_HI) ? &dbp->dba_hip : 
		((pri == BPRI_LO) ? &dbp->dba_lop : &dbp->dba_medp) );
	for ( ; *tmp; tmp = &(*tmp)->se_next )
		if ( (*tmp)->se_func == func && (*tmp)->se_arg == arg )
		{
			sefree(sep);
			splx(s);
			return 1;
		}
	*tmp = sep;
	sep->se_next = NULL;
	sep->se_func = func;
	sep->se_arg = arg;
	splx(s);
	return(1);
}

/******************************************************************************/
/* called by qdetach to remove queues from the bufcall list when the queues
/* are removed from the stream.
*/
chk_bufcall(rq)
register long  rq;
{
	struct strevent  *tmp;
	struct strevent  *lastsep = 0;
	int              class;
	int              savpri;
	extern           qenable();

	ASSERT(valid_rq(rq));
	for ( class = 0; class < NCLASS; class++ )
	{
		tmp = dballoc[class].dba_lop;
		savpri = splstr();
		for ( lastsep = 0; tmp; tmp = tmp->se_next )
		{
			if ( tmp->se_arg == (long)rq ||
			     tmp->se_arg == (long)WR(rq) )
				break;
			lastsep = tmp;
		}
		if ( tmp )
		{
			if ( lastsep )
				lastsep->se_next = tmp->se_next;
			else
				dballoc[class].dba_lop = tmp->se_next;
			sefree(tmp);
			splx(savpri);
			return;
		}
		splx(savpri);

		tmp = dballoc[class].dba_medp;
		savpri = splstr();
		for ( lastsep = 0; tmp; tmp = tmp->se_next )
		{
			if ( tmp->se_arg == (long)rq ||
			     tmp->se_arg == (long)WR(rq) )
				break;
			lastsep = tmp;
		}
		if ( tmp )
		{
			if ( lastsep )
				lastsep->se_next = tmp->se_next;
			else
				dballoc[class].dba_medp = tmp->se_next;
			sefree(tmp);
			splx(savpri);
			return;
		}
		splx(savpri);

		tmp = dballoc[class].dba_hip;
		savpri = splstr();
		for ( lastsep = 0; tmp; tmp = tmp->se_next )
		{
			if ( tmp->se_arg == (long)rq ||
			     tmp->se_arg == (long)WR(rq) )
				break;
			lastsep = tmp;
		}
		if ( tmp )
		{
			if ( lastsep )
				lastsep->se_next = tmp->se_next;
			else
				dballoc[class].dba_hip = tmp->se_next;
			sefree(tmp);
			splx(savpri);
			return;
		}
		splx(savpri);

	}
}

/******************************************************************************/
/*
 * Free a message block and decrement the reference count on its 
 * data block. If reference count == 0 also return the data block.
 */
freeb(bp)
register mblk_t *bp;
{
	register s;
	register class;
	register struct dbalcst *dbp;

	ASSERT(valid_iopm_mblk(bp));

	class = bp->b_datap->db_class;
	dbp = &dballoc[class];
	s = splstr();
	ASSERT(bp->b_datap->db_ref > 0);
	if (--bp->b_datap->db_ref == 0) {
		bp->b_datap->db_freep = dbfreelist[class];
		dbfreelist[class] = bp->b_datap;
		dbp->dba_cnt--;
		if (!strbcflag && ( dbp->dba_hip ||
		    (dbp->dba_medp && (dbp->dba_cnt < dbp->dba_med)) ||
		    (dbp->dba_lop && (dbp->dba_cnt < dbp->dba_lo)))) {
			qrunflag = 1;
			strbcflag = 1;
		}			
		strst.dblock.use--;
		strst.dblk[class].use--;
	}
	bp->b_datap = NULL;
	bp->b_next = mbfreelist;
	mbfreelist = bp;
	strst.mblock.use--;
	splx(s);
	return;
}

/******************************************************************************/
/*
 * Free all message blocks in a message using freeb().  
 * The message may be NULL.
 */
freemsg(bp)
register mblk_t *bp;
{
	register mblk_t *tp;
	register        savpri;

	ASSERT(valid_iopm_mblk(bp) || bp == NULL);

	savpri = splstr();
	while (bp) {
		tp = bp->b_cont;
		freeb(bp);
		bp = tp;
	}
	splx(savpri);
}

/******************************************************************************/
/*
 * Duplicate a message block
 *
 * Allocate a message block and assign proper
 * values to it (read and write pointers)
 * and link it to existing data block.
 * Increment reference count of data block.
 */
mblk_t *
dupb(bp)
register mblk_t *bp;
{
	register s;
	register mblk_t *nbp;

	ASSERT(valid_iopm_mblk(bp));

	s = splstr();
	if (!(nbp = mbfreelist)) {
		splx(s);
		strst.mblock.fail++;
		return(NULL);
	}
	mbfreelist = nbp->b_next;
	(nbp->b_datap = bp->b_datap)->db_ref++;
	splx(s);
	BUMPUP(strst.mblock);

	nbp->b_next = NULL;
	nbp->b_prev = NULL;
	nbp->b_cont = NULL;
	nbp->b_rptr = bp->b_rptr;
	nbp->b_wptr = bp->b_wptr;
	return(nbp);
}

/******************************************************************************/
/*
 * Duplicate a message block by block (uses dupb), returning
 * a pointer to the duplicate message.
 * Returns a non-NULL value only if the entire message
 * was dup'd.
 */
mblk_t *
dupmsg(bp)
register mblk_t *bp;
{
	register mblk_t  *head, *nbp;
	register         savpri;

	ASSERT(valid_iopm_mblk(bp) || bp == NULL);

	if ( !bp || !(nbp = head = dupb(bp)) )
		return(NULL);

	savpri = splstr();
	while (bp->b_cont) {
		if (!(nbp->b_cont = dupb(bp->b_cont))) {
			freemsg(head);
			return(NULL);
		}
		nbp = nbp->b_cont;
		bp = bp->b_cont;
	}
	splx(savpri);
	return(head);
}

/******************************************************************************/
/*
 * copies data from message block to newly allocated message block and
 * data block.  The copy is rounded out to full word boundaries so that
 * the (usually) more efficient word copy can be done.
 * Returns new message block pointer, or  NULL if error.
 */
mblk_t *
copyb(bp)
register mblk_t *bp;
{
	register mblk_t *nbp;
	register dblk_t *dp, *ndp;
	caddr_t base;

	ASSERT(valid_iopm_mblk(bp));
	ASSERT(bp->b_wptr >= bp->b_rptr);

	dp = bp->b_datap;
	if (!(nbp = allocb(dp->db_lim - dp->db_base, BPRI_MED)))
		return(NULL);
	ndp = nbp->b_datap;	
	ndp->db_type = dp->db_type;
	nbp->b_rptr = ndp->db_base + (bp->b_rptr - dp->db_base);
	nbp->b_wptr = ndp->db_base + (bp->b_wptr - dp->db_base);
	base = straln(nbp->b_rptr);
	strbcpy(straln(bp->b_rptr), base,
	        (uint)(straln(nbp->b_wptr + (sizeof(int) - 1)) - base));
	return(nbp);
}

/******************************************************************************/
/*
 * copies data from message to newly allocated message using new
 * data blocks.  
 * returns pointer to new message.
 * NULL if error.
 */
mblk_t *
copymsg(bp)
register mblk_t *bp;
{
	register mblk_t *head, *nbp;
	register        savpri;

	ASSERT(valid_iopm_mblk(bp) || bp == NULL);

	if ( !bp || !(nbp = head = copyb(bp)) )
		return(NULL);

	savpri = splstr();
	while (bp->b_cont) {
		if (!(nbp->b_cont = copyb(bp->b_cont))) {
			freemsg(head);
			splx(savpri);
			return(NULL);
		}
		nbp = nbp->b_cont;
		bp = bp->b_cont;
	}
	splx(savpri);
	return(head);
}

/******************************************************************************/
/* 
 * link a message block to tail of message
 */
linkb(mp, bp)
register mblk_t *mp;
register mblk_t *bp;
{
	register  savpri;

	ASSERT(valid_iopm_mblk(bp));
	ASSERT(valid_iopm_mblk(mp));

	savpri = splstr();
	for ( ; mp->b_cont; mp = mp->b_cont )
		;
	mp->b_cont = bp;
	splx(savpri);
}

/******************************************************************************/
/*
 * unlink a message block from head of message
 * return pointer to new message.
 * NULL if message becomes empty.
 */
mblk_t *
unlinkb(bp)
register mblk_t *bp;
{
	register mblk_t *bp1;
	register         savpri;

	ASSERT(valid_iopm_mblk(bp));

	savpri = splstr();
	bp1 = bp->b_cont;
	bp->b_cont = NULL;
	splx(savpri);
	return(bp1);
}

/******************************************************************************/
/* 
 * remove a message block "bp" from message "mp"
 *
 * Return pointer to new message or NULL if no message remains.
 * Return -1 if bp is not found in message.
 */
mblk_t *
rmvb(mp,bp)
register mblk_t *mp;
register mblk_t *bp;
{
	register mblk_t *tmp;
	register mblk_t *lastp = NULL;
	register        savpri;

	ASSERT(valid_iopm_mblk(mp) || mp == NULL);
	ASSERT(valid_iopm_mblk(mp) || bp == NULL);

	savpri = splstr();
	for (tmp = mp; tmp; tmp = tmp->b_cont) {
		if (tmp == bp) {
			if (lastp) lastp->b_cont = tmp->b_cont;
			else mp = tmp->b_cont;
			tmp->b_cont = NULL;
			splx(savpri);
			return(mp);
		}
		lastp = tmp;
	}
	splx(savpri);
	return((mblk_t *)-1);
}

/******************************************************************************/
/*
 * macro to check pointer alignment
 * (true if alignment is sufficient for worst case)
 */

#define str_aligned(X)	(((uint)(X) & (sizeof(int) - 1)) == 0)

/*
 * Concatenate and align first len bytes of common
 * message type.  Len == -1, means concat everything.
 * Returns 1 on success, 0 on failure
 * After the pullup, mp points to the pulled up data.
 * This is convenient but messy to implement.
 */
pullupmsg(mp, len)
mblk_t *mp;
register len;
{
	register mblk_t *bp;
	register mblk_t *new_bp;
	register uint   n;
	mblk_t *tmp;
	int s;

	ASSERT(valid_iopm_mblk(mp));

	/*
	 * Quick checks for success or failure:
	 */
	if (len == -1) {
		if (mp->b_cont == NULL && str_aligned(mp->b_rptr))
			return(1);
		len = xmsgsize(mp);
	} else {
		ASSERT(mp->b_wptr >= mp->b_rptr);
		if (mp->b_wptr - mp->b_rptr >= len && str_aligned(mp->b_rptr))
			return(1);
		if (xmsgsize(mp) < len)
			return(0);
	}

	/*
	 * Allocate a new mblk header.  It is used later to interchange
	 * mp and new_bp.
	 */
	s = splstr();
	if ((tmp = mbfreelist) == NULL) {
		splx(s);
		return(0);
	}
	mbfreelist = tmp->b_next;
	splx(s);

	/*
	 * Allocate the new mblk.  We might be able to use the existing
	 * mblk, but we don't want to modify it in case its shared.
	 * The new dblk takes on the type of the old dblk
	 */
	if ((new_bp = allocb(len, BPRI_MED)) == NULL) {
		s = splstr();
		tmp->b_next = mbfreelist;
		mbfreelist = tmp;
		splx(s);
		return(0);
	}
	new_bp->b_datap->db_type = mp->b_datap->db_type;

	/*
	 * Scan mblks and copy over data into the new mblk.
	 * Two ways to fall out: exact count match: while (len)
	 * Bp points to the next mblk containing data or is null.
	 * Inexact match: if (bp->b_rptr != ...)  In this case,
	 * bp points to an mblk that still has data in it.
	 */ 
	bp = mp;
	while ( len && bp ) {
		mblk_t *b_cont;

		ASSERT(bp->b_wptr >= bp->b_rptr);
		n = min((uint)(bp->b_wptr - bp->b_rptr), len);
		bcopy((caddr_t)bp->b_rptr, (caddr_t)new_bp->b_wptr, n);
		new_bp->b_wptr += n;
		bp->b_rptr += n;
		len -= n;
		if (bp->b_rptr != bp->b_wptr)
			break;
		b_cont = bp->b_cont;
		if (bp != mp)	/* don't free the head mblk */
			freeb(bp);
		bp = b_cont;
	}

	/*
	 * At this point:  new_bp points to a dblk that
	 * contains the pulled up data.  The head mblk, mp, is
	 * preserved and may or may not have data in it.  The
	 * intermediate mblks are freed, and bp points to the
	 * last mblk that was pulled-up or is null.
	 *
	 * Now the tricky bit.  After this, mp points to the new dblk
	 * and tmp points to the old dblk.  New_bp points nowhere
	 */
	*tmp = *mp;
	*mp = *new_bp;
	new_bp->b_datap = NULL;

	/*
	 * If the head mblk (now tmp) still has data in it, link it to mp
	 * otherwise link the remaining mblks to mp and free the
	 * old head mblk.
	 */
	if (tmp->b_rptr != tmp->b_wptr)
		mp->b_cont = tmp;
	else {
		mp->b_cont = bp;
		freeb(tmp);
	}

	/*
	 * Free new_bp
	 */
	s = splstr();
	new_bp->b_next = mbfreelist;
	mbfreelist = new_bp;
	splx(s);

	return(1);
}

/******************************************************************************/
/*
 * Trim bytes from message
 *  len > 0, trim from head
 *  len < 0, trim from tail
 * Returns 1 on success, 0 on failure
 */
adjmsg(mp, len)
mblk_t *mp;
register int len;
{
	register mblk_t *bp;
	register n;
	int fromhead;

	ASSERT(valid_iopm_mblk(mp));

	fromhead = 1;
	if (len < 0) {
		fromhead = 0;
		len = -len;
	}
	if (xmsgsize(mp) < len)
		return(0);

	if (fromhead) {
		for ( bp = mp; len; bp = bp->b_cont ) {
			ASSERT(bp);
			ASSERT(bp->b_datap->db_type ==  mp->b_datap->db_type);
			ASSERT(bp->b_wptr >= bp->b_rptr);
			n = min((uint)(bp->b_wptr - bp->b_rptr), len);
			bp->b_rptr += n;
			len -= n;
		}
	} else {
		register mblk_t *save_bp;
		register unsigned char type;

		type = mp->b_datap->db_type;
		while ( len ) {
			bp = mp;
			save_bp = NULL;
			while (bp && bp->b_datap->db_type == type) {
				ASSERT(bp->b_wptr >= bp->b_rptr);
				if (bp->b_wptr - bp->b_rptr > 0)
					save_bp = bp;
				bp = bp->b_cont;
			}
			ASSERT(save_bp);
			n = min((uint)(save_bp->b_wptr - save_bp->b_rptr), len);
			save_bp->b_wptr -= n;
			len -= n;
		}
	}
	return(1);
}

/******************************************************************************/
/*
 * Return size of message of block type (bp->b_datap->db_type)
 */
xmsgsize(bp)
register mblk_t *bp;
{
	register unsigned char type;
	register count = 0;
	register        savpri;
	
	ASSERT(valid_iopm_mblk(bp));

	type = bp->b_datap->db_type;

	savpri = splstr();
	for (; bp; bp = bp->b_cont) {
		if (type != bp->b_datap->db_type)
			break;
		ASSERT(bp->b_wptr >= bp->b_rptr);
		count += bp->b_wptr - bp->b_rptr;
	}
	splx(savpri);
	return(count);
}

/******************************************************************************/
/*
 * get number of data bytes in message
 */
msgdsize(bp)
register mblk_t *bp;
{
	register int count = 0;
	register        savpri;

	ASSERT(valid_iopm_mblk(bp) || bp == NULL);

	savpri = splstr();
	for (; bp; bp = bp->b_cont)
		if (bp->b_datap->db_type == M_DATA) {
			ASSERT(bp->b_wptr >= bp->b_rptr);
			count += bp->b_wptr - bp->b_rptr;
		}

	splx(savpri);
	return(count);
}

/******************************************************************************/
/*
 * Get a message off head of queue
 *
 * If queue has no buffers then mark queue
 * with QWANTR. (queue wants to be read by
 * someone when data becomes available)
 *
 * If there is something to take off then do so.
 * If queue falls below hi water mark turn off QFULL
 * flag.  Decrement weighted count of queue.
 * Also turn off QWANTR because queue is being read.
 *
 * If queue count is below the lo water mark and QWANTW
 * is set, enable the closest backq which has a service 
 * procedure and turn off the QWANTW flag.
 */

mblk_t *
getq(q)
register iqueue_t *q;
{
	register mblk_t *bp;
	register mblk_t *tmp;
	register s;

	ASSERT(valid_queue(q));

	s = splstr();
	if (!(bp = q->q_first))
		q->q_flag |= QWANTR;
	else {
		if (!(q->q_first = bp->b_next))	q->q_last = NULL;
		else q->q_first->b_prev = NULL;
		for (tmp = bp; tmp; tmp = tmp->b_cont)
			q->q_count -= bsize[tmp->b_datap->db_class];
		if (q->q_count < q->q_hiwat)
			q->q_flag &= ~QFULL;
		q->q_flag &= ~QWANTR;
		bp->b_next = bp->b_prev = NULL;
	}

	if (q->q_count<=q->q_lowat && q->q_flag&QWANTW) {
		q->q_flag &= ~QWANTW;
		/* find nearest back queue with service proc */
		for (q = backq(q); q && !q->q_qinfo->qi_srvp; q = backq(q));
		if (q)
		{
			PRINT(SSRV, "getq: ");
			qenable(q);
		}
	}
	splx(s);
	return(bp);
}

/******************************************************************************/
/*
 * Remove a message from a queue.  The queue count and other
 * flow control parameters are adjusted and the back queue
 * enabled if necessary.
 */
rmvq(q, mp)
register iqueue_t *q;
register mblk_t *mp;
{
	register s;
	register mblk_t *tmp;

	ASSERT(valid_queue(q));
	ASSERT(valid_iopm_mblk(mp));

	s = splstr();

	if (mp->b_prev) mp->b_prev->b_next = mp->b_next;
	else q->q_first = mp->b_next;

	if (mp->b_next) mp->b_next->b_prev = mp->b_prev;
	else q->q_last = mp->b_prev;

	mp->b_next = mp->b_prev = NULL;

	for (tmp = mp; tmp; tmp = tmp->b_cont)
		q->q_count -= bsize[tmp->b_datap->db_class];

	if (q->q_count < q->q_hiwat)
		q->q_flag &= ~QFULL;

	if (q->q_count<=q->q_lowat && q->q_flag&QWANTW) {
		q->q_flag &= ~QWANTW;
		/* find nearest back queue with service proc */
		for (q = backq(q); q && !q->q_qinfo->qi_srvp; q = backq(q));
		if (q)
			qenable(q);
	}
	splx(s);
}

/******************************************************************************/
/*
 * Empty a queue.  
 * If flag is set, remove all messages.  Otherwise, remove
 * only non-control messages.  If queue falls below its low
 * water mark, and QWANTW was set before the flush, enable
 * the nearest upstream service procedure.
 */
flushq(q, flag)
register iqueue_t *q;
{
	register mblk_t *bp, *nbp;
	register s;
	int wantw;

	ASSERT(valid_queue(q));

	s = splstr();

	wantw = q->q_flag & QWANTW;

	bp = q->q_first;
	q->q_first = NULL;
	q->q_last = NULL;
	q->q_count = 0;
	q->q_flag &= ~(QFULL|QWANTW);
	splx(s);
	while (bp) {
		nbp = bp->b_next;
		bp->b_next = 0;
		if (!flag && !datamsg(bp->b_datap->db_type))
			putq(q, bp);
		else
			freemsg(bp);
		bp = nbp;
	}

	s = splstr();
	if ((q->q_count <= q->q_lowat) && wantw) {
		/* find nearest back queue with service proc */
		for (q = backq(q); q && !q->q_qinfo->qi_srvp; q = backq(q));
		if (q)
			qenable(q);
	}
	else
		q->q_flag |= wantw;
	splx(s);
}

/******************************************************************************/
/*
 * Return 1 if the queue is not full.  If the queue is full, return
 * 0 (may not put message) and set QWANTW flag (caller wants to write
 * to the queue).
 */
canput(q)
iqueue_t *q;
{
	register  savpri;

	if ( !q )
		return 0;

	ASSERT(valid_queue(q));

	savpri = splstr();
	while ( q->q_next && !q->q_qinfo->qi_srvp )
		q = q->q_next;

	if ( q->q_flag & QFULL )
	{
		q->q_flag |= QWANTW;
		splx(savpri);
		return 0;
	}
	splx(savpri);
	return 1;
}

/******************************************************************************/
/*
 * Put a message on a queue.  
 *
 * Messages are enqueued on a priority basis.  The priority classes
 * are PRIORITY (type >= QPCTL) and NORMAL (type < QPCTL). 
 *
 * Add appropriate weighted data block sizes to queue count.
 * If queue hits high water mark then set QFULL flag.
 *
 * QWANTW is reset if the message is not PRIORITY, because the back
 * module will not want to write again unless it indicates such by
 * first calling canput().
 *
 * If QNOENAB is not set (putq is allowed to enable the queue),
 * enable the queue only if the message is PRIORITY,
 * or the QWANTR flag is set (indicating that the service procedure 
 * is ready to read the queue.  This implies that a service 
 * procedure must NEVER put a priority message back on its own
 * queue, as this would result in an infinite loop (!).
 */
putq(q, bp)
register iqueue_t *q;
register mblk_t *bp;
{
	register s;
	register mblk_t *tmp;
	register mcls = queclass(bp);

	ASSERT(valid_queue(q));
	ASSERT(valid_iopm_mblk(bp) || valid_pm_mblk(bp));
	ASSERT(bp->b_next == NULL);

	s = splstr();

	/* 
	 * If queue is empty or queue class of message is less than
	 * that of the last one on the queue, tack on to the end.
	 */
	if ( !q->q_first || (mcls <= queclass(q->q_last)) ){
		if (q->q_first) {
			q->q_last->b_next = bp;
			bp->b_prev = q->q_last;
		} else {
			q->q_first = bp;
			bp->b_prev = NULL;
		}
		bp->b_next = NULL;
		q->q_last = bp;

	} else {
		register mblk_t *nbp = q->q_first;

		while (queclass(nbp) >= mcls) nbp = nbp->b_next;
		bp->b_next = nbp;
		bp->b_prev = nbp->b_prev;
		if (nbp->b_prev) nbp->b_prev->b_next = bp;
		else q->q_first = bp;
		nbp->b_prev = bp;
	}

	for (tmp = bp; tmp; tmp = tmp->b_cont)
		q->q_count += bsize[tmp->b_datap->db_class];
	if (q->q_count >= q->q_hiwat)
		q->q_flag |= QFULL;

	if ( (mcls > QNORM) || (canenable(q) && (q->q_flag & QWANTR)) )
	{
		PRINT(SSRV, "putq: ");
		qenable(q);
	}

	splx(s);
}

/******************************************************************************/
/*
 * Put stuff back at beginning of Q according to priority order.
 * See comment on putq above for details. QWANTW is not reset as
 * this is internal to the module.
 */
putbq(q, bp)
register iqueue_t *q;
register mblk_t *bp;
{
	register s;
	register mblk_t *tmp;
	register mcls = queclass(bp);

	ASSERT(valid_queue(q));
	ASSERT(valid_iopm_mblk(bp));
	ASSERT(bp->b_next == NULL);

	s = splstr();

	PRINT4(SROUT, "putbq: put mp %x on \"%s\" %sq (%x)\n",
	  bp,
	  q->q_qinfo->qi_minfo?q->q_qinfo->qi_minfo->mi_idname:"no module_info",
	  q->q_flag & QREADR ? "r" : "w", q);

	/* 
	 * If queue is empty of queue class of message >= that of the
	 * first message, place on the front of the queue.
	 */
	if ( !q->q_first || (mcls >= queclass(q->q_first))) {
		bp->b_next = q->q_first;
		bp->b_prev = NULL;
		if (q->q_first) q->q_first->b_prev = bp;
		else q->q_last = bp;
		q->q_first = bp;
	}
	else {
		register mblk_t *nbp = q->q_first;

		while ((nbp->b_next) && (queclass(nbp->b_next) > mcls))
				nbp = nbp->b_next;

		if (bp->b_next = nbp->b_next) 
			nbp->b_next->b_prev = bp;
		else
			q->q_last = bp;
		nbp->b_next = bp;
		bp->b_prev = nbp;
	}

	for (tmp = bp; tmp; tmp = tmp->b_cont)
		q->q_count += bsize[tmp->b_datap->db_class];
	if (q->q_count >= q->q_hiwat)
		q->q_flag |= QFULL;

	if ( (mcls > QNORM) || (canenable(q) && q->q_flag & QWANTR) )
		qenable(q);

	splx(s);
}

/******************************************************************************/
/* just like putbq but w/o qenable */
noenable_putbq(q, bp)
register iqueue_t *q;
register mblk_t *bp;
{
	register s;
	register mblk_t *tmp;
	register mcls = queclass(bp);

	ASSERT(valid_queue(q));
	ASSERT(valid_iopm_mblk(bp) || valid_pm_mblk(bp));
	ASSERT(bp->b_next == NULL);

	s = splstr();

	/* 
	 * If queue is empty of queue class of message >= that of the
	 * first message, place on the front of the queue.
	 */
	if ( !q->q_first || (mcls >= queclass(q->q_first))) {
		bp->b_next = q->q_first;
		bp->b_prev = NULL;
		if (q->q_first) q->q_first->b_prev = bp;
		else q->q_last = bp;
		q->q_first = bp;
	}
	else {
		register mblk_t *nbp = q->q_first;

		while ((nbp->b_next) && (queclass(nbp->b_next) > mcls))
				nbp = nbp->b_next;

		if (bp->b_next = nbp->b_next) 
			nbp->b_next->b_prev = bp;
		else
			q->q_last = bp;
		nbp->b_next = bp;
		bp->b_prev = nbp;
	}

	for (tmp = bp; tmp; tmp = tmp->b_cont)
		q->q_count += bsize[tmp->b_datap->db_class];
	if (q->q_count >= q->q_hiwat)
		q->q_flag |= QFULL;

	splx(s);
}

/******************************************************************************/
/*
 * Insert a message before an existing message on the queue.  If the
 * existing message is NULL, the new messages is placed on the end of
 * the queue.  The queue class of the new message is ignored.
 * All flow control parameters are updated.
 */
insq(q, emp, mp)
register iqueue_t *q;
register mblk_t *emp, *mp;
{
	register mblk_t *tmp;
	register        savpri;

	ASSERT(valid_queue(q));
	ASSERT(valid_iopm_mblk(emp) || emp == NULL);
	ASSERT(valid_iopm_mblk(mp));
	ASSERT(mp->b_next == NULL);

	savpri = splstr();
	if (mp->b_next = emp) {
		if (mp->b_prev = emp->b_prev) 
			emp->b_prev->b_next = mp;
		else
			q->q_first = mp;
		emp->b_prev = mp;
	} else {
		if (mp->b_prev = q->q_last) 
			q->q_last->b_next = mp;
		else 
			q->q_first = mp;
		q->q_last = mp;
	}

	for (tmp = mp; tmp; tmp = tmp->b_cont)
		q->q_count += bsize[tmp->b_datap->db_class];

	if (q->q_count >= q->q_hiwat)
		q->q_flag |= QFULL;
	
	if ( canenable(q) && (q->q_flag & QWANTR) )
		qenable(q);

	splx(savpri);
}

/******************************************************************************/
/*
 * Create and put a control message on queue.
 */
putctl(q, type)
iqueue_t *q;
{
	register mblk_t *bp;

	ASSERT(valid_queue(q));

	if (datamsg(type) || !(bp = allocb(0, BPRI_HI)))
		return(0);

	bp->b_datap->db_type = type;

	*LED_CTRL_REG = LED1_OFF;
	*LED_CTRL_REG = LED2_ON;

	ASSERT(valid_iopm_vaddr(q->q_qinfo));
	ASSERT(valid_iopm_vaddr(q->q_qinfo->qi_putp));

	(*q->q_qinfo->qi_putp)(q, bp);

	*LED_CTRL_REG = LED1_ON;
	*LED_CTRL_REG = LED2_OFF;

	return(1);
}

/******************************************************************************/
/*
 * Control message with a single-byte parameter
 */
putctl1(q, type, param)
iqueue_t *q;
{
	register mblk_t *bp;

	ASSERT(valid_queue(q));

	if (datamsg(type) ||!(bp = allocb(1, BPRI_HI)))
		return(0);
	bp->b_datap->db_type = type;
	*bp->b_wptr++ = param;

	*LED_CTRL_REG = LED1_OFF;
	*LED_CTRL_REG = LED2_ON;

	ASSERT(valid_iopm_vaddr(q->q_qinfo));
	ASSERT(valid_iopm_vaddr(q->q_qinfo->qi_putp));

	(*q->q_qinfo->qi_putp)(q, bp);

	*LED_CTRL_REG = LED1_ON;
	*LED_CTRL_REG = LED2_OFF;

	return(1);
}

/******************************************************************************/
/* return 1 for ok, 0 for fail
*/
setnummsg( n4, n16, n64, n128, n256, n512, n1024, n2048, n4096 )
uint n4, n16, n64, n128, n256, n512, n1024, n2048, n4096;
{
	int                savpri = splstr();
	uint               numpg;
	uint               blkpg;
	uint               numdblk;
	uint               nummblk;
	uchar              *base;
	dblk_t             *databp;
	mblk_t             *msgbp;
	int                class;
	int                i;
	int                dblockp;
	extern struct map  memmap[];

	PRINT5(CONFIG, "setnummsg: msg 4-4k; %d %d %d %d %d ",
	  n4, n16, n64, n128, n256);
	PRINT4(CONFIG, "%d %d %d %d\n", n512, n1024, n2048, n4096);

	if ( nmblock )
		return 0;

	numdblk = n4096 + n2048 + n1024 + n512 + n256 + n128 + n64 + n16 + n4;
/*CMW move nmblock set after last return 0 */
	nmblock = nummblk = numdblk + numdblk/4;
	numpg = btoc( numdblk * sizeof(dblk_t) + nummblk * sizeof(mblk_t) );
/*CMW note: caddr_t sb mblk_t* */
	mblkp = (caddr_t)(memalloc((int)numpg) << PNUMSHFT);
	if ( mblkp == 0 )
	{
		splx(savpri);
		return 0;
	}
	dblkp = (caddr_t)mblkp + nummblk * sizeof(mblk_t);
	bzero(mblkp, numpg * NBPP);

	blkpg = btoc( n4096*4096 + n2048*2048 + n1024*1024 +
	        n512*512 + n256*256 + n128*128 + n64*64 + n16*16 + n4*4 );

	dblockp = memalloc((int)blkpg) << PNUMSHFT;
	if ( dblockp == 0 )
	{
		memfree((uint)mblkp, (int)numpg);
		splx(savpri);
		return 0;
	}
	bzero((caddr_t)dblockp, blkpg * NBPP);

	PRINT3(CONFIG,"setnummsg: mblkp at %x, dblkp at %x, msg's at %x\n",
	  mblkp, dblkp, dblockp);

	nblk[0] = n4096;
	nblk[1] = n2048;
	nblk[2] = n1024;
	nblk[3] = n512;
	nblk[4] = n256;
	nblk[5] = n128;
	nblk[6] = n64;
	nblk[7] = n16;
	nblk[8] = n4;

	base = (uchar *)dblockp;

	databp = (dblk_t *)dblkp;
	for (class = NCLASS-1; class >=0; class--) {
		dbfreelist[class] = NULL;
		dballoc[class].dba_lo = (nblk[NCLASS-1-class] * strlofrac)/100;
		dballoc[class].dba_med = (nblk[NCLASS-1-class] * strmedfrac)/100;
		for (i=0; i<nblk[NCLASS-1-class]; i++, databp++) {
			databp->db_class = class;
			databp->db_base = base;
			databp->db_lim = base + rbsize[class];
			base += rbsize[class];
			databp->db_freep = dbfreelist[class];
			dbfreelist[class] = databp;
		}
	}

	mbfreelist = NULL;
	for (i=0; i<nmblock; i++) {
		msgbp = (mblk_t *)mblkp + i;
		msgbp->b_next = mbfreelist;
		mbfreelist = msgbp;
	}

	splx(savpri);
	return 1;
}

/******************************************************************************/
/* Init routine run from main at boot time.  This contains some 
/* machine dependent code.
*/
strinit()
{
	register dblk_t *databp;
	register mblk_t *msgbp;
	register i;
	register unsigned char *base;
	register        savpri;
	int class, *vnp;
	extern char *strbasep;
	extern mblk_t  *mblockp;
	extern dblk_t  *dblockp;
	extern iqueue_t queue[];
	extern uint    nq;

	savpri = splstr();

	/*
	 * Set up initial stream event cell free list.
	 */
	sefreelist = NULL;
	if ( nstrevent )
		for (i=0; i < nstrevent; i++) {
			(streventp + i)->se_next = sefreelist;
			sefreelist = streventp + i;
	}

	for (class = NCLASS-1; class >=0; class--)
	{
		dbfreelist[class] = NULL;
		dballoc[class].dba_lo = 0;
		dballoc[class].dba_med = 0;
	}
	mbfreelist = NULL;
	queuep = 0;
	mblkp = 0;
	dblkp = 0;
	for ( i = 0; i < NCLASS; i++ )
		nblk[i] = 0;
	if ( strbasep )
	{
		extern nb[];
		base = (uchar *)strbasep;

		databp = dblockp;
		for (class = NCLASS-1, vnp = nb; class >=0; class--, vnp++) {
			dbfreelist[class] = NULL;
			dballoc[class].dba_lo = ((*vnp) * strlofrac)/100;
			dballoc[class].dba_med = ((*vnp) * strmedfrac)/100;
			for (i=0; i<*vnp; i++, databp++) {
				databp->db_class = class;
				databp->db_base = base;
				databp->db_lim = base + rbsize[class];
				base += rbsize[class];
				databp->db_freep = dbfreelist[class];
				dbfreelist[class] = databp;
			}
		}

		mbfreelist = NULL;
		for (i=0; i<nmblock; i++) {
			msgbp = mblockp + i;
			msgbp->b_next = mbfreelist;
			mbfreelist = msgbp;
		}

		mblkp = (caddr_t)mblockp;
		dblkp = (caddr_t)dblockp;
		nblk[0] = nb[0];
		nblk[1] = nb[1];
		nblk[2] = nb[2];
		nblk[3] = nb[3];
		nblk[4] = nb[4];
		nblk[5] = nb[5];
		nblk[6] = nb[6];
		nblk[7] = nb[7];
		nblk[8] = nb[8];
	}

	if ( nq )
	{
		queuep = queue;
		nqueue = nq;
	}

	splx(savpri);
	return;
}

/******************************************************************************/
/* returns 1 for ok, 0 for fail
*/
setnumq(numq)
uint numq;	/* number of queue pairs to desired */
{
	int                savpri = splstr();
	uint               numpg;
	uint               i;
	extern uint        nq;
	extern struct map  memmap[];

	if ( nq )
	{
		splx(savpri);
		return 0;	/* can't set queues if already set */
	}

	if ( numq / 2 * 2 != numq )
		numq++;		/* must be even number of queues */

	numpg = btoc(numq * sizeof(iqueue_t));
	queuep = (iqueue_t *)(memalloc((int)numpg) << PNUMSHFT);
	if ( queuep == 0 )
	{
		splx(savpri);
		return 0;
	}

	bzero((caddr_t)queuep, (uint)numpg * NBPP);

	nstrevent = numq + 16;
	streventp = (struct strevent *)(memalloc(
	  (int)btoc(nstrevent * sizeof(struct strevent))) << PNUMSHFT);
	if ( streventp == 0 )
	{
		memfree((uint)queuep, (int)numpg);
		queuep = 0;
		splx(savpri);
		return 0;
	}

	sefreelist = NULL;
	for (i = 0; i < nstrevent; i++) {
		(streventp + i)->se_next = sefreelist;
		sefreelist = streventp + i;
	}

	nq = nqueue = (numpg * NBPP) / (2 * sizeof(iqueue_t)) * 2;

	PRINT1(CONFIG, "setnumq: num queue %d\n", nq);

	splx(savpri);
	return 1;
}

/******************************************************************************/
freeq(rq)
iqueue_t  *rq;
{
	register s;

	ASSERT(valid_rq(rq));

	s = splstr();
	(rq)->q_flag = WR(rq)->q_flag = 0;
	strst.queue.use--;
	splx(s);
}

/******************************************************************************/
/*
 * allocate a pair of queues
 */
iqueue_t *
allocq()
{
	register s;
	register iqueue_t *rq;
	static iqueue_t zeroR =
	  { NULL,NULL,NULL,NULL,NULL,NULL,0,QUSE|QREADR,0,0,0,0,0};
	static iqueue_t zeroW =
	  { NULL,NULL,NULL,NULL,NULL,NULL,0,QUSE,0,0,0,0,0};
	int qused = 0;		/* diagnostics only */

	s = splstr();
	/*
	 * Check each pair of queues until one is found with QUSE off
	 */
	for (rq = queuep; rq < queuep + nqueue; rq += 2) {
		if ((rq->q_flag & QUSE) == 0) {
			*rq = zeroR;
			*WR(rq) = zeroW;
			BUMPUP(strst.queue);
			splx(s);

			PRINT3(SRESRC, "allocq: q %x, %x/%x used\n",
			  rq, qused, nqueue);

			return(rq);
		}
		qused +=2;
	}
	strst.queue.fail++;
	splx(s);
	cmn_err(CE_WARN,"allocq: out of queues.");
	return(NULL);
}

/******************************************************************************/
/*
 * return the queue upstream from this one
 */
iqueue_t *
backq(q)
register iqueue_t *q;
{
	ASSERT(valid_queue(q));

	q = OTHERQ(q);
	if (q->q_next) {
		q = q->q_next;
		return(OTHERQ(q));
	}
	return(NULL);
}

/******************************************************************************/
/*
 * Send a block back up the queue in reverse from this
 * one (e.g. to respond to ioctls)
 */
qreply(q, bp)
register iqueue_t *q;
mblk_t *bp;
{
	ASSERT(valid_queue(q));
	ASSERT(valid_iopm_mblk(bp));

	q = OTHERQ(q);
	ASSERT(valid_queue(q->q_next));

	*LED_CTRL_REG = LED1_OFF;
	*LED_CTRL_REG = LED2_ON;

	ASSERT(valid_iopm_vaddr(q->q_next->q_qinfo));
	ASSERT(valid_iopm_vaddr(q->q_next->q_qinfo->qi_putp));

	(*q->q_next->q_qinfo->qi_putp)(q->q_next, bp);

	*LED_CTRL_REG = LED1_ON;
	*LED_CTRL_REG = LED2_OFF;
}

/******************************************************************************/
/*
 * Streams Queue Scheduling
 * 
 * Queues are enabled through qenable() when they have messages to 
 * process.  They are serviced by queuerun(), which runs each enabled
 * queue's service procedure.  The call to queuerun() is processor
 * dependent - the general principle is that it be run whenever a queue
 * is enabled but before returning to user level.  For system calls,
 * the function runqueues() is called if their action causes a queue
 * to be enabled.  For device interrupts, queuerun() should be
 * called before returning from the last level of interrupt.  Beyond
 * this, no timing assumptions should be made about queue scheduling.
 */

/*
 * Enable a queue: put it on list of those whose service procedures are
 * ready to run and set up the scheduling mechanism.
 */
qenable(q)
register iqueue_t *q;
{
	register s;

	ASSERT(valid_queue_addr(q));

	if (!q->q_qinfo->qi_srvp) return;

	if ( !(q->q_flag & QUSE) )
	{
		cmn_err(CE_WARN,
		  "Attempted qenable on queue that is not in use.");
		return;
	}

	s = splstr();
	/*
	 * Do not place on run queue if already enabled.
	 */
	if (q->q_flag & QENAB) {
		splx(s);
		return;
	}

	PRINT3(SSRV, "qenable: schedule \"%s\" %sq (%x)\n",
	  q->q_qinfo->qi_minfo?q->q_qinfo->qi_minfo->mi_idname:"no module_info",
	  q->q_flag & QREADR ? "r" : "w", q);

	/*
	 * mark queue enabled and place on run list
	 */
	q->q_flag |= QENAB;

	if (!qhead) qhead = q;
	else qtail->q_link = q;
	qtail = q;
	q->q_link = NULL;

	/*
	 * set up scheduling mechanism
	 */
	qrunflag = 1;
	splx(s);
}

/******************************************************************************/
/*
 * Run the service procedures of each enabled queue
 *	-- must not be reentered
 *
 * Called by service mechanism (processor dependent) if there
 * are queues to run.  The mechanism is reset.
 */
queuerun()
{
	register iqueue_t *q;
	register s;
	register i;
	register int *vnp;
	struct strevent *sep;
	int count, total;
	struct dbalcst *dbp;

	s = splstr();
	do {
		if (strbcflag) {
			strbcflag = 0;
			for (i=NCLASS-1, vnp = (int *)nblk; i>=0; i--, vnp++){
				dbp = &dballoc[i];
				count = *vnp - dbp->dba_cnt;
				total = 0;
				while ( mbfreelist && 
					(count > 0) && (sep = dbp->dba_hip)) {
					count--;
					total++;
					dbp->dba_hip = sep->se_next;
					(*sep->se_func)(sep->se_arg);
					sefree(sep);
				}
				count = (dbp->dba_med - dbp->dba_cnt) - total;
				while ( mbfreelist && 
					(count > 0) && (sep = dbp->dba_medp)) {
					count--;
					total++;
					dbp->dba_medp = sep->se_next;
					(*sep->se_func)(sep->se_arg);
					sefree(sep);
				}
				count = (dbp->dba_lo - dbp->dba_cnt) - total;
				while ( mbfreelist && 
					(count > 0) && (sep = dbp->dba_lop)) {
					count--;
					dbp->dba_lop = sep->se_next;
					(*sep->se_func)(sep->se_arg);
					sefree(sep);
				}
			}
		}

		while (q = qhead) {
			ASSERT( valid_iopm_addr(q) );
			if (!(qhead = q->q_link)) qtail = NULL;
			q->q_flag &= ~QENAB;
			if ( !(q->q_flag & QUSE) )
			{
				cmn_err(CE_WARN,
				  "Attempted queuerun on queue that is not in use.");
				continue;
			}
			ASSERT(valid_iopm_vaddr(q->q_qinfo));
			if (q->q_qinfo->qi_srvp) {
			ASSERT(valid_iopm_vaddr(q->q_qinfo->qi_srvp));
				spl1();
				PRINT4(SSRV,
		"queuerun: about to call service (%x) for \"%s\" %sq (%x)\n",
				  q->q_qinfo->qi_srvp,
				  q->q_qinfo->qi_minfo ?
				    q->q_qinfo->qi_minfo->mi_idname :
				    "no module_info",
				  q->q_flag & QREADR ? "r" : "w", q);

				*LED_CTRL_REG = LED1_OFF;
				*LED_CTRL_REG = LED2_ON;

				(*q->q_qinfo->qi_srvp)(q);

				*LED_CTRL_REG = LED1_ON;
				*LED_CTRL_REG = LED2_OFF;

				splstr();
			}
		}
	} while (strbcflag);

	qrunflag = 0;
	splx(s);
}

/******************************************************************************/
chk_runlist(rq)
register iqueue_t  *rq;
{
	register iqueue_t  *q;
	register iqueue_t  *prev = NULL;

	/*
	 * Check if queues are still enabled, and remove from 
	 * runlist if necessary.
	 */
	ASSERT(valid_rq(rq));
	if ((rq->q_flag | WR(rq)->q_flag) & QENAB) {
		for (q = qhead; q; q = q->q_link)  {
			if (q == rq || q == WR(rq)) {
				if (prev)
					prev->q_link = q->q_link;
				else
					qhead = q->q_link;
				if (q == qtail)
					qtail = prev;
			}
			else
				prev = q;
		}
	}
}

/******************************************************************************/
/*
 * Function to kick off queue scheduling for those system calls
 * that cause queues to be enabled (read, recv, write, send, ioctl).
 */
runqueues()
{
	register s;

	s = spl5();
	if (qrunflag & !queueflag) {
		queueflag = 1;
		splx(s);
		queuerun();
		queueflag = 0;
		return;
	}
	splx(s);
}

/******************************************************************************/
/* 
 * find module
 * 
 * return index into fmodsw
 * or -1 if not found
 */
findmod(name)
register char *name;
{
	register int i, j;

	for (i = 0; i < fmodcnt; i++)
		for (j = 0; j < FMNAMESZ + 1; j++) {
			if (fmodsw[i].f_name[j] != name[j]) 
				break;
			if (name[j] == '\0')
				return(i);
		}
	return(-1);
}

/******************************************************************************/
/* 
 * return number of messages on queue
 */
qsize(qp)
register iqueue_t *qp;
{
	register count = 0;
	register mblk_t *mp;
	register        savpri;

	ASSERT(valid_queue(qp));

	savpri = splstr();
	for (mp = qp->q_first; mp; mp = mp->b_next)
		count++;

	splx(savpri);
	return(count);
}

/******************************************************************************/
/*
 * allocate a stream event cell
 */
struct strevent *
sealloc()
{
	register s;
	register struct strevent *sep;

	s = splstr();
	if ( !sefreelist )
	{
		splx(s);
		return NULL;
	}

	sep = sefreelist;
	sefreelist = sep->se_next;
	splx(s);
	sep->se_procp = NULL;
	sep->se_events = 0;
	sep->se_next = NULL;
	return sep;
}

/******************************************************************************/
/*
 * free a stream event cell
 */
sefree(sep)
struct strevent *sep;
{
	register s;

	s = splstr();
	sep->se_next = sefreelist;
	sefreelist = sep;
	splx(s);
}

/******************************************************************************/
/*
 * Set the interface values for a pair of queues (qinit structure,
 * packet sizes, water marks).
 */
setq(rq, rinit, winit)
iqueue_t *rq;
struct qinit *rinit, *winit;
{
	register iqueue_t  *wq;
	register          savpri;

	ASSERT(valid_rq(rq));
	ASSERT(valid_iopm_vaddr(rinit));
	ASSERT(valid_iopm_vaddr(rinit->qi_minfo));
	ASSERT(valid_iopm_vaddr(winit));
	ASSERT(valid_iopm_vaddr(winit->qi_minfo));

	wq = WR(rq);

	savpri = splstr();
	rq->q_qinfo = rinit;
	rq->q_hiwat = rinit->qi_minfo->mi_hiwat;
	rq->q_lowat = rinit->qi_minfo->mi_lowat;
 	rq->q_minpsz = rinit->qi_minfo->mi_minpsz;
	rq->q_maxpsz = rinit->qi_minfo->mi_maxpsz;
	wq->q_qinfo = winit;
	wq->q_hiwat = winit->qi_minfo->mi_hiwat;
	wq->q_lowat = winit->qi_minfo->mi_lowat;
	wq->q_minpsz = winit->qi_minfo->mi_minpsz;
	wq->q_maxpsz = winit->qi_minfo->mi_maxpsz;
	splx(savpri);
}

/******************************************************************************/
/*
 * Attach a stream device or module.
 * qp is a read queue; the new queue goes in so its next
 * read ptr is the argument, and the write queue corresponding
 * to the argument points to this queue.
 * Return dev on success, OPENFAIL on failure. (different from kernel)
 */
qattach(qinfo, qp, dev, flag, sflg)
register struct streamtab *qinfo;
register iqueue_t *qp;
dev_t dev;
{
	register iqueue_t *rq;
	register s;
	int      openresult;

	ASSERT(valid_iopm_vaddr(qinfo));
	ASSERT(valid_rq(qp));
	ASSERT(dev & UNMINOR_BIT);

	if ( !(rq = allocq()) )
		return OPENFAIL;

	PRINT2(SOPEN, "qattach: new rq = %x, new wq = %x\n", rq, WR(rq));
	s = splstr();
	rq->q_next = qp;
	WR(rq)->q_next = WR(qp)->q_next;
	if (WR(qp)->q_next) {
		OTHERQ(WR(qp)->q_next)->q_next = rq;
		sflg = MODOPEN;
	}
	WR(qp)->q_next = WR(rq);
	setq(rq, qinfo->st_rdinit, qinfo->st_wrinit);
 	rq->q_flag |= QWANTR;
	WR(rq)->q_flag |= QWANTR;

	/*
	 * Open the attached module or driver.
	 * The open may sleep, but it must always return here.  Therefore,
	 * all sleeps must set PCATCH or ignore all signals to avoid a 
	 * longjump if a signal arrives.
	 */

	ASSERT(valid_iopm_vaddr(rq->q_qinfo));
	ASSERT(valid_iopm_vaddr(rq->q_qinfo->qi_qopen));

	PRINT4(SOPEN,
	  "qattach: about to call qopen(%x) dev = %x, flag = %x sflag = %x\n",
	  *rq->q_qinfo->qi_qopen, dev, flag, sflg );

	*LED_CTRL_REG = LED1_OFF;
	*LED_CTRL_REG = LED2_ON;

	if ( (openresult = (*rq->q_qinfo->qi_qopen)(rq, dev, flag, sflg)) ==
	  OPENFAIL )
	{
		*LED_CTRL_REG = LED1_ON;
		*LED_CTRL_REG = LED2_OFF;

		qdetach(rq, 0, 0);
		splx(s);
		return OPENFAIL;
	}

	*LED_CTRL_REG = LED1_ON;
	*LED_CTRL_REG = LED2_OFF;

	splx(s);
	/* return dev (may be new from clone open) to sopenproc */
	return openresult;
}

/******************************************************************************/
/*
 * Detach a stream module or device.
 * If clmode == 1 then the module or driver was opened and its
 * close routine must be called.  If clmode == 0, the module
 * or driver was never opened or the <open failed, and so its close
 * should not be called.
 */
qdetach(rq, clmode, flag)
register iqueue_t *rq;
{
	register s;

	ASSERT(valid_rq(rq));

	if (clmode) {
		if (qrunflag) runqueues();
		s = splstr();

		*LED_CTRL_REG = LED1_OFF;
		*LED_CTRL_REG = LED2_ON;

		ASSERT(valid_iopm_vaddr(rq->q_qinfo));
		ASSERT(valid_iopm_vaddr(rq->q_qinfo->qi_qclose));

		(*rq->q_qinfo->qi_qclose)(rq, (rq->q_next ? 0 : flag));

		*LED_CTRL_REG = LED1_ON;
		*LED_CTRL_REG = LED2_OFF;

		chk_runlist(rq);	/* remove q's from queuerun list */
		chk_bufcall((long)rq);	/* remove q's from bufcall list */
		chktimeout(rq, WR(rq));	/* remove q's from timeout list */

		flushq(rq, FLUSHALL);
		flushq(WR(rq), FLUSHALL);
	} else 
		s = splstr();

	if (WR(rq)->q_next)
		backq(rq)->q_next = rq->q_next;

	if (rq->q_next)
		backq(WR(rq))->q_next = WR(rq)->q_next;

	freeq(rq);
	splx(s);
}

/******************************************************************************/
/* 
 * given a queue ptr, follow the chain of q_next pointers until you reach the
 * last queue on the chain and return it
 */
iqueue_t *
getendq(q)
register iqueue_t *q;
{
	ASSERT(valid_queue(q));

	while (q->q_next)
	{
		q = q->q_next;
		ASSERT(valid_queue(q));
	}

	return(q);
}

/******************************************************************************/
noenable(q)
iqueue_t *q;
{ 
	register int s;

	ASSERT(valid_queue(q));

	s = splstr();
	q->q_flag |= QNOENB;
	splx(s);
}

/******************************************************************************/
enableok(q)
iqueue_t *q;
{
	register int s;

	ASSERT(valid_queue(q));

	s = splstr();
	q->q_flag &= ~QNOENB;
	splx(s);
}

/******************************************************************************/
iqueue_t *
RD(q)
iqueue_t  *q;
{
	ASSERT(valid_wq(q));
	return q - 1;
}

iqueue_t *
WR(q)
iqueue_t  *q;
{
	ASSERT(valid_rq(q));
	return q + 1;
}

iqueue_t *
OTHERQ(q)
iqueue_t  *q;
{
	ASSERT(valid_queue(q));
	return q->q_flag & QREADR ? q + 1 : q - 1;
}

/******************************************************************************/
valid_queue_addr(addr)
uint addr;
{
	extern iqueue_t  *queuep;
	extern uint      nqueue;

	return addr >= (uint)queuep &&
	       addr < (uint)(queuep + nqueue) &&
	       !((addr - (uint)queuep) % sizeof(iqueue_t));

}

valid_queue(addr)
uint addr;
{
	extern iqueue_t  *queuep;
	extern uint      nqueue;

	return addr >= (uint)queuep &&
	       addr < (uint)(queuep + nqueue) &&
	       !((addr - (uint)queuep) % sizeof(iqueue_t)) &&
	       ((iqueue_t *)addr)->q_flag & QUSE;
}

valid_rq(q)
iqueue_t  *q;
{
	if ( !valid_queue((uint)q) )
		return 0;

	return q->q_flag & QREADR;
}

valid_wq(q)
iqueue_t  *q;
{
	if ( !valid_queue((uint)q) )
		return 0;

	return !(q->q_flag & QREADR);
}

valid_iopm_mblk(imp)
register mblk_t  *imp;
{
	extern caddr_t  mblkp;
	extern int      nmblock;

	if (  (uint)imp < (uint)mblkp ||
	      (uint)imp >= (uint)((mblk_t *)mblkp + nmblock) ||
	      ((uint)imp - (uint)mblkp) % sizeof(mblk_t) )
		return 0;

	if ( !valid_iopm_addr((uint)imp->b_datap) )
		return 0;

	return imp->b_datap->db_ref;
}

