/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) opt_intr.c: version 25.1 created on 11/27/91 at 14:44:53	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)opt_intr.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
#include "sys/types.h"
#include "sys/param.h"
#include "sys/sysmacros.h"
#include "sys/errno.h"
#include "sys/elog.h"
#include "sys/file.h"
#include "sys/debug.h"

#include "sys/buf.h"
#include "sys/immu.h"

#include "scsi_log.h"
#include "scsi_error.h"
#include "hlvl_macro.h"
#include "opt_cdb.h"
#include "sys/ioctl.h"
#include "scsi_cmdbx.h"
#include "opt.h"

extern struct optiobuf opttbl[];
extern buf_t *o_ownbuf[];
extern get_time_stamp();
extern scsi_send_cmd();

char	*get_add_sense_msg();
char	*get_vendor_uniq_add_msg();

/*
 * A little phiosophy for the interested reader.
 * 
 * optical disk reads and writes are dma'd to user memory (as they are in 
 * all dsdb drivers). There are two cases where this driver needs to access
 * the data in main memory. The first is to swap data on read/write and the
 * second is to check the data for a pretend blank check (both horrible
 * concepts).  * The routines to handle this need to map one page at a time.
 * This means that they need to know the page size (NBPP), which is a
 * machine dependent constant. Currently the value of NBPP is the same for
 * the 020 and 040, but it may change in future systems. Ideally the driver
 * should be machine independent.
 */

/* opt_intr -- disk interrupt service routine	*/
void
opt_intr(cm)
CMD_BOX	*cm;
{
	buf_t *bp;
	struct optiobuf	*dp;
	unchar	unit_id;
	uint	do_retry = 0;

	bp = (buf_t *)cm->tag;
	unit_id = UNIT(bp->b_dev);
	dp = GETDP(unit_id);

	bp->b_flags &= ~B_ERROR;

	if (cm->err_code)
		do_retry = opt_errsetup(dp, bp, cm);	

	if (!do_retry) {
		bp->b_resid += cm->dma_resid;
		if (bp != o_ownbuf[dp->unit_id]) {
			if (bp->b_flags & B_READ) {
				/* this should probably be moved to before errsetup
				 * call, since it may be possible to have an error, which
				 * will cause a retry, but which has already transferred 
				 * blocks up to the blank one.
				 */
				if (dp->b_flags & B_SONY_MF && dp->b_flags & B_ERS)
					optcheckblank(dp, bp, cm);

			}
			/* this will swap read buffers and unswap write buffers which
			 * were swapped in the strategy routine */
			if (dp->b_flags & B_SWAP)
				optswap(dp, cm);
			ASSERT(bp->b_resid <= bp->b_bcount);
			dp->sar->io_resp += (get_time_stamp() - bp->b_start);
			dp->sar->io_act += cm->act;
			dp->sar->io_bcnt += 
				((bp->b_bcount - bp->b_resid) >> BLK_SHIFT);
		}
		/* delete bp from sorting queue if necessary */
		optdeque(dp);
		iodone(bp);
		if (cm == dp->cm[0])
			dp->b_outcnt &= ~CMB0_IN_USE; /* turn off used bit */
		else
			dp->b_outcnt &= ~CMB1_IN_USE;	
		optstart(dp);	/* go get next command to excute	*/
	}
} /* opt_intr */

#ifndef MIN
#define MIN(x, y) (((x) < (y)) ? (x) : (y))
#endif

/*
 * optvcmp() - compare a local buffer with one in virtual memory.
 */
optvcmp(buff, local_buff, size, proc)
unchar	*buff, *local_buff;
uint	size;
void	*proc;
{
	uint count = MIN(MIN(NBPP, size), NBPP - poff(buff));
	uint	saved_map = vmap_save();

	while (size) {
		if (!optcmp((unchar *)vmap(buff, proc), local_buff, count)) {
			vmap_restore(saved_map);
			return(0);
		}
		buff += count;
		size -= count;
		count = MIN(NBPP, size); /* next page */
	}
	vmap_restore(saved_map);
	return(1);
}

/*
 * This is from the Imix driver.
 */
static unchar	sonyblank[] = { 0xcf, 0x23, 0xcf, 0x23 };
/*
 * optcheckblank() - check data read from each block
 * against the format pattern. When format pattern is found treat as 
 * a blank check and invalidate rest of data.
 */
static
optcheckblank(dp, bp, cm)
struct	optiobuf	*dp;
buf_t *bp;
CMD_BOX	*cm;
{
	register uint	data_len;
	uint	blk_len = dp->cfg.blk_len; /* drive block size */
	register unchar	*buff = cm->dma_buffer;

	ASSERT(cm->dma_buf_len >= cm->dma_resid);
	data_len = cm->dma_buf_len - cm->dma_resid;
	/* step through buffer one block at a time */
	while (data_len > 0) {
		if (optvcmp(buff, sonyblank, sizeof(sonyblank), cm->dma_proc)) {
			bp->b_resid += data_len;
			DISPLAY_DRV_ID(cm->unit_id);
			printf("pretend blank check, block %x\n",
				bp->b_sblock + (buff - cm->dma_buffer) / blk_len);
			bp->b_flags |= B_ERROR;
			bp->b_error = EIO;
			return;
		}
		buff += blk_len;
		data_len -= blk_len;
	}
}

/* swap bytes in a buffer. These must be in the same page for raw io */
static void
byte_swap_blk(buff, count)
unchar	*buff;
register	uint		count;
{
	register	unchar	*p = buff - 1;
	register	unchar	*q = buff;
	register	unchar	tmp;

	ASSERT(pnum(buff) == pnum(buff + count - 1)); /* same page */
	ASSERT((count & 1) == 0);
	count /= 2;
	while (count--) {
		tmp = *++p;
		*p++ = *(++q);
		*q++ = tmp;
	}
}

void
swap_buff(buff, size, proc)
unchar	*buff;
uint		size;
void	*proc;
{
	register	uint	count;
	int	odd = (uint)buff & 1;	/* starts on odd boundary */
	uint	saved_map = vmap_save();

	size &= (uint)~1;

	/* first swap to end of page */
	count = MIN(MIN(NBPP, size), NBPP - poff(buff));

	while (size) {
		count -= odd;	/* don't do last if odd */
		byte_swap_blk((unchar *)vmap(buff, proc), count);
		size -= count;
		buff += count;
		count = MIN(NBPP, size); /* next page */
		if (size && odd) {
			/* swap last byte on this page with first on next */
			/* need two temps since can't address both pages together */
			unchar	tmp0 = *(unchar *)vmap(buff, proc);
			unchar	*ptr = (unchar *)vmap(buff + 1, proc);
			unchar	tmp1 = *ptr;

			/* only on page boundary */
			ASSERT((((uint)buff + 1) & (NBPP - 1)) == 0);

			*ptr = tmp0;
			*(unchar *)vmap(buff, proc) = tmp1;
			size -= 2;
			buff += 2;
			count--;	/* first byte is done */
		}
	}
	vmap_restore(saved_map);
}

optswap(dp, cm)
struct	optiobuf	*dp;
CMD_BOX	*cm;
{
	ASSERT(cm->dma_buf_len >= cm->dma_resid);
	swap_buff(cm->dma_buffer, cm->dma_buf_len - cm->dma_resid, cm->dma_proc);
}

/*
 * opt_errsetup -- setup real error code and display error message
 * returns 0 for errors and 1 for commands which are being retried.
 */
opt_errsetup(dp, bp, cm)
struct optiobuf	*dp;
buf_t *bp;
CMD_BOX *cm;
{
	int  do_retry;
	
	if (do_retry = (set_sense_error(dp, cm) || set_scsi_err(dp, cm)))
			bp->b_flags &= ~B_ERROR;
	else
		opterror(bp, EIO);
	return(do_retry);
}

/*
 * set_sense_err()
 *
 * Check for drive sense error. Retry command if reset or acceptable new
 * medium.
 */
static
set_sense_error(dp, cm)
struct optiobuf	*dp;
CMD_BOX *cm;
{
	EX_SENSE_DATA *sd;
	unchar	sense_key;

	sd = (EX_SENSE_DATA *)cm->error;
	if (cm->err_code != SCSIERR_SENSE)
		return(0);
	if ((sd ->err_class & ERR_CLASS_MASK) != ANSI_CLASS) {
		DISPLAY_DRV_ID(cm->unit_id);
		printf( " vendor unique error\n");
	}
	else {
		if ((sense_key = sd->sense_key & SENSE_KEY) == SNSKEY_UNIT_ATTN)
		{
			if (sd->err_code == ASNS_MDMCHNG) { /* handle media change */
				dp->b_flags |= B_NEWMDM;
				if (!(dp->b_flags & B_NEW_MED_OK))
					return(0);	/* very bad */
			}

			/* either a reset or an acceptable new medium */
			if (dp->pg1_len != 0)
				opt_set_page_01(dp, cm);
			else {
				cm->cmd_type = CMD_RELEASE_ATTN;
				opt_retry_cmd(cm);
			}
			return(1);
		}

		if (sense_key == SNSKEY_RECOVERED_ERR && cm->dma_resid == 0) {
				sd->sense_key = SNSKEY_NO_SENSE;
				return(0);
		}
		DISPLAY_DRV_ID(cm->unit_id);
		printf("Sense Error.\n");
		display_sense_detail(dp, sd);
	}
	return(0);
}

/*
 * set_scsi_err()
 *
 * setup scsi error from low level driver
 * return 1 if performing a retry otherwise print error message & return 0
 */
static
set_scsi_err(dp, cm)
struct optiobuf	*dp;
CMD_BOX *cm;
{
 	if (!opt_err_retry(cm)) {
		if (cm->err_code == SCSIERR_SENSE)
			return(0); /* already printed */
		DISPLAY_DRV_ID(cm->unit_id);
		printf("%s", scsi_err_msg(cm->err_code)); /* display error message */
		printf("\n");
		return(0);
	}
	else 	
		return(1);
}

static
display_sense_detail(dp, sense_data)
struct optiobuf	*dp; /* required for vendor quique errors */
EX_SENSE_DATA *sense_data;
{
	int xlen;
	char	*msg;

	printf("   sense key: %x", sense_data->sense_key);
	printf(" %s.\n", scsi_sns_msg(sense_data->sense_key));
	if ((xlen = sense_data->xsense_len) == 0)
		return;
	xlen -= 4;
	if (xlen <= 0)
		return;
	printf("   additional sense: %x", sense_data->err_code);
	if (msg = get_vendor_uniq_add_msg(dp, sense_data->err_code))
		printf(", %s.\n", msg);
	else printf(", %s.\n", get_add_sense_msg(sense_data->err_code));
	if (sense_data->err_class & INFO_VALID) {
		printf("   information bytes: ");
		printf("%x\n",CAT4(sense_data->info_msb, sense_data->info_2,
				sense_data->info_1, sense_data->info_lsb));
	}
	if (sense_data->sense_key == SNSKEY_BLANK_CHECK)
		return;
	if (--xlen <= 0) 	/* skip reserved byte */
		return;
	if (--xlen <= 0) 	/* point to FRU code */
		return;
	if (sense_data->frucode)
		printf("\tField Replaceable Unit\n");
	if (--xlen <= 0)	/* point to flag	*/
		return;
	if (sense_data->flags & FPV) {
		printf("\tfield pointer:");
		printf("byte %x ", 
			CAT2(sense_data->field_hi, sense_data->field_lo));
		if (sense_data->flags & CDBIT)
			printf("in CDB field\n");
		else
			printf("in data field\n");
		if (sense_data->flags & BPV)
			printf ("\tbit pointer: %x\n", 
					(sense_data->flags & BIT_POINTER_MASK));
	}
}


static
opt_retry_cmd(cm)
CMD_BOX	*cm;
{
	cm->cmd_flags = 0;
	bzero(cm->error, SENSE_DATA_SIZE); /* clear error buffer */
	scsi_send_cmd(cm);
}

static
opt_err_retry(cm)
CMD_BOX	*cm;
{
	struct optiobuf	*dp;

	dp = GETDP(cm->unit_id);

	if(cm->err_code == 0)
		return(1);
	
	switch(cm->err_code){
	   case SCSIERR_ATN:
		if (dp->b_flags & B_NEWMDM) /* no retry when media changed */
			return(0);
		cm->cmd_type = CMD_RELEASE_ATTN;
		if (dp->b_flags & B_RESET) {
			dp->reset_q->next = cm;
			cm->next = NULL;
		} else 
			opt_retry_cmd(cm);
		return(1);
		break;
	   case SCSIERR_BUSTIMEOUT:
	   case SCSIERR_CMDTIMEOUT:
	   case SCSIERR_BUSRESET:
	   case SCSIERR_PARITY:
	   case SCSIERR_DISC:
		if (cm->retry_cnt-- == 0) 
			return(0);
		else {
			opt_retry_cmd(cm);
			return(1);
		}
		break;
	   default:
		return(0);
	}
}

static char *additional_sense[] = {
/* 0x00 */	"No Additional Sense Information",
/* 0x01 */	"No Index/Sector signal",
/* 0x02 */	"No Seek Complete",
/* 0x03 */	"Write Fault",
/* 0x04 */	"Drive Not Ready",
/* 0x05 */	"Drive Not Selected",
/* 0x06 */	"No Track Zero found",
/* 0x07 */	"Multiple Drives Selected",
/* 0x08 */	"Logical Unit Communication Failure",
/* 0x09 */	"Track Following error",
/* 0x0A */	"Reserved additional sense code",
/* 0x0B */	"Reserved additional sense code",
/* 0x0C */	"Reserved additional sense code",
/* 0x0D */	"Reserved additional sense code",
/* 0x0E */	"Reserved additional sense code",
/* 0x0F */	"Reserved additional sense code",
/* 0x10 */	"ID CRC or ECC error",
/* 0x11 */	"Unrecovered Read error of data blocks",
/* 0x12 */	"No Address Mark found in ID field",
/* 0x13 */	"No Address Mark found in Data field",
/* 0x14 */	"No record found",
/* 0x15 */	"Seek Positioning error",
/* 0x16 */	"Data Synchronization Mark error",
/* 0x17 */	"Recovered Read data with target's Read retries(not with ECC)",
/* 0x18 */	"Recovered Read data with target's ECC correction(not with retries",
/* 0x19 */	"Defect List error",
/* 0x1A */	"Parameter Overrun",
/* 0x1B */	"Synchronous Transfer error",
/* 0x1C */	"Primary Defect List not found",
/* 0x1D */	"Compare error",
/* 0x1E */	"Recovered ID with target's ECC correction",
/* 0x1F */	"Reserved additional sense code",
/* 0x20 */	"Invalid Command Operation Code",
/* 0x21 */	"Illegal Logical Block Address",
/* 0x22 */	"Illegal function for device type",
/* 0x23 */	"Reserved additional sense code",
/* 0x24 */	"Illegal field in CDB",
/* 0x25 */	"Invalid LUN",
/* 0x26 */	"Invalid field in Parameter List",
/* 0x27 */	"Write Protected",
/* 0x28 */	"Medium  Changed",
/* 0x29 */	"Power On or Reset or Bus Device Reset occured",
/* 0x2A */      "Mode Select Parameters changed",
/* 0x2B */	"Reserved additional sense code",
/* 0x2C */	"Reserved additional sense code",
/* 0x2D */	"Reserved additional sense code",
/* 0x2E */	"Reserved additional sense code",
/* 0x2F */	"Reserved additional sense code",
/* 0x30 */	"Incompatible Cartridge",
/* 0x31 */	"Medium Format corrupted",
/* 0x32 */	"No Defect Spare Location Available",
/* 0x33 */	"Reserved additional sense code",
/* 0x34 */	"Reserved additional sense code",
/* 0x35 */	"Reserved additional sense code",
/* 0x36 */	"Reserved additional sense code",
/* 0x37 */	"Reserved additional sense code",
/* 0x38 */	"Reserved additional sense code",
/* 0x39 */	"Reserved additional sense code",
/* 0x3A */	"Reserved additional sense code",
/* 0x3B */	"Reserved additional sense code",
/* 0x3C */	"Reserved additional sense code",
/* 0x3D */	"Reserved additional sense code",
/* 0x3E */	"Reserved additional sense code",
/* 0x3F */	"Reserved additional sense code",
/* 0x40 */	"RAM failure",
/* 0x41 */	"Data Path diagnostic failure",
/* 0x42 */	"Power On Diagnostic Failure",
/* 0x43 */	"Message Reject Error",
/* 0x44 */	"Internal Controller Error",
/* 0x45 */	"Select/Reselect failed",
/* 0x46 */	"Unsuccessful Soft Reset",
/* 0x47 */	"SCSI Interface Parity Error",
/* 0x48 */	"Initiator Detected Error",
/* 0x49 */	"Inappropriate/Illegal Message"
};

#define MIN_RESERVED	0x4A
#define MAX_RESERVED	0x7F
#define MIN_VDR_UNIQ	0x80
#define MAX_VDR_UNIQ	0xFF
static char *
get_add_sense_msg(err_code)
unchar err_code;
{
	if ((err_code >= MIN_RESERVED ) && (err_code < MAX_RESERVED))
		return("Reserved additional sense code");
	if ((err_code >= MIN_VDR_UNIQ) && (err_code <= MAX_VDR_UNIQ))
		return("Vendor Unique");
	return(additional_sense[err_code]);
}

opt_set_page_01(dp,cm)
struct optiobuf *dp;
CMD_BOX  *cm;
{
	dp->b_flags |= B_RESET;
	ASSERT(dp->reset_q == NULL);
	dp->reset_q = cm;	
	scsi_send_cmd(&dp->ms_cm);
}

opt_reset_done(cm)
CMD_BOX  *cm;
{
	struct optiobuf	*dp;
	CMD_BOX	*next_cm, *wait_cm;
	buf_t	*bp;

	dp = GETDP(cm->unit_id);
	if (cm->err_code != 0) {
		DISPLAY_DRV_ID(cm->unit_id);
		printf("error happened in reset sequence\n");
		for (wait_cm = dp->reset_q; wait_cm != NULL;) {
			bp = (buf_t *)wait_cm->tag;
			bp->b_error = EIO;
			bp->b_flags |= B_ERROR; 
			bp->b_resid = 0;
			optdeque(dp);
			iodone(bp);
			if ( wait_cm == dp->cm[0])
				dp->b_outcnt &= ~CMB0_IN_USE;
			else
				dp->b_outcnt &= ~CMB1_IN_USE;	
			wait_cm = wait_cm->next;
		}
		dp->reset_q = NULL;
		dp->b_flags &= ~B_RESET;
		optstart(dp);
		return;
	}	
	else {
		next_cm = dp->reset_q->next;
		dp->reset_q->next = NULL;
		scsi_send_cmd(dp->reset_q);
		dp->reset_q = NULL;
		dp->b_flags &= ~B_RESET;
		if (next_cm)
			scsi_send_cmd(next_cm);	
		else 
			optstart(dp);
	}
}

static 
optdeque(dp)
struct optiobuf	*dp;
{
	buf_t	*bp;

	if ((bp = dp->b_actf) == NULL)
		return;
	if ((dp->b_actf = bp->av_forw) == NULL) {
		dp->b_acts = NULL;
		dp->b_actl = NULL;
#ifdef C_SCAN
		dp->b_acte = NULL;
#endif /* C_SCAN */
	} else {
		if (dp->b_acts == bp)
			dp->b_acts = dp->b_actf;
#ifdef C_SCAN
		if (dp->b_acte == bp)
			dp->b_acte = dp->b_actf;
#endif /* C_SCAN */
	}
	dp->qcnt--;
	ASSERT(dp->qcnt != -1);
}


/*
 * get_vendor_uniq_add_msg - return any known vendor unique messages
 *
 * In particular required for Sony drives which use reserved codes to
 * announce simple things like no disk present as well as using standard
 * codes to represent vendor unique errors. Horrible.
 */
struct vdr_add_sns {
	unchar	code;
	char	*msg;
};

struct	vdr_add_sns sony_sns[] = {
	0x02, "No Command Complete",
	0x0a,	"No Disk",
	0x0b,	"Load/Unload Failure",
	0x0c,	"Spindle Failure",
	0x0d, "Focus Failure",
	0x0e, "Traceing Failure",
	0x0f, "Bias Magnet Failure",
	0x23, "Illegal Function of Medium Type",
	0x30, "Incompatible Cartridge",
	0x31,	"Medium Format Corrupted",
	0x38, "Recovered with Automatic Reallocation",
	0x39, "Automatic Reallocation Failure",
	0x3a, "Defect List Update Failure",
	0x3d, "Defect List Not Available",
	0x42, "Power On Diagnostic Failure",
	0x80, "Limited Laser Life",
	0x81, "Focus Coil Over-current Failure",
	0x82, "Tracking Coil Over-Current Failure",
	0x83, "Temperature Alarm",
	0x92, "Overwrite Attempted",
	0x93, "Blank Sector Detected during Read of Verify",
	0x94, "Written Sector Detected during Verify",
	0xff, NULL
};

static char	*
get_add_sns(sns_list, err_code)
struct	vdr_add_sns *sns_list;
unchar	err_code;
{
	for ( ; sns_list->msg != NULL; sns_list++)
		if (sns_list->code == err_code)
			return(sns_list->msg);
	return(NULL);
}

static char	*
get_vendor_uniq_add_msg(dp, err_code)
struct optiobuf	*dp;
unchar	err_code;
{
	if (dp->b_flags & B_SONY_MF)
		return(get_add_sns(sony_sns, err_code));
	return(NULL);
}
