/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) cdr_open.c: version 25.1 created on 11/27/91 at 14:41:56	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)cdr_open.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
#include "sys/buf.h"
#include "sys/errno.h"
#include "sys/user.h"
#include "sys/cmn_err.h"
#include "sys/debug.h"
#include "sys/param.h"
#include "sys/file.h"

#include "../head/cdb.h"
#include "../head/hlvl_macro.h"
#include "../head/scsi_cmdbx.h"
#include "cdr_ds.h"
#include "cdr_cdb.h"

#define START 	1
#define STOP	0

extern struct buf *geteblk() ;
extern void cdr_intr() ;
extern struct buf * cdr_pop_head() ;
extern struct buf * cdr_q_head() ;
extern struct cmbox *build_cmbox() ;
extern struct cmbox *scsi_get_cmd() ;
extern void cdr_send_cmd() ;
extern void cdrerr() ;
extern void cdr_drive_close() ;
extern void cdr_sttbl_close() ;
extern void build_mode_select() ;

int ds_init = NOTINIT ;	/* whether data structures initialized */

/*
 * array of statetbl :  one for each cdrom on this controller 
 *			describes current state; this changes each time
 *			a new cd is inserted into the drive
 */
struct statetbl cdr_state[MAXDRVS] ;  

/* buffer built by driver to send commands to scsi */
struct buf *drv_buf[MAXDRVS] ; 

/*
 * only one open allowed on device at a time
 */
static cdrlockopcl (stp)
register STTBL *stp ;
{
	uint oldpri ;

	oldpri = splhlv() ;
	while (stp->cd_flags & CD_OPNIN_CLSIN)
		sleep (stp, PZERO) ;
	stp->cd_flags |= CD_OPNIN_CLSIN ;
	splx (oldpri) ;	
}

static cdrunlockopcl (stp)
register STTBL *stp ;
{
	uint oldpri ;
	
	oldpri = splhlv() ;
	stp->cd_flags &= ~CD_OPNIN_CLSIN ;
	wakeup (stp) ;
	splx (oldpri) ;
}

static void
cdr_lock_buf(unit_id)
unchar unit_id;
{
	uint oldpri ;
	register buf_t *bp = drv_buf[unit_id];
	ASSERT(bp);
	while(bp->b_flags & B_BUSY){
		bp->b_flags |= B_WANTED;
		sleep(bp,PRIBIO);
	}
	ASSERT((bp->b_flags & B_BUSY) == 0);
	bp->b_flags |= B_BUSY;
}

static void
cdr_rel_buf(unit_id)
unchar unit_id;
{
	uint oldpri ;
	register buf_t *bp = drv_buf[unit_id];

	ASSERT(bp->b_flags & B_BUSY);
	bp->b_flags &= ~B_BUSY;
	if(bp->b_flags & B_WANTED){
		bp->b_flags &= ~B_WANTED;
		wakeup(bp);
	}
}

/* 
 * input : state table for this device
 * output : state of scsi cmd_box when entered this routine
 * side effect : if current state is not busy, set to busy
 */
static int
isbusy_cmbox(stp)
register STTBL *stp ;
{
	uint oldpri ;
	int ret ;
	
	oldpri = splhlv() ;
	if (stp->cd_busy == NOTBUSY) {
		stp->cd_busy = BUSY ;
		ret =  NOTBUSY ;
	}
	else
		ret = BUSY ;
	splx (oldpri) ;
	return ret ;
	
}

static void
release_cmbox(stp)
register STTBL *stp ;
{
	stp->cd_busy = NOTBUSY ;
}

static int
dostrncmp(str1, str2, len)
char *str1 ;
char *str2 ;
int len ;
{
	int index ;
	for (index = 0 ; index < len; index++) {
		if (str1[index] != str2[index]) 
			return 1 ;
	}
	return 0 ;
}

static void
get_set_vendor(inq, stp)
register struct inquiry *inq ;
register STTBL *stp ;
{
	if (!(dostrncmp (inq->i_vendor, "SONY", 4)))
		stp->cd_vendor = CD_SONY ;
	else if (!(dostrncmp (inq->i_vendor, "LMS", 3)))
		stp->cd_vendor = CD_LMS ;
}

/*
 * cdr_init()
 * initialize global data structures
 * input : none
 * output : none
 * side effects: initializes statetbl to zero
 */
cdr_init()
{
	(void)bzero((char *)cdr_state, sizeof(cdr_state)) ;
}

static int
cdr_do_open (stp, unit_id)
STTBL *stp ;
int unit_id ;
{
	stp->cd_dev = unit_id ;
	stp->cd_flags |= CD_ONCE ;

	if((stp->cd_cmbox = scsi_get_cmd(unit_id)) == NULL) {
		u.u_error = EBUSY ;
		return FAILURE ;
	}

	drv_buf[unit_id] = geteblk() ;
	drv_buf[unit_id]->b_dev = unit_id << SHFT4 ;
	drv_buf[unit_id]->b_proc = 0 ;
	/* clear busy flag */
	cdr_rel_buf (unit_id) ;
	stp->cd_start_bit = START ;
	stp->cd_retry_cnt = MAXRETRY ;
	stp->cd_rst_retry = MAXRETRY ;
	ASSERT (stp->cd_head == NULL) ;
	ASSERT (stp->cd_tail == NULL) ;

	if (!cdr_device_init(stp)) {
		u.u_error = EIO ;
		brelse (drv_buf[unit_id]) ;
		drv_buf[unit_id] = NULL ;
		cdr_sttbl_close (stp) ;
		return FAILURE ;
	}

	stp->cd_flags |= CD_OPEN ;
	return SUCCESS ;
}


/* 
 * cdr_open()
 *	check device 
 *	get current state of device
 *	update open count
 *		
 * input :	
 *	device
 *	mode
 * 
 * side effects:
 *		if error then u.u_error is set
 */
cdr_open(device, mode, oper)
register ushort device ;
uint mode ;
uint oper ;
{
	unchar unit_id = UNIT(device) ;
	STTBL *stp ;
	uint intpri ;

	if (unit_id >= MAXDRVS) {
		u.u_error = EINVAL ;
		return ;
	}
	stp = (STTBL *)GETSTBL(unit_id) ;
	ASSERT (stp) ;

	cdrlockopcl (stp) ;

	if (!(mode & FREAD)) {
		u.u_error = EINVAL ;
		cdrunlockopcl (stp) ;
		return ;
	}

	if (oper != OTYP_BLK && oper != OTYP_CHR && oper != OTYP_MNT) {
		u.u_error = EINVAL ;
		cdrunlockopcl (stp) ;
		return ;
	}

	if ((stp->cd_flags & CD_MEDIA_CH) || stp->cd_flags & CD_RESET_ERROR) {
		u.u_error = EIO ;
		cdrunlockopcl (stp) ;
		return ;
	}

	if (!ds_init) {
		intpri = splhlv() ;
		cdr_init() ;
		ds_init = ISINIT ;
		splx(intpri) ;
	}

	if (!(stp->cd_flags & CD_OPEN)) {
		if  ((cdr_do_open (stp, unit_id)) != FAILURE) {
			switch (oper) {
				case OTYP_BLK :
					stp->cd_openmap |= CDR_BLK_OTYP ;
					break ;
				case OTYP_CHR :
					stp->cd_openmap |= CDR_CHR_OTYP ;
					break ;
				case OTYP_MNT :
					stp->cd_openmap |= CDR_MNT_OTYP ;
					break ;
				default :
					ASSERT (0) ;
			}
		}
	}

	cdrunlockopcl (stp) ;
}

/* 
 *  cdr_device_init()
 *	initialize the cd_rom drive
 *	needed at each disk change
 *		
 * input : state table for this drive, with current cd in the drive	
 *		need to re_init drive with each new cd 
 *
 * output :
 *	status :  SUCCESS	 1
 *		  FAILURE	 0 
 * 
 * side effects:
 *		test unit ready
 *		start up drive
 *		read capacity
 */
static int
cdr_device_init (stp)
STTBL *stp ;
{
	register struct capacity *cap ;
	register struct inquiry *inq ;
	int cd_size ;
	unchar unit_id = stp->cd_dev ;

	if (!driver_cmd(unit_id, CDB_INQUIRY << SHFT24,sizeof(struct inquiry))) {
		printf ("cdr_device_init: inquiry returns error\n") ;
		return FAILURE ;
	}
	inq = (struct inquiry *)drv_buf[unit_id]->b_un.b_addr ;
	if (inq->i_dev_type != DEV_CDROM) {
		printf("cdr_device_init: inquiry finds incorrect device type\n") ;
		return FAILURE ;
	}
	get_set_vendor(inq, stp) ;

	if (!driver_cmd (unit_id, CDB_START_STOP << SHFT24, 0)) {
		printf ("cdr_device_init: start/stop returns error\n") ;
		return FAILURE ;
	}

	if (!driver_cmd (unit_id, CDB_TEST_UNIT_READY << SHFT24, 0)) {
		printf ("cdr_device_init: test unit ready returns error\n") ;
		return FAILURE ;
	}

	/* default block size is set to 1024 */
	if (!driver_cmd (unit_id, CDB_MODE_SELECT << SHFT24, 
					MODEHEAD_LEN + DSCPTR_LEN)) {
		printf ("cdr_device_init: mode_select returns error\n") ;
		return FAILURE ;
	}

	if (!driver_cmd (unit_id, CDB_READ_CAPACITY << SHFT24, 
					sizeof(struct capacity) )) {
		printf ("cdr_device_init: read_capacity returns error\n") ;
		return FAILURE ;
	}

	cap = (struct capacity *)drv_buf[unit_id]->b_un.b_addr ;
	cd_size = CAT4 (cap->lba_msb, cap->lba_mid0, cap->lba_mid1,
							cap->lba_lsb) ;	
	stp->cd_log_blk_size = CAT4 (cap->bl_msb, cap->bl_mid0, cap->bl_mid1,
					cap->bl_lsb) ;
	stp->cd_max_blks = cd_size + 1 ;


	return SUCCESS ;
}


/* 
 * driver_cmd ()
 *	builds buf structure for local commands 
 *	calls strategy routine with buf structure
 *		
 * input :	
 *	unit_id : device
 *	cmd :	scsi command
 *		first byte : scsi command code
 *		last 3 bytes : command parameters
 *	buf_size : size [in bytes] of transfer
 *		if size = zero, use default
 *	tony ????? default ?????
 *
 * output :
 *	status :  SUCCESS	 1
 *		  FAILURE	 0 
 * 
 * side effects:
 */
static int
driver_cmd (unit_id, cmd, buf_size )
unchar unit_id ;
uint cmd ;
uint buf_size ;
{
	register struct buf *fbp ;
	int status ;

	cdr_lock_buf (unit_id) ;

	fbp = drv_buf[unit_id] ;
	ASSERT (fbp) ;
	fbp->b_error = 0 ;
	fbp->b_bcount = buf_size ;
	fbp->b_cmd = cmd ;

	cdr_strategy(fbp) ;
	iowait (fbp) ;

	fbp->b_flags &= ~B_DONE ;

	if (fbp->b_flags & B_ERROR) {
		status =  FAILURE ;
	}
	else {
		status =  SUCCESS ;
 	}
	cdr_rel_buf (unit_id) ;
	return status ;
}


/* 
 * cdr_strategy ()
 *	called by iopm and cdrcommand routine
 *	calls: 
 *		cdr_put_q 
 *		cdr_send_cmd 
 *
 *		
 * input :	
 *	bp
 *
 * output : none
 * 
 * side effects:
 *
 */
cdr_strategy(bp)
register struct buf *bp ;
{
	uint oldpri ;
	unchar unit_id = (unchar)UNIT(bp->b_dev) ;
	STTBL *stp = (STTBL *)GETSTBL(unit_id) ;
	int blknotread ;

/* 
 * CD_MEDIA_CH = cdrom has been changed : have cleared out queue,
 *		 and am refusing all requests except close
 *		 until cdrom is closed and reopened
 */
	
	ASSERT (bp) ;
	bp->b_resid = 0 ;
	/* bp comes from iopm */
	if (bp != drv_buf[unit_id]) { 
		if ((stp->cd_flags & CD_MEDIA_CH) || stp->cd_flags & CD_RESET_ERROR) {
			cdrerr (bp, EIO) ;
			return ;
		}

		if (!(bp->b_flags & B_READ)) {
			cdrerr (bp, EROFS) ;
			return ;
		}

		if (((bp->b_bcount - bp->b_resid) & (NBPSCTR - 1)) != 0) {
			cdrerr (bp, EIO) ;
			return ;
		}	

		if (bp->b_blkno > stp->cd_max_blks) {
			cdrerr (bp, EIO) ;
			return ;
		}	

		if (bp->b_blkno == stp->cd_max_blks) {
			bp->b_resid += bp->b_bcount ;
			iodone (bp) ;
			return ;
		}

		if (((bp->b_bcount - bp->b_resid) / NBPSCTR) > (stp->cd_max_blks - bp->b_blkno)) {
			bp->b_resid += bp->b_bcount - ((stp->cd_max_blks - bp->b_blkno) * NBPSCTR) ;	

		}
	}
	ASSERT(bp->b_resid <= bp->b_bcount);

	oldpri = splhlv() ;
	/* put the bp on tail of run queue */
	cdr_put_q (bp) ;
	cdr_send_cmd(unit_id) ;
	splx (oldpri) ;	

}

/* 
 * cdr_put_q ()
 *	put bp on queue for this device
 *
 * called by strategy routine
 *		
 * input :	
 *	bp
 *
 * output :
 *	status :  SUCCESS	 1
 *		  FAILURE	 0 
 * 
 * side effects:
 *	puts bp on tail of queue
 */
cdr_put_q (bp)
struct buf *bp ;
{
	STTBL *stp ;
	unchar unit_id = UNIT (bp->b_dev) ;

	stp = (STTBL *)GETSTBL(unit_id) ;

	if (stp->cd_head == NULL) {
		ASSERT (stp->cd_tail == NULL) ;
		stp->cd_head = bp ;
		stp->cd_tail = bp ;
	}
	else {
		ASSERT (stp->cd_tail) ;
		stp->cd_tail->av_forw = bp ;
		stp->cd_tail = bp ;
 	}
	bp->av_forw = NULL ;
}

/* 
 * cdr_send_cmd ()
 *	get bp from head of run queue
 *	build command box
 *	send command box to llvl scsi driver
 *
 * called by strategy routine
 * called by intr routine
 *		
 * input :	
 *	unit_id
 *
 * output :
 * 
 * side effects: if cmbox available, sends cmbox to llvl scsi
 *
 */
void
cdr_send_cmd (unit_id)
unchar unit_id ;
{
	register struct buf *bp ;
	register struct cmbox *cdr_cmbox ;
	STTBL *stp ;

	stp = (STTBL *)GETSTBL(unit_id) ;

	/* check if cmbox is busy */
	if (isbusy_cmbox(stp) == NOTBUSY)
	{
		/* bp = cdr_get_q (unit_id) ; */
		bp = cdr_q_head (unit_id) ;
		if (bp != NULL ) {
			cdr_cmbox = build_cmbox (bp) ;
			scsi_send_cmd (cdr_cmbox) ;
		}
		else
			release_cmbox (stp) ;
	}
}

/* 
 *  cdr_pop_head()
 *		
 * input :	
 *	unit_id
 *
 * output :
 *	ptr to next buffer to process
 * 
 * side effects:
 */
struct buf *
cdr_pop_head(unit_id)
unchar unit_id ;
{
	struct buf *bp ;
	STTBL *stp ;
 	int oldpri ;

	oldpri = splhlv() ;
	stp = (STTBL *)GETSTBL(unit_id) ;

	if ((bp = stp->cd_head) == NULL) {
		ASSERT (stp->cd_tail == NULL) ;
		splx (oldpri) ;
		return NULL  ;
	}

	ASSERT (stp->cd_tail) ;
	if (bp == stp->cd_tail) 
		stp->cd_head = stp->cd_tail = NULL ;
	else
		stp->cd_head = bp->av_forw ;

	splx (oldpri) ;
	return bp ;
}

struct buf *
cdr_q_head(unit_id)
unchar unit_id ;
{
	struct buf *bp ;
	STTBL *stp ;

	stp = (STTBL *)GETSTBL(unit_id) ;

	if (stp->cd_head == NULL) {
		ASSERT (stp->cd_tail == NULL) ;
		return NULL  ;
	}
	else
		return stp->cd_head ;

}



/* 
 *  build_cmbox()
 *		
 * input :	
 *	bp
 *
 * output :
 *	command mailbox : ready to send to llvl scsi driver
 * 
 * side effects:
 */
static struct cmbox *
build_cmbox(bp)
struct buf *bp ;
{
	unchar unit_id ;
	STTBL *stp ;
	struct cmbox *cdr_cmbox ;

	unit_id = UNIT (bp->b_dev) ;
	stp = (STTBL *)GETSTBL(unit_id) ;

	cdr_cmbox =  stp->cd_cmbox ;
	ASSERT (cdr_cmbox) ;
	(void)bzero ((char *)cdr_cmbox, sizeof(* cdr_cmbox)) ;
	cdr_cmbox->next = NULL ;
	/* cdr_cmbox->tag =  (uint)bp ; */
	cdr_cmbox->retry_cnt = MAXRETRY ;	
	cdr_cmbox->unit_id = unit_id ;
	cdr_cmbox->cmd_type = CMD_REGULAR ;

	if (bp == drv_buf[unit_id]) {
		cdrlocalcmd (cdr_cmbox, stp) ;
	}
	else
		cdriopmcmd (cdr_cmbox, stp) ;

	if((cdr_cmbox->dma_buf_len = bp->b_bcount - bp->b_resid) != 0) {
		cdr_cmbox->dma_buffer = (unchar *)bp->b_un.b_addr ;
		cdr_cmbox->dma_proc = bp->b_proc ;
	}

	return cdr_cmbox ;
}

/* 
 *  cdriopmcmd ()
 *		
 * input :	
 *	pointer to command box
 *	pointer to state table for this device
 *
 * output :
 *	status :  SUCCESS	 1
 *		  FAILURE	 0 
 * 
 * side effects:
 */
cdriopmcmd (cdr_cmbox, stp)
struct cmbox *cdr_cmbox ;
STTBL *stp ;
{
	buf_t *bp ;
	int blocks ;

	struct g1_cdb *cdb = (struct g1_cdb *)cdr_cmbox->cmd_dsc_blk ;
	/* bp = (buf_t *)cdr_cmbox->tag ; */
	bp = stp->cd_head ;
	ASSERT (bp) ;
	cdr_cmbox->service = cdr_intr ;
	cdr_cmbox->dma_flags |= DMA_READ ;
	cdr_cmbox->cmd_dsc_len = G1SIZE ;
	cdr_cmbox->dma_buf_len = bp->b_bcount - bp->b_resid ;
	cdr_cmbox->timer = RDCP_TIMER ;

	/* build command descripter block for extended read */
	ASSERT (stp->cd_log_blk_size) ;
	blocks = (bp->b_bcount - bp->b_resid) / stp->cd_log_blk_size ;
	cdb->opcode = CDB_READ_EXTENDED ;
	ITOC4 (cdb->lbadr_msb, cdb->lbadr_2, cdb->lbadr_1,
		cdb->lbadr_lsb, bp->b_blkno) ;
	ITOC2 (cdb->xferlen_msb, cdb->xferlen_lsb, blocks);
}


/* 
 * cdrlocalcmd ()
 *		
 * input :	
 *	pointer to command box
 *	pointer to state table for this device
 *
 * output : ?
 *	status :  SUCCESS	 1
 *		  FAILURE	 0 
 * 
 * side effects:
 */
cdrlocalcmd (cdr_cmbox, stp)
struct cmbox *cdr_cmbox ;
STTBL *stp ;
{
	/* buf_t *cmb_bp = (buf_t *)cdr_cmbox->tag ; */
	buf_t *cmb_bp = stp->cd_head ;
	unchar cmd = cmb_bp->b_cmd >> SHFT24 ;

	ASSERT (cmb_bp) ;
	cdr_cmbox->service = cdr_intr ;
	cdr_cmbox->dma_proc = 0 ;

	switch (cmd & GRPMASK) {
		case GRP0:
			grp0_cmbox(cdr_cmbox, stp) ;
			break ;
		case GRP1:
			grp1_cmbox(cdr_cmbox, stp) ;
			break ;
		/*
			case GRP_AUDIO:
			grpaudio_cmbox(cdr_cmbox, stp) ;
			break ;
		 */
		default:
			printf ("cdrlocalcommand unknown command <%x>\n",
					cmd) ;
	}

}

/* 
 * grp0_cmbox ()
 *	fill in data for cmbox for  group 0 commands
 *		
 * input :	
 *	cmbox : data already filled : service, dma_proc, cmd, next, 
 *				tag, retry_cnt,	unit_id, cmd_type
 *	stp : this unit_id
 *
 *
 * output : none
 * 
 * side effects:
 *	complete cmbox : fields used is command dependent
 */
static
grp0_cmbox (cdr_cmbox, stp)
register struct cmbox *cdr_cmbox ;
STTBL *stp ;
{
	register struct g0_cdb *cdb = (struct g0_cdb *)cdr_cmbox->cmd_dsc_blk ;
	buf_t *bp = stp->cd_head ;
	ASSERT (bp) ;

	cdb->ctl = 0 ;
	cdr_cmbox->cmd_dsc_len = G0SIZE ;

	switch (cdb->opcode = ((bp->b_cmd) >> SHFT24))
	{
		case CDB_TEST_UNIT_READY :
					cdr_cmbox->timer = DEF_TIMER ;
					break ;
		case CDB_START_STOP :
					cdb->start_flags = stp->cd_start_bit ;
					cdr_cmbox->timer = DEF_TIMER ;
					break ;
		case CDB_INQUIRY:
					cdr_cmbox->dma_flags |= DMA_READ ;
					cdr_cmbox->timer = DEF_TIMER ;
					cdb->alloc_len = bp->b_bcount ;
					break ;

		case CDB_MODE_SELECT:
					cdr_cmbox->timer = DEF_TIMER ;
					cdb->alloc_len = bp->b_bcount ;
					build_mode_select (cdr_cmbox, stp) ;
					break ;
		default:
			printf("grp0_cmbox: bad opcode <%x>\n",
					cdb->opcode) ;

	}
}

static void
build_mode_select (cmbox, stp)
struct cmbox *cmbox ;
STTBL *stp ;
{
	register struct mode_head *md_head ;
	register struct blk_dscptr *blk_dsc ;
	register buf_t *bp ;

	/* bp = (buf_t *)cmbox->tag ; */
	bp = stp->cd_head ;
	ASSERT (bp) ;
	bzero ((char *)bp->b_un.b_addr, MODEHEAD_LEN + DSCPTR_LEN) ;
	md_head = (struct mode_head *)bp->b_un.b_addr ;
	blk_dsc = (struct blk_dscptr *)((uint)md_head + MODEHEAD_LEN) ;

	md_head->dscptr_len = 8 ;
	blk_dsc->blk_len = 4 ;		/* set block size to 1024 */
	
}

/* 
 *  ()
 *		
 * input :	
 *
 * output :
 *	status :  SUCCESS	 1
 *		  FAILURE	 0 
 * 
 * side effects:
 */
static
grp1_cmbox (cdr_cmbox, stp)
struct cmbox *cdr_cmbox ;
STTBL *stp ;
{
	register struct g1_cdb *cdb =(struct g1_cdb *)cdr_cmbox->cmd_dsc_blk ;
	/* buf_t *cmb_bp = (buf_t *)cdr_cmbox->tag ; */
	buf_t *cmb_bp = stp->cd_head ;
	ASSERT (cmb_bp) ;

	cdb->ctl = 0 ;
	cdr_cmbox->cmd_dsc_len = G1SIZE ;

	/* switch (cdb->opcode = (uint)cdr_cmbox->tag->b_cmd >> SHFT24) */
	switch (cdb->opcode = (uint)(cmb_bp->b_cmd >> SHFT24) )
	{
		case CDB_READ_CAPACITY :
					cdr_cmbox->dma_flags |= DMA_READ ;
					cdr_cmbox->timer = RDCP_TIMER ;
					break ;
		default:
			printf("grp1_cmbox: bad opcode <%x>\n",
					cdb->opcode) ;

	}
}

/* 
 *  ()
 *		
 * input :	
 *
 * output :
 *	status :  SUCCESS	 1
 *		  FAILURE	 0 
 * 
 * side effects:
 */
static
grpaudio_cmbox (cdr_cmbox, stp)
struct cmbox *cdr_cmbox ;
STTBL *stp ;
{
	printf ("audio commands not supported\n") ;
}
/*	cdr_cmbox->dma_proc = 
	cdr_cmbox->cmd_dsc_len
	cdr_cmbox->cmd_dsc_blk
	cdr_cmbox->dma_flags
	cdr_cmbox->dma_buf_len
	cdr_cmbox->dma_buffer 
	cdr_cmbox->timer 
*/

/* 
 *  cdr_close()
 *		
 * input : minor number
 *
 * output : none
 * 
 * side effects:
 *	if (last close)
 *		mark device closed
 *		release scsi command mailbox
 *		spin down drive
 *	else
 *		mark this_type_open closed in state table[this unit id]
 */
cdr_close (device, flag, oper)
int device ;
uint flag ;
uint oper ;
{
	unchar unit_id = UNIT(device) ;
	STTBL *stp ;

	stp = (STTBL *)GETSTBL(unit_id) ;
	ASSERT (stp) ;

	cdrlockopcl (stp) ; 
	switch (oper) {
		case OTYP_BLK :
			stp->cd_openmap &= ~CDR_BLK_OTYP ;
			break ;
		case OTYP_CHR :
			stp->cd_openmap &= ~CDR_CHR_OTYP ;
			break ;
		case OTYP_MNT :
			stp->cd_openmap &= ~CDR_MNT_OTYP ;
			break ;
		default :
			cdrunlockopcl (stp) ;
			return ;
	}

	if (!(stp->cd_openmap &  CDR_ANYOPEN)) {
	/* side effect of cdr_drive_close is that the run queue
	 * is guaranteed empty; must have some driver_cmd to keep
	 * this necessary side effect
	 */
		cdr_drive_close (stp, unit_id) ;
		/* clean state table for this cdrom */
		cdr_sttbl_close (stp) ;
		brelse (drv_buf[unit_id]) ;
		drv_buf[unit_id] = NULL ;
	}
	cdrunlockopcl (stp) ; 
}

static void
cdr_drive_close (stp, unit_id)
STTBL *stp ;
unchar unit_id ;
{
	stp->cd_start_bit = STOP ;
	driver_cmd (unit_id, CDB_START_STOP << SHFT24, 0) ;
}

static void
cdr_sttbl_close (stp)
STTBL *stp ;
{
	stp->cd_flags &= ~(CD_OPEN | CD_MEDIA_CH | CD_RESET_ERROR) ;
	stp->cd_max_blks = 0 ;
	stp->cd_log_blk_size = 0 ;
	scsi_put_cmd (stp->cd_cmbox) ;
	stp->cd_cmbox = NULL ;
}

static void
cdrerr(bp, error)
struct buf *bp ;
unchar error ;
{
	bp->b_flags |= B_ERROR ;
	bp->b_error = error ;
	iodone (bp) ;
}

