/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) flp.c: version 25.1 created on 11/27/91 at 14:35:09	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)flp.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/

/*
 * file: flp.c
 *		File offers the user interface to the floppy driver,
 *		this includes the floppy and floppy test menus, as
 *		well as rd_flp_dev subroutine.
 *		
 *		This file makes calls to routines in flp_exec.c only.
 *
 *  890819  Redo floppy driver 
 *
 *	flp.c		--  User interface
 *	flp_exec.c	--  Floppy command start
 *	flp_result.c	--  Process floppy command
 *	flp_check.c	--  Analyze floppy response
 *	flp_test.c	--  Floppy tests -- from i_flop_test menu
 *	head/flp.h	--  General defines used by all
 *	head/flp_chk.h  --  Floppy result checking include
 */

#include "types.h"
#include "global.h"
#include "menu.h"
#include "flp.h"

#define BUFPOINT	0x0103e000

struct	flp_cmd	 flp_cmd;
union	flp_res  flp_res;

/*
	Global variables used by the floppy driver.  
	Most are used only in the routines they affect,
	They contain the status of the floppy drive.
*/
uchar	*flp_buf;			/* address of the buffer used 
					   by read and write */

uchar	spinning;			/* motor status, checked by mon/moff
					   set/reset by flp_motor_on and
					   flp_motor_off respectively	*/
uchar	current_cyl;			/* accurate cylinder, initialized to
					   0xFF on powerup, checked by seek,
					   set by result phase of recal or
					   seek only, set to 0xFF when 
					   ready change detected on floppy */
uchar	flp_status;			/* Status returned by a sense drive
					   status command		*/

uint	xfer_size;			/* Size of xfer to be performed by the
					   assembly code in flp_xfer.s	*/

extern	uint	flp_int;		/* flag used to indicate a floppy
					   interrupt has occured.	*/

extern	uint	flp_err;		/* flag used to indicate a floppy
					   error has occured during result
					   phase.			*/

unsigned int flp_errors[4];		/* error array */

extern migrate();

int	fcmdinit_req();
int	mon_req();
int	moff_req();
int	recal_req();
int	fcmdinit();
int	status ();
int	seek_req();
int	format_track_req();
int	format_disk_req();
int	block_req();
int	sector_req();
int	floppy_req();
int	rdtrack_req();
int	rd_flp_dev();
void	disp_cc();
void	floppy_init();

/*
	Floppy commands found in flp_exec.c
*/
extern	void 	mon();
extern	void	moff();
extern	int	seek();
extern	int	recal();
extern	int	rdsec();
extern	int	rdtrack();
extern	int	wrsec();
extern	int	format_track();
extern  int	flp_dev_init();
extern  int	specify();

/*
	Test routines found in flp_test.c and accessed through
	the i_flop_test menu below
*/
extern	int	lseek();
extern	int	lrds();
extern	int	lwrs();
extern	int	lwrrd();
extern	int	rdall();

struct init_menu i_diagnostics [] = {

	"ft(ests)", "",
	"Floppy test",'m',0,0,(PFI)i_flop_test, (char *)0,

	"if(loppy)", "",
	"Initialize floppy and driver", 'c',0,0,fcmdinit_req, (char *)0,

	"stat(us)", "<arg>",
	"Report or clear error status",'c',0,1,status, (char *)0,

	"mon", "",
	"Motor on", 'c',0,0,(PFI)mon_req, (char *)0,

	"moff", "",
	"Motor off", 'c',0,0,(PFI)moff_req, (char *)0,

	"seek", "cylinder",
	"Seek to cylinder", 'c',1,1,seek_req, (char *)0,

	"recal(ibrate)", "",
	"Recalibrate floppy heads", 'c',0,0,recal_req, (char *)0,

	"fmtd(isk)", "",
	"Format a floppy disk", 'c',0,0,format_disk_req, (char *)0,

	"fmtt(rack)", "cyl head",
	"Format track",'c',2,2,format_track_req, (char *)0,

	"rdblk", "blk blk_cnt <buff>",
	"Read floppy block to buff",'c',2,3,block_req, (char *)0,

	"rdsec", "cyl head sect sect_cnt <buff>",
	"Read sect_cnt sectors into memory",'c',4,5,sector_req, (char *)0,

	"rdtrack", "cyl head <buff>",
	"Read track into buffer",'c',2,3,rdtrack_req, (char *)0,

	"wrsec", "cyl head sect sect_cnt <buff>",
	"Write sect_cnt sectors to disk",'c',4,5,sector_req, (char *)0,

	"wrblk", "blk blk_cnt <buff>",
	"Write floppy block to disk",'c',2,3,block_req, (char *)0,

	"cc"," ",
	"Display current cylinder",'c',0,0,(PFI)disp_cc, (char *)0,

	"", "", "", '\0', 0, 0, 0, (char *)0,
};

/*
	Floppy Test menu
*/
struct init_menu i_flop_test [] = {

	"lseek", "cyl1 cyl2", 
	"Loop seeks between two cylinders",'c',2,2,lseek, (char *)0,

	"lwrs(ector)", "cyl head sect sect_cnt <buff>",
	"Loop write a sector",'c',4,5,lwrs, (char *)0,

	"lrds(ector)", "cyl head sect sect_cnt <buff>",
	"Loop read a sector",'c',4,5,lrds, (char *)0,

	"lwrcmp", "cyl head sect sect_cnt buff_wr buff_rd",
	"Loop Write, read, compare", 'c',6,6,lwrrd, (char *)0,

	"rdall", "buff_wr buff_rd",
	"Loop write, read, compare entire disk",'c',2,2,rdall, (char *)0,

	"", "", "", '\0', 0, 0, 0, (char *)0,
};

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int 
fcmdinit_req()
{
	
	printf ("Initialize floppy drive controller and disk.\n");
	fcmdinit();
	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int 
mon_req()
{
	
	printf ("Enable floppy motor.\n");
	mon();
	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int 
moff_req()
{

	printf ("Disable floppy motor.\n");
	moff();
	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int 
recal_req()
{
	
	printf ("Recalibrate floppy disk, set to track 0.\n");
	recal();
	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int 
fcmdinit()	/* initialize the floppy chip and s/w */
{
	int	ret_code;

	current_cyl = (uchar)NEED_RECAL;	/* Initialize to an out of range
						   value, this should be set 
						   properly by recal below */
	flp_int = 0;
	flp_err = 0;
	spinning = 0;

	flp_dev_init ();

	if (ret_code = specify ())
		return(ret_code);

	if (ret_code = recal ())
		return(ret_code);

	WRERR = RDERR = SKERR = CPERR = 0;
	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
status(comm_str,arg_cnt)
char *comm_str;
int arg_cnt;
{
	if(arg_cnt) 
		WRERR = RDERR = SKERR = CPERR = 0;
	else{
		printf("\nRead errors %d  Seek errors %d\n",RDERR,SKERR);
		printf("Write errors %d Compare errors %d\n",WRERR,CPERR);
	}
	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
format_track_req(comm_str, arg_cnt)
char	*comm_str;
int	arg_cnt;
{
	uchar	cyl;
	uchar	head;
	int	ret_code;

	cyl = (uchar)atox(comm_args[1]);
	head = (uchar)atox(comm_args[2]);

	if (cyl > MAX_CYL_NUM) {
		printf ("\nInvalid cylinder (%d); Max cylinder (%d).\n",
			cyl, MAX_CYL_NUM);
		return(INVL_CMD);
	}

	if (head > (uchar)1) {
		printf ("\nInvalid head (%d); Max head (1).\n", head);
		return(INVL_CMD);
	}

	mon();				/* turn on motor if off */

#ifdef	DEBUG
	printf ("Format_track_req:  cyl:  %d; head:  %d.\n", cyl, head);
#endif

	if (ret_code = format_track(cyl, head)) {
		printf ("Failed\n");
		return(ret_code);
	}

	printf ("Completed\n");
	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
format_disk_req()
{

	uchar cyl; 
	uchar head; 
	int ret_code;

	mon();				/* turn on motor if off */

	printf ("\nFormat Floppy Disk\n");

#ifdef	DEBUG
	printf ("format_disk_req:  begin format.\n");
#endif

	for (cyl = 0; cyl != NUMBER_CYLINDERS; cyl++) {

		if(ret_code = format_track(cyl, (head = 0))) {
			printf ("Failed\n");
			return(ret_code);
		}
		printf ("\r");

		if(ret_code = format_track(cyl, (head = 1))) {
			printf ("Failed\n");
			return(ret_code);
		}
		printf ("\r");

		if(con_in() != -1)	/* Check if key hit */
			break;
	}

	if (cyl == NUMBER_CYLINDERS)
		printf("\nCompleted.\n");
	else
		printf("\nTerminated.\n");

	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
sector_req(comm_str,arg_cnt)
char *comm_str;
int arg_cnt;
{
	uchar cyl;
	uchar head;
	uchar sect;
	uint sect_cnt;
	int ret_code;

	cyl = (uchar)atox(comm_args[1]);
	head = (uchar)atox(comm_args[2]);
	sect= (uchar)atox(comm_args[3]);
	sect_cnt = (uint)atox(comm_args[4]);

	if (arg_cnt == 5)
		flp_buf = (uchar *)atox(comm_args[5]);
	else
		flp_buf = (uchar *)BUFPOINT;

	printf ("%s:  cyl  %d; head  %d; sct  %d; cnt  %d; buff  %X.\n",
			comm_args[0], cyl, head, sect, sect_cnt, flp_buf);

	if (comm_str[0] == 'r') {
		if (ret_code = floppy_req (cyl, head, sect, sect_cnt, FLP_READ))
			return(ret_code);
	} 
	else {
		if(ret_code = floppy_req (cyl, head, sect, sect_cnt, FLP_WRITE))
			return(ret_code);
	}

	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
block_req(comm_str,arg_cnt)
char *comm_str;
int arg_cnt;
{
	uint block;
	uint block_cnt;
	uchar cyl;
	uchar head;
	uchar sect;
	uint sect_cnt;
	int ret_code;

	block = (uint)atox(comm_args[1]);
	block_cnt = (uint)atox(comm_args[2]);

	if (arg_cnt == 3)
		flp_buf = (uchar *)atox(comm_args[3]);
	else
		flp_buf = (uchar *)BUFPOINT;

	sect_cnt = block_cnt * 2;
	block *=2;			/* 512 byte to 1k byte sectors. */
	sect = block % EOT;		
	sect++;				/* sectors from 1-15, not 0-14. */
	block = block/EOT;		/* Get head and cyl information */
	head = block & 0x01;		
	cyl = block >> 1;		/* shift away the head number. */

	printf ("%s:  block:  %d; block_cnt:  %d.\n", 
			comm_args[0], block, block_cnt);
	printf ("%s:  cyl  %d; head  %d; sct  %d; cnt  2; buff  %X.\n",
			comm_args[0], cyl, head, sect, flp_buf);

	if (comm_str[0] == 'r') {
		if (ret_code = floppy_req (cyl, head, sect, sect_cnt, FLP_READ))
			return(ret_code);
	}
	else {
		if(ret_code = floppy_req (cyl, head, sect, sect_cnt, FLP_WRITE))
			return(ret_code);
	}
	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
rd_flp_dev(blk, buf)	
uint blk; 		/* block number to read. */
uchar *buf;		/* buffer to read sector into .*/
{
	uchar	cyl;
	uchar   head;
	uchar	sect;
	uint	sect_cnt;
	int 	ret_code;

	flp_buf = buf;

	blk *=2;			/* 512 byte to 1k byte sectors. */
	sect_cnt = 2;
	sect = blk % EOT;		
	sect++;				/* sectors from 1-15, not 0-14. */
	blk = blk/EOT;			/* Get head and cyl information */
	head = blk & 0x01;		
	cyl = blk >> 1;			/* shift away the head number. */

#ifdef	DEBUG
	printf ("rd_flp_dev:  block:  %d;  buffer:  %X.\n", blk, flp_buf);
	printf ("rd_flp_dev:  cyl:  %d; head:  %d; sct:  %d; cnt:  %d.\n",
			cyl, head, sect, sect_cnt);
#endif
	if (ret_code = floppy_req (cyl, head, sect, sect_cnt, FLP_READ))
		return(ret_code);

	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
floppy_req (cyl, head, sect, sect_cnt, rd_or_wr)
uchar	cyl;
uchar	head;
uchar	sect;
uint	sect_cnt;
uchar	rd_or_wr;
{
	
	int	ret_code;
	uint	start_sect;

	if (cyl > (uchar)MAX_CYL_NUM) {
		printf ("\nInvalid cylinder (%d); Max cylinder (%d).\n",
			cyl, MAX_CYL_NUM);
		return(INVL_CMD);
	}

	if (head > (uchar)1) {
		printf ("\nInvalid head (%d); Max head (1).\n", head);
		return(INVL_CMD);
	}

	if (sect > (uchar)EOT) {
		printf ("\nInvalid sector (%d); Max sector (%d).\n",
			sect, EOT);
		return(INVL_CMD);
	}

	start_sect = (uint)((cyl * SECT_CYL) + (head * SECT_HEAD) + sect);

	if ((start_sect + sect_cnt) > (TOTAL_SECT + 1)) {
		printf ("\n Starting sector and count will overrun floppy.\n");
		return(INVL_CMD);  
	}

	mon(); 

#ifdef	DEBUG
	printf ("floppy_req:  rd_or_wr:  %d.\n", rd_or_wr);
	printf ("floppy_req:  cyl:  %d; head:  %d; sct:  %d; cnt:  %d.\n",
			cyl, head, sect, sect_cnt);
#endif
	while (sect_cnt) {

		if (rd_or_wr == FLP_READ) {	/* Reading (command = rdsec) */
			if (ret_code = rdsec(cyl, head, sect))
				return(ret_code);
		}
		else {
			if (ret_code = wrsec(cyl, head, sect))
				return(ret_code);
		}

		if (--sect_cnt) {
			flp_buf += SECT_SZ;
			if (sect == (uchar)EOT) {
				sect = 1;
				if(head == (uchar)1) {
					head = 0;
					cyl++;
				}	
				else {
					head = 1;
				}
			}
			else {
				sect++;
			}
		}
	}
	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
seek_req(comm_str,arg_cnt)
char *comm_str;
int arg_cnt;
{
	uchar	cyl;

	cyl = (uchar)atox(comm_args[1]);	/* get cylinder num */

	if (cyl > MAX_CYL_NUM) {
		printf ("\nInvalid cylinder (%d); Max cylinder (%d).\n",
			cyl, MAX_CYL_NUM);
		return(INVL_CMD);
	}

	mon(); 

	printf ("%s:  requested cyl:  %d.\n", comm_args[0], cyl);

	if (seek(cyl) != cyl)
		return(INVL_CMD);

	return(0);
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
int
rdtrack_req(comm_str,arg_cnt)
char *comm_str;
int arg_cnt;
{
	uchar	cyl;
	uchar	head;
	int	ret_code;

	cyl = (uchar)atox(comm_args[1]);
	head = (uchar)atox(comm_args[2]);

	if (arg_cnt == 3)
		flp_buf	= (uchar *)atox(comm_args[3]);
	else
		flp_buf = (uchar *)BUFPOINT;

	if (cyl > MAX_CYL_NUM) {
		printf ("\nInvalid cylinder (%d); Max cylinder (%d).\n",
			cyl, MAX_CYL_NUM);
		return(INVL_CMD);
	}

	if (head > (uchar)0x01) {
		printf ("\nInvalid head (%d); Max head (1).\n", head);
		return(INVL_CMD);
	}

	mon();

	printf ("%s:  Requested track  cyl:  %d; head:  %d.\n", cyl, head);

	if (ret_code = rdtrack(cyl, head))
		return(ret_code);

	return(0);
}

void
disp_cc ()
{
	printf ("Current Cylinder is %d.\n", current_cyl);
	return;
}

/*------------------------------------------------------------------------------
------------------------------------------------------------------------------*/
void
floppy_init()
{
/*
**	This is called only after a hard reset is done to the SPM
**	This routine will initialize the floppy controller chip to
**	be in PC AT mode.
*/
	
	*OPS = (uchar)CNTRLwd;
	return;
}
