/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) boot.c: version 25.1 created on 11/27/91 at 14:33:44	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)boot.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/

/*------------------------------------------------------------------------------
	boot.c: This program will handle the current simple boot.
------------------------------------------------------------------------------*/
#include "types.h"
#include "icb.h"
#include "iom.h"
#include "spm.h"
#include "globl.h"
#include "global.h"
#include "devcmd.h"
#include "vreg.h"
#include "icb_config.h"
#include "rtb.h"
#include "misc.h"
#include "disp.h"
#include "sa_dir.h"
#include "novram.h"
#include "filehdr.h"
#include "scnhdr.h"
#include "aouthdr.h"
#include "a.out.h"
#include "dev.h"
#include "iopm_comm.h"
#include "boot.h"
#include "iopm_err.h"
#include "sdk_disk.h"

#define FD_INT_VECT	0x1000078
#define BUFPOINT	0x0103e000
#define	LINES_PER_SCREEN	16
#define	DISK_RETRY      	15
#define	TAPE_RETRY      	45
#define	IOPM_RETRY      	5
#define	NO_ENTRY_FOUND      	0xFF

struct fhdr {
	FILHDR filhdr;
	AOUTHDR saouthdr;
	SCNHDR sctnhdr[6];
};

struct flp_boot {
	int	st_block;
	int	st_off;
	int	blocks;
	int	addr;
	} flp_blk[6];

extern unsigned char 		*buff;		/* download buffer */
extern char 			diskflg;	/* if disk reads,set this. */
char 				booting;
char 				lastblk;	/* set this, if on last blk */

extern	uchar	css_slot;
extern	uchar	sub_slot;
extern	uchar	phys_dev;
extern	uchar	log_dev;
extern	char	filename[MAX_DIR_NAME];
extern	uint	boot_type;
extern	struct	icb_config	*icb_ptr;

int	curr_blk;

void	bootme ();
void	print_entry ();
void	print_entries ();
uint	find_entry ();
void	boot ();
void	boot_edt ();
void	boot_iopm ();
void	boot_floppy ();
uchar	*mov_flp();

extern void get_device_info ();
extern int rd_iopm_dev ();

void
bootme(comm_str, arg_cnt)
char *comm_str;
int arg_cnt;
{
	register struct novram *ptr = (struct novram *)NOVRAM;

	ptr->diag_fly_by = 1;

	if (*(comm_str) == 'b')
		ptr->autoboot = 1;
	else
		ptr->autoboot = 0;

	softreset();
}

void
boot (bootstr, print_only, save_path)
char	*bootstr;
uint	print_only;
char	save_path;
{
}


void
boot_edt(print_only) /* edt (smd or scsi) boot */
uint print_only;
{
	struct		dtreq_call 	dtreq_struct;
	struct		dtreq_call 	*dtreq_ptr = &dtreq_struct;
	struct		sa_dir 		*dir= (struct sa_dir *)ICB_RD_ADDR; 
	uchar 		mem_slot;
	unsigned	i=0, i1;
	uint		entry;
	unsigned 	block, memaddr = 0;
	unsigned 	last_block_read, stat;

	curr_blk = -1;		/* show tape device position unknown */

/*******	get mem_slot - code passed to MM then loaded to SPM *******/
	if((mem_slot = (unsigned char)findmem(1)) == 18) {
		printf("Can't ICB boot without Memory Board.\n");
		return;
	}

/*******	set flags, and maps for using IOM and MM (ie CSS bus) *******/
	diskflg = booting = 1;
	sp_int_set();  /* set SPM to receive int itself */
	cssmap(MAP01,mem_slot,(char)0x00); /* setup mem slot*/
	cssmap(MAP00,css_slot,(char)0x00); /* setup io slot*/
	
/*******	set block to the directory block number    ********/
	if (boot_type == DV_RV)
		block = DIR_BLK_NUM;	/* set for hard disk boots. */
	else
		block = 0;		/* set for tape boots. */

/*******	Setup the dtreq_call structure		********/
	dtreq_ptr->icbp = icb_ptr;
	dtreq_ptr->mem_slot = mem_slot;
	dtreq_ptr->drive = phys_dev;
	dtreq_ptr->memaddr = memaddr;
	dtreq_ptr->block = block;
	dtreq_ptr->count = BLK_SIZE;

/*******	read in directory block set pointer check magic     *******/
	do {
		if (ifesc())  /* to escape between directories */
			return;

		if (read_edt_dir(dtreq_ptr))
			return;

		if (boot_type == DV_9T)
			for (i = 0; i != 0x80000; i++); /* delay for tape */

		if(dir->magic_num != 0x5341) {		
			printf("Invalid magic number\n"); 
			return;
		}

/*******	dirme only prints........this is it      ********/
		if (print_only)  /* display the directory; return */
			print_entries (dir);
		else
			entry = find_entry (dir, filename);

		dtreq_ptr->block = block + dir->next_dir_block;

	} while (dir->next_dir_block && 
				(print_only || (entry == NO_ENTRY_FOUND)));

	if (print_only || (entry == NO_ENTRY_FOUND)) {
		if (boot_type != DV_RV) {
			printf("\nRewinding Tape....");
			dtreq_ptr->command = REWIND;
			dtreq_ptr->type = DTTAPE;
			dtreq(dtreq_ptr);
			printf("\n");
		}
		return;
	}

	block = block + dir->entry[entry].start_block_num; /* starting block */
	last_block_read = block + ((dir->entry[entry].byte_count - 1) / dir->block_size);

/*******	if a tape.......skip to first block with dummy reads *******/
	if ((boot_type == DV_MT) || (boot_type == DV_9T))
		seek_tape(dtreq_ptr, block);

/*******	read in the complete file, headers and all, to the MM  ******/
	if (boot_type == DV_RV) {
		dtreq_ptr->command = PLREAD;
		dtreq_ptr->type = DTDISK;
		for(; block <= last_block_read; block++) {	/* read all blocks.. */
			if((i1++ % 10) == 0)
				printf("."); 	/* show me right now.. */
			
			dtreq_ptr->memaddr = memaddr;
			dtreq_ptr->block = block;
			if (stat=dtreq(dtreq_ptr)) {
				printf("\nDisk read failed, stat:  %4X\n",stat);
				return;
			}
			memaddr += BLK_SIZE;
		}
	}
	else {
		dtreq_ptr->command = TPREAD;
		dtreq_ptr->type = DTTAPE;
		for(; block <= last_block_read; block++) {	/* read all blocks.. */
			if((i1++ % 10) == 0) 
				printf("."); 	/* show me right now.. */
			
			if (last_block_read == block)
				lastblk = 1;
			dtreq_ptr->memaddr = memaddr;
			dtreq_ptr->block = block;
			if (stat=dtreq(dtreq_ptr)) {
				printf("\nTape read failed, stat:  %4X\n",stat);
				return;
			}
			curr_blk++;
			lastblk = 0;
			memaddr += BLK_SIZE;
		}
		block = 0;		/* rewind tape after tape read done */
		printf("\nRewinding Tape....");
		dtreq_ptr->command = REWIND;
		dtreq_ptr->type = DTTAPE;
		dtreq(dtreq_ptr);
	}

/*******	reset some flags, move code, and execute it  ********/
	diskflg = booting=0;	/* show not in bootup mode now. */

	for (i = 0; i != 0x200000; i++);

#ifdef	SPM_PROM
	mov_code();	/* move the code(tape/disk), and jump to it */
#endif

#ifdef	SPM_IMAGE
	vbrmv(0);	/* move vector base register to point at PROM....0 */
	trap1();	/* trap SPM PROM to move the executable */
#endif

	return;		/* should never get here...... */
}

void
boot_iopm(print_only) /* boot from iopm. */
uint print_only;
{
	struct		sa_dir 		*dir = (struct sa_dir *)IOPM_RD_ADDR; 
	uint  		entry;
	unsigned  int	block;
	unsigned  int	block_sav;
	unsigned  int	memaddr = 0;
	unsigned  int 	blk_cnt = 1;		/* directory size */
	unsigned  int	stat;
	unsigned  int	count;
	unsigned  int	retry;

/*	
	Currently the iopm boot looks a lot like the edt boot, in the 
	future it will be made to look like the floppy boot.
*/

/*******	set flags, and maps for using IOM and MM (ie CSS bus) *******/
	diskflg = booting = 1;
	cssmap(MAP02,css_slot,(char)0x0F); /* setup io slot*/

/*******	setup block to be directory block     *********/
	switch (boot_type) {
		case DV_IOPM_DK:	/* 32 slice IOPM disk */
			if ((block = get_iopm_blk_num(LD_BOOT_SLICE)) == 0xFFFFFFFF)
				return;
			break;
		case DV_IOPM_RV:	/* 16 slice 'A1000' disk */
			block = DIR_BLK_NUM;
			break;
		default:		/* Tape device */
			block = 0;
			break;
	}
	block_sav = block;

/*******	read directory....and setup pointer   ********/
	do {

		for (retry = 0; retry != IOPM_RETRY; retry++)	{ /* 5 times */
			if (stat = rd_iopm_dev (block, blk_cnt)) {
				for(count = 0; count < 180000; count++) ;
				if(ifesc())
					return;  /* exit */
			}
			else
				break;

			if (stat == UNIT_INIT_FAIL) {
				printf ("Device non-existent or defective.\n");
				return;
			}
		}

		if(retry == IOPM_RETRY) {
			printf("IOPM read failed, stat:  %X\n", stat);
			return;
		}
	
	
/*******	Check directory, and scan for filename  *******/
		if(dir->magic_num != 0x5341) {		
			printf("Invalid magic number\n"); 
			printf("Was the proper device type selected?\n");
			check_iopm_dev_type (block, blk_cnt, boot_type);
			return;
		}
	
/*******	dirme only prints the directory........so here it is *******/
		if (print_only)  /* display the directory */
			print_entries (dir);
		else
			entry = find_entry (dir, filename);
		
		block = block_sav + dir->next_dir_block;

	} while (dir->next_dir_block && (print_only || (entry == NO_ENTRY_FOUND)));

	if (print_only || (entry == NO_ENTRY_FOUND)) {
		return;
	}

/*******	offset relative to directory block...so set it up right ******/
	block = block + dir->entry[entry].start_block_num; /* starting block */
	blk_cnt = ((dir->entry[entry].byte_count + BLK_SIZE - 1) / BLK_SIZE);

/*******	read the file to the MM.....headers and all  ********/
	for (retry = 0; retry != IOPM_RETRY; retry++)	{ /* 30 times.. */
		if (stat = rd_iopm_dev (block, blk_cnt)) {
			for(count = 0; count < 180000; count++) ;
			if(ifesc())
				return;  /* exit */
		}
		else
			break;
	}
	if(retry == IOPM_RETRY) {
		printf("IOPM boot read failed, stat:  %X\n", stat);
		return;
	}

/*******	setup some flags and move code then execute it  *******/
	diskflg = booting=0;	/* show not in bootup mode now. */

#ifdef	SPM_PROM
	mov_iopm_code();	/* move the code(tape/disk), and jump to it */
#endif

#ifdef	SPM_IMAGE
	vbrmv(0);		/* move vbr to point at PROM */
	trap2();		/* trap the SPM PROM to move the executable */
#endif

	return;			/* should never get here */
}

#ifdef	SPM_PROM
void
boot_floppy (print_only) 	/* boot floppy */
uint print_only;
{
	struct		sa_dir 		*dir = (struct sa_dir *)BUFPOINT;
	unsigned 	i=0;
	uint		entry;
	int		num_sect;
	int		scn;
	int		blk;
	uchar		*rdbuff;
	unsigned 	block = 0;		/* directory starts at 0 */
	unsigned 	last_block_read, stat;
	struct fhdr 	*fhptr = (struct fhdr *)BUFPOINT;
	struct scnhdr 	*scn_ptr;
	unsigned 	*dest, *src, *limit, final, *startp, save, save1;

/*******	initialize floppy device **********/

	if (fcmdinit ())
		return;

/*******	read the sa directory    **********/
	do {
		if (stat = rd_flp_dev(block, buff)) { /* Read block in buffer */
			printf("Floppy directory read failed, stat:  %x\n", stat);
			moff();
			return;
		}

/*******	check directory, find file ********/
		if(dir->magic_num != 0x5341) {		
			printf("Invalid magic number in directory\n"); /* tell user. */
			moff();
			return;
		}

/*******	dirme only wants to print the directory, do it return ******/
		if (print_only)  /* Used by dirme to display the directory; returns 0*/
			print_entries (dir);
		else
			entry = find_entry (dir, filename);

		block = dir->next_dir_block;

	} while (dir->next_dir_block && (print_only || (entry == NO_ENTRY_FOUND)));

	if (print_only || (entry == NO_ENTRY_FOUND)) {
		recal();
		moff();
		return;
	}

	block = dir->entry[entry].start_block_num; /* starting block */
	last_block_read = block + ((dir->entry[entry].byte_count - 1) / dir->block_size);

/*******	read in first block of file -- headers  **********/
	if(stat = rd_flp_dev (block, buff)) { 	/* read a block */
		printf("Floppy 1st block read failed, stat:  %x\n", stat);
		moff();
		return;
	}

	scn_ptr = fhptr->sctnhdr;
	startp = (unsigned *)(scn_ptr->s_paddr + 0x4);
	num_sect = set_flp_ptr(block, buff);

	for (scn = 0; scn != num_sect; scn++) {

		rdbuff = (uchar *)BUFPOINT;
		block = flp_blk[scn].st_block;
		if(stat = rd_flp_dev (block, rdbuff)) {	/* read a block */
			printf("Floppy block read failed, stat:  %x\n", stat);
			moff();
			return;
		}

		block++;		

		if (!scn) /* assume only section 0 (text) overlaps vectors */
			save = *(unsigned *)FD_INT_VECT;    /* PROM vector */

		rdbuff = mov_flp (scn);

		if (!scn) {
			save1 = *(unsigned *)FD_INT_VECT;   /* Image vector */
			*(unsigned *)FD_INT_VECT = save;    /* replace vector */
		}
		printf ("+");

		for (blk = 0; blk != flp_blk[scn].blocks;) {
			if(stat = rd_flp_dev (block, rdbuff)) {	/* read block */
				printf("Floppy block read failed, stat:  %x\n",
					stat);
				moff();
				return;
			}

			if (!(++blk % 10))
				printf (".");

			rdbuff += BLK_SIZE;
			block++;
		}
	}

/*******	setup fd interrrupt, turn off motor, and execute  ********/
	recal();
	moff();			/* turn motor off, just for fun. */

	*(unsigned *)FD_INT_VECT = save1;
	final = *(unsigned *)startp;
	(*(PFI)final)(); 	/* Jump to it!!! */
}

int
set_flp_ptr(block, fhdr)
int	block;
struct	fhdr *fhdr;
{

	struct scnhdr 	*scn_ptr;
	int	cnt;
	int	scn;
	int	size;

	scn_ptr = fhdr->sctnhdr;
	for (cnt = 0, scn = 0; cnt != fhdr->filhdr.f_nscns; cnt++, scn_ptr++) {

		if (!scn_ptr->s_scnptr)
			continue;

		flp_blk[scn].addr = scn_ptr->s_paddr;
		flp_blk[scn].st_block = block + 
					(scn_ptr->s_scnptr / BLK_SIZE);
		flp_blk[scn].st_off = (scn_ptr->s_scnptr % BLK_SIZE);
		size = (scn_ptr->s_size - BLK_SIZE + flp_blk[scn].st_off);
		flp_blk[scn].blocks = ((size + BLK_SIZE - 1) / BLK_SIZE);
		scn++;
	}
	return (scn);
}

uchar *
mov_flp (scn)
int	scn;
{

	unsigned  *dest = (unsigned *)flp_blk[scn].addr;
	unsigned  *src  = (unsigned *)(BUFPOINT + flp_blk[scn].st_off);
	unsigned  *limit = (unsigned *)(BUFPOINT + BLK_SIZE);

	for (; src < limit; src++, dest++) 
		*dest = *src;
	return((uchar *)dest);
}

#endif


void
print_entry (dir, element)
struct	sa_dir 	*dir;	/* stand alone dir pointer. */
uint	element;
{
	
	if ((element % LINES_PER_SCREEN) == 0)
		printf ("\n FILENAME              VERSION   START_BLK     SIZE        SPM_EXEC\n\n");

	printf (" %16s       %2d.%2d     %6d       %6d       %s\n", 
	dir->entry[element].name,
	dir->entry[element].major_version, dir->entry[element].minor_version,
	dir->entry[element].start_block_num, dir->entry[element].byte_count,
	(dir->entry[element].spm_exec) ? " SPM_EXEC  " : "NO_SPM_EXEC");
}

uint
find_entry (dir, file) /* see if file is in dir */
struct	sa_dir   *dir;
char	*file;
{
	int entry;

	for(entry = 0; entry < MAX_DIR_ENTRY; entry++) { 
			if(!strcmp(file, dir->entry[entry].name)) /* same */
				break;
	}
	
	if(entry == MAX_DIR_ENTRY) { /* if failed */
		if (! dir->next_dir_block) /* no more directories */
			printf("Did not find (%s) code. \n", file);
		return(NO_ENTRY_FOUND);
	}

	if(!dir->entry[entry].spm_exec) { /* if not allowed to exec. */
		printf("\nFile (%s) is not executable by SPM.\n", file);
		return(NO_ENTRY_FOUND); 		/* exit */
	}
	
	return (entry);
}


void
print_entries (dir)
struct  sa_dir  *dir;
{
	uchar  entry;

	for(entry = 0; entry < MAX_DIR_ENTRY; ) {	/* all entries */

		if (dir->entry[entry].name[0] == '\0')
			break;

		print_entry (dir, entry);
		entry++;
		if ((entry % LINES_PER_SCREEN) == 0) {
			printf ("Depress ESC to quit... ");
			if (anykey () == ESCAPE) {
				printf ("\n");
				return;
			}	
			clscrn (25);
		}
	}
	return;
}

int
read_edt_dir (dtreq_ptr)
struct		dtreq_call 	*dtreq_ptr; 
{

	int	i;
	int	i1;
	int	stat = 0;

	if (boot_type != DV_RV) {
		seek_tape(dtreq_ptr, dtreq_ptr->block);

		dtreq_ptr->command = TPREAD;
		if(stat=dtreq(dtreq_ptr)) {
			printf("Tape directory read failed, stat:  %4X\n",stat);
			return(1);
		}
		curr_blk++;
	}
	else {
		dtreq_ptr->command = PLREAD;
		dtreq_ptr->type = DTDISK;
		while(i++ != DISK_RETRY)	{ /* retries 45 times.. */
	   		if(stat=dtreq(dtreq_ptr)) {
				for(i1 = 0; i1 < 180000; i1++) ;
				if(ifesc())
					return(1);  /* exit */
			}
			else
				break;
		}
		if(i == DISK_RETRY) {
			printf("Disk read failed, stat:  %4X\n", stat);
			return(1);
		}
	}
	return(0);
}

int
seek_tape (dtreq_ptr, blk_num)
struct		dtreq_call 	*dtreq_ptr; 
int	blk_num;
{
	int	stat = 0;
	
	dtreq_ptr->type = DTTAPE;

	if (curr_blk == -1)
		rewind_tape (dtreq_ptr);

	if (curr_blk > blk_num)
		rewind_tape (dtreq_ptr);

	dtreq_ptr->command = TPREAD;
	for (; curr_blk != blk_num; curr_blk++) {
		if(stat=dtreq(dtreq_ptr)) {
			printf("\nTape seek failed, stat:  %4X\n",stat);
			dtreq_ptr->command = REWIND;
		  	dtreq(dtreq_ptr);
			return(1);
		}
	}
	return(0);
}

int
rewind_tape(dtreq_ptr)
struct		dtreq_call 	*dtreq_ptr; 
{
	int	i = 0;
	int	stat = 0;
	int	i1;

	dtreq_ptr->command = REWIND;
	dtreq_ptr->type = DTTAPE;
	while(i++ != TAPE_RETRY)	{/* retries 15 times.. */
		if(stat=dtreq(dtreq_ptr)) {
			for(i1 = 0; i1 < 180000; i1++) ;
			if(ifesc())
				curr_blk = -1;
				return(1);	
		}
		else
			break;
	}

	if(i == TAPE_RETRY) {
		printf("Tape rewind failed, stat:  %4X\n", stat);
		curr_blk = -1;
		return(1);
	}
	curr_blk = 0;
	return(0);
}

/*							*/
