/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) lpdvr.c: version 25.1 created on 11/27/91 at 14:32:25	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)lpdvr.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
/*------------------------------------------------------------------*
 *	ioplpdvr.c - streams lp driver
 *
 *	04-01-89 - gil hunt - original version
 *	05-23-90 - gil hunt - set RESET bit high in control reg
 *			      (didn't used to have to do this)
 *			    - IOPMLPVER2
 *	07-13-90 - gil hunt - call 'wakeup' if closing and no output from
 *			      write service so ilp_close can't sleep forever
 *			    - also change close timer so it is repeatedly called
 *			      if there is output really occurring
 *			    - add more debug
 *			    - IOPMLPVER3
 *
 *	this is a streams lp driver that resides either:
 *	- in the async/only iopm, or
 *	- in the pm talking to the spm
 *
 *	- on the upstream size it interfaces with modules that handle most
 *	  of the character processing
 *	- on the downstream side it interfaces with either:
 *	  - the hardware directly (if on the iopm), or
 *	  - a program on the spm that talks to the hardware
 *
 *	the primary functions of this driver are:
 *	- move output data from streams messages to print buffers
 *	- start the output of data using the DMA on the useq
 *	  or let the spm start the output
 *	- periodically monitor the status of the output
 *
 *------------------------------------------------------------------*/
#include "sys/types.h"
#include "sys/user.h"
#include "sys/param.h"
#include "sys/errno.h"
#include "sys/sysmacros.h"
#include "sys/systm.h"
#include "sys/stream.h"
#include "sys/stropts.h"
#include "sys/signal.h"
#include "sys/file.h"
#include "sys/strlog.h"
#include "sys/termio.h"
#include "sys/debug.h"
#include "sys/cmn_err.h"
#include "sys/ilp.h"
#include "sys/ms_lp.h"
#include "sys/ilp_clog.h"
#include "sys/cir_log.h"
#include "sys/str_conf.h"
/*----------------------------------------------------------------------*/
int	IOPMLPVER3;	/* version of stream lp driver for async iopm	*/
/*----------------------------------------------------------------------*/
int	ilp_cir_log_debug = 1;		/* set non-zero to enable ilp_cir_log */
int	ilp_cir_log_init_flag = 0;	/* non-zero if cur_log initialized */

int	ilp_cir_index;		/* number of next ilp_cir_log_buf entry */

struct  cir_log	*ilp_cir_ptr;  /* ptr to next ilp_cir_log_buf entry */
struct	cir_log	ilp_cir_log_buf[ILP_CIR_LOG_SIZE];
/*------------------------------------------------------------------*/
int	ilp_debug = 0;		/* set non-zero to enable ilp_log */
/*------------------------------------------------------------------*/
/*	the following 2 lines are needed for the BUFLOG macros */
#define LOG_DEBUG	ilp_debug
#define	LOG_RTN		ilp_log

/*	the following 2 lines are needed for the macro BUFCIR_LOG */
#define CIRLOG_DEBUG	ilp_cir_log_debug
#define	CIRLOG_RTN	ilp_cir_logit

#include "sys/debuglog.h"
/*------------------------------------------------------------------*/
#define ILP_MOD		129
#define ILP_MSIZE	1024		/* max data msg size we will send up */
#define ILP_NUM_PORTS	1		/* one port per iopm/spm */

/*------------------------------------------------------------------*/
#define ILP_CLOSE_TIME	15	/* number of seconds to allow a single
				   output to complete if closing */
/*------------------------------------------------------------------*/
int ilp_open(), ilp_close(), ilp_rput(), ilp_wput(), ilp_wsrv();

struct module_info	ilp_minfo =
{
/*			 packet size
   mod id   module name  min    max	hi-water  lo_water
   ------   -----------  -------------  --------  --------*/
   ILP_MOD, "lpdvr",     0,     INFPSZ, 4096,     1024
};

struct qinit ilp_ri =
{
/* put       service   open      close      unused  mod_info   stats
   ---       -------   ----      -----      ------  --------   ----- */
   ilp_rput, NULL,     ilp_open, ilp_close, NULL,  &ilp_minfo, NULL
};
struct qinit ilp_wi =
{
/* put       service   open      close      unused  mod_info   stats
   ---       -------   ----      -----      ------  --------   ----- */
   ilp_wput, ilp_wsrv, NULL,     NULL,      NULL,  &ilp_minfo, NULL
};

struct streamtab	 lpdvrinfo = { &ilp_ri, &ilp_wi, NULL, NULL };
/*------------------------------------------------------------------*/
int	ilp_timer();
int	ilp_close_timer_expired();

ilp_t	ilp[ILP_NUM_PORTS];		/* one struct per lp port */
ilp_t	*ilp_lp_tab[ILP_NUM_PORTS];	/* check for activity in poll*/
int	ilp_poll_timer_id;			/* id of ilp_timer */
int	ilp_poll_timer_init = 0;
/*------------------------------------------------------------------*/
/*
 *		OPEN
 *
 *------------------------------------------------------------------*
 *
 * ilp_open -- open a stream lp driver
 *
 * Arguments:
 *	rq	if stream open, read queue pointer
 *	dev	major/minor device number
 *		module open		- 0
 *		regular driver open	- major/minor
 *		clone driver open	- major only
 *	flag	flags device was opened with
 *	sflag	open type flag
 *		0		- driver open
 *		MODOPEN		- normal module open
 *		CLONEOPEN	- clone driver open
 *
 * Returns minor device number or OPENFAIL
 */
ilp_open(rq, dev, flag, sflag)
queue_t	*rq;
dev_t	dev;
int	flag, sflag;
{
	ilp_t	*lp;
	queue_t	*wq;
	dev_t	min_dev;

#if	! defined(IOPM)
	ilp_init();
#endif	/* IOPM */
	wq = OTHERQ(rq);
	min_dev = minor(dev);
	BUGCIR_LOG(min_dev, flag, sflag, ILP_CIR_OPEN, wq);
	if (min_dev >= ILP_NUM_PORTS)
		return (OPENFAIL);
	if (sflag != 0)
	{
		BUGLOG1(NULL,BUG_LERR,"ilp_open bad sflag=%x", sflag);
		return (OPENFAIL);
	}
	if ((lp = (ilp_t *)rq->q_ptr) != NULL)
	{		/* already attached */
		if ((lp == (ilp_t *)wq->q_ptr)
		 && (lp->t_device == dev)	/* major/minor device no. */
		 && (lp->state & TTUSE))	/* double check */
			return(min_dev);	/* flag successful open */
		else
		{
			cmn_err(CE_WARN,"ilp_open: multiple open\n");
			rq->q_ptr = wq->q_ptr = NULL;
			return(OPENFAIL);
		}			
	}
	lp = &ilp[min_dev];		/* one struct per lp port */
	BUGLOG2(lp, BUG_LOPEN, "ilp_open wq=%x, flag=%x", wq, flag);
	bzero(lp, sizeof(ilp_t));
	lp->myname[0] = 'I';
	lp->myname[1] = 'L';
	lp->myname[2] = 'P';
	lp->myname[3] = (unchar) min_dev;

	rq->q_ptr = (char *)lp;
	wq->q_ptr = (char *)lp;
	lp->rq = rq;				/* queue for upstream */
	lp->wq = wq;				/* queue for downstream */
	lp->state = TTUSE;			/* double check */
	lp->t_openf = flag;			/* type of open - info only */
	lp->t_device = dev;			/* major/minor device no. */
	lp->min_dev = min_dev;			/* minor device no. */
	ilp_msinit(lp);				/* init useq pointers */
	lp->state |= ISOPEN;			/* flag port open */

	ilp_lp_tab[min_dev] = lp;		/* check in ilp_timer */
	return(min_dev);			/* flag successful open */
}/*------------------------------------------------------------------*/
/*
 *		CLOSE
 *
 *------------------------------------------------------------------*
 *
 * ilp_close -- close a stream lp driver
 *
 * Arguments:
 *	rq	read queue pointer
 *	flag	flags device opened with, 0 for modules
 */
ilp_close(rq, flag)
register queue_t	*rq;
{
	register ilp_t	*lp;
	register queue_t *wq;
	int	sleep_return;		/* return code from sleep
					   0 = requested item occurred
					   1 = unexpected signal occurred */

	wq = OTHERQ(rq);
	lp = (ilp_t *)rq->q_ptr;
	ASSERT(lp != NULL);
	BUGCIR_LOG(lp->min_dev, flag, 0, ILP_CIR_CLOSE, wq);
	BUGLOG2(lp, BUG_LCLOSE, "ilp_close wq=%x, flag=%x", wq, flag);

	if (lp->state & ISOPEN)
	{
		lp->t_closing = 1;
		ilp_check_if_output_empty(lp);
		/* 'ilp_close_timer_expired' is called only if the timer that
		   'ilp_start_close_timer()' starts, is allowed to expire,
		   which can only happen if the output is 'stuck' */
		ilp_start_close_timer(lp);
		sleep_return = sleep((caddr_t)&lp->output_in_progress,
							STIPRI|PCATCH);
		/* return from sleep if closing and:
		   1. write service routine called and no output in progress,
		   2. 'ilp_close_timer_expired' called because timer expired, or
		   3. signal received from stream head
		*/
	}
	BUGCIR_LOG(lp->min_dev,lp->output_in_progress,0,ILP_CIR_CLOSE_END,wq);
	BUGLOG2(lp, BUG_LCLOSE, "ilp_close_end wq=%x output_in_progress=%d",
						wq, lp->output_in_progress);
	ilp_lp_tab[lp->min_dev] = NULL;	/* don't check in ilp_timer anymore */
	lp->state &= ~TTUSE;
	if (lp->timer_enable & OUTPUT_EMPTY_TIMER)
		untimeout(lp->output_empty_id);
	if (lp->timer_enable & CANCEL_CHECK_TIMER)
		untimeout(lp->cancel_id);
	rq->q_ptr = NULL;
	wq->q_ptr = NULL;
}/*------------------------------------------------------------------*/
/*
 *		READ PUT
 *
 *------------------------------------------------------------------*
 *
 * ilp_rput -- read put routine
 *
 *	should never be called since 'ilp_read()' puts msgs onto the
 *	read queue directly. Included only to scream if anyone trys it.
 *
 * Arguments:
 *	rq	read queue pointer
 *	mp	message pointer
 */
ilp_rput(rq, mp)
queue_t *rq;
mblk_t	*mp;
{
	ilp_t	*lp;

	lp = (ilp_t *)rq->q_ptr;
	ASSERT(lp != NULL);
	if (mp == NULL)
	{
		cmn_err(CE_NOTE, "NULL mp received in ilp_rput, q = %x", rq);
		return;
	}
	BUGCIR_LOG(lp->min_dev, mp->b_datap->db_type, *mp->b_rptr,
							ILP_CIR_RPUT,mp);
	BUGLOG2(NULL,BUG_LERR,"ilp_rput rq=%x mp=%x  ???", rq, mp);
	freemsg(mp);
}/*------------------------------------------------------------------*/
/*
 *		WRITE PUT
 *
 *------------------------------------------------------------------*
 *
 * ilp_wput -- write put routine
 *
 *	processes M_FLUSH and M_IOCTL
 *	pass M_DATA to write service routine
 *
 * Arguments:
 *	wq	write queue pointer
 *	mp	message pointer
 */
ilp_wput(wq, mp)
register queue_t	*wq;
register mblk_t		*mp;
{
	register ilp_t	*lp;
	register type;
	struct	iocblk		*iocbp;
	int	cmd;
	unchar	arg;

	lp = (ilp_t *)wq->q_ptr;
	ASSERT(lp != NULL);
	if (mp == NULL)
	{
		cmn_err(CE_NOTE, "NULL mp received in ilp_wput, q = %x", wq);
		return;
	}
	if (lp == NULL  || !(lp->state & TTUSE))
	{
		freemsg(mp);
		return;		/* port is not open */
	}
	type = mp->b_datap->db_type;
	BUGLOG3(lp, BUG_LGEN, "ilp_wput wq=%x, mp=%x, type=%x",wq, mp, type);
	switch (type)
	{
	default:
		BUGCIR_LOG(lp->min_dev,type,*mp->b_rptr,ILP_CIR_WSRV,mp);
		freemsg(mp);	/* throw away what we don't understand*/
		break;
	case M_DATA:
		putq(wq, mp);		/* pass M_DATA to service */
		break;
	case M_FLUSH:		/* flush read and/or write bufs */
		BUGCIR_LOG(lp->min_dev,0,*mp->b_rptr,ILP_CIR_WRITE_FLUSH,mp);
		if (*mp->b_rptr & FLUSHW)
		{
			flushq(lp->wq, FLUSHALL); /* flush all ofwrite queue */
			ilp_flush_write_side(lp);
		}
		if (*mp->b_rptr & FLUSHR)
		{
			flushq(lp->rq, FLUSHALL);	/* flush read queue */
			*mp->b_rptr &= ~FLUSHW;
			qreply(wq, mp);
		}
		else	freemsg(mp);
		break;
	case M_IOCTL:
		iocbp = (struct iocblk *)mp->b_rptr;  /*addr of ioctl control */
		cmd = iocbp->ioc_cmd;
		switch(cmd)
		{
		default:
			cmd &= IOCTYPE;
			if (cmd == LDIOC || cmd == TIOC)
				mp->b_datap->db_type = M_IOCACK;
			else	mp->b_datap->db_type = M_IOCNAK;
			iocbp->ioc_count = 0;
			qreply(wq, mp);
			break;
		case TCSBRK:
			lp->delay_msg = mp;
			/*block write from moving data to printer output buf */
			lp->state |= NO_OUTPUT_ALLOWED;
			ilp_check_if_output_empty(lp);
			break;
		case TCFLSH:
			if (mp->b_cont == NULL)
				goto nak_ioctl;
			arg = *(int *)mp->b_cont->b_rptr;
			BUGCIR_LOG(lp->min_dev,cmd,arg,ILP_CIR_WRITE_IOCTL, mp);
			switch (arg)
			{
			default:
nak_ioctl:
				mp->b_datap->db_type = M_IOCNAK;
				iocbp->ioc_error = EINVAL;
				qreply(wq, mp);
				return;
			case 0:
				flushq(lp->rq, FLUSHDATA);/* flush read queue */
				break;
			case 1:
				flushq(wq, FLUSHDATA);	/* flush write queue*/
				ilp_flush_write_side(lp);
				break;
			case 2:
				flushq(lp->rq, FLUSHDATA);/* flush read queue */
				flushq(wq, FLUSHDATA);	/* flush write queue*/
				ilp_flush_write_side(lp);
				break;
			} /* switch (*(int *)mp->b_cont->b_rptr) */
			mp->b_datap->db_type = M_IOCACK;
			iocbp->ioc_count = 0;
			qreply(wq, mp);
			break;
		}	/* switch (cmd) */
		break;
	} /* switch (type) */
}/*------------------------------------------------------------------*/
/*
 *		WRITE SERVICE
 *
 *------------------------------------------------------------------*
 *
 * ilp_wsrv -- write service routine
 *
 * process M_DATA messages from upstream
 *
 * Arguments:
 *	wq	write queue pointer
 */
ilp_wsrv(wq)
register queue_t	*wq;
{
	ilp_t		*lp;

	lp = (ilp_t *)wq->q_ptr;
	ASSERT(lp != NULL);
	if (!(lp->state & NO_OUTPUT_ALLOWED) && lp->output_in_progress == 0)
	{
		ilp_get_output_msgs(lp);	/* get msgs from write queue */
		if (lp->first_mp != NULL)
			ilp_move_data_to_useq(lp); /* move to output buffer */
	}
	if (lp->output_in_progress == 0 && lp->t_closing)
		wakeup((caddr_t)&lp->output_in_progress);
}/*------------------------------------------------------------------*/
/*
 *	get messages off the queue until we either run out of msgs
 *	or we have enough M_DATA data bytes to fill up the output buffer
 */
ilp_get_output_msgs(lp)
register ilp_t	*lp;
{
	register mblk_t	*bp;
	int bytes_in_this_msg;		/* no. bytes in this message */

	lp->bytes_so_far = 0;
	bytes_in_this_msg = 0;
	lp->first_mp = lp->last_mp = NULL; 
	while ((bp = getq(lp->wq)) != NULL)
	{
		/* count the number of data bytes in this message */
		bytes_in_this_msg = msgdsize(bp);
		if ((lp->bytes_so_far + bytes_in_this_msg) >= lp->max_size)
		{
			BUGCIR_LOG(lp->min_dev,0,0,ILP_CIR_WPB, bp);
			putbq(lp->wq, bp);
			break;
		}
		else
		{
			if (lp->first_mp == NULL)
				lp->first_mp = bp;
			else	lp->last_mp->b_next = bp;
			lp->last_mp = bp;
			lp->bytes_so_far += bytes_in_this_msg;
		}
	}
}/*------------------------------------------------------------------*/
/*	move data to output buffer on microsequencer on iopm */
ilp_move_data_to_useq(lp)
register ilp_t	*lp;
{
	register unchar *ptr;
	register i;
	mblk_t	*mp, *next, *bp;
	unchar *first_addr;		/* addr of first char in print buf*/
	unchar junk;

	/* disable printer DMA transfer*/
	*(lp->ms_control_reg) = MS_LPC_KEEP_HI;

	/* the hardware counts up until the last address of the hardware
	   buffer is reached, so the first addres to store at is the
	   last address +1 minus the number of data bytes */
	first_addr = lp->end_of_lp_buf_hi + 1 - lp->bytes_so_far;
	ptr = first_addr;
	BUGCIR_LOG(lp->min_dev, lp->bytes_so_far, 0, ILP_CIR_MOVE, ptr);
	mp = lp->first_mp;
	/* copy data from streams message block to the printer buffer */
	while (mp != NULL)
	{
		for (bp = mp; bp != NULL; bp = bp->b_cont)
		{
			for (i = bp->b_wptr - bp->b_rptr; i > 0; i--)
				*(ptr++) = *(bp->b_rptr++);
		}
		next = mp->b_next;
		freemsg(mp);
		mp = next;
	}
	junk = *first_addr;	/* do this so the last access */
	*first_addr = junk;	/* by the IOPM is to the 1st byte of data*/
				/* writing back isn't necessary for the useq
				   but if we don't, the compiler optimizer
				   deletes the read into junk */
	lp->output_in_progress = 1;
	lp->first_mp = NULL;
	*(lp->ms_clear_done) = 0;/* what we write is not important, but
				   the act of writing clears the F/F */
	/* enable printer DMA transfer*/
	*(lp->ms_control_reg) = MS_LPC_DMAEN | MS_LPC_KEEP_HI;
	if (lp->t_closing)
		ilp_start_close_timer(lp);
}/*------------------------------------------------------------------*/
/*
 *		start/restart close timer
 */
ilp_start_close_timer(lp)
register ilp_t	*lp;
{
	/* if the timer goes off, it means nothing happened for
	   ILP_CLOSE_TIME seconds. 'ilp_close_timer_expired' wakes up
	   'ilp_close' so the close will exit */
	if (lp->timer_enable & CANCEL_CHECK_TIMER)
		untimeout(lp->cancel_id);
	lp->timer_enable |= CANCEL_CHECK_TIMER;
	lp->cancel_id = timeout(ilp_close_timer_expired,lp,ILP_CLOSE_TIME*HZ);
}/*------------------------------------------------------------------*/
/* cancel waiting for output to complete */
ilp_close_timer_expired(lp)
register ilp_t	*lp;
{
	lp->timer_enable &= ~CANCEL_CHECK_TIMER;
	BUGCIR_LOG(lp->min_dev, 0, 0, ILP_CIR_CANCEL, lp->wq);
	BUGLOG1(lp, BUG_LCLOSE, "ilp_close_timer_expired wq=%x", lp->wq);
	wakeup((caddr_t)&lp->output_in_progress);
}/*------------------------------------------------------------------*/
/*
 *	if output is empty, continue processing
 *	else timeout and check again later
 */
ilp_check_if_output_empty(lp)
register ilp_t	*lp;
{
	ASSERT(lp != NULL);
	if (lp->timer_enable & OUTPUT_EMPTY_TIMER)
	{
		lp->timer_enable &= ~OUTPUT_EMPTY_TIMER;
		untimeout(lp->output_empty_id);
	}
	if (lp->output_in_progress == 0)
		ilp_process_msg_output_empty(lp);	/*do output empty step*/
	else
	{
		lp->output_empty_id = timeout(ilp_check_if_output_empty,lp,HZ);
		lp->timer_enable |= OUTPUT_EMPTY_TIMER;
	}
}/*------------------------------------------------------------------*/
/*
 *	the output buffer is now empty, so process message
 *
 *	all data that came before this message has been output to the
 *	device, since this is now the first message in the queue
 */
ilp_process_msg_output_empty(lp)
register ilp_t	*lp;
{
	register mblk_t		*bp;
	register struct	iocblk	*iocbp;
	register type;

	ASSERT(lp != NULL);
	if ((bp = lp->delay_msg) != NULL)
	{
		type = bp->b_datap->db_type;
		BUGCIR_LOG(lp->min_dev,type,*bp->b_rptr,ILP_CIR_WOUT,bp);
		BUGLOG3(lp, BUG_LGEN, "ilp_process_msg_output_empty wq=%x bp=%x type=%x",lp->wq, bp, type);
		switch(type)
		{
		default:
			freemsg(bp);	/* throw away what we don't understand*/
			break;
		case M_IOCTL:
			iocbp = (struct iocblk *)bp->b_rptr; /*ioctl info addr*/
			switch(iocbp->ioc_cmd)
			{
			default:
				freemsg(bp);	/* disregard unknown msgs*/
				break;
			case TCSBRK:		/* process same as M_BREAK */
				bp->b_datap->db_type = M_IOCACK;
				iocbp->ioc_count = 0;
				qreply(lp->wq, bp);
				break;
			} /* switch(iocbp->ioc_cmd) */
		} /* switch(bp->b_datap->db_type) */
		lp->delay_msg = NULL;
	}
	lp->state &= ~NO_OUTPUT_ALLOWED; /*allow output*/
	qenable(lp->wq);		/* do in case we have data pending */
}/*------------------------------------------------------------------*/
/*
 *	INTERRUPT FROM TIMER
 *
 *------------------------------------------------------------------*
 *
 *	ilp_timer - check for output complete
 */
ilp_timer()
{
	ilp_t	*lp;
	int i;
	unchar status;

	for (i = 0; i < ILP_NUM_PORTS; i++)
	{
		if ((lp = ilp_lp_tab[i]) == NULL)
			continue;
		if ((status = *(lp->ms_status_reg)) != lp->status)
		{
			BUGCIR_LOG(lp->min_dev, lp->status, status,
							ILP_CIR_STATUS, lp);
			lp->status = status;
		}
		if (lp->output_in_progress != 0)
			if (*(lp->ms_status_reg) & MS_LPS_DONE)
			{
				BUGCIR_LOG(lp->min_dev,0,0,ILP_CIR_DONE,lp);
				lp->output_in_progress = 0;
				qenable(lp->wq); /* enable write service*/
			}
	}
	/* call this same routine on the next clock tick */
	ilp_poll_timer_id = timeout(ilp_timer, (ilp_t *)NULL,1);
}/*------------------------------------------------------------------*/
/*
 *	flush handling
 *
 *------------------------------------------------------------------*/
ilp_flush_write_side(lp)
register ilp_t	*lp;
{
	mblk_t	*bp, *next;

	*(lp->ms_control_reg) = MS_LPC_PRIME | MS_LPC_KEEP_HI;
	lp->output_in_progress = 0;
	lp->bytes_so_far = 0;
	bp = lp->first_mp;
	lp->first_mp = NULL;
	while (bp != NULL)
	{
		next = bp->b_next;
		freemsg(bp);
		bp = next;
	}
}/*------------------------------------------------------------------*/
/*
 *	general initialization
 */
ilp_msinit(lp)
register ilp_t	*lp;
{
	lp->max_size		= MS_LP_BUF_SIZE;
	lp->ms_control_reg	= MS_LP_CONTROL_REG;	/* addr of control reg*/
	lp->end_of_lp_buf_hi	= END_OF_MS_LP_BUF_HI;	/* addr of end of buf*/
	lp->ms_clear_done	= MS_LP_CLEAR_DONE;	/* write clears F/F */
	lp->ms_status_reg	= MS_LP_STATUS_REG;	/* addr of status reg */
	*(lp->ms_control_reg)	= MS_LPC_SETDONE | MS_LPC_ACKNLG;
	*(lp->ms_control_reg)	= MS_LPC_KEEP_HI;
}/*------------------------------------------------------------------*/
ilp_fail(a, f, l)
register char *a, *f;
int	l;
{
	cmn_err(CE_NOTE, "assertion failed: %s, file: %s, line: %d", a, f, l);
}/*------------------------------------------------------------------*/
ilp_log(lp, level, fmt, ARGS)
ilp_t	*lp;
int		level;
char		*fmt;
unsigned	ARGS;
{
	int p_level;
	dev_t	dev;

	if (lp == NULL)
		dev = ILP_NUM_PORTS;
	else	dev = lp->min_dev;

	p_level = (ilp_debug & 0xf0000) >> 16;
	if (ilp_debug != 0 && (p_level == 0 || level <= p_level))
	{
		if ((ilp_debug & 0xfff) == 0x123)
			ilp_print_no_log(fmt, ARGS);
		else	strlog(ILP_MOD,dev,level, SL_TRACE, fmt, ARGS);
	}
}/*------------------------------------------------------------------*/
ilp_print_no_log(fmt, ARGS)
char		*fmt;
unsigned	ARGS;
{
	printf(fmt, ARGS);
	if ((ilp_debug & 0x1000) == 0)
		printf("\n");
	else	printf(" ** ");
}/*------------------------------------------------------------------*/
ilp_cir_logit(dev, parm3, parm4, log_type, parm)
unchar dev, parm4, log_type;
ushort parm3;
int parm;
{
	register struct cir_log	*ptr;  /* ptr to next ilp_cir_log_buf entry */
	int s;

	s = splstr();
	ptr = ilp_cir_ptr;
	ptr->rel_time = (dev << 24) + (lbolt & 0x7fffff);
	ptr->log_type = log_type;
	ptr->parm_1 = parm >> 16;
	ptr->parm_2 = parm & 0xffff;
	ptr->parm_3 = parm3;
	ptr->parm_4 = parm4;
	if (++ilp_cir_index >= ILP_CIR_LOG_SIZE)
	{
		ilp_cir_index = 0;		/* wrap */
		ilp_cir_ptr = ilp_cir_log_buf;
	}
	else	ilp_cir_ptr++;
	ilp_cir_ptr->log_type = -1;	/* end of buf */
	splx(s);
}/*------------------------------------------------------------------*/
/*
 *	initialize the circular log buffer
 *
 *	- set the 'next entry pointer' to the start of the buffer
 *	- flag that entry as the 'end of buffer' entry (log_type = -1)
 */
ilp_cir_log_init()
{
	if (ilp_cir_log_init_flag == 0)
	{
		ilp_cir_log_init_flag++;	/* flag 'initialization done */
		ilp_cir_index = 0;		/* next entry pointer */
		ilp_cir_ptr   = ilp_cir_log_buf; /* next entry pointer */
		ilp_cir_ptr->log_type = -1;	/* end of buf */
	}
}/*------------------------------------------------------------------*/
#if	defined(IOPM)
/*
 *	driver initialization
 */
lpdvrinit()
{
	static struct str_config  strconf;

	strconf.sc_version = SC_VERSION;
	strconf.sc_strtab = &lpdvrinfo;
	strconf.sc_start = 0;
	attach_streams_driver( &strconf );

	ilp_init();
}/*------------------------------------------------------------------*/
#endif	/* IOPM */
ilp_init()
{
	ilp_cir_log_init();		/* init circular logging */
	if (ilp_poll_timer_init == 0)
	{	/* start pollint timer */
		ilp_poll_timer_init++;
		ilp_poll_timer_id = timeout(ilp_timer, (ilp_t *)NULL,1);
	}
}/*------------------------------------------------------------------*/
