/*
 *	@(#) dialVA3450.c 22.1 89/11/14 
 *
 *	Copyright (C) The Santa Cruz Operation, 1984, 1985, 1986, 1987, 1988.
 *	Copyright (C) Microsoft Corporation, 1984, 1985, 1986, 1987, 1988.
 *	This Module contains Proprietary Information of
 *	The Santa Cruz Operation, Microsoft Corporation
 *	and AT&T, and should be treated as Confidential.
 */
/*
 *  Function:	dialer program for the Racal Vadic VA3450 Auto Dial
 *
 *  Usage:	dial ttyname telnumber speed 
 *		dial -h ttyname speed
 *
 *  Returns:	0x80	bit = 1 if connection failed
 *		0x10	bit = 1 if line is also used for dialin #ifndef HDUU
 *		0x0f	if msb=1: error code
 *			if msb=0: connected baud rate (0=same as dialed baud)
 *
 *		Note: getty calls the dialer with -h whenever it starts up
 *		on a line enabled in /etc/ttys and listed in Devices with
 *		this dialer.
 *
 *		Error codes are split into two catagories: 1) (codes 0-11)
 *		Local problems are defined as tty port, or modem problems:
 *		problems that can be worked around by using a different device.
 *		2) (codes 12-15) Remote problems are phone busy, no answer, etc.
 *		: attempts to connect to this remote system should be stoped.
 *
 *  Documents:	Publication No. 18008-103
 *
 *  Operational notes:
 *		The 3450 must have DTR controled by the computer or
 *		it will not hangup properly.  So pull your modem apart
 *		and make sure switch B2 is OFF.
 *
 *  Note:	This source file can be used both for the old UUCP and
 *		for the new HoneyDanBer UUCP.  For the latter, HDUU may
 *		be defined to avoid calls to the HD ungetty program - which
 *		assumes that uugetty is used, and so simply returns SUCCESS.
 *		However, dialer binaries for the old UUCP are equally valid
 *		for the new HoneyDanBer UUCP (but make an unnecessary call
 *		to the null ungetty, supplied for backward compatibility).
 */

static char sccsid[] = "@(#)dialVA3450.c	22.1 89/11/14 ";

#include <stdio.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <termio.h>
#include <pwd.h>
#include <errno.h>
#include <ctype.h>

/* return codes: these are set up so that an abort signal at any time can */
/* set the fail bit and return to the caller with the correct status */
#define	RC_BAUD		0x0f	/* CBAUD connected at (0=same as dialed speed)*/
#define	RC_ENABLED	0x10	/* enabled flag: 1 = ungetty -r required to */
				/* restore the line */
#define	RC_FAIL		0x80	/* 1 = failed to connect */

/* return code error codes */
#define	RCE_NULL	0	/* general purpose or unknown error code */
#define	RCE_INUSE	1	/* line in use */
#define	RCE_SIG		2	/* signal aborted dialer */
#define	RCE_ARGS	3	/* invalid arguments */
#define	RCE_PHNO	4	/* invalid phone number */
#define	RCE_SPEED	5	/* invalid dial baud rate */
#define	RCE_OPEN	6	/* can't open line */
#define	RCE_IOCTL	7	/* ioctl error */
#define	RCE_TIMOUT	8	/* timeout */
#define	RCE_NOTONE	9	/* no dial tone */
#define	RCE_BUSY	13	/* phone is busy */
#define	RCE_NOCARR	14	/* no carrier */
#define	RCE_ANSWER	15	/* no answer */

#define	SUCCESS	0

/* ungetty return codes */
#define	UG_NOTENAB	0
#define	UG_ENAB		1
#define	UG_RESTART	1
#define	UG_FAIL		2

#define SAME		0
#define MAXLINE		80
#define	UNGETTY		"/usr/lib/uucp/ungetty"

#define DEBUG(l, f, s)	if (Debug >= l) fprintf(stderr, f, s)
#ifndef DBG
#define	DBG	0
#endif

/*
 *  MDVALID - Allow only these characters to reach the modem.
 */
#define MDVALID		"0123456789K"

/*
 *  MDHANGUP - Vadic won't hangup, this just takes us out of command mode.
 */
#define	MDHANGUP	"I\r"

/*
 *  MDRESET - Put the modem into command mode.
 */
#define	MDRESET		"\005\r" 

/*
 *  These defines are used to determine how long the
 *  dialer timeout should be.
 */
#define	MDPAUSCHR	'K'
#define	MDPAUSDLY	5

#define	DIAL_RETRY	4

/*
 *  Possible messages produced by modem.
 */
#define	HELLO		0
#define	INVALID		1
#define	ONLINE		2
#define	FAILEDCALL	3
#define	IDLE		4
#define	TIMEOUT		5
#define DIALING		6
#define NUMBER		7
#define PHONENUM	8

char *mdmsgs[] = {
/* 0 */	"HELLO: I'M READY\r\n*",
/* 1 */	"INVALID\r\n*",
/* 2 */	"ON LINE",
/* 3 */	"FAILED CALL",
/* 4 */	"IDLE",
/* 5 */	"TIME OUT",
/* 6 */	"DIALING:",
/* 7 */	"NUMBER?",
/* 8 */	 0,		/* empty slot for phone number */
	0
};

char *strchr();
int alrmint();
struct termio term;
int Debug = DBG;			/* set when debug flag is given	*/
int dialing;				/* set while modem is dialing	*/
int fd = -1;				/* file descriptor for acu	*/
int retcode = RC_FAIL;			/* return code			*/
char User[10] = "unknown";		/* user id for logfile entries	*/
char Rmtname[] = "unknown";		/* remote system for logfile	*/


#define	toprint(x)	((x)<' '?((x)+'@'):'?')

/* vgets - Format one character in "always printable" format (like cat -v)
 */

char *
vgets(c, f)
unsigned char c;
FILE *f;
{
	static char buffer[10];
	char *pnt;

	pnt = buffer;
	if (iscntrl(c) || !isprint(c)) {
		if (!isascii(c)) {			/* Top bit is set */
			*pnt++ = 'M';
			*pnt++ = '-';
			c = toascii(c);			/* Strip it */
		}
		if (iscntrl(c)) {			/* Not printable */
			*pnt++ = '^';
			c = toprint(c);			/* Make it printable */
		}
	}
	*pnt++ = c;
	*pnt = '\0';
	return(buffer);
}


/*
 * translate the pairs of characters present in the first
 * string whenever the first of the pair appears in the second
 * string.
 */
static void
translate(ttab, str)
register char *ttab, *str;
{
	register char *s;

	for(;*ttab && *(ttab+1); ttab += 2)
		for(s=str;*s;s++)
			if(*ttab == *s)
				*s = *(ttab+1);
}


abort(sig)
int sig;
{
	signal(SIGINT, SIG_IGN);
	if (fd != -1) {
		ioctl(fd, TCGETA, &term);
		term.c_cflag |= HUPCL;		/* make sure modem hangs up */
		ioctl(fd, TCSETA, &term);
		close(fd);
	}
	if (sig)  retcode |= (RC_FAIL | RCE_SIG);
	exit(retcode);
}


main(argc,argv)
	int argc;
	char *argv[];
{
	char *acu;			/* device to dial through	*/
	char *phone;			/* phone number to dial		*/
	char command[MAXLINE];		/* modem command buffer		*/
	int baudrate;			/* baud rate of modem		*/
	int errflag = 0;		/* set on errors		*/
	int hflag = 0;			/* set to hangup modem		*/
	int timeout;			/* how long to wait for alarm	*/
	int dial_retry = DIAL_RETRY;	/* dial retry count		*/
	extern int optind;
	extern char *optarg;
	int c, pid;
	char *p;
	struct passwd *pwd, *getpwuid();

	/*
	 *  Reenable all those signals we want to know about
	 */

	signal(SIGILL, SIG_DFL);
	signal(SIGIOT, SIG_DFL);
	signal(SIGEMT, SIG_DFL);
	signal(SIGFPE, SIG_DFL);
	signal(SIGBUS, SIG_DFL);
	signal(SIGSEGV, SIG_DFL);
	signal(SIGSYS, SIG_DFL);
	signal(SIGTERM, SIG_DFL);

	while ((c = getopt(argc, argv, "hx:")) != EOF)
		switch(c) {
			case 'h':
				hflag++;
				break;
			case 'x':
				Debug = atoi(optarg);
				break;
			case '?':
				errflag++;
				break;
		}

	if (Debug) {
		fprintf(stderr, "dialer args ");
		for (c=0; c<argc; c++)  fprintf(stderr, ":%s", argv[c]);
		fprintf(stderr, "\n");
	}

	if (hflag) {
		if (argc - optind != 2)  errflag++ ;
	} else {
		if (argc - optind != 3)  errflag++ ;
	}

	if (errflag) {
		if (hflag)
			fprintf(stderr,"Usage: dial -h devicename speed\n");
		else
			fprintf(stderr,"Usage: dial devicename number speed\n");
		exit(RC_FAIL | RCE_ARGS);
	}

	acu = argv[optind++];

	if (!hflag) {
		phone = argv[optind++];
		translate("=K-K", phone);
		if (strlen(phone) != strspn(phone, MDVALID)) {
			fprintf(stderr, "dial: Bad phone number %s\n", phone);
			exit(RC_FAIL | RCE_PHNO);
		}
	}

	switch(atoi(argv[optind])) {
		case 300:
			baudrate = B300;
			break;
		case 1200:
			baudrate = B1200;
			break;
		default:
			fprintf(stderr, "dial: Bad speed: %s\n", argv[optind]);
			exit(RC_FAIL | RCE_SPEED);
	}		

#ifndef HDUU
	if (!hflag) {
		if ((pid = fork()) == 0) {
			execl(UNGETTY, "ungetty", acu, NULL);
			fprintf(stderr, "ungetty exec error\n");
			exit(-1);
		}
		while (((c = wait(&errflag)) != pid) && c != -1) ;
		switch ((errflag>>8) & 0xff) {
		case UG_NOTENAB:	/* line acquired: not enabled */
			retcode = SUCCESS;
			break;
		case UG_ENAB:	/* line acquired: need ungetty -r when done */
			retcode = RC_ENABLED;
			break;
		case UG_FAIL:		/* could not acquire line */
			exit(RC_FAIL | RCE_INUSE);
		case 255:
			exit(RC_FAIL);
		}
	}
#else
	retcode = SUCCESS;	/* uugetty does not require ungetty */
#endif

	/*
	 *  Must open with O_NDELAY set or the open may hang.
	 */
	if ((fd = open(acu, O_RDWR | O_NDELAY)) < 0) {
		fprintf(stderr, "dial: Can't open device: %s\n", acu);
		exit(RC_FAIL | RCE_OPEN | retcode);
	}
	/*
	 * set line for no echo and correct speed.
	 */
	signal(SIGINT, abort);
	errflag = ioctl(fd, TCGETA, &term);
	term.c_cflag &= ~(CBAUD | HUPCL);
	term.c_cflag |= CLOCAL | baudrate | (hflag ? HUPCL : 0);
	term.c_lflag &= ~ECHO;
	term.c_cc[VMIN] = '\1';
	term.c_cc[VTIME] = '\0';
	errflag = ioctl(fd, TCSETA, &term);
	if (errflag) {
		char buf[16];
		DEBUG(1, "dial: ioctl error on %s", acu);
		DEBUG(1, " errno=%d\n", errno);
		cleanup(RC_FAIL | RCE_IOCTL | retcode);
	}
	/*
	 *  Reopen line with clocal so we can talk without carrier present
	 */
	c = fd;
	if ((fd = open(acu, O_RDWR)) < 0) {
		fprintf(stderr, "dial: Can't open device local: %s\n", acu);
		exit(RC_FAIL | RCE_OPEN | retcode);
	}
	close(c);

	/*
	 *  Timeout after 10 seconds if no response
	 */
	timeout = 10;
	signal(SIGALRM, alrmint);

	/*
	 * Hangup and exit if requested
	 */
	if (hflag) {
#ifndef	HDUU
		/* call ungetty to see if we need to switch to dialin */

		if ((pid = fork()) == 0) {
			execl(UNGETTY, "ungetty", "-t", acu, NULL);
			exit(-1);
		}
		while (((c = wait(&errflag)) != pid) && c != -1) ;
		if (((errflag>>8) & 0xff) != UG_RESTART)  cleanup(SUCCESS);
#endif
		/* setup modem for dialin */
		hangup(timeout);
#ifndef	HDUU
		if ((pid = fork()) == 0) {
			execl(UNGETTY, "ungetty", "-r", acu, NULL);
			exit(-1);
		}
		while (((c = wait(&errflag)) != pid) && c != -1) ;
		cleanup( ((errflag>>8) & 0xff) ? RC_FAIL : SUCCESS ) ;
#else
		cleanup(SUCCESS);
#endif
	}


	/*
	 *  Reset and put modem into command mode
	 */
reset:
	if (mdwrite(MDRESET) == -1)
		cleanup(RC_FAIL | retcode);
	if (mdread(timeout) != HELLO) {
		if (hangup(timeout) == -1)
			cleanup(RC_FAIL | retcode);
		goto reset;
	}

	/*
	 *  Build up the phone number and install it in
	 *  mdmsgs to make it a valid modem response.
	 */
	sprintf(command, "%s\r", phone);
	mdmsgs[PHONENUM] = phone;

redial:
	if (mdwrite("D\r") == -1)
		cleanup(RC_FAIL | retcode);
	if (mdread(timeout) != NUMBER)
		goto redial;
	if (mdwrite(command) == -1)
		cleanup(RC_FAIL | retcode);
	if (mdread(timeout) != PHONENUM)
		goto redial;
	if (mdwrite("\r") == -1)
		cleanup(RC_FAIL | retcode);
	if (mdread(timeout) != DIALING)
		goto redial;

	/*
	 *  Set up a timeout for the connect.
	 *    MDPULSDLY seconds more for each pause character
	 *    Vadic only pulse dials.
	 */
	timeout = 6 * strlen(phone) + 15;
	for (p = phone; (p = strchr(p, MDPAUSCHR)) != NULL; p++)
		timeout += MDPAUSDLY;

	DEBUG(6, "wait for connect - timeout %d\n", timeout);
	dialing = 1;

	switch (mdread(timeout)) {
		case TIMEOUT:
			goto reset;
		case HELLO:
		case INVALID:
			if (dial_retry--) goto redial;
			cleanup(RC_FAIL | RCE_NULL | retcode);
		case ONLINE:
			cleanup(retcode);
		case FAILEDCALL:
			cleanup(RC_FAIL | RCE_NOCARR | retcode);
		default:
			cleanup(RC_FAIL | retcode);
	}
}


/*
 *  hangup(htime)
 *
 *  Function:	Should force the modem to drop carrier and hang up
 *		the phone, but the Vadic cannot do this, so we just
 *		try and get out of command mode.  If we can do this
 *		much, the connection is dropped and the phone hung up.
 *		Reads are allowed htime seconds before timeout.
 *
 *  Returns:	0 if disconnect completed.
 *		-1 on failure, phone may still be connected.
 *
 */
hangup(htime)
	int htime;
{
	int c;

	DEBUG(4, "hangup - timeout %d\n", htime);
	if (mdwrite(MDRESET) == -1)
		return(-1);
	if ((c = mdread(htime)) != HELLO && c != INVALID)
		return(-1);
	if (mdwrite(MDHANGUP) == -1)
		return(-1);
	if (mdread(htime) != IDLE)
		return(-1);
	sleep(1);
	return(0);
}


/*
 *  mdread(rtime)
 *
 *  Function:	Reads from the ACU until it finds a valid response (found
 *		in mdmsgs) or times out after rtime seconds.
 *
 *  Returns:	The index in mdmsgs of the modem response found.
 *		-1 on timeout.
 *
 */
mdread(rtime)
	int rtime;
{
	int c;
	char **mp;
	register char *bp;
	char buf[MAXLINE];

	bp = buf;
	alarm(rtime);
	DEBUG(6, "MODEM returned %s", "<<");
	while (read(fd, &c, 1) == 1) {
		c &= 0177;
		if ((*bp = c) != '\0')
			*++bp = '\0';
		DEBUG(6, "%s", vgets(c));
		if (bp >= buf + MAXLINE) {
			alarm(0);
			DEBUG(4,">>-%s\n","FAIL");
			return(-1);
		}
		for (mp = mdmsgs; *mp; ++mp)
			if (substr(*mp, buf) == 0) {
				alarm(0);
				DEBUG(6,">>-%s\n", "OK");
				DEBUG(4,"got %s\n", mdmsgs[mp - mdmsgs]);
				return(mp - mdmsgs);
			}
	}
	alarm(0);
	DEBUG(6,">>-%s","FAIL");
	DEBUG(4, " no response\n", 0);
	return(-1);
}


/*
 *  mdwrite(c)
 *
 *  Function:	Outputs the string pointed to by c to the ACU device.
 *
 *  Returns:	0 on completion.
 *		-1 on write errors.
 *
 */
mdwrite(c)
	register char *c;
{
	int err;
	/*
	 *  Give modem a chance to recover before writing.
	 */
	sleep(1);
	DEBUG(6, "Sent MODEM %s", "<<");
	while (*c) {
		nap(100l);	/* silly Vadic can't handle chars too fast */
		if ((err = write(fd, c, 1)) != 1) {
			char buf[16];
			DEBUG(6, ">>-%s\n", "FAIL");
			if (err == -1)
				DEBUG(1, "ACU write error (errno=%d)\n", errno);
			return(-1);
		}
		DEBUG(6, "%s", vgets(*c));
		c++;
	}
	DEBUG(6, ">>-%s\n", "OK");
	return(0);
}


/* ------------- */


/*  mdflush()
 *
 *  Function:	Flushes input clists for modem
 */
mdflush()
{
	ioctl(fd, TCFLSH, 0) ;
}


/*
 *  substr(s, l)
 *
 *  Function:	Checks for the presence of the string pointed to by s
 *		somewhere within the string pointed to by l.
 *
 *  Returns:	0 if found.
 *		-1 if not found.
 */
substr(s, l)
	char *s;
	register char *l;
{
	int len;

	len = strlen(s);
	while ((l = strchr(l, *s)) != NULL) {
		if (strncmp(s, l, len) == SAME)
			return(0);
		l++;
	}
	return(-1);
}


/*
 *  alrmint()
 *
 *  Function:	Catches alarm calls (signal 14) and exits.
 *
 *  Returns:	No return.  Exits with status RC_FAIL.
 */
alrmint()
{
	DEBUG(4, "\nTimeout waiting for %s\n", dialing ? "carrier" : "acu");
	cleanup(RC_FAIL | RCE_TIMOUT | retcode);
}


/*
 *  cleanup(stat)
 *
 *  Function:	Closes device file and exits.
 *
 *  Returns:	No return.  Exits with status stat.
 */
cleanup(stat)
	int stat;
{
	if (stat & RC_FAIL) {	/* if we failed, drop DTR (in abort) */
		retcode = stat;
		abort(0);
	} else {		/* else, return */
		exit(stat);
	}
}
