/*	Begin SCO_C2TCB		*/
/*
 *	@(#) dlvr_audit.c 1.2 89/05/11 
 *
 *	Copyright (C) The Santa Cruz Operation, 1989.
 *	This Module contains Proprietary Information of
 *	The Santa Cruz Operation, and should be treated as Confidential.
 */
/*
 * Copyright (c) 1988 SecureWare, Inc.  All rights reserved.
 * This is proprietary source code of SecureWare, Inc.
 */

/* #ident "@(#)dlvr_audit.c	2.3 12:42:34 3/4/89 SecureWare, Inc." */
#pragma comment(exestr, "@(#) dlvr_audit.c 1.2 89/05/11 ")

/*
 * This program passes on audit information from a protected subsystem
 * to the audit daemon via a message queue.
 */

#include <stdio.h>
#include <signal.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <sys/stat.h>

#include <sys/security.h>
#include <sys/audit.h>
#include <prot.h>


struct aud_hdr_info  {
	time_t tstamp;
	int event;
	int record;
	int pid;
	char *cmd_name;
	int code;
};


static int verbose = 0;
static char *default_args = "[-v] tstamp event record pid cmd code subsys fd";
static double alignment;
static char audit_buffer[sizeof(struct msgbuf) + sizeof(struct audit_header) +
			 BUFSIZ];

static int proper_subsystem();
static void get_mq();
static void db_integ();
static void fs_integ();
static void db_resrc();
static void sec_fail();
static void subsys();
static void lock_unlock();
static void audit_rec();


extern char *optarg;	/* arg pointer for getopt */
extern int optind;	/* option index for getopt */


extern struct group *getgrgid();


main(argc, argv)
	int argc;
	char *argv[];
{
	register int option;
	int daemon_pid;
	int audit_mq;
	int start_arg;
	struct aud_hdr_info passed_in;
	int error = 0;

	set_auth_parameters(argc, argv);

	while ((option = getopt(argc, argv, "v")) != EOF)
		switch (option) {
			/*
			 * Tell about all actions.
			 */
			case 'v':
				verbose++;
				break;

			case '?':
				error = 1;
				break;
			}

	start_arg = optind + AUTH_AUD_CNST_ARGS;

	if (error || (argc < start_arg)) {
		fprintf(stderr,
			"Usage:  %s %s <special args>\n",
			command_name, default_args);
		exit(1);
	}


	if (!proper_subsystem(argv[optind+6], argv[optind+7]))  {
		if (verbose)
			fprintf(stderr,
		"%s: not of the right subsystem to submit audit records\n",
				command_name);
		exit(1);
	}


	/*
	 * Save the data always put in the audit header.
	 */
	passed_in.tstamp 	= atol(argv[optind]);
	passed_in.event  	= atoi(argv[optind + 1]);
	passed_in.record 	= atoi(argv[optind + 2]);
	passed_in.pid    	= atoi(argv[optind + 3]);
	passed_in.cmd_name	= argv[optind + 4];
	passed_in.code    	= atoi(argv[optind + 5]);

	/*
	 * Get information about the audit message queue.
	 */
	get_mq(&daemon_pid, &audit_mq);

	/*
	 * Determine the record type to use and call the appropriate routine.
	 */
	switch (passed_in.event)  {
	  case ET_DATABASE:
	    switch (passed_in.code)  {
		case ES_DB_INTEGRITY:
			if (argc != start_arg + 3)  {
				fprintf(stderr,
			"Usage:  %s %s db_code ent_name prob\n",
					command_name, default_args);
				exit(1);
			}
			db_integ(audit_mq, &passed_in, atoi(argv[start_arg]),
				argv[start_arg+1], argv[start_arg+2]);
			break;
		case ES_FS_INTEGRITY:
			if (argc != start_arg + 2)  {
				fprintf(stderr,
			"Usage:  %s %s file prob\n",
					command_name, default_args);
				exit(1);
			}
			fs_integ(audit_mq, &passed_in,
				argv[start_arg], argv[start_arg+1]);
			break;
		case ES_DB_RESOURCE:
			if (argc != start_arg + 3)  {
				fprintf(stderr,
			"Usage:  %s %s object name prob\n",
					command_name, default_args);
				exit(1);
			}
			db_resrc(audit_mq, &passed_in, atoi(argv[start_arg]),
				argv[start_arg+1], argv[start_arg+2],
				argv[start_arg+3]);
			break;
		case ES_SEC_FAILURE:
			if (argc != start_arg + 5)  {
				fprintf(stderr,
			"Usage:  %s %s object exp curr action result\n",
					command_name, default_args);
				exit(1);
			}
			sec_fail(audit_mq, &passed_in, atoi(argv[start_arg]),
				atol(argv[start_arg+1]),
				atol(argv[start_arg+2]),
				argv[start_arg+3], argv[start_arg+4]);
			break;
		default:
			fprintf(stderr, "%s: unknown record type\n",
				command_name);
			exit(1);
	    };
	    break;

	  case ET_SUBSYSTEM:
		if (argc != start_arg + 2)  {
			fprintf(stderr, "Usage:  %s %s action result\n",
				command_name, default_args);
			exit(1);
		}
		subsys(audit_mq, &passed_in,
			argv[start_arg], argv[start_arg+1]);
		break;

	  case ET_SYS_ADMIN:
		switch (passed_in.record) {
		case RT_LOCK:
			if (argc != start_arg + 1) {
				fprintf (stderr, "Usage:  %s %s user/term\n",
				 command_name, default_args);
				exit(1);
			}
			lock_unlock(audit_mq, &passed_in, argv[start_arg]);
			break;
		case RT_SUBSYSTEM:
			if (argc != start_arg + 2)  {
				fprintf(stderr, "Usage:  %s %s action result\n",
					command_name, default_args);
				exit(1);
			}
			subsys(audit_mq, &passed_in,
				argv[start_arg], argv[start_arg+1]);
			break;
		case RT_AUDIT:
			if (argc != start_arg + 1) {
				fprintf (stderr, "Usage: %s %s text\n",
				 command_name, default_args);
				exit(1);
			}
			audit_rec(audit_mq, &passed_in, argv[start_arg]);
			break;
		case RT_DATABASE:
			switch (passed_in.code) {
			case	ES_DB_INTEGRITY:
				if (argc != start_arg + 3)  {
					fprintf(stderr,
			"Usage:  %s %s object name prob\n",
						command_name, default_args);
					exit(1);
				}
				db_resrc(audit_mq, &passed_in,
				  atoi(argv[start_arg]),
				  argv[start_arg+1], argv[start_arg+2],
				  argv[start_arg+3]);
				break;
			case	ES_SEC_FAILURE:
				if (argc != start_arg + 5)  {
					fprintf(stderr,
			"Usage:  %s %s object exp curr action result\n",
					  command_name, default_args);
					exit(1);
				}
				sec_fail(audit_mq, &passed_in,
				  atoi(argv[start_arg]),
				  atol(argv[start_arg+1]),
				  atol(argv[start_arg+2]),
				  argv[start_arg+3], argv[start_arg+4]);
				break;
			}
		}
		break;
	  default:
		fprintf(stderr, "%s: unknown event type\n",
			command_name);
		exit(1);
	}

	/*
	 * It is important that all the routines that perform a msgsnd()
	 * actually check the return value.  If the operation failed, this
	 * process must fail then.  By the time we get here, a message
	 * has been queued up to the audit daemon, so we can notify it.
	 * If we notify the daemon when we really don't have anything, the
	 * daemon will hang for an unknown period on the msgrcv().  That's
	 * why, when we get here, all past actions in this program must
	 * be successful.
	 *
	 *
	 * If we cannot signal the audit daemon, it just means that the
	 * daemon may not get to our record for now, but it is bound to when
	 * it gets the next signal or it is cleaning up before dying.
	 */
	if (kill(daemon_pid, SIGUSR1) != 0)
		if (verbose)
			fprintf(stderr, "Could not signal audit daemon\n");
}


/*
 * This routine checks permissions to submit audit records.
 * The invoking subsystem opens a file only accessible to that subsystem.
 * This program verifies that the file opened is the one corresponding
 * to the subsystem, and treats the file descriptor passed as a "ticket"
 * allowing that subsystem to audit.  Without a ticket, no audit is possible.
 */

static char *subsys_filename = AUTH_SUBSYS_TICKET;

static int
proper_subsystem(subsys_name, fd_string)
char *subsys_name;
char *fd_string;
{
	struct stat file_sb;
	struct stat fd_sb;

	if (starting_ruid() == 0)
		return 1;

	strcpy(strrchr(subsys_filename, '/')+1, subsys_name);
	if (stat(subsys_filename, &file_sb) < 0)
		return(0);
	if (fstat(atoi(fd_string), &fd_sb) < 0)
		return(0);
	return(file_sb.st_dev == fd_sb.st_dev &&
		file_sb.st_ino == fd_sb.st_ino);
}


/*
 * This routines reads the AUDIT_DMNINFO file to get the PID of the
 * audit daemon and the message queue ID it is using.  It then
 * returns the descriptor and PID of the audit daemon.
 */
static void
get_mq(daemon_pid, queue_fd)
	register int *daemon_pid;
	register int *queue_fd;
{
	register FILE *info_file;
	struct dmninfo audit_queue_info;

	info_file = fopen(AUDIT_DMNINFO, "r");
	if (info_file == (FILE *) 0)  {
		perror(AUDIT_DMNINFO);
		exit(1);
	}

	if (fread((char *) &audit_queue_info, sizeof(audit_queue_info),
		  1, info_file) != 1)  {
		if (verbose)
			fprintf(stderr, "Format problem with %s\n",
				AUDIT_DMNINFO);
		exit(1);
	}
	(void) fclose(info_file);

	*daemon_pid = audit_queue_info.pid;
	*queue_fd = audit_queue_info.msgqid;
}


/*
 * Format and send the message denoting a problem with the Authentication
 * database integrity.
 */
static void
db_integ(mqid, std_data, db_code, ent, purpose)
	int mqid;
	struct aud_hdr_info *std_data;
	int db_code;
	char *ent;
	char *purpose;
{
	register struct database_activity *record;
	register char *record_trailer;
	register int tot_size;
	register struct msgbuf *msgp;
	int ent_size;
	int cmd_size;
	int purpose_size;
	int calloced = 0;


	if (std_data->cmd_name == (char *) 0)
		cmd_size = 0;
	else
		cmd_size = strlen(std_data->cmd_name) + 1;

	if (ent == (char *) 0)
		ent_size = 0;
	else
		ent_size = strlen(ent) + 1;

	if (purpose == (char *) 0)
		purpose_size = 0;
	else
		purpose_size = strlen(purpose) + 1;

	/*
	 * If the audit record we have is bigger than the static
	 * space, first try to calloc() the space, and then
	 * start throwing away  variable fields.
	 */
	tot_size = sizeof(msgp->mtype) + sizeof(struct database_activity) +
		ent_size + cmd_size + purpose_size;
	if (tot_size > sizeof(audit_buffer))  {
		msgp = (struct msgbuf *) calloc(1, tot_size);
		if (msgp == (struct msgbuf *) 0)  {
			msgp = (struct msgbuf *) audit_buffer;
			tot_size -= purpose_size;
			purpose_size = 0;
			if (tot_size > sizeof(audit_buffer))  {
				tot_size -= cmd_size;
				cmd_size = 0;
				if (tot_size > sizeof(audit_buffer))  {
					tot_size -= ent_size;
					ent_size = 0;
				}
			}
		}
		else
			calloced = 1;
	}
	else
		msgp = (struct msgbuf *) audit_buffer;

	msgp->mtype = 1;

	record = (struct database_activity *) msgp->mtext;

	/*
	 * Fill in fixed login audit record.
	 */
	record->aud_hdr.rec_length = tot_size;
	record->aud_hdr.tstamp = std_data->tstamp;
	record->aud_hdr.event_type = std_data->event;
	record->aud_hdr.record_type = std_data->record;
	record->aud_hdr.pid = std_data->pid;
	record->command.count = cmd_size;
	record->code = std_data->code;
	record->object = (ushort) db_code;
	record->expected_val = 0L;
	record->present_val = 0L;
	record->action.count = ent_size;
	record->result.count = purpose_size;

	/*
	 * Start putting in the variable length strings
	 * immediately after the audit record, so we
	 * can do this in one write operation.
	 */
	record_trailer = (char *) (record+1);
	if (std_data->cmd_name != (char *) 0)
		(void) strncpy(record_trailer, std_data->cmd_name, cmd_size);
	record_trailer += cmd_size;
	if (ent != (char *) 0)
		(void) strncpy(record_trailer, ent, ent_size);
	record_trailer += ent_size;
	if (purpose != (char *) 0)
		(void) strncpy(record_trailer, purpose, purpose_size);

	/*
	 * Not much we can do if auditing fails.  However, so we don't
	 * accidentally send a signal to the audit daemon, we die here
	 * if the send fails.
	 */
	if (msgsnd(mqid, msgp, tot_size, 0) < 0)  {
		if (verbose)
			perror("msgsnd");
		exit(1);
	}

	if (calloced)
		free((char *) msgp);
}


/*
 * Format and send the message denoting a problem with the file
 * system integrity.  This could be a missing file, bad permissions,
 * or ownership problems.
 */
static void
fs_integ(mqid, std_data, file, prob)
	int mqid;
	struct aud_hdr_info *std_data;
	char *file;
	char *prob;
{
	register struct database_activity *record;
	register char *record_trailer;
	register int tot_size;
	register struct msgbuf *msgp;
	int cmd_size;
	int file_size;
	int prob_size;
	int calloced = 0;


	if (std_data->cmd_name == (char *) 0)
		cmd_size = 0;
	else
		cmd_size = strlen(std_data->cmd_name) + 1;

	if (file == (char *) 0)
		file_size = 0;
	else
		file_size = strlen(file) + 1;

	if (prob == (char *) 0)
		prob_size = 0;
	else
		prob_size = strlen(prob) + 1;

	/*
	 * If the audit record we have is bigger than the static
	 * space, first try to calloc() the space, and then
	 * start throwing away  variable fields.
	 */
	tot_size = sizeof(msgp->mtype) + sizeof(struct database_activity) +
		cmd_size + file_size + prob_size;
	if (tot_size > sizeof(audit_buffer))  {
		msgp = (struct msgbuf *) calloc(1, tot_size);
		if (msgp == (struct msgbuf *) 0)  {
			msgp = (struct msgbuf *) audit_buffer;
			tot_size -= prob_size;
			prob_size = 0;
			if (tot_size > sizeof(audit_buffer))  {
				tot_size -= file_size;
				file_size = 0;
				if (tot_size > sizeof(audit_buffer))  {
					tot_size -= cmd_size;
					cmd_size = 0;
				}
			}
		}
		else
			calloced = 1;
	}
	else
		msgp = (struct msgbuf *) audit_buffer;

	msgp->mtype = 1;

	record = (struct database_activity *) msgp->mtext;

	/*
	 * Fill in fixed login audit record.
	 */
	record->aud_hdr.rec_length = tot_size;
	record->aud_hdr.tstamp = std_data->tstamp;
	record->aud_hdr.event_type = std_data->event;
	record->aud_hdr.record_type = std_data->record;
	record->aud_hdr.pid = std_data->pid;
	record->command.count = cmd_size;
	record->code = std_data->code;
	record->expected_val = 0L;
	record->present_val = 0L;
	record->action.count = file_size;
	record->result.count = prob_size;

	/*
	 * Start putting in the variable length strings
	 * immediately after the audit record, so we
	 * can do this in one write operation.
	 */
	record_trailer = (char *) (record+1);
	if (std_data->cmd_name != (char *) 0)
		(void) strncpy(record_trailer, std_data->cmd_name, cmd_size);
	record_trailer += cmd_size;
	if (file != (char *) 0)
		(void) strncpy(record_trailer, file, file_size);
	record_trailer += file_size;
	if (prob != (char *) 0)
		(void) strncpy(record_trailer, prob, prob_size);

	/*
	 * Not much we can do if auditing fails.  However, so we don't
	 * accidentally send a signal to the audit daemon, we die here
	 * if the send fails.
	 */
	if (msgsnd(mqid, msgp, tot_size, 0) < 0)  {
		if (verbose)
			perror("msgsnd");
		exit(1);
	}

	if (calloced)
		free((char *) msgp);
}


/*
 * Format and send the message denoting a problem with a resource
 * allocation during a database operation.
 */
static void
db_resrc(mqid, std_data, object, name, prob)
	int mqid;
	struct aud_hdr_info *std_data;
	int object;
	char *name;
	char *prob;
{
	register struct database_activity *record;
	register char *record_trailer;
	register int tot_size;
	register struct msgbuf *msgp;
	int cmd_size;
	int name_size;
	int prob_size;
	int calloced = 0;


	if (std_data->cmd_name == (char *) 0)
		cmd_size = 0;
	else
		cmd_size = strlen(std_data->cmd_name) + 1;

	if (name == (char *) 0)
		name_size = 0;
	else
		name_size = strlen(name) + 1;

	if (prob == (char *) 0)
		prob_size = 0;
	else
		prob_size = strlen(prob) + 1;

	/*
	 * If the audit record we have is bigger than the static
	 * space, first try to calloc() the space, and then
	 * start throwing away  variable fields.
	 */
	tot_size = sizeof(msgp->mtype) + sizeof(struct database_activity) +
		cmd_size + name_size + prob_size;
	if (tot_size > sizeof(audit_buffer))  {
		msgp = (struct msgbuf *) calloc(1, tot_size);
		if (msgp == (struct msgbuf *) 0)  {
			msgp = (struct msgbuf *) audit_buffer;
			tot_size -= prob_size;
			prob_size = 0;
			if (tot_size > sizeof(audit_buffer))  {
				tot_size -= name_size;
				name_size = 0;
				if (tot_size > sizeof(audit_buffer))  {
					tot_size -= cmd_size;
					cmd_size = 0;
				}
			}
		}
		else
			calloced = 1;
	}
	else
		msgp = (struct msgbuf *) audit_buffer;

	msgp->mtype = 1;

	record = (struct database_activity *) msgp->mtext;

	/*
	 * Fill in fixed login audit record.
	 */
	record->aud_hdr.rec_length = tot_size;
	record->aud_hdr.tstamp = std_data->tstamp;
	record->aud_hdr.event_type = std_data->event;
	record->aud_hdr.record_type = std_data->record;
	record->aud_hdr.pid = std_data->pid;
	record->command.count = cmd_size;
	record->code = std_data->code;
	record->object = (ushort) object;
	record->expected_val = 0L;
	record->present_val = 0L;
	record->action.count = name_size;
	record->result.count = prob_size;

	/*
	 * Start putting in the variable length strings
	 * immediately after the audit record, so we
	 * can do this in one write operation.
	 */
	record_trailer = (char *) (record+1);
	if (std_data->cmd_name != (char *) 0)
		(void) strncpy(record_trailer, std_data->cmd_name, cmd_size);
	record_trailer += cmd_size;
	if (name != (char *) 0)
		(void) strncpy(record_trailer, name, name_size);
	record_trailer += name_size;
	if (prob != (char *) 0)
		(void) strncpy(record_trailer, prob, prob_size);

	/*
	 * Not much we can do if auditing fails.  However, so we don't
	 * accidentally send a signal to the audit daemon, we die here
	 * if the send fails.
	 */
	if (msgsnd(mqid, msgp, tot_size, 0) < 0)  {
		if (verbose)
			perror("msgsnd");
		exit(1);
	}

	if (calloced)
		free((char *) msgp);
}


/*
 * Format and send the message denoting the failure of an expected event
 * during a security operation.
 */
static void
sec_fail(mqid, std_data, object, exp, curr, action, result)
	int mqid;
	struct aud_hdr_info *std_data;
	int object;
	long exp;
	long curr;
	char *action;
	char *result;
{
	register struct database_activity *record;
	register char *record_trailer;
	register int tot_size;
	register struct msgbuf *msgp;
	int cmd_size;
	int action_size;
	int result_size;
	int calloced = 0;


	if (std_data->cmd_name == (char *) 0)
		cmd_size = 0;
	else
		cmd_size = strlen(std_data->cmd_name) + 1;

	if (action == (char *) 0)
		action_size = 0;
	else
		action_size = strlen(action) + 1;

	if (result == (char *) 0)
		result_size = 0;
	else
		result_size = strlen(result) + 1;

	/*
	 * If the audit record we have is bigger than the static
	 * space, first try to calloc() the space, and then
	 * start throwing away  variable fields.
	 */
	tot_size = sizeof(msgp->mtype) + sizeof(struct database_activity) +
		cmd_size + action_size + result_size;
	if (tot_size > sizeof(audit_buffer))  {
		msgp = (struct msgbuf *) calloc(1, tot_size);
		if (msgp == (struct msgbuf *) 0)  {
			msgp = (struct msgbuf *) audit_buffer;
			tot_size -= result_size;
			result_size = 0;
			if (tot_size > sizeof(audit_buffer))  {
				tot_size -= action_size;
				action_size = 0;
				if (tot_size > sizeof(audit_buffer))  {
					tot_size -= cmd_size;
					cmd_size = 0;
				}
			}
		}
		else
			calloced = 1;
	}
	else
		msgp = (struct msgbuf *) audit_buffer;

	msgp->mtype = 1;

	record = (struct database_activity *) msgp->mtext;

	/*
	 * Fill in fixed login audit record.
	 */
	record->aud_hdr.rec_length = tot_size;
	record->aud_hdr.tstamp = std_data->tstamp;
	record->aud_hdr.event_type = std_data->event;
	record->aud_hdr.record_type = std_data->record;
	record->aud_hdr.pid = std_data->pid;
	record->command.count = cmd_size;
	record->code = std_data->code;
	record->object = (ushort) object;
	record->expected_val = exp;
	record->present_val = curr;
	record->action.count = action_size;
	record->result.count = result_size;

	/*
	 * Start putting in the variable length strings
	 * immediately after the audit record, so we
	 * can do this in one write operation.
	 */
	record_trailer = (char *) (record+1);
	if (std_data->cmd_name != (char *) 0)
		(void) strncpy(record_trailer, std_data->cmd_name, cmd_size);
	record_trailer += cmd_size;
	if (action != (char *) 0)
		(void) strncpy(record_trailer, action, action_size);
	record_trailer += action_size;
	if (result != (char *) 0)
		(void) strncpy(record_trailer, result, result_size);

	/*
	 * Not much we can do if auditing fails.  However, so we don't
	 * accidentally send a signal to the audit daemon, we die here
	 * if the send fails.
	 */
	if (msgsnd(mqid, msgp, tot_size, 0) < 0)  {
		if (verbose)
			perror("msgsnd");
		exit(1);
	}

	if (calloced)
		free((char *) msgp);
}


/*
 * Format and send the message denoting a problem with the file
 * system integrity.  This could be a missing file, bad permissions,
 * or ownership problems.
 */
static void
subsys(mqid, std_data, action, result)
	int mqid;
	struct aud_hdr_info *std_data;
	char *action;
	char *result;
{
	register struct subsystem_activity *record;
	register char *record_trailer;
	register int tot_size;
	register struct msgbuf *msgp;
	int cmd_size;
	int act_size;
	int res_size;
	int calloced = 0;


	if (std_data->cmd_name == (char *) 0)
		cmd_size = 0;
	else
		cmd_size = strlen(std_data->cmd_name) + 1;

	if (action == (char *) 0)
		act_size = 0;
	else
		act_size = strlen(action) + 1;

	if (result == (char *) 0)
		res_size = 0;
	else
		res_size = strlen(result) + 1;

	/*
	 * If the audit record we have is bigger than the static
	 * space, first try to calloc() the space, and then
	 * start throwing away  variable fields.
	 */
	tot_size = sizeof(msgp->mtype) + sizeof(struct subsystem_activity) +
		cmd_size + act_size + res_size;
	if (tot_size > sizeof(audit_buffer))  {
		msgp = (struct msgbuf *) calloc(1, tot_size);
		if (msgp == (struct msgbuf *) 0)  {
			msgp = (struct msgbuf *) audit_buffer;
			tot_size -= res_size;
			res_size = 0;
			if (tot_size > sizeof(audit_buffer))  {
				tot_size -= act_size;
				act_size = 0;
				if (tot_size > sizeof(audit_buffer))  {
					tot_size -= cmd_size;
					cmd_size = 0;
				}
			}
		}
		else
			calloced = 1;
	}
	else
		msgp = (struct msgbuf *) audit_buffer;

	msgp->mtype = 1;

	record = (struct subsystem_activity *) msgp->mtext;

	/*
	 * Fill in fixed login audit record.
	 */
	record->aud_hdr.rec_length = tot_size;
	record->aud_hdr.tstamp = std_data->tstamp;
	record->aud_hdr.event_type = std_data->event;
	record->aud_hdr.record_type = std_data->record;
	record->aud_hdr.pid = std_data->pid;
	record->command.count = cmd_size;
	record->code = std_data->code;
	record->action.count = act_size;
	record->result.count = res_size;

	/*
	 * Start putting in the variable length strings
	 * immediately after the audit record, so we
	 * can do this in one write operation.
	 */
	record_trailer = (char *) (record+1);
	if (std_data->cmd_name != (char *) 0)
		(void) strncpy(record_trailer, std_data->cmd_name, cmd_size);
	record_trailer += cmd_size;
	if (action != (char *) 0)
		(void) strncpy(record_trailer, action, act_size);
	record_trailer += act_size;
	if (result != (char *) 0)
		(void) strncpy(record_trailer, result, res_size);

	/*
	 * Not much we can do if auditing fails.  However, so we don't
	 * accidentally send a signal to the audit daemon, we die here
	 * if the send fails.
	 */
	if (msgsnd(mqid, msgp, tot_size, 0) < 0)  {
		if (verbose)
			perror("msgsnd");
		exit(1);
	}

	if (calloced)
		free((char *) msgp);
}

/* lock or unlock user or terminal from a system administrator action */
static
void
lock_unlock(mqid, std_data, user_term)
	int	mqid;
	struct aud_hdr_info	*std_data;
	char	*user_term;
{
	register struct msgbuf *msgp;
	register struct lock_audit *record;

	msgp = (struct msgbuf *) audit_buffer;
	msgp->mtype = 1;
	record = (struct lock_audit *) msgp->mtext;
	record->aud_hdr.rec_length = sizeof (struct audit_actions);
	record->aud_hdr.tstamp = std_data->tstamp;
	record->aud_hdr.event_type = std_data->event;
	record->aud_hdr.record_type = std_data->record;
	record->aud_hdr.pid = std_data->pid;
	record->code = std_data->code;
	record->trys = 0;
	strncpy(record->username, user_term, sizeof(record->username));

	if (msgsnd(mqid, msgp, sizeof(*record) + sizeof (msgp->mtype), 0) < 0)
		exit(1);
}

/* audit an action done on behalf of the audit subsystem.  */
static
void
audit_rec(mqid, std_data, action)
	int mqid;
	struct aud_hdr_info *std_data;
	char *action;
{
	register struct msgbuf *msgp;
	int action_size;
	int tot_size;
	register struct audit_actions *record;
	int calloced = 0;
	char *record_trailer;

	if (action == (char *) 0)
		action_size = 0;
	else	action_size = strlen(action) + 1;
	tot_size = sizeof(msgp->mtype) + sizeof(struct audit_actions) +
	  action_size;
	if (tot_size > sizeof(audit_buffer)) {
		msgp = (struct msgbuf *) calloc (1, tot_size);
		if (msgp == (struct msgbuf *) 0) {
			msgp = (struct msgbuf *) audit_buffer;
			action_size = 0;
			tot_size -= action_size;
		}
		else
			calloced = 1;
	} else
		msgp = (struct msgbuf *) audit_buffer;
	msgp->mtype = 1;
	record = (struct audit_actions *) msgp->mtext;
	record->aud_hdr.rec_length = tot_size;
	record->aud_hdr.tstamp = std_data->tstamp;
	record->aud_hdr.event_type = std_data->event;
	record->aud_hdr.record_type = std_data->record;
	record->aud_hdr.pid = std_data->pid;
	record->code = std_data->code;
	record->text1.count = action_size;
	record_trailer = (char *) (record + 1);
	if (action_size > 0)
		strncpy (record_trailer, action, action_size);
	if (msgsnd(mqid, msgp, tot_size, 0) < 0)
		exit(1);
	if (calloced)
		free ((char *) msgp);
}
